;;
;; Core Wars system in Scheme this time (last time was in 6502
;; assembly and it never did work quite right...)
;;
;; Author: Sam Shen (sls@soda.berkeley.edu)
;; Date:   4/15/89 plus bugs fixes and hacks until 4/91.
;;

(module mars (with util core))
(include "cw.sch")

(define *n-instr-executed* #f)

;;
;; the redcode interpreter (mars)
;;
;; programs is a list of programs.  Runs the programs until only
;; one remains or more than limit steps is reached.
;;
(define (mars programs core limit)
  
  ; main loop
  (define (loop count the-remaining-programs)
    (cond ((>= count limit)
	   (newline)
	   (display "Time limit has been reached.  A tie between ")
	   (map (lambda (p) (display (program-name p)) (display " "))
		the-remaining-programs)
	   (newline)
	   (display "The others have lost.")
	   (newline))
	  ((null? the-remaining-programs)
	   (newline)
	   (display "All programs have died.  A tie.")
	   (newline))
	  ((null? (cdr the-remaining-programs))
	   (newline)
	   (display (program-name (car the-remaining-programs)))
	   (display " is the sole survivor.")
	   (newline))
	  (else
	   (loop
	    (inc count)
	    (remove-falses
	     (map (lambda (p) (execute-once p core count))
	      the-remaining-programs))))))
  
  ; main body of mars

  (set! *n-instr-executed* 0)
  (let load-loop ((pl programs))
	(if pl
	  (begin (put-program-in-core (car pl) core)
			 (load-loop (cdr pl)))
      (begin (display "Loaded programs.")
	     (newline)
	     (display-init programs)
	     (loop 0 programs)))))

(define (remove-falses lst)
  (let loop ((result #f)
	     (l lst))
    (if l
	(loop (if (car l)
		  (if (not result)
		      (let ((x (cons (car l) '())))
		        (cons x x))
		      (begin (set-cdr! (cdr result) (cons (car l) '()))
			     result))
		  result)
	      (cdr l))
	(car result))))

(define-external (lookup-instruction-by-value v) instr)
(define-external (print-execution c p i ma oa ea mb ob eb) print)

;;
;; execute one instruction of the given program
;;
(define (execute-once p core count)
  (set! *n-instr-executed* (inc *n-instr-executed*))
  (let* ((pc           (current-pc p))
	 (current-word (vector-ref core pc))
	 (i-object     (lookup-instruction-by-value
			(current-word 'opcode)))
	 (proc         (i-object 'proc))
	 (n-ops        (i-object 'n-ops))
	 (op-a         (current-word 'operand-a))
	 (op-b         (current-word 'operand-b))
	 (mode-a       (current-word 'mode-a))
	 (mode-b       (current-word 'mode-b))
	 (effective-a  (if (> n-ops 0)
			   (evaluate-operand mode-a op-a pc core)
			   'nothing))
	 (effective-b  (if (> n-ops 1)
			   (evaluate-operand mode-b op-b pc core)
			   'nothing)))
    (print-execution count p i-object
		     mode-a op-a effective-a
		     mode-b op-b effective-b)
    (if (or (not proc) (not effective-a) (not effective-b))
	(begin
	  (format #t " ~s[~s]@~s died.~%" (program-name p)
					  (current-pc-number p)
					  (current-pc p))
	  (if (not (abort-current-program! p))
	      (begin
		(format #t " ~s out of competition.~%"
			(program-name p))
		#f)
	      p))
	(begin
	  (proc effective-a effective-b p)
	  (inc-current-pc! p)
	  (next-pc! p)
	  p))))

;;
;; Evaluate an operand.  Return the effective operand unless it's
;; illegal in which case return #f
;;
(define (evaluate-operand mode operand pc core)
  (case mode
	((0)
	 ;; relative mode
	 (vector-ref core (scale-to-core (+ pc operand))))
	((1)
	 ;; indirect mode.  Use operand-a because dat's where da "dat"
	 ;; instruction stores its value.  The address is relative to
	 ;; the word.
	 (let ((word (vector-ref core
				 (scale-to-core (+ pc operand)))))
	   (vector-ref core (scale-to-core
			     (+ (word 'operand-a)
				(word 'address))))))
	((2)
	 ;; immediate mode.  This is a bit tricky.  Return a
	 ;; procedure that emulates a core word only for reads.
	 (lambda (m)
	   (cond ((eq? m 'operand-a) operand)
		 ((eq? m 'read)
		  (vector 0 2 operand 0 0))
		 (else
		  (error
		   "Attempt to do something other than read an immediate operand."
		   "The operand was created at pc = " pc
		   "The operand itself was " operand
		   "What you tried to do was " m)))))
	(else
	 (display "Unsuported addressing mode (mode and then pc):"
		  mode pc)
	 #f)))

;;
;; helper procedure for mars
;;
(define (put-program-in-core p core)
  (define (iter addr list-of-code)
    (cond ((null? list-of-code) #f)
	  (else
	   (((vector-ref core addr) 'write) (car list-of-code))
	   (iter (scale-to-core (inc addr)) (cdr list-of-code)))))
  (let ((loc (random **core-size**)))
    (display "Putting program ")
    (display (program-name p))
    (display " at address ")
    (display loc) (newline)
    ;; adjust the program's starting location to loc + current-pc
    (set-current-pc! p (scale-to-core (+ loc (current-pc p))))
    ;; and copy it into core
    (iter loc (program-code p))
    #t))
