;;
;; implementation of instructions
;;
;; non-executable instructions have null procedures
;;

(module instr)
(include "cw.sch")

;;
;;  The code for the actual instructions.
;;
;;  an instruction is a procedure of 3 arguments: operand a,
;;  operand b, and the program.
;;
;;  The operands are the effective core word locations, unless they
;;  are immediate when they are "fake" core words which give errors
;;  for writes.
;;
(define (mov a b p)
  ((b 'write) (a 'read)))

(define (add a b p)
  ((b 'write) (vector 0 2
		      (scale-to-core
		       (+ (a 'operand-a)
			  (b 'operand-a)))
		      0 0)))

(define (sub a b p)
  ((b 'write) (vector 0 2
		      (scale-to-core
		       (- (b 'operand-a)
			  (a 'operand-a)))
		      0 0)))

;;
;; jump to -1+ the address of operand a because we're going to
;; increment the pc in the mars main loop.
;;
(define (jmp a b p)
  (set-current-pc! p (scale-to-core (dec (a 'address)))))

(define (jmpz a b p)
  (if (zero? (a 'operand-a))
      (set-current-pc! p (scale-to-core (dec (b 'address))))))

(define (split a b p)
  (add-pc! p (a 'address)))

(define (djz a b p)
  ;; we need to save op-b in case decrementing op-a changes op-b
  (let ((place-to-jump-to (b 'address)))
    ((a 'write) (vector 0 2
			(scale-to-core (dec (a 'operand-a)))
			0 0))
    (if (zero? (a 'operand-a))
	(set-current-pc! p (scale-to-core (dec place-to-jump-to))))))

(define (djnz a b p)
  ;; we need to save op-b in case decrementing op-a changes op-b
  (let ((place-to-jump-to (b 'address)))
    ((a 'write) (vector 0 2
			(scale-to-core (dec (a 'operand-a)))
			0 0))
    (if (not (zero? (a 'operand-a)))
	(set-current-pc! p (scale-to-core (dec place-to-jump-to))))))

(define (cmp a b p)
  (if (not (equal? (a 'read) (b 'read)))
      (inc-current-pc! p)))

(define (prot a b p)
  (a 'protect))

(define (the-nothing-instruction word d p)
  #f)

;;
;; This makes an instruction object.
;;
(define (make-instruction name value n-ops proc)
  (lambda (m)
    (case m
	  ((name) name)
	  ((value) value)
	  ((n-ops) n-ops)
	  ((proc) proc)
	  ((print) (display name)))))

;;
;; Store the instruction objects in two places, one keyed on name
;; and the other keyed on value.
;;
(define *instruction-table-by-name* (make-table))
(define *instruction-vector* (make-vector 16 #f))
(define *max-opcode* 15)

(define (add-instruction! i)
  (insert! (i 'name)  i *instruction-table-by-name*)
  (vector-set! *instruction-vector* (i 'value) i))

;;
;; Notice that I have more than one instruction per opcode.  This is
;; mostly so that you can have different names for the same instruction
;; for the assembler.
;;
(add-instruction! (make-instruction 'halt  0 0 #f))
(add-instruction! (make-instruction 'data  0 1 #f))
(add-instruction! (make-instruction 'dat   0 1 #f))
(add-instruction! (make-instruction 'spl   1 1 split))
(add-instruction! (make-instruction 'split 1 1 split))
(add-instruction! (make-instruction 'move  2 2 mov))
(add-instruction! (make-instruction 'mov   2 2 mov))
(add-instruction! (make-instruction 'add   3 2 add))
(add-instruction! (make-instruction 'sub   4 2 sub))
(add-instruction! (make-instruction 'jump  5 1 jmp))
(add-instruction! (make-instruction 'jmp   5 1 jmp))
(add-instruction! (make-instruction 'pct   6 1 prot))
(add-instruction! (make-instruction 'prot  6 1 prot))
(add-instruction! (make-instruction 'djz   7 2 djz))
(add-instruction! (make-instruction 'djnz  8 2 djnz))
(add-instruction! (make-instruction 'nothing 9 0 the-nothing-instruction))
(add-instruction! (make-instruction 'nop   9 0 the-nothing-instruction))
(add-instruction! (make-instruction 'cmp  10 2 cmp))

(define (lookup-instruction-by-value v)
  (let ((i
	 (if (> v *max-opcode* 15)
	     #f
	     (vector-ref *instruction-vector* v))))
    (if i
	i
	(error "Unknown instruction, opcode:" v))))

(define (lookup-instruction-by-name n)
  (let ((i (lookup n *instruction-table-by-name*)))
    (if i
	i
	(error "Uknown instruction:" n))))

(define (one-op-instruction? instr)
  (= 1  (instr 'n-ops)))

(define (zero-op-instruction? i)
  (zero? (i 'n-ops)))

(define (opcode i)
  (i 'value))
