'From Smalltalk-80, Version 2.2 of July 4, 1987 on 16 November 1988 at 5:28:08 pm'!


OrderedCollection variableSubclass: #ChangeTree
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'System-Changes'!


!ChangeTree methodsFor: 'accessing'!

addLeaf: aValue type: typeSymbol
	self addLast: (ChangeTreeLeaf value: aValue type: typeSymbol)!

addNode: aValue type: typeSymbol sons: aChangeTree
	self addLast: (ChangeTreeNode value: aValue type: typeSymbol sons: aChangeTree)! !

!ChangeTree methodsFor: 'printing'!

printOn: aStream
	self printOn: aStream level: 0!

printOn: aStream level: level
	self do: [:each | each printOn: aStream level: level]! !

!ChangeTree methodsFor: 'fileIn/Out'!

fileOut
	self fileOutInClass: nil!

fileOutInClass: aClass
	self do: [:node | node fileOutInClass: aClass]! !

!ChangeTree methodsFor: 'archiving'!

archiveChanges
	| tree |
	tree _ ChangeTree buildFrom: SystemOrganization.
	self prune: tree.
	tree prune: self.
	tree fileOut.
	self graft: tree.
	Smalltalk changes initialize!

graft: aChangeTree
	| match | 
	aChangeTree do: [:node |
		match _ self matchThisLevel: node.
		match == nil
			ifTrue: [self addLast: node]
			ifFalse: [match graft: node]]!

matchThisLevel: matchNode
	^ self detect: [:node | node value == matchNode value] ifNone: [nil]!

prune: aChangeTree
	| match | 
	aChangeTree do: [:node |
		match _ self matchThisLevel: node.
		(match ~~ nil and: [node isLeaf not]) ifTrue:
			[match prune: node]]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ChangeTree class
	instanceVariableNames: ''!


!ChangeTree class methodsFor: 'instance creation'!

buildFrom: aSystemOrganizer
	^ aSystemOrganizer changeTree! !


Object variableSubclass: #ChangeTreeLeaf
	instanceVariableNames: 'type value '
	classVariableNames: 'FileOutMessages '
	poolDictionaries: ''
	category: 'System-Changes'!


!ChangeTreeLeaf methodsFor: 'accessing'!

isLeaf
	^ true!

type
	^ type!

value
	^ value!

value: aValue type: typeSymbol
	value _ aValue.
	type _ typeSymbol! !

!ChangeTreeLeaf methodsFor: 'printing'!

printOn: aStream
	self printOn: aStream level: 0!

printOn: aStream level: level
	(type == #classChanges)
		ifTrue:
			[value do: [:change |
				change == #rename
					ifFalse:
						[self printSpacingOn: aStream level: level.
						aStream nextPutAll: '<', change, '>'; cr
						"<rename> won't appear, but <oldName: xxxxx> will."]]]
		ifFalse:
			[self printSpacingOn: aStream level: level.
			type == #methodRemove
				ifTrue: [aStream nextPutAll: value, ' <remove>']
				ifFalse: [aStream nextPutAll: value].
			type == #classCategory
				ifTrue: [aStream cr.  1 to: value size do: [:i | aStream nextPut: $-]].
			aStream cr]!

printSpacingOn: aStream level: level
	"Possibly print some blank lines, unless this is the first line.  The print 'level' tabs."
	aStream isEmpty ifFalse:
		[type == #classCategory ifTrue: [aStream cr; cr].
		type == #class ifTrue: [aStream cr]].
	"Print one tab per level"
	1 to: level do: [:i | aStream nextPut: Character tab]! !

!ChangeTreeLeaf methodsFor: 'fileIn/Out'!

fileOutClass: className
	(Smalltalk at: className) fileOut!

fileOutClassCategory: cat
	SystemOrganization fileOutCategory: cat!

fileOutClassChanges: classChanges inClass: class

	| stream | 
	stream _ FileStream fileNamed: class name , '.def'.

	(classChanges includes: #rename) ifTrue:
		[stream nextChunkPut: (self oldNameIn: classChanges), ' rename: #', class name; cr].

	(classChanges includes: #change) ifTrue:
		[stream emphasis: 5; nextChunkPut: class definition; cr; emphasis: 1].

	(classChanges includes: #comment) ifTrue:
		[class organization putCommentOnFile: stream
			numbered: nil moveSource: false.
		stream cr].

	(classChanges includes: #reorganize) ifTrue:
		[class fileOutOrganizationOn: stream.
		stream cr]!

fileOutInClass: aClass
	| selector |
	selector _ FileOutMessages at: type.
	aClass == nil
		ifTrue: [self perform: selector with: value]
		ifFalse: [self perform: selector with: value with: aClass]!

fileOutMethod: selector inClass: aClass
	aClass fileOutMessage: selector!

fileOutMethodCategory: cat inClass: aClass
	aClass fileOutCategory: cat!

fileOutMethodRemove: selector inClass: aClass
	| fileStream | 
	fileStream _ FileStream fileNamed: 'removes'.
	fileStream setToEnd.
	fileStream position > 0 ifTrue: [fileStream nextPutAll: '.
'].
	fileStream nextPutAll: aClass name, 'removeSelector: ', selector storeString! !

!ChangeTreeLeaf methodsFor: 'archiving'!

graft: aLeaf
	"Do nothing"!

prune: aNode
	aNode become: self copy! !

!ChangeTreeLeaf methodsFor: 'private'!

classNamed: className 
	"className is either a class name or a class name followed by ' class'.  Answer the class or metaclass it names."

	| meta name class |
	(className size > 6 
		and: [(className copyFrom: className size - 5 to: className size) = ' class'])
		ifTrue: 
			[meta _ true.
			name _ className copyFrom: 1 to: className size - 6]
		ifFalse: 
			[meta _ false.
			name _ className].
	class _ Smalltalk at: name asSymbol.
	meta
		ifTrue: [^class class]
		ifFalse: [^class]!

oldNameIn: classChanges
	| name |
	name _ classChanges asOrderedCollection detect:
				[:x | 'oldName: *' match: x].
	^(Scanner new scanTokens: name) last! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ChangeTreeLeaf class
	instanceVariableNames: ''!


!ChangeTreeLeaf class methodsFor: 'class initialization'!

initialize
	"ChangeTreeLeaf initialize"
	FileOutMessages _ Dictionary new.
	FileOutMessages at: #classCategory put: #fileOutClassCategory:.
	FileOutMessages at: #class put: #fileOutClass:.
	FileOutMessages at: #classChanges put: #fileOutClassChanges:inClass:.
	FileOutMessages at: #methodCategory put: #fileOutMethodCategory:inClass:.
	FileOutMessages at: #method put: #fileOutMethod:inClass:.
	FileOutMessages at: #methodRemove put: #fileOutMethodRemove:inClass:! !

!ChangeTreeLeaf class methodsFor: 'instance creation'!

value: aValue type: typeSymbol
	^ self new value: aValue type: typeSymbol! !

ChangeTreeLeaf initialize!


ChangeTreeLeaf variableSubclass: #ChangeTreeNode
	instanceVariableNames: 'sons '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'System-Changes'!


!ChangeTreeNode methodsFor: 'accessing'!

isLeaf
	^ false!

sons
	^ sons!

value: aValue type: typeSymbol sons: aChangeTree
	value _ aValue.
	type _ typeSymbol.
	sons _ aChangeTree! !

!ChangeTreeNode methodsFor: 'printing'!

printOn: aStream level: level
	super printOn: aStream level: level.
	sons printOn: aStream level: level+1! !

!ChangeTreeNode methodsFor: 'fileIn/Out'!

fileOutInClass: aClass
	type == #class
		ifTrue: [sons fileOutInClass: (self classNamed: value)]
		ifFalse: [sons fileOutInClass: aClass]! !

!ChangeTreeNode methodsFor: 'archiving'!

graft: aNode
	self sons graft: aNode sons!

prune: aNode
	self sons prune: aNode sons! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ChangeTreeNode class
	instanceVariableNames: ''!


!ChangeTreeNode class methodsFor: 'instance creation'!

value: aValue type: typeSymbol sons: aChangeTree
	^ self new value: aValue type: typeSymbol sons: aChangeTree! !


!ChangeSet methodsFor: 'change management'!

classChangesAt: aClass
	^ classChanges at: aClass name ifAbsent: [Set new]!

methodChangesAt: aClass
	^ methodChanges at: aClass name ifAbsent: [Set new]! !


!ClassDescription methodsFor: 'change trees'!

changesToClass
	"Return a ChangeTree of all the changes I have undergone.  If I am a new class, then return #all.  If I have not changed at all, return #none."

	| changeSet tree classChanges methodChanges allChanged changesToMethodCat |
	changeSet _ Smalltalk changes.
	tree _ ChangeTree new.
	classChanges _ changeSet classChangesAt: self.
	methodChanges _ changeSet methodChangesAt: self.
	classChanges isEmpty
		ifTrue: [methodChanges isEmpty
			ifTrue: [^ #none]]
		ifFalse:
			[(classChanges includes: #add)
				ifTrue: [^ #all].
			tree addLeaf: classChanges copy type: #classChanges].

	methodChanges isEmpty ifFalse:
		[self organization categories do: [:cat |
			changesToMethodCat _ self changesToMethodCat: cat.
			changesToMethodCat == #all
				ifTrue: [tree addLeaf: cat type: #methodCategory]
				ifFalse: [changesToMethodCat ~~ #none
					ifTrue: [tree addNode: cat type: #methodCategory sons: changesToMethodCat]]]].
	^ tree!

changesToMethodCat: cat
	"Return a ChangeTree of all changes to the methods in cat.  If all methods have changed, then return #all.  If none have changed, return #none."

	| changeSet tree allChanged methodChange | 
	changeSet _ Smalltalk changes.
	tree _ ChangeTree new.
	allChanged _ true.
	(self organization listAtCategoryNamed: cat) do: [:selector |
		methodChange _ (changeSet atSelector: selector class: self).
		methodChange == #none
			ifTrue: [allChanged _ false]
			ifFalse: [(methodChange == #change) | (methodChange == #add)
				ifTrue: [tree addLeaf: selector type: #method]
				ifFalse: [methodChange == #remove
					ifTrue: [tree addLeaf: selector type: #methodRemove]]]].
	allChanged ifTrue: [^ #all].
	tree size == 0 ifTrue: [^ #none].
	^ tree! !


!SystemOrganizer methodsFor: 'change trees'!

changesToCategory: cat
	"Return a ChangeTree of all changes to the classes in cat.  If all classes have changed entirely, then return #all.  If none have changed at all, return #none."

	| tree allChanged aClass changesToClass changesToMeta | 
	tree _ ChangeTree new.
	allChanged _ true.
	(self listAtCategoryNamed: cat) do: [:className |
		aClass _ Smalltalk at: className.
		changesToClass _ aClass changesToClass.
		changesToClass == #all
			ifTrue:
				[tree addLeaf: className type: #class]
			ifFalse:
				[changesToMeta _ aClass class changesToClass.
				changesToClass ~~ #none
					ifTrue: [tree addNode: className type: #class sons: changesToClass].
				changesToMeta ~~ #none
					ifTrue: [tree addNode: className, ' class' type: #class sons: changesToMeta]].
		allChanged _ allChanged & (changesToClass == #all)].
	allChanged ifTrue: [^ #all].
	tree size == 0 ifTrue: [^ #none].
	^ tree!

changeTree
	| tree changesToCategory |
	tree _ ChangeTree new.
	self categories do: [:cat |
		changesToCategory _ self changesToCategory: cat.
		changesToCategory == #none ifFalse:
			[changesToCategory == #all
				ifTrue: [tree addLeaf: cat type: #classCategory]
				ifFalse: [tree addNode: cat type: #classCategory sons: changesToCategory]]].
	^ tree! !


!SystemOrganizer reorganize!
('fileIn/Out' fileOutCategory: fileOutCategory:on: printOutCategory: superclassOrder:)
('accessing' changeFromString:)
('change trees' changesToCategory: changeTree)
!


!SystemDictionary methodsFor: 'change management'!

newArchive
	self at: #Archive put: (ChangeTree buildFrom: SystemOrganization).
	self changes initialize! !


