
;;;
;;;   P A T T E R N S   T O O L B O X
;;;   = = = = = = = =   = = = = = = =
;;;
;;; Depends on the following toolboxes:
;;;    Systems
;;;
;;; This toolbox contains routines to match patterns. The patterns
;;; may contain elements that match any single atom/list or a
;;; sequence of atoms/lists. It is possible to bind such a part of
;;; the matching string to a variable. It is also possible to provide
;;; additional restrictions that must be fufilled for a match to
;;; succeed.
;;;

; User interface:
; ==== =========
;
;    (Match pattern string)
; or
;    (Match pattern string AList)
;
; where pattern consists of a list of atoms, lists or matching
; elements; string  is a list which will be tested for conformance
; with the pattern. Match with only two parameters uses an initially
; empty AList. However, optionally a third parameter AList may be
; supplied to provide initial variable bindings. This AList may have
; for example been returned from a previous call to Match. Both
; functions return #f is the match fails and #t or an AList if the
; match succeeds. If the pattern specified bindings then an
; association list (alist) giving the binding will be returned on
; success rather than #t. If an initial AList was suppied to Match
; then the returned alist also includes these initial bindings.
;
; The form of the matching elements are:
;
; ?          match any single atom or list
; ?+         match any number of atoms and/or lists
; (? var)    match any single atom or list and bind it to
;            var in the returned alist
; (?+ var)   match any sequence of atoms and/or lists and
;            append them to any existing list bound to var
;            or create a new list and bind to var in the
;            returned alist
; (<-? var)  match against the value (atom or list) bound
;            to var by a previous matching
; (? var predicate1? predicate2? ...)
; (?+ var predicate1? predicate2? ...)
; (<-? var predicate1? predicate2? ...)
;            match and bind as above but also require that
;            none of the expressions
;            (predicate1? var), (predicate2? var), ...
;            evaluate to #f for the match to succeed.
;            Each predicate is called with the current
;            association list as a parameter
;
; Examples
; ========
;
; (Match (PATTERN "xxx" 123 'a)
;        '("xxx" 123 a))            ==> #t
; (Match (PATTERN "xxx" 123 'a)
;        '("xxx" 124 a))            ==> #f
; (Match (PATTERN "xxx" ? 'a)
;        '("xxx" 123 a))            ==> #t
; (Match (PATTERN "xxx" ?+)
;        '("xxx" 123 a))            ==> #t
; (Match (PATTERN "xxx" (? 'x) 'a)
;        '("xxx" 123 a))            ==> ((x 123))
; (Match (PATTERN "xxx" (? 'x) (? 'y))
;        '("xxx" 123 a))            ==> ((x 123) (y a))
; (Match (PATTERN "xxx" (?+ 'x))
;        '("xxx" 123 a))            ==> ((x (123 a)))
; (Match (PATTERN "xxx" (? 'x) (<-? 'x))
;        '("xxx" 123 123))          ==> ((x 123))
; (Match (PATTERN "xxx" (?+ 'x) (<-? 'x))
;        '("xxx" 123 124 123 124))  ==> ((x (123 124)))
; (Match (PATTERN "xxx" (? 'x isxnumber? isxpositive?))
;        '("xxx" 123))              ==> ((x 123))
; (Match (PATTERN "xxx" (? 'x isxnumber? isxpositive?))
;        '("xxx" -123))             ==> #f
; (Match (PATTERN "xxx" (PATTERN 1 (? 'x)) (? 'y))
;        '("xxx" (1 3) 123))        ==> ((x 3) (y 123))
; (Match (PATTERN "xxx" (<-? 'x))
;                 '("xxx" 123)
;                 '((x 123)))       ==> ((x 123))
;

; Data Structures
; ==== ==========
;
;   STRING              - a list of symbols (atoms or lists)
;   ASSOCIATION         - a <name value> list, where atoms or
;                         lists may be used as values
;   ALIST               - a list of ASSOCIATIONs 
;   PATTERN             - a list of symbols, including
;                         matching elements
;   MATCHING-ELEMENT    - an <indicator var predicate> list
;
; ========== fns on STRINGs =================================
; lists of symbols (atoms or lists)
;
; implemented through the usual list processing functions

; ========== fns on ASSOCIATIONs ============================
; (symbol value) lists 

(define MakeAssoc
  (lambda (aSymbol aValue) (list aSymbol aValue)))
                         
(define GetSymbol
  (lambda (anAssociation) (car anAssociation)))

(define GetValue
  (lambda (anAssociation) (cadr anAssociation)))

; ========== fns on ALISTs ==================================
; lists of ASSOCIATIONs

(define MakeAList (lambda anAssocList anAssocList))

(define GetNextAssoc (lambda (anAList) (car anAList)))

(define GetRestOfAssocs (lambda (anAList) (cdr anAList)))

(define StoreAssocAsItem
  (lambda (aSymbol anItem anAlist)
    ; used for "(? var ...)" matches
    (append anAlist (list (MakeAssoc aSymbol anItem)))))

(define StoreAssocAsList
  (lambda (aSymbol anItem anAlist)
    ; used for "(?+ var ...)" matches
    (cond ((null? anAlist)
           ; no binding for symbol yet - make new entry
           (MakeAList (MakeAssoc aSymbol (list anItem))))
          ((equal? aSymbol (GetSymbol (GetNextAssoc anAlist)))
           ; old binding already on the list - if a list append
           ; new values else make a fresh list
           (cons (MakeAssoc
                   aSymbol
		   (if (pair? (GetValue (GetNextAssoc anAlist)))
		       (append (GetValue (GetNextAssoc anAlist)) (list anItem))
		       (list anItem)))
                 (GetRestOfAssocs anAlist)))
          (else (cons (GetNextAssoc anAlist)
		      ; remember first association and keep looking
		      (StoreAssocAsList aSymbol
					anItem
					(GetRestOfAssocs anAlist)))))))

(define GetAssociation
  (lambda (aPatternVariable anAlist)
    ; retrieves binding for "aPatternVariable" (if any)
    (assoc aPatternVariable anAlist)))

(define GetAssociationValue
  (lambda (aPatternVariable anAlist)
    ; retrieves value for "aPatternVariable" giving error if none
    (let ((association (GetAssociation aPatternVariable anAlist)))
      (if (not association)
          (Fatal-Error "GetAssociationValue:"
                       "no previous binding for"
                       aPatternVariable)
          (GetValue association)))))

; ========== fns on PATTERNs ==================================
; lists of symbols, some of which are interpreted as
; MATCHING-ELEMENTs                                                             

(define PATTERN (lambda symbols symbols))

(define BuildNewPattern
  ; used for "(<-? var ...)" elements. If old binding is a list
  ; prepend all the previous sequence to pattern (to be matched again)
  (lambda (anItemOrList aPattern)
    (if (pair? anItemOrList)
        (append anItemOrList aPattern)
        (cons anItemOrList aPattern))))

(define ? (lambda (var . predicates) (list ? var predicates)))

(define ?+ (lambda (var . predicates) (list ?+ var predicates)))

(define <-? (lambda (var . predicates) (list <-? var predicates)))

(define ?Symbol? (lambda (aSymbol) (equal? aSymbol ?)))

(define ?+Symbol? (lambda (aSymbol) (equal? aSymbol ?+)))

(define <-?Symbol? (lambda (aSymbol) (equal? aSymbol <-?)))

(define IsMatchingOperator?
  (lambda (anOperator)
    (or (?Symbol? anOperator)
        (?+Symbol? anOperator)
        (<-?Symbol? anOperator))))

(define PrintMatchingOperator
  (lambda (anOperator)
    (cond ((?Symbol? anOperator)   (display "?"))
          ((?+Symbol? anOperator)  (display "?+"))
          ((<-?Symbol? anOperator) (display "<-?")))))

(define PrintPattern
  (lambda (aPattern)
    
    (define printPatternAux
      (lambda (aPattern)
        
        (define printPatternElement
          (lambda (aPatternElement)
            (cond ((IsMatchingOperator? aPatternElement)
                   (PrintMatchingOperator aPatternElement))
                  ((IsMatchingElement? aPatternElement)
                   (PrintMatchingElement aPatternElement))
                  ((pair? aPatternElement)
                   (PrintPattern aPatternElement))
                  (else (display aPatternElement)))))
        
        (if (null? aPattern)
            #f
            (begin (printPatternElement (car aPattern))
                   (display " ")
                   (printPatternAux (cdr aPattern))))))
    
    (display "(")
    (printPatternAux aPattern)
    (display ")")))

(define MakeVariableSubst
  (lambda (aPattern anAList)
    ; replace all (<-? var) elements in pattern with the
    ; value bound to the variable in anAList
    (if (null? aPattern)
        (PATTERN)
        (let ((element (car aPattern)))
          (cons (if (and (IsMatchingElement? element)
                         (<-?Symbol? (MatchingOperator element)))
                    (GetAssociationValue (MatchingVariable element) anAList)
                    element)
                (MakeVariableSubst (cdr aPattern) anAList))))))


; ========== fns on MATCHING-ELEMENTs =======================
; (indicator variable) or (indicator variable predicatelist) lists;
; where indicator is one of: ?, ?+, <-?, or <-?+.

(define MatchingOperator
  (lambda (aMatchingElement) (car aMatchingElement)))

(define MatchingVariable
  (lambda (aMatchingElement) (cadr aMatchingElement)))

(define MatchingPredicateList
  (lambda (aMatchingElement) (caddr aMatchingElement)))

(define IsMatchingElement?
  (lambda (aMatchingElement)
    (and (pair? aMatchingElement)
         (>= (length aMatchingElement) 2)
         (IsMatchingOperator? (MatchingOperator aMatchingElement)))))

(define SatisfiesPredicates?
  (lambda (anAList aPredicateList)
    ; returns the (boolean) result of applying ALL
    ; restricting fns to the restricted symbol
    (cond ((null? aPredicateList) #t)
          (((car aPredicateList) anAList)
           (SatisfiesPredicates? anAList (cdr aPredicateList)))
          (else #f))))

(define PrintMatchingElement
  (lambda (aMatchingElement)
    
    (define printPredicateList
      (lambda (aPredicateList)
        (if (null? aPredicateList)
            #f
            (begin (display (car aPredicateList))
                   (printPredicateList (cdr aPredicateList))))))
    
    (display "(")
    (PrintMatchingOperator (car aMatchingElement))
    (display " ")
    (display (cadr aMatchingElement))
    (if (caddr aMatchingElement)
        (begin (display " ")
               (printPredicateList (caddr aMatchingElement)))
        #f)
    (display ")")))

(define ProcessMatchingElement
  (lambda (aPattern aString anAList) 
    
    (let* ((patternElement (car aPattern))
           (operator       (MatchingOperator patternElement))
           (variable       (MatchingVariable patternElement))
           (predicateList  (MatchingPredicateList patternElement))
           (restPattern    (cdr aPattern))
           (stringElement  (car aString))
           (restString     (cdr aString))
           (newAList       anAList))
      ; build new ALIST if (? ...) of (?+ ...)
      (cond ((?Symbol? operator)
             (set! newAList (StoreAssocAsItem variable
                                              stringElement
                                              anAList)))
            ((?+Symbol? operator) 
             (set! newAList (StoreAssocAsList variable
                                              stringElement
                                              anAList)))
            (else #f))
      ; check predicates first and then attempt match
      (if (or (null? predicateList)
              (SatisfiesPredicates? newAList predicateList))
          (cond ((?Symbol? operator)
                 ; matches and binds any SINGLE symbol to the
                 ; pattern variable
                 (MatchWithAList restPattern restString newAList))
                ((?+Symbol? operator)
                 ; matches and binds any SEQUENCE of symbols
                 (or (MatchWithAList restPattern
                                     restString
                                     newAList)
                     (MatchWithAList aPattern
                                     restString
                                     newAList)))
                ((<-?Symbol? operator)
                 ; retrieves a previous binding for the pattern
                 ; variable and THEN tries to match that VALUE
                 ; against the current symbol.
                 (MatchWithAList (BuildNewPattern
                                  (GetAssociationValue variable anAList)
                                  restPattern)
                                 aString
                                 newAList)))
          #f))))

; ========== fns for control of pattern matching ============

(define MatchWithAList
  (lambda (aPattern aString anAList)
    ; scans a string for correspondence with a pattern using initial
    ; bindings given by anAList. If match is successful, it will
    ; return #t or a list of associations (bindings for pattern variables)
    
    (define ReportSuccess
      (lambda (anAList)
        ; indicates a successful match, returning variable bindings (if any)
        (if anAList anAList #t)))
    
    (define ReportFailure (lambda () #f))
    
    (cond ((and (null? aPattern) (null? aString))
           (ReportSuccess anAList))
          ((or (null? aPattern) (null? aString))
           (ReportFailure))
          ((?Symbol? (car aPattern))
           ; match against any SINGLE symbol
           (MatchWithAList (cdr aPattern) (cdr aString) anAList))
          ((?+Symbol? (car aPattern))
           ; match against any SEQUENCE of symbols
           (or (MatchWithAList (cdr aPattern) (cdr aString) anAList)
               (MatchWithAList aPattern (cdr aString) anAList)))
          ((IsMatchingElement? (car aPattern))
           (ProcessMatchingElement aPattern aString anAList))
          ((and (pair? (car aString)) (pair? (car aPattern)))
           ; try matching sublist recursively
           (let ((newAList (MatchWithAList (car aPattern)
					   (car aString)
					   anAList)))
             (if newAList (MatchWithAList (cdr aPattern)
					  (cdr aString)
					  newAList)
                          (ReportFailure))))
          ; see if elements are equal?
          ((equal? (car aPattern) (car aString))
           (MatchWithAList (cdr aPattern) (cdr aString) anAList))
          (else (ReportFailure)))))

(define Match
  (lambda (aPattern aString . optionalAList)
    ; scans a string for correspondence with a pattern. If optionalAList is not
    ; supplied then no initial bindings are used. If match is successful, it
    ; will return #t or a list of associations (bindings for pattern variables)
    ; including any initial bindings if the optionalAList was supplied
    (cond ((null? optionalAList)
           (MatchWithAList aPattern aString (MakeAList)))
          ((null? (cdr optionalAList))
           (MatchWithAList aPattern aString (car optionalAList)))
          (else
           (Fatal-Error "Match:"
                        "Too many args"
                        (+ 2 (length optionalAList)))))))

