;;;
;;;   K N O W L E D G E   B A S E S   T O O L B O X
;;;   = = = = = = = = =   = = = = =   = = = = = = =
;;;
;;; Depends on the following toolboxes:
;;;    Systems
;;;    Patterns
;;;
;;; This toolbox contains routines to search an associative knowledge
;;; base for entries that contain a match for a given pattern. It is built on top of the
;;; pattern matching toolbox.

; Data Structures
; ==== ==========
;
; FACT               - arbitary item. Some part of the FACT will
;                      contain either a STRING or a PATTERN in the
;                      pattern matching toolbox sence
; KB                 - a list of FACTs
; RETRIEVED-ELEMENT  - a list of the form (aFact anAList)
; RETRIEVED-LIST     - a list of RETRIEVED-ELEMENTs

; ========== fns on FACTs ===================================
; Not defined by this toolbox. The usual list processing functions
; can be used to build and manipulate these

; ========== fns on KBs =====================================
; a KB is a list of FACTs.

(define MakeKB (lambda initialFacts initialFacts))

(define EmptyKB? (lambda (aKB) (null? aKB)))

(define FirstInKB (lambda (aKB) (car aKB)))

(define RestOfKB (lambda (aKB) (cdr aKB)))

(define KnownFact? (lambda (aFact aKB) (member aFact aKB)))

(define AddFact
  (lambda (aKB aFact)
    ; add a new fact to end of KB if not already present
    (if (KnownFact? aFact aKB)
        aKB
        (append aKB (list aFact)))))

(define RemoveFact
  (lambda (aKB aFact)
    ; remove a fact from the KB
    (cond ((null? aKB) #f)
          ((equal? (FirstInKB aKB) aFact) (RestOfKB aKB))
          (else (cons (FirstInKB aKB)
		      (RemoveFact (RestOfKB aKB) aFact))))))

(define PrintKB
  (lambda (aKB aDisplayProcedure)
    (if (null? aKB)
        #f
        (begin (aDisplayProcedure (FirstInKB aKB))
               (newline)
               (PrintKB (RestOfKB aKB) aDisplayProcedure)))))

; ========== fns on RETRIEVED-ELEMENTs ======================
; a list of the form (fact aList)

(define MakeRetrievedElement
  (lambda (aFact anAList)
    (if (not anAList)
        #f
        (list aFact anAList))))

(define GetFact
  (lambda (aRetrievedElement) (car aRetrievedElement)))

(define GetAList
  (lambda (aRetrievedElement) (cadr aRetrievedElement)))

; ========== fns on RETRIEVED-LISTs ==========================
; lists of RETRIEVED-ELEMENTs

(define MakeRetrievedList
  (lambda aRetrievedList aRetrievedList))

(define EmptyRetrievedList?
  (lambda (aRetrievedList) (null? aRetrievedList)))

(define CurrentRetrievedElement
  (lambda (aRetrievedList) (car aRetrievedList)))

(define RestOfRetrievedList
  (lambda (aRetrievedList) (cdr aRetrievedList)))

(define AddRetrievedElement
  (lambda (aRetrievedElement aRetrievedList)
    ; if aRetrievedElement is not empty append it to aRetrievedList
    (if (not aRetrievedElement)
        aRetrievedList
        (append aRetrievedList (MakeRetrievedList aRetrievedElement)))))

(define CombineRetrievedLists
  (lambda (aRetrievedList1 aRetrievedList2)
    (cond ((EmptyRetrievedList? aRetrievedList1) aRetrievedList2)
          ((EmptyRetrievedList? aRetrievedList2) aRetrievedList1)
          (else (append aRetrievedList1 aRetrievedList2)))))

; ========== matching fns on KBs ============================
; Each of these retrieving functions may optioanlly contain a selector
; function that can be applied to a FACT in the KB to extract either
; a STRING (for RetrieveByPattern or RetrieveAllByPattern) or a
; PATTERN (for RetrieveByString or RetrieveAllByString). If none is
; supplied then the whole FACT is used. There may also be an
; optionally supplied an initial AList which will be used as a
; parameter to Match. If none is supplied then it defaults to #f.
; The distinction between these two optional parameters is achieved
; by using the procedure? predicate function.

(define SelectWholeFact (lambda (aFact) aFact))

(define RetrieveByPattern
  (lambda (aKB aPattern . optionalArguments )
    (let ((aSelectorFN SelectWholeFact)
          (anAList ()))
      
      ; Find first fact in 'aKB' that matches 'aPattern'. Only the
      ; part of the fact selected by 'aSelectorFN' is involved in the
      ; match. The list 'anAList' ispassed to each call of Match
      ; Returns #f or a RETRIVED-ELEMENT
      
      (define retrieveAux
        (lambda (aKB)
          (if (EmptyKB? aKB)
              #f
              (let* ((fact (FirstInKB aKB))
                     (string (aSelectorFN fact))
                     (result (Match aPattern string anAList)))
                (if result
                    (MakeRetrievedElement fact result)
                    (retrieveAux (RestOfKB aKB)))))))
      
      (if (> (length optionalArguments) 2)
          (Fatal-Error "RetrieveByPattern:"
                       "Too many optional arguments"
                       optionalArguments)
          #f)
      (do ((args optionalArguments (cdr args))
           (arg #f))
          ((null? args) #f)
          (set! arg (car args))
          (if (procedure? arg)
              (set! aSelectorFN arg)
              (set! anAList arg)))
      (retrieveAux aKB))))

(define RetrieveAllByPattern
  (lambda (aKB aPattern . optionalArguments )
    (let ((aSelectorFN SelectWholeFact)
          (anAList ()))
      ; Find all facts in 'aKB' that matches 'aPattern'. Only the
      ; part of the fact selected by 'aSelectorFN' is involved in the
      ; match. The list 'anAList' ispassed to each call of Match
      ; Returns #f or a RETRIVED-LIST
      
      (define retrieveAux
        (lambda (remaingKB prevResults)
          (if (EmptyKB? remaingKB)
              prevResults
              (let* ((fact (FirstInKB remaingKB))
                     (string (aSelectorFN fact))
                     (result (Match aPattern string anAList)))
                (retrieveAux (RestOfKB remaingKB)
                             (AddRetrievedElement
                              (MakeRetrievedElement fact result)
                              prevResults))))))
      
      (if (> (length optionalArguments) 2)
          (Fatal-Error "RetrieveByPattern:"
                       "Too many optional arguments"
                       optionalArguments)
          #f)
      (do ((args optionalArguments (cdr args))
           (arg #f))
          ((null? args) #f)
          (set! arg (car args))
          (if (procedure? arg)
              (set! aSelectorFN arg)
              (set! anAList arg)))
      (retrieveAux aKB (MakeRetrievedList)))))

(define RetrieveByString
  (lambda (aKB aString . optionalArguments )
    (let ((aSelectorFN SelectWholeFact)
          (anAList ()))
      ; find first fact in 'aKB' with a pattern that matches 'aString'.
      ; Only the part of the fact selected by 'aSelectorFN' is
      ; involved in the match. The list 'anAList' ispassed to each
      ; call of Match. Returns #f or a RETRIVED-ELEMENT
      
      (define retrieveAux
        (lambda (aKB)
          (if (EmptyKB? aKB)
              #f
              (let* ((fact    (FirstInKB aKB))
                     (pattern (aSelectorFN fact))
                     (result  (Match pattern aString anAList)))
                (if result
                    (MakeRetrievedElement fact result)
                    (retrieveAux (RestOfKB aKB)))))))
      
      (if (> (length optionalArguments) 2)
          (Fatal-Error "RetrieveByPattern:"
                       "Too many optional arguments"
                       optionalArguments)
          #f)
      (do ((args optionalArguments (cdr args))
           (arg #f))
          ((null? args) #f)
          (set! arg (car args))
          (if (procedure? arg)
              (set! aSelectorFN arg)
              (set! anAList arg)))
      (retrieveAux aKB))))

(define RetrieveAllByString
  (lambda (aKB aString . optionalArguments )
    (let ((aSelectorFN SelectWholeFact)
          (anAList ()))
      ; Find all facts in 'aKB' with a pattern that matches 'aString'.
      ; Only the part of the fact selected by 'aSelectorFN' is
      ; involved in the match. The list 'anAList' ispassed to each
      ; call of Match Returns #f or a list of RETRIVED-ELEMENTs
      
      (define retrieveAux
        (lambda (remaingKB prevResults)
          (if (EmptyKB? remaingKB)
              prevResults
              (let* ((fact (FirstInKB remaingKB))
                     (pattern (aSelectorFN fact))
                     (result (Match pattern aString anAList)))
                (retrieveAux (RestOfKB remaingKB)
                             (AddRetrievedElement
                              (MakeRetrievedElement fact result)
                              prevResults))))))
      
      (if (> (length optionalArguments) 2)
          (Fatal-Error "RetrieveByPattern:"
                       "Too many optional arguments"
                       optionalArguments)
          #f)
      (do ((args optionalArguments (cdr args))
           (arg #f))
          ((null? args) #f)
          (set! arg (car args))
          (if (procedure? arg)
              (set! aSelectorFN arg)
              (set! anAList arg)))
      (retrieveAux aKB (MakeRetrievedList)))))
