;;;
;;;   G A M E S    T O O L B O X
;;;   = = = = =    = = = = = = =
;;;
;;; Depends on the following toolboxes:
;;;    Systems
;;;    Search

; This toolbox is built upon the lower levels of the search toolbox.
; It provides routines to be called to select the best move from the
; current game position for the machine and also provides a framework
; to play a game with the user. The game tree is an AND/OR tree to be
; evaluated to a particular ply level. It assumes the existence of an
; 'evaluation' function that can be used to rate each game position
; from the point of view of the machine. The higher the value, the
; better position (as far as the machine is concerned).

; ========== operations on Game Problems ====================
;
; A game problem is shares some of the structure of a Search Problem.
; The tracing, announcing, count, initial state and evaluation slots
; are used in the same way except the evaluation function is now
; passed a second parameter which is #t if the next move from the
; state is the machines. The goal function slot is now used to hold a
; user function that detects special states such as won, lost or
; drawn states or states that the game allows an extra move to the
; last player. It is passed both the state and a flag indicating who moves next
; (#t for machine) and should return one of the following:
;
; "machine"  - This state is a won position for the machine.
; "opponent" - This state is a won position for the opponent.
; "draw"     - This state is a draw position.
; "again"    - This state is one which allows an extra move for the
;              last player.
; #f         - There is nothing special about this state.
; The actions are split into two sets, the machines moves and the opponents
; moves. A slot is used to hold an optional function to accept moves from
; the user, check their validity and then return the state produced after
; a move (which may be the same state if a pass is allowed) or #f to indicate
; resignation. If no such function is supplied a simple interface to request
; a new move is used.

(define *specialStateIndex    6)
(define *machineActionsIndex  7)
(define *opponentActionsIndex 8)
(define *getAMoveIndex        9)

(define *lengthGameVector    10)

(define MakeGameProblem
 (lambda (aState aSpecialStateFN anEvalFN
          machineActions opponentActions)
   (let ((aProblem (make-vector *lengthGameVector)))
     (vector-set! aProblem *traceIndex            #f)
     (vector-set! aProblem *announceIndex         #f)
     (vector-set! aProblem *countIndex            0)
     (vector-set! aProblem *stateIndex            aState)
     (vector-set! aProblem *printStateIndex       display)
     (vector-set! aProblem *evalIndex             anEvalFN)
     (vector-set! aProblem *specialStateIndex     aSpecialStateFN)
     (vector-set! aProblem *machineActionsIndex   machineActions)
     (vector-set! aProblem *opponentActionsIndex  opponentActions)
     (vector-set! aProblem *getAMoveIndex         #f)
     aProblem)))
    
(define SpecialState
  (lambda (aState aGameProblem machineMove?)
    ((vector-ref aGameProblem *specialStateIndex)
     aState machineMove?)))

(define GetMachineActions
  (lambda (aGameProblem)
    (vector-ref aGameProblem *machineActionsIndex)))

(define GetOpponentActions
  (lambda (aGameProblem)
    (vector-ref aGameProblem *opponentActionsIndex)))

(define SetGetAMove!
  (lambda (aGameProblem aUserFN)
    (vector-set! aGameProblem *getAMoveIndex aUserFN)))

(define GetGetAMove
  (lambda (aGameProblem)
    (vector-ref aGameProblem *getAMoveIndex)))

(define PrintGameProblem
  (lambda (aGameProblem)
    (newline) (display "Initial State     : ")
    (PrintState (GetInitialState aGameProblem) aGameProblem)
    (newline)
    (DisplayLine "Move(s)           :" (GetCount aGameProblem))
    (DisplayLine "Eval function     :" (GetEvalFN aGameProblem))
    (DisplayLine "Machine Action(s) :"
                 (GetMachineActions aGameProblem))
    (DisplayLine "Opponent Action(s):"
                 (GetOpponentActions aGameProblem))
    (DisplayLine "GetAMove function :" (GetGetAMove aGameProblem))
    (if (GetTraceFlag aGameProblem)
        (DisplayLine "Tracing           : on")
        (DisplayLine "Tracing           : off"))
    (if (GetAnnounceFlag aGameProblem)
        (DisplayLine "Announcing        : on")
        (DisplayLine "Announcing        : off"))))

; ========== operations on GAMELEVELS =======================
; A list of (score move ...)
;
(define MakeGameLevel
  (lambda (aScore aMoveList)
    (if (null? aMoveList) #f (cons aScore aMoveList))))

(define GetLevelScore
  (lambda (aGameLevel) (car aGameLevel)))

(define GetLevelMoves
  (lambda (aGameLevel) (cdr aGameLevel)))

(define EmptyGameLevel?
  (lambda (aGameLevel) (null? aGameLevel)))

(define DisplayScore
  (lambda (score)
    (display
     (cond ((= score NEG-INFINITY) "-INF")
           ((= score INFINITY)     "+INF")
           (else                   score)))))

(define ChooseRandomMove
  (lambda (aLevel aGameProblem)
    ; Return one of the moves at random
    (newline)
    (DisplayList "Move chosen from among"
                 (length (GetLevelMoves aLevel))
                 "of value")
    (DisplayScore (GetLevelScore aLevel))
    (DisplayList " after generating" (GetCount aGameProblem))
    (list-ref (GetLevelMoves aLevel)
              (MyRandom (length (GetLevelMoves aLevel))))))

(define PrintGameLevel
  (lambda (aGameLevel aGameProblem)
    
    (define printGameLevelAux
      (lambda (aMoveList)
        (if (null? aMoveList)
            #f
            (begin (PrintState (car aMoveList) aGameProblem)
                   (display " ")
                   (printGameLevelAux (cdr aMoveList))))))
    
    (if (EmptyGameLevel? aGameLevel)
        (display "Empty game level")
        (begin (display "Best score: ")
	       (DisplayScore (GetLevelScore aGameLevel))
	       (DisplayLine " Best moves:")
	       (printGameLevelAux (GetLevelMoves aGameLevel))))))


; ======== fns to control playing of a GameProblem ==========

(define MakeStateGenerator
  (lambda (aState aGameProblem machineMove?)
    ; Return a function that will return successor states of 'aState'
    ; using the machine or opponents actions of aGameProblem. After
    ; the last state has been returned the function returns #f.
    (let ((remainingActions (if machineMove?
				(GetMachineActions aGameProblem)
				(GetOpponentActions aGameProblem))))
      (lambda ()
        (do ((nextState #f))
            ((or (EmptyActions? remainingActions) nextState)
             (if nextState
                 (IncCount! aGameProblem)
                 #f)
             nextState)
            (set! nextState ((FirstAction remainingActions) aState))
            (set! remainingActions (RestOfActions remainingActions)))))))

(define FindBestMachineMove
  (lambda (aState aGameProblem alphabeta? aPly)
    ; Return the best moves for the machine from 'aState' expanding
    ; the game tree to level 'aPly'. If 'alphabeta' is #t then apply
    ; alpha-beta pruning when evaluating moves.
    
    (define scoreState
      (lambda (aState aDepth machineMove? alpha beta tracing?)
        ; Return the value of this position.
        ; The value of 'aDepth' gives the current number of levels
        ; from the bottom of the 'aPly' game tree. 'machineMove?' is
        ; #t if the next move is to be made by the machine. The values
        ; alpha and beta are used by the alpha-beta method and simply
        ; hold NEG-INFINITY and INFINITY for the minimax method.
        (let ((result 0)
              (pruneRest #f)
              (status (SpecialState aState aGameProblem machineMove?)))
          (if tracing?
              (begin (DisplayList "Enter scoreState"
                                  (if machineMove? "(MAX)" "(MIN)")
                                  "state=" aState
                                  " depth=" aDepth)
                     (if alphabeta?
                         (begin (display " alpha=") (DisplayScore alpha)
                                (display " beta=") (DisplayScore beta))
                         #f)
                     (newline))
              #f)
          (cond ((and status (string=? status "machine"))
                 (set! result INFINITY))
                ((and status (string=? status "opponent"))
                 (set! result NEG-INFINITY))
                ((and status (string=? status "draw"))
                 (set! result 0))
                ((and (not status) (zero? aDepth))
                 (set! result
                       ((GetEvalFN aGameProblem) aState
                                                 machineMove?)))
                (else
                 ; At interior node of tree. Build new set of states
                 ; and then call this function recursively to evaluate
                 ; them and return the best score. If status was
                 ; "again" then swap whose move and do not decrement
                 ; depth.
                 (let ((stateGenerator #f)
                       (newDepth 0))
                   (if status
                       (begin
                        (if (string=? status "again")
                            #f
                            (Fatal-Error "FindBestMachineMove:"
                                         "Unexpected value"
                                         status
                                         "returned from user fn"))
                        (set! machineMove? (not machineMove?))
                        (set! newDepth aDepth))
                       (set! newDepth (- aDepth 1)))
                   (set! stateGenerator
                         (MakeStateGenerator aState
                                             aGameProblem
                                             machineMove?))
                   (do ((nextState (stateGenerator) (stateGenerator))
                        (nextScore 0)
                        (bestScore (if machineMove? alpha beta)))
                       ((or pruneRest (not nextState))
                        (set! result bestScore))
                       (set! nextScore
                             (scoreState nextState
                                         newDepth
                                         (not machineMove?)
                                         alpha
                                         beta
                                         tracing?))
                       (if machineMove?
                           (begin ; MAX level
                            (set! bestScore (max bestScore nextScore))
                            (if alphabeta?
                                (begin
                                 (set! alpha bestScore)
                                 (if (>= alpha beta)
                                     (begin (set! bestScore beta)
                                            (set! pruneRest #t))
                                     #f))
                                #f))
                           (begin ; MIN level
                            (set! bestScore (min bestScore nextScore))
                            (if alphabeta?
                                (begin
                                 (set! beta bestScore)
                                 (if (>= alpha beta)
                                     (begin (set! bestScore alpha)
                                            (set! pruneRest #t))
                                     #f))
                                #f)))))))
          
          (if tracing?
              (begin (DisplayList "Exit scoreState"
                                  (if machineMove? "(MAX)" "(MIN)")
                                  (if pruneRest "(PRUNED)" "")
                                  "state=" aState "depth=" aDepth
                                  "-> ")
                     (DisplayScore result)
                     (newline))
              #f)
          result)))
    
    (define announceMoves
      (lambda (aGameLevel)
        (DisplayLine "Best moves for machine from state:")
        (PrintState aState aGameProblem)
        (DisplayLine " found after trying"
                     (GetCount aGameProblem)
                     "moves at" aPly "ply.")
        (PrintGameLevel aGameLevel aGameProblem)
        (newline)))
    
    (ZeroCount! aGameProblem)
    (let ((stateGenerator (MakeStateGenerator aState aGameProblem #t))
          (resultLevel #f))
      (do ((nextState (stateGenerator) (stateGenerator))
           (nextScore 0)
           (bestScore NEG-INFINITY)
           (bestMoves #f))
          ((not nextState)
           (set! resultLevel (MakeGameLevel bestScore bestMoves)))
          (set! nextScore
                (scoreState nextState
                            (- aPly 1)
                            #f
                            (if (and alphabeta?
                                     (not (= bestScore NEG-INFINITY)))
                                (- bestScore 1)
                                NEG-INFINITY)
                            INFINITY
                            (GetTraceFlag aGameProblem)))
          (cond ((> nextScore bestScore)
                 (set! bestScore nextScore)
                 (set! bestMoves (list nextState)))
                ((= nextScore bestScore)
                 (set! bestMoves (cons nextState bestMoves)))))
      (if (GetAnnounceFlag aGameProblem)
          (announceMoves resultLevel)
          #f)
      resultLevel)))

(define GetAMove
  (lambda (aState aGameProblem)
    ; Get a move from the user. Use the user supplied function is one
    ; is supplied, otherwise just request the state of the game after
    ; the move from the user.
    (let ((userFN (GetGetAMove aGameProblem))
          (newState #f))
      (newline)
      (if userFN
          (set! newState (userFN aState))
          (begin 
           (DisplayLine "Enter the board state after your move.")
           (DisplayLine "(#f if resign, same state if pass):" aState)
           (set! newState (read))))
      newState)))
  
(define PlayGame
  (lambda (aGameProblem)
    ; Play the game defined by 'aGameProblem' with the user.
    ; Request the ply level, who is to move first, and whether
    ; alpha-beta pruning is to performed from the player.
    
    (define announceEnd
      (lambda (aState status)
        (newline)
        (cond ((string=? status "machine") 
               (DisplayLine "Machine wins. Bad luck!"))
              ((string=? status "resign")
               (DisplayLine "You resign, machine wins. Bad luck!"))
              ((string=? status "opponent")
               (DisplayLine "You win. Well done!"))
              ((string=? status "draw")
               (DisplayLine "Game is draw. Good game!"))
              (else
               (Fatal-Error "PlayGame:"
                            "Unexpected value"
                            status
                            "returned from user fn")))
        (if (not aState)
            #f
            (begin (DisplayLine "Final state of game is:")
                   (PrintState aState aGameProblem)))))
    
    (let ((aPly (do ((n 0))
                    ((and (number? n) (positive? n)) n)
                    (newline)
                    (DisplayLine "Give ply level (>0):")
                    (set! n (read))))
          (alphabeta? (do ((reply "invalid"))
                          ((boolean? reply) reply)
                          (newline)
                          (DisplayLine "Use alpha-beta pruning? (y/n):")
                          (set! reply (read))
                          (cond ((eq? reply 'y) (set! reply #t))
                                ((eq? reply 'n) (set! reply #f)))))
          (machineMove? (do ((reply "invalid"))
                            ((boolean? reply) reply)
                            (newline)
                            (DisplayLine "Is machine to move first? (y/n):")
                            (set! reply (read))
                            (cond ((eq? reply 'y) (set! reply #t))
                                  ((eq? reply 'n) (set! reply #f)))))
          (state (GetInitialState aGameProblem)))
      (do ((stateStatus #f)
           (gameOver #f))
          (gameOver (announceEnd state stateStatus))
          (newline)
          (DisplayLine "Current game state ("
                       (if machineMove? "Machine's" "Your")
                       "move):")
          (PrintState state aGameProblem)
          (newline)
          ; get machine or user to make a move
          (set! state
                (if machineMove?
                    (ChooseRandomMove (FindBestMachineMove
                                        state
					aGameProblem
					alphabeta?
					aPly) aGameProblem)
                    (GetAMove state aGameProblem)))
          ; ensure next move is next player's
          (set! machineMove? (not machineMove?))
          ; is it a special state?
          (set! stateStatus
                (if (not state)
                    "resign"
                    (SpecialState state aGameProblem machineMove?)))
          ; look for end of game or repeated move
          (if stateStatus
              (cond ((or (string=? stateStatus "machine")
                         (string=? stateStatus "resign")
                         (string=? stateStatus "opponent")
                         (string=? stateStatus "draw"))
                     (set! gameOver #t))
                    ((string=? stateStatus "again")
                     (set! machineMove? (not machineMove?))))
              #f)))))
