; ******** The 8-Puzzle ***********
; This is a problem involving 8 numbered, movable tiles set
; in a 3 x 3 frame. One cell of the frame is always empty
; thus making it possible to move an adjacent tile into the
; empty cell. Let the initial and goal configurations be:
;               +---+---+---+             +---+---+---+
;               | 2 | 8 | 3 |             | 1 | 2 | 3 |
;               +---+---+---+             +---+---+---+
;  Initial:     | 1 | 6 | 4 |     Goal:   | 8 |   | 4 | 
;               +---+---+---+             +---+---+---+
;               | 7 |   | 5 |             | 7 | 6 | 5 |
;               +---+---+---+             +---+---+---+

(define 8-SearchProblem #f)

(define SetUp8Puzzle
  (lambda ()
    
    ; represent the state of the puzzle as a list of the
    ; contents of squares in the top row, then middle and
    ; then bottom row. Use 'e' to represent the empty cell.
    ; Thus the initial state is
    
    (define initialState '(2 8 3 1 6 4 7 e 5))
    
    ; the goal state is:
    
    (define goalState '(1 2 3 8 e 4 7 6 5)) ; see figure 3.7
    
    ; the goal state is recognized by:
    
    (define goalFN (lambda (aState) (equal? aState goalState)))
    
    ; the moves are easier to visualize if we think of the
    ; empty cell moving one position up, down, left or right
    ; under the restriction than it may not move off the
    ; frame. The following routine is passed a list of where
    ; the empty cell can move to from each of the cell, with
    ; #f indicating it would move off the frame. A function
    ; is then returned that would make such a move.
    
    (define makeMove
      (lambda (emptyTo)
        ; returns a function to move the empty cell as
        ; indicated by the list of moves in emptyTo
        (lambda (aState)
          (let* ((emptyCell (FindFirstSlotOfSymbol 'e aState))
                 (newCell   (FindSymbolInSlot emptyCell emptyTo)))
            (if (not newCell)
                #f ; would move empty cell off frame
                (FillSlot
                  newCell   ; empty the vacated spot
		  'e
		  (FillSlot emptyCell  ; fill the destination
			    (FindSymbolInSlot newCell aState)
			    aState)))))))

    ; the evaluation function measures the distance in terms
    ; of number of moves of each tile is from its final
    ; position. The negative of the number of moves is used.
    
    (define evalFN
      (lambda (aState)
        
        (define movesRequired
          (lambda (source dest)
            ; how many moves does it take to get from
            ; 'source' to 'dest'?
            (vector-ref
             (vector-ref
              (vector
               ; dest  1 2 3 4 5 6 7 8 9
               ;       - - - - - - - - -
               (vector 0 1 2 1 2 3 2 3 4) ; source 1 to others
               (vector 1 0 1 2 1 2 3 2 3) ; source 2 to others
               (vector 2 1 0 3 2 1 4 3 2) ; source 3 to others
               (vector 1 2 3 0 1 2 1 2 3) ; source 4 to others
               (vector 2 1 2 1 0 1 2 1 2) ; source 5 to others
               (vector 3 2 1 2 1 0 3 2 1) ; source 6 to others
               (vector 2 3 4 1 2 3 0 1 2) ; source 7 to others
               (vector 3 2 3 2 1 2 1 0 1) ; source 8 to others
               (vector 4 3 2 3 2 1 2 1 0) ; source 9 to others
               )
              (- source 1)) ; vector indicies start at 0!
             (- dest 1))))  ; vector indicies start at 0!
        
        ; accumulate for each of tiles 1 thru to 8
        (do ((tile 1 (+ 1 tile))  ; start & increment
             (moves 0))
            ((> tile 8) moves)    ; termination and val. returned
            (let ((source (FindFirstSlotOfSymbol tile aState))
                  (dest (FindFirstSlotOfSymbol tile goalState)))
              (set! moves
                    ; Manhattan dist. is negative (!) of sum
                    (- moves (movesRequired source dest)))))))
    
    ; print state in sets of three
    
    (define 8-printState
      (lambda (aState)
        
        (define printIn3s
          (lambda (aList)
            (if (null? aList)
                #f
                (begin (display (list (car aList)
                                      (cadr aList)
                                      (caddr aList)))
                       (printIn3s (cdddr aList))))))
        
        (display "[")
        (printIn3s aState)
        (display "]")))
    
    ; set up the search problem
    
    (set! 8-SearchProblem
          (MakeSearchProblem
           initialState
           goalFN
           evalFN
           (MakeActionList
            (makeMove '(#f #f #f 1 2 3 4 5 6)) ; up
            (makeMove '(4 5 6 7 8 9 #f #f #f)) ; down
            (makeMove '(#f 1 2 #f 4 5 #f 7 8)) ; left
            (makeMove '(2 3 #f 5 6 #f 8 9 #f)) ; right
            )))
    
    (SetPrintState! 8-SearchProblem 8-printState)))

(begin (SetUp8Puzzle)
       #t)
