; ********** Noughts and crosses ************
; Simple test of game playing toolbox

; Represent the board as a list of the contents of each
; square in following order with 'x', 'o' or 'e' in each
; position:
;            1 | 2 | 3
;           ---+---+---
;            4 | 5 | 6
;           ---+---+---
;            7 | 8 | 9
;
; The machine is always 'x'

(define X+OGameProblem #f)

(define SetUpX+O
  (lambda ()
    
    ; possible lines of three are as follows:
    (define lines '( (1 2 3)
                     (4 5 6)
                     (7 8 9)
                     (1 4 7)
                     (2 5 8)
                     (3 6 9)
                     (1 5 9)
                     (3 5 7)))
    
    (define evalFN
      (lambda (aState machineMove?)
        (let ((score 0))
          
          (define scoreOf
            (lambda (sym val)
              ; +ve for 'x , -ve for 'o, and zero for 'e
              (cond ((eq? sym 'x) val)
		    ((eq? sym 'o) (- 0 val))
		    (else 0))))
          
          ; encourage moving to corners and center
          (do ((posList aState (cdr posList))
               (valueList '(5 0 5 0 5 0 5 0 5)
                          (cdr valueList)))
              ((null? posList))
              (set! score (+ score
                             (scoreOf (car posList)
                                      (car valueList)))))
          ; encourage getting two in row with other empty
          (for-each (lambda (aLine)
                      ; look for two in line with other
                      ; empty
                      (let
                        ((x1 (list-ref aState
                                       (- (car aLine) 1)))
                         (x2 (list-ref aState
                                       (- (cadr aLine) 1)))
                         (x3 (list-ref aState
                                       (- (caddr aLine) 1))))
                        (set! score
                              (+ score
                                 (cond ((and (eq? x1 x2)
                                             (eq? 'e x3))
                                        (scoreOf x1 50))
                                       ((and (eq? x1 x3)
                                             (eq? 'e x2))
                                        (scoreOf x1 50))
                                       ((and (eq? x2 x3)
                                             (eq? 'e x1))
                                        (scoreOf x2 50))
                                       (else 0))))))
                    lines)
          score)))
          
          
    (define status
      (lambda (aState machineMove?)
        ; Look for three in a row (won position) or all
        ; squares occupied (drawn position).
        
        (define allSame
          (lambda (aLine)
            ; return 'x or 'o if all same on a line else #f
            (let ((x1 (list-ref aState (- (car aLine) 1)))
                  (x2 (list-ref aState (- (cadr aLine) 1)))
                  (x3 (list-ref aState (- (caddr aLine) 1))))
              (if (and (eq? x1 x2)
                       (eq? x2 x3)
                       (not (eq? x1 'e)))
                  x1
                  #f))))
        
        (do ((winner #f)
             (restLines lines (cdr restLines)))
        ((or winner
             (null? restLines))
         (if winner
             winner
             (do ((list aState (cdr list))
                  (numEs 0))
                 ((null? list) (if (zero? numEs)
                                   "draw" #f))
                 (if (eq? 'e (car list))
                     (set! numEs (+ numEs 1))
                     #f))))
         (let ((sym (allSame (car restLines))))
	   (cond ((eq? sym 'x) (set! winner "machine"))
		 ((eq? sym 'o) (set! winner "opponent"))
		 (else #f))))))
    
    (define moveTo
      (lambda (aPosition XorO)
        ; Return a function that attempts to move to
        ; 'aPosition' with a 'x' or 'o' depending on the
        ; value of XorO.
        
        (define list-set
          (lambda (list k obj)
            ; Return list with k'th object in list replaced
            ; by obj
            (if (= 0 k)
                (cons obj (cdr list))
                (cons (car list)
                      (list-set (cdr list) (- k 1) obj)))))
        
        (lambda (aState)
          (let ((curr (list-ref aState (- aPosition 1))))
            (if (not (eq? curr 'e))
                #f ; can't move into non empty position
                (list-set aState (- aPosition 1) XorO))))))
    
    (define getMove
      (lambda (aState)
        (if (not (memq 'e aState))
            aState ; no moves left so pass
            
            (do ((pos -1))
                ((and (>= pos 0)
                      (<= pos 9)
                      (eq? 'e (list-ref aState (- pos 1))))
                 ((moveTo pos 'o) aState))
                (newline)
                (display "Give postion 1-9: ")
                (newline)
                (set! pos (read))))))
    
    (define printX+O-MacVersion
      (lambda (aState)
        ; Use MacScheme graphics is available. Use must
        ; ensure that the grpahics heap is used and the
        ; graphics window is open by typing:
        ; (start-graphics 'half)
        
        (define paintSym
          (lambda (place asym apos)
            (let ((x (car apos))
                  (y (cadr apos)))
              (cond ((eq? asym 'o)
                     (paint-circle (+ x 20) (+ y 20) 10))
                    ((eq? asym 'x)
                     (paint-rect (+ x 10) (+ y 10)
                                          (+ x 30)
                                          (+ y 30)))
                    ((eq? asym 'e)
                     (move-to (+ x 20) (+ y 20))
                     (draw-char (number->string place)))))))
        
        (define paintBoard
          (lambda (place symList posList)
            (if (null? symList)
                #f
                (begin
                 (paintSym place (car symList)
                                 (car posList))
                 (paintBoard (+ place 1) (cdr symList)
                                         (cdr posList))))))
        
        (clear-graphics)
        (move-to 40 0) (line-to 40 120)
        (move-to 80 0) (line-to 80 120)
        (move-to 0 40) (line-to 120 40)
        (move-to 0 80) (line-to 120 80)
        (paintBoard 1 aState '( (0   0) (40  0) (80  0)
                                (0  40) (40 40) (80 40)
                                (0  80) (40 80) (80 80)))))
    
    (define printX+O
      (lambda (aState)
        
        (define displaysym
          (lambda (asym)
            (if (eq? asym 'e)
                (display ".")
                (display asym))))
        
        (if (null? aState)
            (newline)
            (begin (newline)
                   (displaysym (car aState))
                   (displaysym (cadr aState))
                   (displaysym (caddr aState))
                   (printX+O (cdddr aState))))))
    
    (set! X+OGameProblem (MakeGameProblem
                          '(e e e  e e e  e e e)
                          status
                          evalFN
                          (MakeActionList (moveTo 1 'x)
                                          (moveTo 2 'x)
                                          (moveTo 3 'x)
                                          (moveTo 4 'x)
                                          (moveTo 5 'x)
                                          (moveTo 6 'x)
                                          (moveTo 7 'x)
                                          (moveTo 8 'x)
                                          (moveTo 9 'x))
                          (MakeActionList (moveTo 1 'o)
                                          (moveTo 2 'o)
                                          (moveTo 3 'o)
                                          (moveTo 4 'o)
                                          (moveTo 5 'o)
                                          (moveTo 6 'o)
                                          (moveTo 7 'o)
                                          (moveTo 8 'o)
                                          (moveTo 9 'o))))
    
    (SetPrintState! X+OGameProblem
                    (if (and AllowGraphics
                             (string=? *SchemeVersion*
                                       "MacScheme"))
                        printX+O-MacVersion
                        printX+O))
    
    (SetGetAMove! X+OGameProblem getMove)))
    
(begin (SetUpX+O)
       (newline)
       (display "Type (PlayGame X+OGameProblem) to play")
       (display " noughts and crosses")
       (newline))
