; ****** D O D G E M ********
;
; This game, invented by Colin Vout, is played on a 3x3
; board, two white counters and two black counters. The
; initial state of the board where B represents a black
; counter and W a white counter is:
;
;          white off here
;          +---+---+---+
;          | B |   |   |
;   |      +---+---+---+ black
;   B -    | B |   |   | off
;   |      +---+---+---+ here
;          |   | W | W |
;          +---+---+---+
;                |
;               -W-
;
; The black counters can move one position up, down, or
; right while the white counters can move one position left,
; right or up. The aim is to move both your counters off the
; board, black to the right and white off the top. Only one
; counter is permitted on each square and you lose if you
; prevent your opponent from moving.

(define DodgemPerfectGameProblem #f)

(define SetUpDodgemPerfect
  (lambda ()
    ; Represent the board as a list of four values, the
    ; position of the two black counters and then the two
    ; white counters. The positions are numbered as follows
    ; will 0 representing a counter moved off the board.
    ; Board positions:
    ;          +---+---+---+
    ;          | 1 | 2 | 3 |
    ;          +---+---+---+
    ;          | 4 | 5 | 6 |
    ;          +---+---+---+
    ;          | 7 | 8 | 9 |
    ;          +---+---+---+
    ;
    ; Thus the initial state is represented by:
    (define initial '(1 4 8 9))
    
    (define Black1 0)
    (define Black2 1)
    (define White1 2)
    (define White2 3)
    
    (define getPos
      ; Return the square occupied by counter 'which'
      (lambda (aState which)
        (list-ref aState which)))
    
    (define setPos
      (lambda (aState which pos)
        ; Return state with position of counter 'which'
        ; replaced by 'pos'. If this results in two counters
        ; in same position then return #f
        
        (define setList
          (lambda (list k obj)
            (cond ((null? list) #f)
                  ((zero? k) (cons obj (cdr list)))
                  (else (cons (car list)
                              (setList (cdr list)
                                       (- k 1) obj))))))
        
        (define recurrs
          (lambda (list)
            ; does the first in list recurr latter in list?
            ; allow zero (off board) to recurr
            (and (positive? (car list))
                 (memq (car list) (cdr list)))))
        
        (let ((newState (setList aState which pos)))
          (cond ((null? newState) #f)
                ((recurrs (list-tail newState Black1)) #f)
                ((recurrs (list-tail newState Black2)) #f)
                ((recurrs (list-tail newState White1)) #f)
                (else newState)))))
    
    
    (define blackPos?
      (lambda (position)
        (<= position Black2)))
    
    (define whitePos?
      (lambda (position)
        (>= position White1)))
    
    (define offBoard? zero?)
    
    (define evalFN
      (lambda (aState machineMove?)
        ; exact table look-up of winn/loss positions table entries are:
        ;  + win for black,
        ;  - win for white,
        ;  0 win for first mover
        ;  *  won for second mover
        ;  x  illegal board position
        ; return +1 for won position and -1 for lost position
        
        (define winloss
          (vector
           "---------------------xxx---------xxx----xx00x" ;03
           "---------xxx-xx--------------x--x--x--x---000" ;02
           "-xx--x--x--x-----x--x--x--x---------------000" ;01
           "---------xxx-xx------xxx-----x--xxxx00x0xx++x" ;23
           "-xx--x--x--x-----x--xxxx--x------xxx00-0xx++x" ;13
           "-xx--x--xxxx-xx--x--x--x--x--x--x--x00x000+++" ;12
           "------------------xxx--0-----0xxx00*0-0x+x+x+" ;06
           "------xxx---x-x-----------0-x0-x0-x00x+000+++" ;05
           "x-x-x--x--x-----x0-x0-x0-x*---------000000+++" ;04
           "------------------xxxxxx0-00-0xxxxxx+0*xxx+xx" ;36
           "------xxx---x-x------xxx0--0x-0x-xxx+x0+xx++x" ;35
           "x-x-x--x--x-----x--x-xxx0x-00-00-xxx++0+xx++x" ;34
           "---------xxx-xx---xxx00-0--0-xxxx++x+0xx+x+x+" ;26
           "------xxxxxxxxx---------0--0xx0xx0xx+xx++++++" ;25
           "x-x-x--x-xxx-xx-x--x--x-0x000x00x00x++x++++++" ;23
           "-xx--x--x--x-00--xxxx00x0-x0-0xxx++0+00x+x+x+" ;16
           "-xx--xxxx--xx0x--x--x--x0-x0x+0x+0x++x+++++++" ;15
           "xxx-xx-xx-xx----xx-xx-xx0xx000000000+++++++++" ;14
           "-------00--0-00xxx0++00+0-0xxx0++++0*0+xx+x++" ;09
           "--0xxx0++-00xx0-00-00-00-0+x++x++x++x++++++++" ;08
           "xx+x-0x+0x+0---x++x++x++x+0000000000+++++++++" ;07
           "-------0----0-0xxx0+0xxx-0-xxx+++xxx0+0xxxx+x" ;39
           "---xxx------xx--0--0-xxx-00x+0x+0xxxx+++xx++x" ;38
           "xx-x0-x0-x--0--x+0x+0xxxx+0++0++0xxx++++xx++x" ;37
           "-------0-xxx0xxxxx0+00+0-0-xxx++x++x0+xxx+x++" ;29
           "---xxx---xxxxxx-0--0--0--0-x+xx+xx+xx+x++++++" ;28
           "xx-x0-x0-xxx0xxx+0x+0x+0x+0++x++x++x++x++++++" ;27
           "-xx--x-0x00x0++xxx0+x0+x-0xxxx++++++0++xx+x++" ;19
           "-xxxxx--x-0xxx+-0x-0x-0x-0xx++x++x++x++++++++" ;18
           "xxxx0xx0xx+x000x+xx+xx+xx+x++++++++++++++++++" ;17
           "------xxx00-x0x0--xxx++0+00+x0xxx+x++x+x+x+x+" ;56
           "x-x-x--x-0x0-000x-xxx+x++x0+00xxx++++++x+x+x+" ;46
           "x-x-x0xxx0x*x+x0x00x00x0+x++x++x++x++x+++++++" ;45
           "----0-0-0000*0+xxxxxx+++000xxxxxx++++++xxxxx+" ;69
           "---xxx---000xx0-00xxx+++-0+x++xxxx++x++x+x+x+" ;68
           "xx-x+0x0-x00000x+0xxxx++x+++++xxx++++++x+x+x+" ;67
           "-00--0xxx++0x+xxxx++++++0-0xxx+x++x++x+xx+x++" ;59
           "--0xxxxxx0++xxx0+00+00+0--0xx+xx+xx+xx+++++++" ;58
           "xx0x+0xxxx+0x+xx++x++x++x0++x++x++x++x+++++++" ;57
           "x-x-x00x00x+0++xxx+x++x+0x+xxx+++++++++xx+x++" ;49
           "x-xxxx-x+-x0xx+0x+0x+0x++x+x++x++x++x++++++++" ;48
           "xxxxx+xx+xx++++xx+xx+xx+xx+++++++++++++++++++" ;47
           "-00xxx000+++xx+xxx++++++-0+xxxx++x++x++xx+x++" ;89
           "xx0x++x+0x+++++xxxx++x++x++xxx+++++++++xx+x++" ;79
           "xx+xxxx++x++xx+x++x++x++x++x++x++x++x++++++++" ;78
           ))
        
        (define findString
          (lambda (aState)
            ; given white positions return correct row of table
            (let ((w1 (min (getPos aState White1) (getPos aState White2)))
                  (w2 (max (getPos aState White1) (getPos aState White2))))
              (vector-ref
               winloss
               (cdr (assoc (list w1 w2)
                           (list (cons '(0 3) 0)  (cons '(0 2) 1)
                            (cons '(0 1) 2)  (cons '(2 3) 3)
                            (cons '(1 3) 4)  (cons '(1 2) 5)
                            (cons '(0 6) 6)  (cons '(0 5) 7)
                            (cons '(0 4) 8)  (cons '(3 6) 9)
                            (cons '(3 5) 10) (cons '(3 4) 11)
                            (cons '(2 6) 12) (cons '(2 5) 13)
                            (cons '(2 4) 14) (cons '(1 6) 15)
                            (cons '(1 5) 16) (cons '(1 4) 17)
                            (cons '(0 9) 18) (cons '(0 8) 19)
                            (cons '(0 7) 20) (cons '(3 9) 21)
                            (cons '(3 8) 22) (cons '(3 7) 23)
                            (cons '(2 9) 24) (cons '(2 8) 25)
                            (cons '(2 7) 26) (cons '(1 9) 27)
                            (cons '(1 8) 28) (cons '(1 7) 29)
                            (cons '(5 6) 30) (cons '(4 6) 31)
                            (cons '(4 5) 32) (cons '(6 9) 33)
                            (cons '(6 8) 34) (cons '(6 7) 35)
                            (cons '(5 9) 36) (cons '(5 8) 37)
                            (cons '(5 7) 38) (cons '(4 9) 39)
                            (cons '(4 8) 40) (cons '(4 7) 41)
                            (cons '(8 9) 42) (cons '(7 9) 43)
                            (cons '(7 8) 44))))))))
        
        (define findIndex
          (lambda (aState)
            ; find position in vector of black counters
            (let ((b1 (min (getPos aState Black1) (getPos aState Black2)))
                  (b2 (max (getPos aState Black1) (getPos aState Black2))))
              (cadr (assoc (list b1 b2)
                           '(((4 7) 0)  ((1 7) 1)  ((1 4) 2)  ((7 8) 3)
                             ((4 8) 4)  ((1 8) 5)  ((5 7) 6)  ((4 5) 7)
                             ((1 5) 8)  ((2 7) 9)  ((2 4) 10) ((1 2) 11)
                             ((5 8) 12) ((2 8) 13) ((2 5) 14) ((7 9) 15)
                             ((4 9) 16) ((1 9) 17) ((6 7) 18) ((4 6) 19)
                             ((1 6) 20) ((3 7) 21) ((3 4) 22) ((1 3) 23)
                             ((0 7) 24) ((0 4) 25) ((0 1) 26) ((8 9) 27)
                             ((5 9) 28) ((2 9) 29) ((6 8) 30) ((5 6) 31)
                             ((2 6) 32) ((3 8) 33) ((3 5) 34) ((2 3) 35)
                             ((0 8) 36) ((0 5) 37) ((0 2) 38) ((6 9) 39)
                             ((3 9) 40) ((3 6) 41) ((0 9) 42) ((0 6) 43)
                             ((0 3) 44))))))) 
        
        (let ((entry (string-ref (findString aState) (findIndex aState))))
          (cond ((char=? entry #\+) +1)
                ((char=? entry #\-) -1)
                ((char=? entry #\0) (if machineMove? +1 -1))
                ((char=? entry #\*) (if machineMove? -1 +1))
                (else (Fatal-Error "illegal board state"))))))
    
    ; Moves from:   0  1  2  3  4  5  6  7  8  9
    (define left  '(#f #f 1  2  #f 4  5  #f 7  8))
    (define right '(#f 2  3  b  5  6  b  8  9  b))
    (define up    '(#f w  w  w  1  2  3  4  5  6))
    (define down  '(#f 4  5  6  7  8  9  #f #f #f))
    
    (define moveCounter
      (lambda (which moves)
        ; Return a function which moves the counter in
        ; 'which' position of the state according to the
        ; "coded instructions" in 'moves'. It is assumed
        ; that the machine is always black. The moves
        ; consist of a list accessed by current position
        ; giving the new position if legal, #f if not, and
        ; 'b' or 'w' if it is only legal for a black or
        ; white counter. It is necessary to check the new
        ; position is free.
        (lambda (aState)
          (let* ((currPos (getPos aState which))
                 (newPos  (list-ref moves currPos))
                 (newState #f))
            (cond ((null? newPos) #f)
                  ((eq? newPos 'b)
                   (if (blackPos? which)
                       (setPos aState which 0)
                       #f))
                  ((eq? newPos 'w)
                   (if (blackPos? which)
                       #f
                       (setPos aState which 0)))
                  (else (setPos aState which newPos)))))))
    
    (define b1u (moveCounter Black1 up))
    (define b1r (moveCounter Black1 right))
    (define b1d (moveCounter Black1 down))
    (define b2u (moveCounter Black2 up))
    (define b2r (moveCounter Black2 right))
    (define b2d (moveCounter Black2 down))
    (define w1l (moveCounter White1 left))
    (define w1u (moveCounter White1 up))
    (define w1r (moveCounter White1 right))
    (define w2l (moveCounter White2 left))
    (define w2u (moveCounter White2 up))
    (define w2r (moveCounter White2 right))
  
    (define specialStatus
      (lambda (aState machineMove?)
        
        (define noBlackMove?
          (lambda (aState)
            (not (or (b1u aState)
                     (b1r aState)
                     (b1d aState)
                     (b2u aState)
                     (b2r aState)
                     (b2d aState)))))
        
        (define noWhiteMove?
          (lambda (aState)
            (not (or (w1l aState)
                     (w1u aState)
                     (w1r aState)
                     (w2l aState)
                     (w2u aState)
                     (w2r aState)))))
        
        (cond ((and (offBoard? (getPos aState Black1))
                    (offBoard? (getPos aState Black2)))
               "machine")
              ((and (offBoard? (getPos aState White1))
                    (offBoard? (getPos aState White2)))
               "opponent")
              ((and machineMove?
                    (noBlackMove? aState)) "machine")
              ((and (not machineMove?)
                    (noWhiteMove? aState)) "opponent")
              (else #f))))
    
    (define printDodgem
      (lambda (aState)
        ; Build a vector and then print in rows of three
        
        (define in3
          (lambda (list)
            (if (null? list)
                (newline)
                (begin (newline)
                       (display (car list))
                       (display (cadr list))
                       (display (caddr list))
                       (in3 (cdddr list))))))
        
        (if (null? aState)
            (display "Invalid board state")
            (begin
             (let ((board (make-vector 10 ".")))
               (vector-set! board (getPos aState Black1) "b")
               (vector-set! board (getPos aState Black2) "b")
               (vector-set! board (getPos aState White1) "w")
               (vector-set! board (getPos aState White2) "w")
               (in3 (cdr (vector->list board)))
               (newline))))))
    
    (define printDodgem-MacVersion
      (lambda (aState)
        ; Use MacScheme graphics is available. Use must
        ; ensure that the grpahics heap is used and the
        ; graphics window is open by typing:
        ; (start-graphics 'half)
        
        (define paintSym
          (lambda (place asym apos)
            (let ((x (car apos))
                  (y (cadr apos)))
              (cond ((eq? asym #\b)
                     (paint-circle (+ x 20) (+ y 20) 10))
                    ((eq? asym #\w)
                     (frame-circle (+ x 20) (+ y 20) 10))
                    ((eq? asym #\.)
                     (move-to (+ x 20) (+ y 20))
                     (draw-char (number->string place)))))))
        
        (define paintBoard
          (lambda (place symList posList)
            (if (null? symList)
                #f
                (begin (paintSym place (car symList)
                                       (car posList))
                       (paintBoard (+ place 1)
                                   (cdr symList)
                                   (cdr posList))))))
        
        (clear-graphics)
        (move-to 40 0) (line-to 40 120)
        (move-to 80 0) (line-to 80 120)
        (move-to 0 40) (line-to 120 40)
        (move-to 0 80) (line-to 120 80)
        (if (null? aState)
            (display "Invalid board state")
            (begin
             (let ((board (make-vector 10 #\.)))
               (vector-set! board (getPos aState Black1) #\b)
               (vector-set! board (getPos aState Black2) #\b)
               (vector-set! board (getPos aState White1) #\w)
               (vector-set! board (getPos aState White2) #\w)
               (paintBoard 1
                           (cdr (vector->list board))
                           '( (0   0) (40  0) (80  0)
                              (0  40) (40 40) (80 40)
                              (0  80) (40 80) (80 80))))))))
    
    (define getMove
      (lambda (aState)
        (let* ((which
                (do ((ok #f)
                     (pos 0))
                    (ok pos)
                    (newline)
                    (display "Give postion (1-9): ")
                    (newline)
                    (set! pos (read))
                    (cond ((not (number? pos))
                           (display "invalid number"))
                          ((or (< pos 0)
                               (> pos 9))
                           (display "invalid position "))
                          ((= pos (getPos aState White1))
                           (set! pos White1)
                           (set! ok #t))
                          ((= pos (getPos aState White2))
                           (set! pos White2)
                           (set! ok #t))
                          (else
                           (display "Not a white counter")))))
               (direction
                (do ((ok #f)
                     (dir #f))
                    (ok dir)
                    (newline)
                    (display "Give direction ")
                    (display "(u(p), (l)eft,  (r)ight): ")
                    (newline)
                    (set! dir (read))
                    (set! ok #t)
                    (cond
                     ((eq? dir 'u)
                      (set! dir up))
                     ((eq? dir 'l)
                      (set! dir left))
                     ((eq? dir 'r)
                      (set! dir right))
                     (else
                      (display "Invalid direction! ")
                      (set! ok #f))))))
          ((moveCounter which direction) aState))))
    
    (set! DodgemPerfectGameProblem
      (MakeGameProblem initial
                       specialStatus
                       evalFN
                       (MakeActionList
                        b1u b1r b1d b2u b2r b2d)
                       (MakeActionList
                        w1l w1u w1r w2l w2u w2r)))
    
    (SetPrintState! DodgemPerfectGameProblem
                    (if (and AllowGraphics
                             (string=? *SchemeVersion*
                                       "MacScheme"))
                        printDodgem-MacVersion
                        printDodgem))
    
    (SetGetAMove! DodgemPerfectGameProblem getMove)))

(begin (SetUpDodgemPerfect)
       (newline)
       (display "Type (PlayGame DodgemPerfectGameProblem) to")
       (display " play Dodgem")
       (newline))
