
% OL(P): Object Layer for Prolog -- system objects (project-oriented)
% Version 1.1 for SICStus Prolog and QUINTUS Prolog
% (C) 1993 Markus P.J. Fromherz.  All Rights Reserved.
% (C) 1993 Xerox Corporation.     All Rights Reserved.


% project object project

object project.

   :- publish create/2, switch/1, rename/1,
              add_files/1, remove_files/1,
              add_library/1, remove_library/1,
              do/1, do/2, do/3, do/4,
              info/0, info/1, info/2, info/3, external_info/1.

   create(Project, Files) :-
      ol_new_project(Project, Files).

   rename(NewProject) :-
      ol_rename_project(NewProject).

   add_files(Files) :-
      ol_add_project_files(Files).

   remove_files(Files) :-
      ol_remove_project_files(Files).

   add_library(Project) :-
      ol_add_library(Project).

   remove_library(Project) :-
      ol_remove_library(Project).

   switch(Project) :-
      ol_switch_project(Project).

   do(Operation) :-
      this::do(Operation, _).

   do(Operation, WhichFiles) :-
      ( WhichFiles == all ; WhichFiles == new ),
      atom(Operation),
      ol_project(Project),
      ol_do_project(Operation, Project, WhichFiles, _, _).

   do(PO, Extension) :-
      ( var(Extension) ; Extension = [_|_] ),
      !,
      ( atom(PO) -> ol_project(Project), Operation = PO
                  ; PO =.. [Operation,Project] ),
      ol_do_project(Operation, Project, all, _, Extension).

   do(Operation, WhichFiles, BaseLanguage) :-
      this::do(Operation, WhichFiles, BaseLanguage, _).

   do(PO, WhichFiles, BaseLanguage, Extension) :-
      ( atom(PO) -> ol_project(Project), Operation = PO
                  ; PO =.. [Operation,Project] ),
      ol_do_project(Operation, Project, WhichFiles, BaseLanguage, Extension).

   info :-
      this::(info(project),info(library)).

   info(project) :-
      ol_project(Project),
      this::print_project_files("Current project", Project).
   info(library) :-
      \+ ( ol_project_files(Project, Files),
           \+ ol_project(Project),
           \+ this::print_project_files("Library", Project) ).

   print_project_files(Text, Project) :-
      ol_project_path(Project, PathString, ""),
      ( PathString == "" ->
           format("~s ~w has files:~n", [Text,Project])
         ;
           format("~s ~w in ~s has files:~n", [Text,Project,PathString])
      ),
      ol_project_files(Project, Files),
      \+ ( ol_member(File, Files),
           findall(Object, ol_object(Object,Project,File), Objects),
           \+ format("   ~q  ~q~n", [File,Objects]) ).

   info(project, [Project,Path]) :-
      ol_project(Project),
      ol_project_path(Project, PathString, ""),
      name(Path, PathString).
   info(library, [Project,Path]) :-
      ol_project_files(Project, _),
      \+ ol_project(Project),
      ol_project_path(Project, PathString, ""),
      name(Path, PathString).
   info(files, Files) :-
      ol_project(Project),
      ol_project_files(Project, Files).
   info(files(Project), Files) :-
      ol_project_files(Project, Files).
   info((object), [Object,File]) :-
      ol_project(Project),
      ol_object(Object, Project, File).
   info(object(Project), [Object,File]) :-
      ol_object(Object, Project, File).
   info(method, [Method,Object,File]) :-
      ol_project(Project),
      ol_info(Object, def, Method),
      ol_object(Object, Project, File).
   info(method(Project), [Method,Object,File]) :-
      ol_info(Object, def, Method),
      ol_object(Object, Project, File).

   info(text, Text, Options) :-
      ol_project(Project),
      ol_project_files(Project, Files),
      this::files_string(Files, Project, FilesString),
      ol_append([32,34|Text], [34,32|FilesString], CS1),    % " 
      ( Options == "" ->
           CS2 = CS1
         ;
           ol_append([32,45|Options], CS1, CS2)    % -
      ),
      ol_append("grep", CS2, CommandString),
      name(Command, CommandString),
      unix(shell(Command)).

   files_string([], _, []).
   files_string([File|Files], Project, FilesString) :-
      this::files_string(Files, Project, AccFilesString),
      ol_project_path(Project, PFString, FileString),
      ol_path_file(File, PFString, FileString, PathFile),
      % name(PathFile, FileString1),
      ol_append(PFString, ".ol", FileString2),
      ol_append(FileString2, [32|AccFilesString], FilesString).

   external_info(Project) :-
      ol_append_suffix(Project, ".pl", InFile),
      ol_path_project(Project, ProjectName, _, _),
      open(InFile, read, InStream),
      repeat,
        read(InStream, Term),
        ( Term == end_of_file
         ;
          ( Term = ol_project_files(ProjectName,Files) ->
               format("Project ~w has files:~n   ~q~nwith object~n  ", [ProjectName,Files]) ;
            Term = ol_project_files(Library,Files) ->
               format("~nLibrary ~w has files:~n   ~q~nwith objects:~n  ", [Library,Files]) ;
            Term = ol_object(O,_,_) ->
               format(" ~q", [O])
          ),
          fail
        ),
      !,
      nl,
      close(InStream).


   :- publish compile/0, compile/1, compile/2, optimize/0.

   compile :-
      this::info(files, Files),
      this::compile(Files).

   compile(Files) :-
      this::compile(Files, [verbose]).

   compile(Files, Options) :-
      ol_compile_project(Project, Files, Options).

   % does not work in the current configuration (objects have to be dynamic)
   optimize :-
      retract(ol_state(O,compiled)),
      assert(ol_state(O,interface)),
      ol_object_trafo(interface, O),
      !,
      this::optimize.
   optimize.


   :- publish help/0, help/1, portray/0, noportray/0.

   portray :-
      retractall(ol_no_portray).

   noportray :-
      retractall(ol_no_portray), assert(ol_no_portray).

   help :-
      format("
The following methods are available in object `project'.
Use `project::help(Name)' to get specific information.
   create          -- create a new project
   rename          -- rename a project
   switch          -- switch to an existing project
   add_files       -- add files to the current project
   remove_files    -- remove files from the current project
   add_library     -- add a library to the current project
   remove_library  -- remove a library from the current project
   compile         -- compile files of the current project
   do              -- perform some operation on the project's files
   info            -- get information about the current project
   external_info   -- get information about another project/library
   portray         -- switch pretty-printing on and off
", []).

   help(create) :- format("
create(+Project, +Files)

   `Project' is an atom (project name, possibly with path); `Files' is a list
   of atoms (OL(P) file names without extension `.ol').

   Creates a new project that consists of these files. All the files must be
   in the directory indicated by the path of `Project' (the current directory
   if no path is given). File names are expected to have one extension indica-
   ting the base language; `.ol' is automatically appended.
   A file `Project.pl' is generated, holding the project's information. The
   idea is that one program consists of a set of files, and that certain
   operations can or have to be performed on all files (and their objects).
", []).
   help(rename) :- format("
rename(+Project)

   `Project' is an atom (new project name).

   Renames the current project to `Project'. The old project file is deleted.
", []).
   help(switch) :- format("
switch(+Project)

   `Project' is an atom (project name, possibly with path).

   Switches to another project by removing the current project's information
   and loading the information of that project.
", []).
   help(add_files) :- format("
add_files+(Files)

   `Files' is a list of atoms (OL(P) file names without extension `.ol').

   Extends the current project to include these files. The files must be in
   the same directory as the project's other files.
", []).
   help(remove_files) :- format("
remove_files(+Files)

   `Files' is a list of atoms (OL(P) file names without extension `.ol').

   Removes these files from the list of files that make up the current
   project. The files' object information is also removed.
", []).
   help(add_library) :- format("
add_library(+Project)

   `Project' is an atom (project name, possibly with path).

   Adds the project as library to the current project. The project's files
   are added to the current program, but most files operations such as
   compilation are not performed on these files. The idea is that this
   project has been set up and tested as a library of objects, and is not
   changed within the current project.
   The added project itself can include libraries (i.e. other projects).
   Note that it makes a difference whether you add a library with absolute
   or relative path. Using a path relative to the project's directory makes
   the project portable. However, if you set up a library which includes
   other libraries, the paths to those libraries should be absolute, or
   they won't be found if you add the library to another project.
", []).
   help(remove_library) :- format("
remove_library(+Project)

   `Project' is an atom (project name).

   Removes this library from the current project. The library's object
   information is also removed. (Note that libraries loaded together with
   this library are not removed.)
", []).
   help(compile) :- format("
compile
compile(Files)
compile(Files, Options)

   `Files' is a list of atoms (file names), `Options' a list of atoms.

   Compiles all files of the current project, or the ones given in `Files',
   with options if given. For each file `File', a file `File.ol' must exist,
   which will be compiled to `File. Currently, the only available option is
   `verbose', which lists some information during compilation.
   `compile(Files)' is equivalent to `compile(Files,[verbose])'.
", []).
   help(do) :- format("
   `Operation' is an atom (name of a unary predicate); `Project' is an atom
   (project name, possibly with path) or a variable; `WhichFiles' is either
   `all' or `new'; `BaseLanguage' is a variable or an atom; `Extensions' is
   a variable or a list of atoms (extensions to be concatenated).

do(+Operation)
do(+Operation(Project))

   Switches to `Project' first if not current project or added library; then
   calls `Operation(File)' for all files of the project (project and all
   added libraries if `Project' is a variable, libraries first).
   `Operation' might be `consult', `fcompile', `load', `translate' etc.

do(+Operation, +WhichFiles)

   Calls `Operation(File)' for all files of the current project, if
   `WhichFiles' is `all', or for the changed files only, if `WhichFiles' is
   `new'. A file is marked as changed if it is compiled by OL(P); this mark
   is removed if the file is `consult'ed or `load'ed.

do(+Operation, ?Extension)
do(+Operation(Project), ?Extension)

   Switches to `Project' first if not current project or added library; then
   calls `Operation(File)' for all files of the project (all projects if
   `Project' is a variable). If `Extension' is a list, the elements of the
   list will be concatenated and replace the files' extensions.
   (E.g., with files ['a.pl','b.cc'], `project::do(load,[ql])' will call
   ""load('a.ql')"" and ""load('b.ql')"".)

do(+Operation, +WhichFiles, ?BaseLanguage)
do(+Operation, +WhichFiles, ?BaseLanguage, +Extension)
do(+Operation(Project), +WhichFiles, ?BaseLanguage)
do(+Operation(Project), +WhichFiles, ?BaseLanguage, +Extension)

   Switches to `Project' first if not current project or added library; then
   calls `Operation(File)' for all files of the current project that are
   defined in the base language, if `WhichFiles' is `all', or for the changed
   files only, if `WhichFiles' is `new' (cf. above; all projects if `Project'
   is a variable).
   (E.g., with files ['a.pl','b.cc'], `project::do(translate,all,cc)' will
   call ""translate('b.cc')"".)
   If `Extension' is given and a list, the elements of the list will be
   concatenated and replace the files' extensions (see above). `BaseLanguage'
   can also be a variable, and this variable can be used in `Extension'.
   (E.g., with files ['a.pl','b.cc'], `project::do(list,all,BL,[BL,'.ol])' will
   call ""list('a.pl.ol')"" and ""list('b.cc.ol')"".)
", []).
   help(info) :- format("
info

   Equivalent to `info(project)'.

info(+ProjectType)

   `ProjectType' is either `project' or `library'.

   Lists information about files and objects of the current project if
   `ProjectType' is `project', or of its libraries if `ProjectType' is
   `library'.

info(+Type, ?Information)

   `Type' is either `project', `library', `files', `files(Project)', `object',
   `object(Project)', `method', or `method(Project)' (`Project' an atom, name
   of a library, or a variable). `Information' will be instantiated to a list.

   Returns information depending on the requested type as follows:
     `project'  : `[Project,Path]' (of current project)
     `library'  : `[Project,Path]' (of added library; all on backtracking)
     `files'    : list of files of the current project
     `files(P)' : list of files of added library `P' (all projects if variable)
     `object'   : `[Object,File]' (of current project; all on backtracking)
     `object(P)': `[Object,File]' (of library `P' (all projects if variable);
                  all on backtracking)
     `method'   : `[Method,Object,File]' (of current project; all on backtr.)
     `method(P)': `[Method,Object,File]' (of `P' (all projects if variable);
                  all on backtracking)

info(+Type, +Argument, +Options)

   `Type' is `text', `Argument' a string, `Options' depends.

   Lists information depending on the requested type and argument as follows:
     `text', Text: lists files and lines of current project where Text occurs
                   (uses grep with Options (characters); useful Options are
                   c - number of lines only, h - no file names, i - ignore
                   case, l - file names only, n - with line number;
                   `Argument' can actually be any expression for grep)
", []).
   help(external_info) :- format("
external_info(+Project)

   `Project' is an atom (project name, possibly with path).

   Lists information about files and objects of this external project without
   switching to it.
", []).
   help(portray) :- format("
portray
noportray

   Switches pretty-printing of OL(P) terms (e.g., messages) on and off.
", []).

end_object project.


:- dynamic ol_trafo_interface_unfolded/2.

ol_object_trafo(interface, O) :-
   retractall(ol_trafo_interface_unfolded(_,_)),
   ol_interface_head(O, S, Def, Inst, M, IH),
   retract((IH:-H)),
   clause(H, B),
   assert(ol_trafo_interface_unfolded(IH, B)),
   fail.
ol_object_trafo(interface, _) :-
   retract(ol_trafo_interface_unfolded(Head, Body)),
   assert((Head:-Body)),
   fail.
ol_object_trafo(interface, _).
