/*
 * state.c
 * Routines for state information
 *
 * AUTHOR:  David Roch
 *
 * CONTRIBUTORS:
 *	DAR - David Roch
 *
 * HISTORY:
 *	7/21/88 - created DAR
 *	7/29/88 - New aliasing routines DAR
 *	8/02/88 - ACCESS flag and varcheck added DAR
 *	8/04/88 - Bug fixed in alias DAR
 */

/* header files */
#include "typedefs.h"
#include "pgm_typedefs.h"
#include "lattice.h"
#include "state.h"

/* macros */
#include "pgm_macros.c"
#include "gpmacros.c"
#include "macros.c"
#include "state_macros.c"

/* prototypes */
#include "protos/lattice.p"
#include "protos/callpat.p"

/* externals */
extern SLOT		ListFunc, NilAtom, *tuple;
extern Procedure	procedures[];
extern SHORT		QueryFunc;


/* globals */
SHORT	alias_to;	/* when processing any term, all nonground
			   variables are assumed to be aliased.
			   alias_to contains the first nonground variable that
			   is found during the processing of the term.
			   All subsequent variables are aliased to this one */



/* debugging */
/*#define HIBUG	1*/	/* High level debugging */
/*#define LOBUG	1*/	/* Low level debugging */

/* local symbols */
#define	END_OF_CHAIN	-1	/* used in union/find algorithm */
#define NO_VARS_YET	-1	/* used when determining aliasing */


/*
 * print_state(STATETBL *state)
 * Displays a state table on standard out
 */
void print_state(state)
     STATETBL *state;
{
  SHORT	i;

  if (state == NULL)
    printf("Null state -- no printout\n");
  else {
    printf("\t\t\tFlags\n");
    printf("Var\tType\t\t(octal)\tALIAS\n");
    for (i=0; i < state->vars; i++) {
      printf("%d\t", state->st[i].var);
      switch(state->st[i].type) {
      case EMPTY:
	printf("EMPTY\t\t");
	break;
      case GROUND:
	printf("GROUND\t\t");
	break;
      case PARTIAL:
	printf("PARTIAL\t\t");
	break;
      case DONTKNOW:
	printf("DONTKNOW\t");
	break;
      case FREEVAR:
	printf("FREEVAR\t\t");
	break;
      case FREE_SCALAR:
	printf("FREE_SCALAR\t");
	break;
      case FREE_GROUND:
	printf("FREE_GROUND\t");
	break;
      case PARTIAL_GROUND:
	printf("PARTIAL_GROUND\t");
	break;
      }
      printf("%o\t%d\n", state->st[i].flags, state->st[i].alias);
    }
  } /* end else */
} /* end print_state */

/* copy_statetbl(STATETBL *orig, SHORT num_vars)
 * Given a pointer to a state table and the maximum
 * number of variables that may be in the table,
 * returns a pointer to a newly created copy of that
 * state table.
 */
STATETBL *copy_statetbl(orig, num_vars)
     STATETBL	*orig;
     SHORT	num_vars;
{ 
  SHORT	i; 
  STATE	*st1, *st2;
  STATETBL *copy;
  
  Malloc_StateTBL(copy); 
  Malloc_State(copy->st, num_vars);
  copy->vars = orig->vars; 
  for (i=0, st1=copy->st, st2=orig->st; 
       i < orig->vars; i++, st1++, st2++) 
     *st1 = *st2;
  return copy;
} /* copy_statetbl */



/*
 * SHORT varlookup(SLOT *variable, STATETBL *state)
 * Search for variable in state table.  Return
 * the index of the variable.  If the variable
 * does not exist, create it and assume it is of
 * type FREE.  Also, the ACCESSED bit in the flags
 * field is set.
 * There should be no danger of table overflow as
 * the state table has been allocated for the total
 * number of variables in the clause.
 */

SHORT varlookup(variable, state)
     SLOT *variable;
     STATETBL *state;
{
  register SHORT	i;

  for (i=0;
       i < state->vars && state->st[i].var != *variable;
       i++);

  if (i < state->vars) {
    SET_FLAG(state->st[i].flags, ACCESSED);
    return i; /* variable exists, return index */
  }
  else {
    /* new variable */
    state->vars++;
    state->st[state->vars - 1].var = *variable;
    state->st[state->vars - 1].flags = 0;
    SET_FLAG(state->st[state->vars - 1].flags, ACCESSED);
    state->st[state->vars - 1].type = FREEVAR;	/* default */
    state->st[state->vars - 1].alias = END_OF_CHAIN; /* no alias */
    return (state->vars - 1);
  } /* end if i < state->vars */
} /* end varlookup */

/* void clear_access(STATETBL *state)
 * Given a state table, clears the access bit for every
 * entry.  Useful for when we wish to know which variables
 * have been used in a particular literal
 */
void clear_access(state)
     STATETBL	*state;
{
  SHORT		i;

  for (i=0; i <state->vars; i++)
    CLEAR_FLAG(state->st[i].flags, ACCESSED);
} /* end clear_access */

/*
 * SHORT varcheck(SLOT var, STATETBL *state)
 * Used for looking up variables in a state table
 * using the .var field of type STATE rather than
 * a pointer to a slot as in varlookup.
 * Unlike varlookup, varcheck does:
 *	NOT insert if the variable is not found.
 *	  instead, DOES_NOT_EXIST is returned
 *	NOT set the ACCESSED bit
 *
 * The primary purpose of this routine is to allow the
 * user to find identical variables in different state
 * tables.  Hence var is a SLOT rather than a pointer to
 * a SLOT.
 */
SHORT varcheck(var, state)
     SHORT	var;
     STATETBL	*state;
{
  register SHORT	i;

  for (i=0;
       i < state->vars && state->st[i].var != var;
       i++);

  if (i < state->vars) {
    return i; /* variable exists, return index */
  } else
    return DOES_NOT_EXIST;
} /* end varcheck */
  
/*
 * void alias(SHORT var1, var2, STATETBL *state)
 * Given the indices to two variables in a state table,
 * indicates that they may be aliased to one another.
 * Any variables previously or subsequently aliased
 * to one of these variables will also be aliased to
 * the other.
 * Uses the union/find algorithm to maintain sets
 */
void alias(var1, var2, state)
     SHORT	var1, var2;
     STATETBL	*state;
{
  SHORT		set1, set2;

  set1 = var1;	/* find root of var1 */
  while (state->st[set1].alias != END_OF_CHAIN)
    set1 = state->st[set1].alias;

  set2 = var2; /* find root of var2 */
  while (state->st[set2].alias != END_OF_CHAIN)
    set2 = state->st[set2].alias;

  if (set1 != set2) /* if not in same set, merge */
    state->st[set1].alias = set2;
  
} /* end alias */

/*
 * void remove_alias(SHORT var, STATETBL *state)
 * Given the index of a variable in the state table,
 * removes any aliases that may have been placed on
 * the variable.
 */
void remove_alias(var, state)
     SHORT	var;
     STATETBL	*state;
{
  SHORT		i, next;

  if (state->st[var].alias != END_OF_CHAIN)
    next = state->st[var].alias;	/* if not root of a set */
  else {
    i = 0;				/* if root, other members? */
    while (state->st[i].alias != var && i < state->vars)
      i++;
    if (i < state->vars) {	/* if there is an alias to this */
      state->st[i].alias = END_OF_CHAIN;   /* set first member to new root */
      next = i;
    }
    else
      next = END_OF_CHAIN;	/* no other members */
  } /* end if (state->st[var... */

  if (next != END_OF_CHAIN)	/* if set must be updated */
    for (i=0; i < state->vars; i++) {	/* remove all references to var */
      if (state->st[i].alias == var)
	state->st[i].alias = next;
    } /* end for (i=0... */
} /* end remove_alias() */


/*
 * SHORT alias_query(SHORT var1, var2, STATETBL *state)
 * Given the indices to two variables in a state table,
 * determine whether or not they might be aliased.
 * Returns non zero if aliasing is possible
 * Uses union/find algorithm with collapse
 */
SHORT alias_query(var1, var2, state)
     SHORT	var1, var2;
     STATETBL	*state;
{
  SHORT	set1, set2, tmp;

  set1 = var1;
  set2 = var2;

  /* find */
  while (state->st[set1].alias != END_OF_CHAIN)
    set1 = state->st[set1].alias;
  while (state->st[set2].alias != END_OF_CHAIN)
    set2 = state->st[set2].alias;

  /* collapse */
  while (state->st[var1].alias != END_OF_CHAIN) {
    tmp = var1;
    var1 = state->st[var1].alias;
    state->st[tmp].alias = set1;
  }
  while (state->st[var2].alias != END_OF_CHAIN) {
    tmp = var2;
    var2 = state->st[var2].alias;
    state->st[tmp].alias = set2;
  }
    
  return (set1 == set2);
} /* end alias_query */

/*
 * SHORT scan_functor(SLOT *functor, STATETBL *state)
 * Given a pointer to a functor and a state, determine
 * if the functor is GROUND.  If so, return GROUND,
 * else return PARTIAL indicating that the functor
 * is partially ground.
 */
SHORT scan_functor(functor, state)
     SLOT	*functor;
     STATETBL	*state;
{
  SHORT	i, var, term_result, result, arity;
  SLOT	*term;

  DEREFPTR(functor);
  result = GROUND;	/* assume it's ground */
  arity = ARITY(functor);
  for (i=0; i < arity; i++) {
    /* loop through all terms of the functor */
    term = ++functor;	/* move to next argument */
    DEREFPTR(term);
    if (TAG_IS_FUNCTOR(term)) {
      /* term is a functor, compute it's type */
      term_result = scan_functor(term, state);
      if (lub_callpat(GROUND, term_result) != GROUND)
	/* if the term is not fully ground, then it is partially bound */
	result = PARTIAL;
    } else 
      if (VARIABLE_PTR(term)) {
	/* term is a variable, is it ground? */
	var = varlookup(term, state);
	if (lub_callpat(GROUND, state->st[var].type) != GROUND) {
	  /* if not ground, then make sure it is aliased */
	  result = PARTIAL;
	  if (alias_to == NO_VARS_YET)	/* if first free var */
	    alias_to = var;
	  else
	    alias(alias_to, var, state);	/* set alias */
	} /* end if (lub_callpat(GROUND... */
	/* if it's ground, we do nothing */
      };
  } /* end for (i=0; i<ARITY... */
  return result;
} /* end scan_functor */


/*
 * void termmap(SLOT *head, CALSUCPR calsuc, STATETBL *state)
 * Given a call/success pair and a term, map the pair to
 * all variables in the term.  Results are stored in state.
 */
void termmap(head, calsuc, state)
     SLOT 	*head;
     CALSUCPR 	calsuc;
     STATETBL	*state;
{
  SHORT 	i, arity;
  SLOT		*next, *term;
  
  term = head;
  DEREFPTR(term);
# ifdef LOBUG
  printf("termmap:  Current term being mapped:  ");
  PrintTerm(term, tuple);
  OsPrint("\n");
# endif  
  
  /* check if functor of arity x where x>0 */
  arity = ARITY(term);
  if (TAG_IS_FUNCTOR(term) && arity > 0) {
    if (calsuc.call == PARTIAL)
      calsuc.call = DONTKNOW;
    for (i=0; i < arity; i++) {
      next = ++term;
      DEREFPTR(next);
      termmap(next, calsuc, state);
    } /* end for (i=0... */
  }
  else
    /* check if variable */
    if (VARIABLE_PTR(term)) {
      i = varlookup(term, state);
      state->st[i].type = lub_instantiation(calsuc.call, state->st[i].type);
      /* If the term contains any free variables, it might be aliased */
      if (lub_callpat(GROUND, state->st[i].type) != GROUND)
	if (alias_to == NO_VARS_YET)	/* if first free var */
	  alias_to = i;
	else
	  alias(alias_to, i, state);	/* set alias */
    }
} /* end termmap */
  

/*
 * STATETBL headmap(SLOT *head, SHORT num_vars)
 * Given a predicate head, examines the established call
 * patterns and computes a least upper bound.  Then, the
 * calling pattern is mapped to the variables in the clause.
 */
      
STATETBL *headmap(head, num_vars)
     SLOT *head;
     SHORT num_vars;
{
  SHORT		proc, i;
  CALSUCPR	*lub;
  STATETBL	*state;
  SLOT		*next;

  DEREFPTR(head);
  alias_to = NO_VARS_YET;
  if (GET_FUNCTOR(head) == QueryFunc) {
    printf("Top level query, not processing\n");
    return NULL;
  }
  else {
    head++;
    DEREFPTR(head);
    proc = GET_FUNCTOR(head);
    Malloc_StateTBL(state);
    Malloc_State(state->st, num_vars);
    state->vars = 0;
# ifdef LOBUG
    printf("headmap():  %s/%d ", procedures[proc].head, procedures[proc].arity);
    PrintTerm(head, tuple);
    OsPrint("\n");
# endif
    lub = lubOfProc(proc);
    for (i=0; i < procedures[proc].arity; i++) {
      next = ++head;	/* move to first argument */
      termmap(next, lub[i], state);
    }
# ifdef LOBUG
    print_state(state);
# endif
    return state;
  } /* end if head == NULL */
} /* end headmap() */



/*
 * void update_state(SLOT *term, STATETBL *state, SHORT instantiation)
 * Given that a term will be instantiated at least to instantiation,
 * determines the new instantiations of any variables contained
 * in the term
 */
void update_state(term, state, instantiation)
     SLOT	*term;
     STATETBL	*state;
     SHORT	instantiation;
{
  SLOT	*member, *tmp;
  SHORT	arity, i;


  DEREFPTR(term);
  if (TAG_IS_FUNCTOR(term)) {
    arity = ARITY(term);
    member = tmp = term;
    for (i=0; i < arity; i++) {
      member = ++tmp;
      update_state(member, state, instantiation);
    } /* end for (i=0; i < arity... */
  } else
    if (VARIABLE(term)) {
      i = varlookup(term, state);
      state->st[i].type = lub_instantiation(state->st[i].type, instantiation);
      if (lub_callpat(GROUND, state->st[i].type) == GROUND)
	remove_alias(i, state);
    } /* end if TAG_IS_FUNCTOR else... */
} /* end update_state */


/*
 * SHORT arcmap(SLOT *functor, STATETBL *state)
 * Given a term and the current state of its variables,
 * determines the state of the variables should the
 * term return successfully.
 * The function returns non zero when one of the calling patterns
 * is an exact match, otherwise zero.
 */
SHORT arcmap(functor, state)
     SLOT	*functor;
     STATETBL	*state;
{
  SHORT		arity, i, var, top, match;
  SLOT		*term, *tmp;
  CALSUCPR	*calsuc;
  
  DEREFPTR(functor);
  alias_to = NO_VARS_YET;

# ifdef HIBUG
  printf("Call pat for ");
  PrintTerm(functor, tuple);
  OsPrint("\n");
# endif

  arity = ARITY(functor);
  if (arity) {
    Malloc_CalSucPR(calsuc, arity);
    
    /* calculate call pattern */
    term = tmp = functor + 1;
    for (i=0; i < arity; i++) {
      /* process each term */
      DEREFPTR(term);
      if (TAG_IS_FUNCTOR(term))
	calsuc[i].call = scan_functor(term, state);
      else
	if (VARIABLE(term)) {
	  top = state->vars;
	  var = varlookup(term, state);
	  if (var == top)	{ /* if a new variable, is FREEVAR */
	    state->st[var].type = FREEVAR;
	  } /* end if var == top */
	  if (alias_to == NO_VARS_YET)	/* if first free var */
	    alias_to = var;
	  else
	    alias(alias_to, var, state);	/* set alias */
	  calsuc[i].call = state->st[var].type;
	}
	else
	  /* term is number or other GROUND term */
	  calsuc[i].call = GROUND;
      term = ++tmp;
    } /* end for (i=0;...) */
    
    match = success_pattern(functor, calsuc);	/* determine success pattern */
  } else
    /* arity = 0 */
    match = 1;	/* arity 0 always matches */
# ifdef HIBUG
  printf("Call\tSuccess:\n");
  for (i=0; i < arity; i++)
    printf("%d\t%d\n", calsuc[i].call, calsuc[i].succ);
  printf("\n");
# endif

  if (arity) {
    /* update state if anything to update */
    term = tmp = functor;
    for (i=0; i < arity; i++) {
      /* process each term */
      term = ++tmp;
      update_state(term, state, calsuc[i].succ);
    } /* end for (i=0; i < arity... */
    Free_Block(calsuc, sizeof(CALSUCPR)*arity);
  }
# ifdef HIBUG
  printf("State right before arcmap returns:\n");
  print_state(state);
# endif
  return match;
} /* end arcmap */

      
/*
 * void further_bind(STATETBL *dest, STATETBL *binding)
 * If anything in binding is more instantiated than
 * in dest, change it to the more instantiated binding.
 */
void *further_bind(dest, binding)
     STATETBL	*dest, *binding;
{
  register SHORT	i, j;
  SHORT			found;

  /* merge binding into state */
  for (i = 0; i < binding->vars; i++) {
    found = 0;
    j = 0;
    while (! found && j < dest->vars) {
      if (binding->st[i].var == dest->st[j].var) {
	found = 1;
	dest->st[j].type = lub_instantiation(dest->st[j].type,
					       binding->st[i].type);
	if (lub_callpat(dest->st[j].type, GROUND) == GROUND)
	  remove_alias(j, dest);
      } /* end if (binding->st[i].var == ... */
      j++;
    } /* end while (! found ... */
    if (! found) {	/* binding[i] must be added to dest */
      dest->st[dest->vars++] = binding->st[i];
    } /* end if (! found) */
  } /* end for (i = 0... */

# ifdef HIBUG
  printf("further_bind: updated destination state\n");
  print_state(dest);
# endif
  
} /* end further_bind */
  

