
 /******************************************************************
 *  Pgm.c  - Parser and program manipulation module               *
 *  Written by:  David Sehr                                       *
 *  Date: April 10, 1988                                          *
 ******************************************************************/

#include "typedefs.h"
#include "pgm_typedefs.h"
#include "macros.c"
#include "pgm_macros.c"
#include "bit_macros.c"

/*
 *  The following are the pre-defined predicate names for the parser.
 */

extern SHORT VLEN_N, VLEN_A, VLEN_V; /* Lengths of bit-vectors */

extern SLOT ListFunc, CommaFunc, NilAtom, ArrowFunc;

extern LONG NullIf, ClauseIf, QueryFunc, ParaFunc, SemiFunc,
	     ColonFunc, GoesToFunc, DetFunc, GuardFunc1, GuardFunc2;
extern SLOT *tuple;

/*
 *  The following are used to build the graph of the clause literals.
 */

extern SHORT from[MaxLits],
	     to[MaxLits],
	     det[MaxLits],
	     cannot_fail[MaxLits],
	     litcount;
extern SLOT  *pos[MaxLits];

SLOT *PreUnify;
SHORT DetFlag;
SHORT MaxEmbVars;
extern CLAUSE_VAR_TYPE varinfo[MaxVarTbl];

extern Procedure procedures[MaxProcs];
extern Procedure *procs;
extern char **TopNames;

#define ProcedureAlloc(p) (p = procs++);

/*
 * Utility routines.
 */

/* extern SLOT *next_struct_elt;     don't need it anymore */

extern char *VarTbl[MaxVarTbl];
extern char PrNames[MaxPrNames];
extern char *PrNext;
extern SHORT  ClausesRead;
extern SHORT  NextOffset;
extern SHORT  Tokens[MaxTokens];
extern SHORT  SpPrec[MaxTokens];
extern union {
	SHORT ival;
	char *chval;
} Lvals[MaxTokens];

extern NameEntry *NmTable[NmHashVal], Names[MaxNames], *NextName;

extern SHORT TokensInTerm;
extern SHORT At;

/*
 *  NameHash(s) returns a modulus hash value for the identifier passed in
 *  as s.
 */

SHORT NameHash(s)
unsigned char *s;
{
	SHORT length;

	length = strlen(s);
	switch(length) {
		case 1 : return s[0] % NmHashVal;
		case 2 : return (s[1] * 26 + s[0]) % NmHashVal;
		case 3 : return ((s[2] * 26 + s[1]) * 26 + s[0]) % NmHashVal;
		default: return (((s[length-2] * 26 + s[length-1]) * 26 +
						   s[1]) * 26 + s[0]) % NmHashVal;
	}
}

/*
 *  PgmLookupAtom(s) returns the index of the symbol table entry
 *  of the string s, which is an atom name.
 */

SHORT PgmLookupAtom(s)
char *s;
{
	NameEntry *a;
	SHORT hashval;

	hashval = NameHash(s);
	for(a = NmTable[hashval]; a != NULL; a = a->next)
		if(!strcmp(s,a->name))
			return a - Names;

	a = NextName++;
	a->name = PrNext;
	a->next = NmTable[hashval];
	NmTable[hashval] = a;
	while((*PrNext++ = *s++) != '\0');
	return a - Names;
}

/*
 *  ProcFor(atom,arity)  returns the index SHORT procedures of the entry
 *  with predicate name 'atom' and arity 'arity'.
 */

SHORT ProcFor(atom,arity)
SHORT atom,arity;
{
	Procedure *p;
	for(p=Names[atom].procs; p != NULL; p = p->hash_chain)
		if(p->arity == arity)
			return p - procedures;
	ProcedureAlloc(p);
	p->arity = arity;
	p->head = Names[atom].name;
	p->builtin = FALSE;
	p->hash_chain = Names[atom].procs;
	p->flags = 0;
	Names[atom].procs = p;
	return p - procedures;
}


/*
 * niProcFor(SHORT atom, SHORT arity)
 * No Insert ProcFor
 * Given an index into the symbol table Names[atom]
 * and an arity, returns the index into the procedures
 * table for the procedure associated with atom and arity.
 * If the procedure does not exist, -1 is returned
 */
SHORT niProcFor(atom, arity)
     SHORT atom, arity;
{
  Procedure *p;

  if (atom < 0)
    return(-1);
  else {
    for (p = Names[atom].procs; p != NULL; p = p->hash_chain)
      if (p->arity == arity)
	return p - procedures;
    return -1;
  }
}

/*
 *  VarLookup(s) returns the offset (activation record position) of
 *  the variable whose name is in s.
 */

SHORT VarLookup(s)
char *s;
{
	struct NameEntry *a;
	SHORT hashval;

	hashval = NameHash(s);
	for(a = NmTable[hashval]; a != NULL; a = a->next)
		if(!strcmp(s,a->name)) {
			if(a->clause != ClausesRead) {
				VarTbl[NextOffset] = a->name;
				a->offset = NextOffset++;
				a->clause = ClausesRead;
			}
			return a->offset;
		}

	a = NextName++;
	a->next = NmTable[hashval];
	VarTbl[NextOffset] = a->name = PrNext;
	a->offset = NextOffset++;
	a->clause = ClausesRead;
	NmTable[hashval] = a;
	while((*PrNext++ = *s++) != '\0');

	return a->offset;
}

/*
 *  NextClause() is used to reset the offset pointer, and to increment
 *  the number of clauses read.  It returns the number of variables
 *  in the clause just completed.
 */

SHORT NextClause()
{
	SHORT offset;
	offset = NextOffset;
	NextOffset = 1;
	ClausesRead++;
	litcount = 1;
	return offset;
}

/*
 *  CopyTopLevelVars() creates a table of the names of the top-level
 *  variables of the query.  This table is used for print names when
 *  displaying the bindings created.
 */

char **CopyTopLevelVars()
{
	SHORT i;
	char **table;
	Malloc_Block(table,NextOffset * sizeof(char *),char *);	
	for(i=0;i<NextOffset;i++)
		table[i] = VarTbl[i];
	return table;
}

/*
 *  The tokenizer follows.
 */

#define UC	1		/* Upper case alphabetic */
#define UL	2		/* Underline */
#define LC	3		/* Lower case alphabetic */
#define N	4		/* Digit */
#define QT	5		/* Single quote */
#define DC	6		/* Double quote */
#define SY	7		/* Symbol character */
#define SL	8		/* Solo character */
#define BK	9		/* Brackets & friends */
#define BS	10		/* Blank space */

static char CharType[] = {
/* nul soh stx etx eot enq ack bel  bs  ht  nl  vt  np  cr  so  si */
    BS, BS, BS, BS, BS, BS, BS, BS, BS, BS, BS, BS, BS, BS, BS, BS,
/* dle dc1 dc2 dc3 dc4 nak syn etb can  em sub esc  fs  gs  rs  us */
    BS, BS, BS, BS, BS, BS, BS, BS, BS, BS, BS, BS, BS, BS, BS, BS,
/*  sp   !   "   #   $   %   &   '   (   )   *   +   ,   -   .   /  */
    BS, SL, DC, SY, LC, SL, SY, QT, BK, BK, SY, SY, BK, SY, SY, SY,
/*  0   1   2   3   4   5   6   7   8   9   :   ;   <   =   >   ? */
    N,  N,  N,  N,  N,  N,  N,  N,  N,  N, SY, SL, SY, SY, SY, SY,
/*  @   A   B   C   D   E   F   G   H   I   J   K   L   M   N   O */
   SY, UC, UC, UC, UC, UC, UC, UC, UC, UC, UC, UC, UC, UC, UC, UC,
/*  P   Q   R   S   T   U   V   W   X   Y   Z   [   \   ]   ^   _ */
   UC, UC, UC, UC, UC, UC, UC, UC, UC, UC, UC, BK, SY, BK, SY, UL,
/*  `   a   b   c   d   e   f   g   h   i   j   k   l   m   n   o */
   SY, LC, LC, LC, LC, LC, LC, LC, LC, LC, LC, LC, LC, LC, LC, LC,
/*  p   q   r   s   t   u   v   w   x   y   z   {   |   }   ~  del */
   LC, LC, LC, LC, LC, LC, LC, LC, LC, LC, LC, BK, BK, BK, SY,  BS };

static char *read_stream;  /* FILE */

#define PREFIX	0
#define INFIX	1
#define POSTFIX	2

#define LEFT_LESS	0x2000
#define RIGHT_LESS	0x1000
#define PREC		0x0fff

SHORT IsOp(atom,kind,prio,left,right)
LONG atom,kind,*prio,*left,*right;
{
	SHORT info;
	switch(kind) {
		case PREFIX:	info = Names[atom].prefix;
						break;
		case INFIX:		info = Names[atom].infix;
						break;
		case POSTFIX:	info = Names[atom].postfix;
						break;
		default:		return FALSE;
	}
	if(info <= 0)
		return FALSE;
	*prio = info & PREC;
	*left = (info & LEFT_LESS) ? *prio-1 : *prio;
	*right = (info & RIGHT_LESS) ? *prio-1 : *prio;
	return TRUE;
}

union {
	SHORT ival;
	char *chval;
} info;

static SHORT is_eof = 0;

#define NextCh		*(read_stream++) 
#define UnGet(c)        *(--read_stream) = c
#define NameChar(c)	((CharType[c] == UC) || (CharType[c] == UL) || \
					 (CharType[c] == LC) || (CharType[c] == N))
#define VNAME	0
#define ANAME	1
#define NUMBER	2
#define STRING	3
#define PUNC	4
#define STOP	5

static SHORT GetToken(bs_precedes)
SHORT *bs_precedes;
{
	SHORT c,flag;
	char *buf;

	buf = PrNext;
	*bs_precedes = 0;
next:
	c = NextCh;
	if(c == EOF) {
		strcpy(buf,"end_of_file");
		info.ival = PgmLookupAtom(buf);
		is_eof = TRUE;
		return ANAME;
	}

	switch(CharType[c]) {
		case BS:	*bs_precedes = 1;
					goto next;
		case UC:
		case UL:	flag = TRUE;
					goto name;
		case LC:	flag = FALSE;
			name:	do {
						*buf++ = c;
						c = NextCh;
					} while(NameChar(c));
					*buf = '\0';
					UnGet(c);
					if(flag) {
						info.ival = VarLookup(PrNext);
						return VNAME;
					}
					else {
						info.ival = PgmLookupAtom(PrNext);
						return ANAME;
					}
		case N:		info.ival = 0;
					do {
						info.ival = info.ival * 10 + (c - '0');
						c = NextCh;
					} while(CharType[c] == N);
					UnGet(c);
					return NUMBER;
		case DC:	flag = DC;
					goto quote;
		case QT:	flag = QT;
			quote:	c = NextCh;
					while(CharType[c] != flag) {
						*buf++ = c;
						c = NextCh;
					}
					*buf = '\0';
					if(flag == DC){
						info.chval = PrNext;
						PrNext += strlen(info.chval) + 1;
						return STRING;
					}
					else {
						info.ival = PgmLookupAtom(PrNext);
						return ANAME;
					}
		case SY:	if(c == '/') {
						if((info.ival = NextCh) == '*') {
							c = NextCh;
							for(;;) {
								if(c == '*') {
									if((c = NextCh) == '/')
										goto next;
								}
								else
									c = NextCh;
							}
						}
						else
							UnGet(info.ival);
					}
					else if(c == '.') {
						if((info.ival = NextCh) == EOF) {
							UnGet(info.ival);
							return STOP;
						}
						else if(CharType[info.ival] == BS)
							return STOP;
						else
							UnGet(info.ival);
					}
					do {
						*buf++ = c;
						c = NextCh;
					} while(CharType[c] == SY);
					*buf = '\0';
					UnGet(c);
					info.ival = PgmLookupAtom(PrNext);
					return ANAME;
		case SL:	if(c == '%') {
						while((c = NextCh) != '\n')
							if(c == EOF) {
								UnGet(c);
								goto next;
							}
						goto next;
					}
					*buf = c;
					buf[1] = '\0';
					info.ival = PgmLookupAtom(PrNext);
					return ANAME;
		case BK:	*buf++ = c;
					if(c == '[') {
						if((info.ival = NextCh) == ']') {
							*buf++ = ']';
							*buf++ = '\0';
							info.ival = PgmLookupAtom("[]");
							return ANAME;
						}
						UnGet(info.ival);
					}
					info.ival = c;
					return PUNC;
	}
}

/*
 *  GetTerm(stream)  reads tokens from stream into Tokens
 *  with information values in Lvals, until end of file or a full stop is
 *  encountered.
 */

SHORT GetTerm(stream)
char *stream;
{
	SHORT token;
	SHORT bs_precedes;

	is_eof = 0;

	read_stream = stream;

	token = GetToken(&bs_precedes);
	for(TokensInTerm=0;!is_eof && TokensInTerm < MaxTokens ;TokensInTerm++) {
		Tokens[TokensInTerm] = token;
		Lvals[TokensInTerm].ival = info.ival;
		SpPrec[TokensInTerm] = bs_precedes;
		if(token == STOP)
			break;
		token = GetToken(&bs_precedes);
	}

	if(TokensInTerm == MaxTokens) {
		fprintf(stderr,"Buffer overflow.\n");
		exit(1);
	}

	At = 0;

	return TokensInTerm;
}

/*
 *  SyntaxError() displays a syntax error which occurred in
 *  position At in Tokens.
 */

SyntaxError()
{
	SHORT i;

	fprintf(stderr,"Syntax Error: ");
	for(i=0;i<TokensInTerm;i++) {
		switch(Tokens[i]) {
			case ANAME:	fprintf(stderr,"%s",Names[Lvals[i].ival].name);
						break;
			case VNAME:	fprintf(stderr,"%s",VarTbl[Lvals[i].ival]);
						break;
			case STOP:	fprintf(stderr,".");
						break;
			case NUMBER:fprintf(stderr,"%d",Lvals[i].ival);
						break;
			case STRING:fprintf(stderr,"\"%s\"",Lvals[i].chval);
						break;
			case PUNC:	fprintf(stderr,"%c",Lvals[i].ival);
						break;
		}
		if(i==At)
			fprintf(stderr," <HERE> ");
	}
	fprintf(stderr,"\n");
	exit();
}

SLOT *ReadArgs(atom, termclosed)
SHORT atom;
BOOLEAN *termclosed;
{
	SLOT *ParseTerm();
	SLOT *block;
	SLOT temp[1024];
	SHORT subterms;
	BOOLEAN closed;

	subterms = 0;
	*termclosed = TRUE;
	do {
		At++;
		temp[subterms++] = (SLOT)ParseTerm(999, &closed);
		if (TAG_IS_FUNCTOR(&(temp[subterms - 1])) && closed)
		   MAKE_CLOSED_TERM(&(temp[subterms - 1]));
		*termclosed = *termclosed && closed;
	} while((Tokens[At] == PUNC) && (Lvals[At].ival == ','));

	if((Tokens[At] != PUNC) || (Lvals[At].ival != ')'))
		SyntaxError();

	At++;
	MkNaryTerm(block,MkFunctor(ProcFor(atom,subterms),subterms),subterms,temp, *termclosed);
	return block;
}

/*
 *  StrToList returns a list term corresponding to the characters in
 *  a string read in.
 */

SLOT *StrToList()
{
	SLOT *last;
	char *p;
	SHORT length;

	p = Lvals[At++].chval;

	length = strlen(p);
	for(last = (SLOT *)NilAtom; length > 0; length--)
		MkBinaryTerm(last,ListFunc,MkIntg(p[length-1]),(SLOT)last, TRUE);

	return last;
}

SLOT *ReadList(listclosed)
BOOLEAN *listclosed;

{
	SLOT *t,*first,*temp;
	SLOT *ParseTerm();
	SHORT n;
	BOOLEAN closed, termclosed;

	*listclosed = TRUE;
	Malloc_Slots(first,3);
	t = first;
	t[0] = ListFunc;

	for(n=1;;n++) {
		t[1] = (SLOT)ParseTerm(999, &closed);
		if (TAG_IS_FUNCTOR(&(t[1])) && closed)
		   MAKE_CLOSED_TERM(&(t[1]));
		termclosed = closed;
		if((Tokens[At] == PUNC) && (Lvals[At].ival == ',')) {
			At++;
			if((Tokens[At]==ANAME)&&!strcmp(Names[Lvals[At].ival].name,"..")){
				At++;
				t[2] = (SLOT)ParseTerm(999, &closed);
				if (TAG_IS_FUNCTOR(&(t[2])) && closed)
				   MAKE_CLOSED_TERM(&(t[2]));
				termclosed = termclosed && closed;
				break;
			}
		}
		else {
			if((Tokens[At] == PUNC) && (Lvals[At].ival == '|')) {
				At++;
				t[2] = (SLOT)ParseTerm(999, &closed);
				if (TAG_IS_FUNCTOR(&(t[2])) && closed)
				   MAKE_CLOSED_TERM(&(t[2]));
				termclosed = termclosed && closed;
				break;
			}
			else {
				t[2] = (SLOT)NilAtom;
				if (termclosed)
				    MAKE_CLOSED_TERM(t);
				*listclosed = *listclosed && termclosed;
				return first;
			}
		}
		if (termclosed)
		   MAKE_CLOSED_TERM(t);
		*listclosed = *listclosed && termclosed;
		Malloc_Slots(temp,3);
		t[2] = (SLOT)temp;
		t = (SLOT *)t[2];
		t[0] = ListFunc;
	}
	return first;
}


SLOT *ReadSet(setclosed)
BOOLEAN *setclosed;

{
	SLOT *t,*first,*temp;
	SLOT *ParseTerm();
	SHORT n;
	BOOLEAN closed, termclosed;

	*setclosed = TRUE;
	Malloc_Slots(first,3);
	t = first;
	t[0] = ListFunc;

	for(n=1;;n++) {
		t[1] = (SLOT)ParseTerm(999, &closed);
		if (TAG_IS_FUNCTOR(&(t[1])) && closed)
		   MAKE_CLOSED_TERM(&(t[1]));
		termclosed = closed;
		if((Tokens[At] == PUNC) && (Lvals[At].ival == ',')) {
			At++;
			if((Tokens[At]==ANAME)&&!strcmp(Names[Lvals[At].ival].name,"..")){
				At++;
				t[2] = (SLOT)ParseTerm(999, &closed);
				if (TAG_IS_FUNCTOR(&(t[2])) && closed)
				   MAKE_CLOSED_TERM(&(t[2]));
				termclosed = termclosed && closed;
				break;
			}
		}
		else	{
			t[2] = (SLOT)NilAtom;
			if (termclosed)
			    MAKE_CLOSED_TERM(t);
			*setclosed = *setclosed && termclosed;
			return first;
		}
		if (termclosed)
		   MAKE_CLOSED_TERM(t);
		*setclosed = *setclosed && termclosed;
		Malloc_Slots(temp,3);
		t[2] = (SLOT)temp;
		t = (SLOT *)t[2];
		t[0] = ListFunc;
	}
	return first;
}



SLOT *ParseTerm(prec, termclosed)
SHORT prec;
BOOLEAN *termclosed;
{
	LONG p,lp,rp,tp,t;
	SLOT *exp1,*exp2;
	BOOLEAN closed;

	*termclosed = TRUE;
	p = 0;
	switch(Tokens[At]) {
		case ANAME:	if((Tokens[At+1] == PUNC) && (Lvals[At+1].ival == '(')
						&& (!SpPrec[At+1])) {
						exp1 = ReadArgs(Lvals[At++].ival, &closed);
						*termclosed = *termclosed && closed;
						break;
					}
					if(IsOp(Lvals[At].ival,PREFIX,&p,&lp,&rp)) {
						t = Lvals[At++].ival;
						MkAtom(exp1,ProcFor(t,0)); 
						if(Tokens[At] == PUNC &&
							(Lvals[At].ival != '(' && Lvals[At].ival != '{' &&
							 Lvals[At].ival != '[') || Tokens[At] == STOP) {
								if(p > prec)
									SyntaxError();
							At++;
							break;
						}
						exp2 = ParseTerm(rp, &closed);
						*termclosed = *termclosed && closed;
						if(!strcmp(Names[t].name,"-") && Tokens[At-1]==NUMBER)
							exp1 = (SLOT *)MkIntg(-Lvals[At-1].ival);
						else
							MkUnaryTerm(exp1,MkFunctor(ProcFor(t,1),1),
										(SLOT)exp2, *termclosed);
						break;
					}
					MkAtom(exp1,ProcFor(Lvals[At++].ival,0)); 
					break;
		case NUMBER:exp1 = (SLOT *)MkIntg(Lvals[At++].ival);
					break;
		case VNAME:	exp1 = (SLOT *)MkVarb(Lvals[At++].ival);
		    		*termclosed = FALSE;
		    		break;
		case STRING:exp1 = StrToList();
					break;
		case PUNC:	if(Lvals[At].ival == '(') {
						At++;
						exp1 = ParseTerm(1200,&closed);
						*termclosed = *termclosed && closed;
						if(Tokens[At] != PUNC || Lvals[At].ival != ')')
							SyntaxError();
						At++;
						break;
					}
					if(Lvals[At].ival == '[') {
						At++;
						exp1 = ReadList(&closed);
						*termclosed = *termclosed && closed;
						if(Tokens[At] != PUNC || Lvals[At].ival != ']')
							SyntaxError();
						At++;
						break;
					}
					if(Lvals[At].ival == '{') {
						At++;
						exp1 = ReadSet(&closed);
						*termclosed = *termclosed && closed;
						if(Tokens[At] != PUNC || Lvals[At].ival != '}')
							SyntaxError();
						At++;
						break;
					}
		case STOP:	SyntaxError();
	}

loop:
	if(Tokens[At] == ANAME) {
		if(IsOp(Lvals[At].ival,INFIX,&tp,&lp,&rp)) {
			if(tp <= prec && lp >= p) {
				MkBinaryTerm(exp1,MkFunctor(ProcFor(Lvals[At++].ival,2),2),
							 (SLOT)exp1,(SLOT)ParseTerm(rp, &closed), (*termclosed && closed));
				p = tp;
				goto loop;
			}
		}
		if(IsOp(Lvals[At].ival,POSTFIX,&tp,&lp,&rp)) {
			if(tp <= prec && lp >= p) {
				MkUnaryTerm(exp1,MkFunctor(ProcFor(Lvals[At++].ival,1),1),
							(SLOT)exp1, *termclosed);
				p = tp;
				goto loop;
			}
		}
		return exp1;
	}
	if(Tokens[At] == STOP)
	    return exp1;
	if(Tokens[At] != PUNC || Lvals[At].ival == '(' || Lvals[At].ival == '[')
		SyntaxError();
	if(Lvals[At].ival == ',' && prec >= 1000 && p <= 999) {
		At++;
		MkBinaryTerm(exp1,CommaFunc,(SLOT)exp1,(SLOT)ParseTerm(1000, &closed), (*termclosed && closed));
		if((p = 1000) < prec)
			goto loop;
	}
	return exp1;
}

SHORT Op(prec,kind,name)
SHORT prec;
SLOT *kind,*name;
{
	static char *Kinds[]={"xfx","xfy","yfx","xf","yf","fx","fy"};
	SHORT i;

	if((prec <= 0) || (prec > 1200))
		return FALSE;

	for(i=0; i < 7; i++)
		if(!strcmp(Kinds[i],procedures[INTVALUE(kind)].head)) {
			if((i == 1) || (i == 6))
				prec |= LEFT_LESS;
			else if((i == 2) || (i == 4))
				prec |= RIGHT_LESS;
			if(i < 3)
				Names[PgmLookupAtom(procedures[INTVALUE(name)].head)].infix
					= prec;
			else if(i < 5)
				Names[PgmLookupAtom(procedures[INTVALUE(name)].head)].postfix
					= prec;
			else
				Names[PgmLookupAtom(procedures[INTVALUE(name)].head)].prefix
					= prec;
			return TRUE;
		}
	return FALSE;
}

extern SLOT FUNCTOR_PLUS;
extern SLOT FUNCTOR_MINUS;
extern SLOT FUNCTOR_MULTIPLY;
extern SLOT FUNCTOR_DIVIDE;
extern SLOT FUNCTOR_MOD;
extern SLOT FUNCTOR_DIV;
extern SHORT CONS;
extern SHORT NILATOMINDEX;

void PgmInit(opsbuf, size)
char *opsbuf;
LONG size;
{
	LONG pre,in,post;
	LONG i, count;

	Malloc_Slots(tuple,100);
	for(i=0;i<100;i++)
		INSERT_SLOT_TAG((tuple+i),UNBOUND_VAR);
	INSERT_TUPLE_SIZE(tuple,100);

	NullIf		=	ProcFor(PgmLookupAtom(":-"),1);
	GuardFunc1	=	ProcFor(PgmLookupAtom("@:"),2);
	GuardFunc2	=	ProcFor(PgmLookupAtom("@::"),2);
	ClauseIf	=	ProcFor(PgmLookupAtom(":-"),2);
	ListFunc	=	MkFunctor(ProcFor(PgmLookupAtom("."),2),2);
	CommaFunc	=	MkFunctor(ProcFor(PgmLookupAtom(","),2),2);
	ArrowFunc       =       MkFunctor(ProcFor(PgmLookupAtom("->"),2),2);
	ParaFunc        =       ProcFor(PgmLookupAtom("//"),2);
	SemiFunc        =       ProcFor(PgmLookupAtom(";"),2);
	ColonFunc       =       ProcFor(PgmLookupAtom(":"),2);
	GoesToFunc      =       ProcFor(PgmLookupAtom("==>"),2);
	DetFunc		=       ProcFor(PgmLookupAtom("@"),2);
	INSERT_CLOSED_TERM(&NilAtom,ProcFor(PgmLookupAtom("[]"),0), 0);
	QueryFunc	=	ProcFor(PgmLookupAtom("?-"),1);
	CONS 		=	ProcFor(PgmLookupAtom("."),2);  /* global */
	NILATOMINDEX	=	ProcFor(PgmLookupAtom("[]"),0); /* constants */

	/* Functors for be_builtins. */

	FUNCTOR_PLUS	=       MkFunctor(ProcFor(PgmLookupAtom("+"),2),2);
	FUNCTOR_MINUS	=	MkFunctor(ProcFor(PgmLookupAtom("-"),2),2);
	FUNCTOR_MULTIPLY=	MkFunctor(ProcFor(PgmLookupAtom("*"),2),2);
	FUNCTOR_DIVIDE	=	MkFunctor(ProcFor(PgmLookupAtom("/"),2),2);
	FUNCTOR_MOD	=	MkFunctor(ProcFor(PgmLookupAtom("mod"),2),2);
	FUNCTOR_DIV	=	MkFunctor(ProcFor(PgmLookupAtom("div"),2),2);

	{
		LONG prec;
		char assoc[5],name[80];
		static char *Kinds[]={"xfx","xfy","yfx","xf","yf","fx","fy"};
		BOOLEAN NextString();
		
		count = size;
		while(NextString(name, &opsbuf, &count))
		{
		    for(prec=0, i=0;isdigit(name[i]);i++)
		        prec = prec * 10 + name[i] - '0';
		    NextString(assoc, &opsbuf, &count);
		    NextString(name, &opsbuf, &count);

		    for(i=0;i<7;i++)
		        if(!strcmp(Kinds[i],assoc)) {
			    if((i == 1) || (i == 6))
			        prec |= RIGHT_LESS;
			    else if((i == 2) || (i == 4))
			        prec |= LEFT_LESS;
			    if(i < 3)
			        Names[PgmLookupAtom(name)].infix = prec;
			    else if(i < 5)
			        Names[PgmLookupAtom(name)].postfix = prec;
			    else
			        Names[PgmLookupAtom(name)].prefix = prec;
			}
		}
	}
	InstallBuiltin("write",1,1);
	InstallBuiltin("nl",0,2);
	InstallBuiltin("is",2,3);
	InstallBuiltin("=\\=",2,4);
	InstallBuiltin("<",2,5);
	InstallBuiltin(">",2,6);
	InstallBuiltin("=:=",2,7);
	InstallBuiltin("div",3,8);
	InstallBuiltin("=<",2,9);
	InstallBuiltin(">=",2,10);
	InstallBuiltin("var",1,11);
	InstallBuiltin("nonvar",1,12);
	InstallBuiltin("functor",3,13);
	InstallBuiltin("atom",1,14);
	InstallBuiltin("arg",3,15);
	InstallBuiltin("integer",1,16);
	InstallBuiltin("=",2,17);
	InstallBuiltin("\\=",2,18);
	InstallBuiltin("=..",2,19);
	InstallBuiltin("print_vars",1,20);
	InstallBuiltin("halt",0,21);
	InstallBuiltin("null",0,22);
	InstallBuiltin("!",0,23);
}



BOOLEAN NextString(string, buffer, size)
char *string;
char **buffer;
LONG *size;
{
    int  i;

    if (*size > 1)
    {
	for (i=0; ((*buffer)[i] != '\n') && ((*buffer)[i] != ' ') && 
	          (*size > 0); i++, (*size)--)
	    string[i] = (*buffer)[i];

	string[i] = '\0';   /* terminate the string */
	(*size)--;
	*buffer += i + 1;  /* ignore the "\n" or the " " */
	return(TRUE);
    }
    else return(FALSE);
}



char *PgmGetAtom(atom)
SLOT atom;
{
    return(procedures[atom].head);
}

int PgmBuiltinIndex(fa)
SLOT *fa;
{
	return(procedures[GET_FUNCTOR(fa)].builtin);
}

CLAUSE *PgmNextClause(lastclause)
CLAUSE *lastclause;
{
	return(lastclause->next);
}

CLAUSE *PgmClauses(fa)
SLOT *fa;
{
	return(procedures[GET_FUNCTOR(fa)].clauses);
}

static InstallBuiltin(func,arity,index)
char *func;
SHORT arity,index;
{
	procedures[ProcFor(PgmLookupAtom(func),arity)].builtin = index;
}

/*
 *  DisplayArcs prints out the arcs which are incident to the given
 *  literal as its predecessors.
 */

static void DisplayArcs(clause,lit_number)
CLAUSE *clause;
SHORT lit_number;
{  
	int i;
	for(i=0;i<clause->num_nodes;i++)
		if(SUCCESSOR(clause->succ_arcs[i],lit_number))
			OsPrintf("\t(%d,%d)\n",i,clause->succ_node[lit_number]);
}

/*
 *  PrintClause prints the literals and arc connections of a clause.
 */

void PrintClause(clause)
CLAUSE *clause;
{  
	SHORT bodylit,i;

	if(PgmIsClauseDet(clause)) OsPrintf("$det(");
	PrintTerm(clause->lit_position[0],tuple);
	if(PgmIsClauseDet(clause)) OsPrintf(")");

	OsPrintf((clause->num_arcs > 0) ? " :-\n" : ".\n");

	for(bodylit = 1; bodylit <= clause->num_arcs; bodylit++) {  
		OsPrintf("\t");
		if(PgmLitIsDet(clause,bodylit-1)) OsPrintf("$det(");
		PrintTerm(clause->lit_position[bodylit],tuple);
		if(PgmLitIsDet(clause,bodylit-1)) OsPrintf(")");
		OsPrintf((bodylit < clause->num_arcs) ? "," : ".");
		DisplayArcs(clause,bodylit-1);
	}

#ifdef PRIORITY
	OsPrintf("ClauseBits %d, ClauseNum %d, SeqLitBits %d\n", clause->numClauseBits, clause->clauseNum, clause->numSeqLitBits);
  	for (i=0; i<clause->num_nodes; i++) 
		OsPrintf("i %d, clause->node_distance[i] = %d\n", i, clause->node_distance[i]);
#endif
  	OsPrintf("\n");

}

/*
 *  PgmListing prints the entire clause database by cycling through the 
 *  procedures array, displaying each predicate's clauses.
 */

void PgmListing()
{  
	Procedure *p;
	CLAUSE *c;
	int i;
	for(p = procedures; p < procs; p++)
		for(c = p->clauses; c != NULL; c = c->next)
			PrintClause(c);
}

/*
 *  PgmMaxClauseBits() computes the number of log N bits needed for
 *  N clauses that match procedure name and arity. The bits are
 *  recorded in the CLAUSE data structure of each clause that match.
 */

#ifdef	PRIORITY
void PgmMaxClauseBits()
{
	Procedure *p;
	CLAUSE *c;
	int bits,count;
	extern int	ceil_log2();
	for(p = procedures; p < procs; p++)
	{
		count = 0;
		for(c = p->clauses; c != NULL; c = c->next)
			if ( c->num_arcs > 0 )
				count++;

		if ( count == 0 )
			bits = 0;
		else
			bits = ceil_log2( count-1 );

		count = count - 1;
		for(c = p->clauses; c != NULL; c = c->next)
			if ( c->num_arcs > 0 )
			{
				c->numClauseBits = bits;
				c->clauseNum = count;
				count--;
			}
			else
			{
				c->numClauseBits = -1;
				c->clauseNum = 0;
			}
	}
}
#endif


/*
 * SHORT ParseGraph(SLOT *term, CLAUSE *clause)
 * Constructs the explicit representation of the user specified
 * graph and parses preunification and clause directives.
 * Graph construction of unspecified sets is handled later.
 * term is a pointer to the clause body
 * clause is the CLAUSE structure for this particular clause
 */

SHORT ParseGraph(term, clause)
     SLOT *term;
     CLAUSE *clause;

{
SHORT i;
SHORT BuildGraph();

/* Initializations */
  for(i=1; i<litcount; i++)	{
    det[i] = FALSE;
    cannot_fail[i] = FALSE;
  }

  clause->num_nodes = BuildGraph(term);
  clause->num_arcs = litcount - 1;
  clause->num_vars = NextOffset - 1;
  clause->top = 0;
  clause->clause_det = DetFlag;
  clause->PreUnify = PreUnify;

  clause->numSeqLitBits = 0;
  clause->clauseNum = 0;
  clause->numClauseBits = 0;
}


SHORT BuildGraph(term)
SLOT *term;
{
	SHORT i, result;
	SHORT BuildGraphArbit(), BuildGraphParen();

/*	TRACE2(printf("%s\n",procedures[GET_FUNCTOR(term)].head));*/
	if (GET_FUNCTOR(term)==GuardFunc1)	{
		MaxEmbVars = -1;
		DetFlag = FALSE;
		for(i=1; i<MaxVarTbl; i++)	{
			varinfo[i].CanImport = 1;
			varinfo[i].CanExport = 1;
			}
		GetCompDir(*(term+1));
		result = BuildGraph(*(term+2));
		}
	else if (GET_FUNCTOR(term)==GuardFunc2)	{
		PreUnify = (SLOT *) (*(term+1));
		/*PreUnifyTraverse(PreUnify);*/
		result = BuildGraph(*(term+2));
		}
	else if ((GET_FUNCTOR(term)==SemiFunc)||(GET_FUNCTOR(term)==ColonFunc)){
		result = BuildGraphArbit(term, -1) + 2;
	}
	else 	{
		result = BuildGraphParen(term, 0, 1, 2);
	}
	return result;
}

/* GetCompDir()
 *
 * Process the compiler directives found in the body of a clause.
 * These are located before guard functor 1, and they are separated 
 * by commas.  They are:
 *
 * 	maxEmbVar(num)
 *	embVar(var_name, can_import, can_export)
 *      det
 * 
 * Anything else before guard functor 1 is an error.
 */

GetCompDir(term)
SLOT *term;
	{
	SLOT functor, *next, *nameterm, *ImportTerm, *ExportTerm;
	SHORT offset;

/*	TRACE2(printf("%s\n",procedures[GET_FUNCTOR(term)].head));*/
	if (GET_FUNCTOR(term) == GET_FUNCTOR(&CommaFunc))	{
		GetCompDir(*(term+1));
		GetCompDir(*(term+2));
		}
	else {
		functor = GET_FUNCTOR(term);
		if (strcmp(procedures[functor].head,"maxEmbVar")==0)	{
			next = term + 1;
			MaxEmbVars = INTVALUE(next);
			TRACE2(printf("MaxEmbVars = %d\n",MaxEmbVars));
			}
		else if (strcmp(procedures[functor].head,"embVar")==0)	{
			nameterm = term+1;
			ImportTerm = term+2;
			ExportTerm = term+3;
			offset = XtrVarb(nameterm);
			TRACE2(printf("offset = %d\n",offset));
			TRACE2(printf("VarName = %s\n",VarTbl[offset]));
			varinfo[offset].CanImport = INTVALUE(ImportTerm);
			varinfo[offset].CanExport = INTVALUE(ExportTerm);
			TRACE2(printf("%d  %d\n",varinfo[offset].CanImport,
					  varinfo[offset].CanExport));
			}
		else if (strcmp(procedures[functor].head,"det")==0) {
			DetFlag = TRUE;
			}
		else {
			fprintf(stderr,"Bad Input\n");
			exit(1);
			}
		}
	}

/* SetLitFlags()
 * 
 * Process the call-site determinism annotations.  (t@df)
 */

void SetLitFlags(term)
SLOT *term;
{
	char temp[5];
	strcpy(temp,procedures[GET_FUNCTOR(term)].head);
	if (temp[0] == 'd')	{
		det[litcount] = TRUE;
		}
	else if ((temp[0] == 'n') || (temp[0] == '_'))	{
		det[litcount] = FALSE;
		}
	else 		{
		fprintf(stderr,"ERROR -- determinism character '%c' not allowed\n",temp[0]);
		exit(1);
		}

	if (temp[1] == 's')	{
		cannot_fail[litcount] = TRUE;
		}
	else if ((temp[1] == 'f')||(temp[1]=='_'))	{
		cannot_fail[litcount] = FALSE;
		}
	else		{
		fprintf(stderr,"ERROR -- cannot_fail character '%c' not allowed\n",temp[1]);
		exit(1);
		}
}

/* PreUnifyTraverse()
 *
 * Traverse the parse tree for the pre-unification part of a 
 *    graph.  This is the part that appears before the second
 *    guard functor, but before the first guard functor. 
 */

SHORT PreUnifyTraverse(term)
SLOT *term;
{
	TRACE2(printf("%s\n",procedures[GET_FUNCTOR(term)].head));
	if((GET_FUNCTOR(term) == GET_FUNCTOR(&CommaFunc)) || 
	   (GET_FUNCTOR(term) == GET_FUNCTOR(&ArrowFunc))) {
		PreUnifyTraverse(*(term+1));
		PreUnifyTraverse(*(term+2));
	}
	else 	{
		/* Perform check */
	}
}

/* BuildGraphParen()
 * 
 * Build the graph for a clause, using the parsed representation of 
 * the clause.  Assume that the clause is in parenthetical notation.
 */

SHORT BuildGraphParen(term,start,end,temp)
SLOT *term;
SHORT start,end,temp;
{
	SHORT after;
/*	TRACE2(printf("%s\n",procedures[GET_FUNCTOR(term)].head));*/
	if((GET_FUNCTOR(term) == GET_FUNCTOR(&CommaFunc)) || 
	   (GET_FUNCTOR(term) == GET_FUNCTOR(&ArrowFunc))) {
		after = BuildGraphParen(*(term+1),start,temp,temp+1);
		after = BuildGraphParen(*(term+2),temp,end,after);
		return after;
	}
	else if(GET_FUNCTOR(term) == ParaFunc) {
		after = BuildGraphParen(*(term+1),start,end,temp);
		after = BuildGraphParen(*(term+2),start,end,after);
		return after;
	}
	else if (GET_FUNCTOR(term) == DetFunc)	{
		BuildGraphParen(*(term+1),start,end,temp);
		SetLitFlags(*(term+2));
		return temp;
	}
	else {
		pos[litcount] = term;
		from[litcount] = start;
		to[litcount++] = end;
		return temp;
	}
}

/* BuildGraphArbit()
 * 
 * Build the graph for a clause, using the parsed representation of 
 * the clause.  Assume that the clause is in arbitrary-graph notation.
 */

SHORT BuildGraphArbit(term,node)
SLOT *term;
SHORT node;
{
	SHORT t1, t2;

/*	TRACE2(printf("%s\n",procedures[GET_FUNCTOR(term)].head));*/
	if(GET_FUNCTOR(term) == SemiFunc)	{
  		t1 = BuildGraphArbit(*(term+1),node);
		t2 = BuildGraphArbit(*(term+2),node);
		return(t1+t2+1);
  		}
	if(GET_FUNCTOR(term) == GET_FUNCTOR(&CommaFunc))	{
  		t1 = BuildGraphArbit(*(term+1),node);
		t2 = BuildGraphArbit(*(term+2),node);
		return(t1+t2);
  		}
	else if (GET_FUNCTOR(term) == ColonFunc)	{
  		t1=BuildGraphArbit(*(term+2),INTVALUE((term+1)) ); 
		return(t1);
		}
	else if (GET_FUNCTOR(term) == GoesToFunc)	{
		BuildGraphArbit(*(term+1),node);
		from[litcount] = node;
		to[litcount] = INTVALUE((term+2));
		litcount++;
		return(0);
		}
	else if (GET_FUNCTOR(term) == DetFunc)	{
		BuildGraphArbit(*(term+1), node);
		SetLitFlags(*(term+2));
		}
	else		{
		pos[litcount] = term;
		}
}

/* PgmAllocClauseVectors()
 *
 * Allocate space for clause bit-vectors
 */

PgmAllocClauseVectors(c)
CLAUSE *c;
{
	c->BlockSize = (c->num_arcs / LONGLENGTH + 2) * (2);
	palloc(LONG, c->BlockSize, c->FreeBlock);
}

/* PgmInitClauseVectors()
 *
 * Initialize the multiple-word bit-vectors in the clause data structure.
 */

int PgmInitClauseVectors(c)
CLAUSE *c;
{
	VectorAlloc(c->FreeBlock,c->top,c->BlockSize,c->lit_det,VLEN_A);
	VectorAlloc(c->FreeBlock,c->top,c->BlockSize,c->lit_cannot_fail,VLEN_A);
/*
	ArrayAlloc(c->FreeBlock,c->top,c->BlockSize,c->succ_arcs,
			VLEN_A,c->num_nodes);
	ArrayAlloc(c->FreeBlock,c->top,c->BlockSize,c->pred_arcs,
			VLEN_A,c->num_nodes);
*/ 
}

int PgmConsult(msgbuffer, size)   

char *msgbuffer;
int size;
{
	LONG tokens,t;
	SLOT *term,*tail_term,*write_term,*query_term;
/*	FILE *in_stream;  */
	CLAUSE *c;
	SHORT i,p;
	BOOLEAN closed;
	extern BOOLEAN mainread;

	read_stream = msgbuffer;      	/* to get source program from host */
        if(size == 0) { 
                OsPrintf("PgmConsult: Did not receive program from host.\n");
		exit(1);
	}
	tokens = 0;
	while((t = GetTerm(read_stream)) > 0) {
		tokens += t;
		term = ParseTerm(1200, &closed);
		if(GET_FUNCTOR(term) == ClauseIf) {
		        Malloc_Clause(c);
			ParseGraph(term[2], c);
			set_and_state(term, c);
			PgmAllocClauseVectors(c);
			PgmInitClauseVectors(c);
			palloc(SLOT *,litcount,c->lit_position);
			if(c->num_vars > 0)	
				Malloc_ClauseVarInfo(c->var_info, (c->num_vars+1));
			if(litcount > 1) {
				palloc(SHORT,c->num_arcs,c->succ_node);
				palloc(SHORT,c->num_arcs,c->pred_node);
				palloc(SHORT,c->num_arcs+1,c->builtin_index);
#ifdef PRIORITY
				palloc(LONG,c->num_arcs+1,c->node_distance);
#endif
			}
			if(c->num_nodes > 0) {
				palloc(LONG,c->num_nodes,c->succ_arcs);
				palloc(LONG,c->num_nodes,c->pred_arcs);
				palloc(SHORT,c->num_nodes,c->succ_count);
				palloc(SHORT,c->num_nodes,c->pred_count);
				palloc(SHORT,c->num_nodes,c->shared_tuple);
			}
			for(i=1;i<litcount;i++) {
				c->lit_position[i] = pos[i];
				if (det[i] == TRUE)
					InsInVector(c->lit_det, i-1);
				if (cannot_fail[i] == TRUE)
					InsInVector(c->lit_cannot_fail,i-1);
				c->builtin_index[i-1] = 0;
			}
			for(i=0;i<c->num_nodes;i++) {
				c->pred_count[i] = 0;
				c->succ_count[i] = 0;
				c->shared_tuple[i] = 0;
			}
			for(i=1;i<litcount;i++) {
				ADD_PRED(c->pred_arcs[to[i]],i-1);
				c->pred_count[to[i]]++;
				ADD_SUCC(c->succ_arcs[from[i]],i-1);
				c->succ_count[from[i]]++;
				c->succ_node[i-1] = to[i];
				c->pred_node[i-1] = from[i];
			}
			c->MaxEmbVars = MaxEmbVars;
			TRACE2(printf("MaxEmbVars = %d\n",MaxEmbVars));
			for (i=1; i<=c->num_vars; i++)	{
				c->var_info[i].CanImport = varinfo[i].CanImport;
				c->var_info[i].CanExport = varinfo[i].CanExport;
				TRACE2(printf("%d) %d  %d\n",i,
				  varinfo[i].CanImport, varinfo[i].CanExport));
				}
			c->lit_position[0] = (SLOT *)term[1];
			p = XtrFunc(*(c->lit_position[0]));
			c->query_size = RopmGetQuerySize(c);
	{int i; for(i = 1; i < c->num_arcs+1; i++)
		c->builtin_index[i] = PgmBuiltinIndex(c->lit_position[i]); 
	}
			c->next = procedures[p].clauses;
			procedures[p].clauses = c;
			TRACE(PrintClause(c));
			NextClause();
		}
		else if(GET_FUNCTOR(term) == NullIf) {
			ParseDirective(term);
			NextClause();
		}
		else if(GET_FUNCTOR(term) == QueryFunc) {
			mainread = TRUE;
			Malloc_Clause(c);
			if(NextOffset > 1) {
				Malloc_Slots(tail_term,NextOffset+1);
				*tail_term = MkFunctor(ProcFor(PgmLookupAtom("vars"),
											   NextOffset-1),NextOffset-1);
				for(i=1;i<NextOffset;i++) {
					INSERT_INDEX((tail_term+i),i,TUPLE_INDEX_1);
				}
				MkUnaryTerm(write_term,
							MkFunctor(ProcFor(PgmLookupAtom("print_vars"),1),
									  1),
						    (SLOT) tail_term, FALSE);
				MkBinaryTerm(query_term,ArrowFunc,term[1],(SLOT)write_term, FALSE);
			}
			else
				query_term = (SLOT *)term[1];
			ParseGraph(query_term, c);
			set_and_state(term, c);
			palloc(SLOT *,litcount,c->lit_position);
			PgmAllocClauseVectors(c);
			PgmInitClauseVectors(c);
			if(c->num_vars > 0)	
				Malloc_ClauseVarInfo(c->var_info,(c->num_vars + 1));
			if(litcount > 1) {
				palloc(SHORT,c->num_arcs,c->succ_node);
				palloc(SHORT,c->num_arcs,c->pred_node);
				palloc(SHORT,c->num_arcs+1,c->builtin_index);
#ifdef PRIORITY
				palloc(LONG,c->num_arcs+1,c->node_distance);
#endif
			}
			if(c->num_nodes > 0) {
				palloc(LONG,c->num_nodes,c->succ_arcs);
				palloc(LONG,c->num_nodes,c->pred_arcs);
				palloc(SHORT,c->num_nodes,c->succ_count);
				palloc(SHORT,c->num_nodes,c->pred_count);
				palloc(SHORT,c->num_nodes,c->shared_tuple);
			}
			for(i=1;i<litcount;i++) {
				c->lit_position[i] = pos[i];
/*	incorrect - Ram 	InsInVector(c->lit_det, i-1);
		7/29/89		InsInVector(c->lit_cannot_fail, i-1);  */
				c->builtin_index[i-1] = 0;
			}
			for(i=0;i<c->num_nodes;i++) {
				c->pred_count[i] = 0;
				c->succ_count[i] = 0;
				c->shared_tuple[i] = 0;
			}
			for(i=1;i<litcount;i++) {
				ADD_PRED(c->pred_arcs[to[i]],i-1);
				c->pred_count[to[i]]++;
				ADD_SUCC(c->succ_arcs[from[i]],i-1);
				c->succ_count[from[i]]++;
				c->succ_node[i-1] = to[i];
				c->pred_node[i-1] = from[i];
			}
			c->MaxEmbVars = MaxEmbVars;
			TRACE2(printf("MaxEmbVars = %d\n",MaxEmbVars));
			for (i=1; i<=c->num_vars; i++)	{
				c->var_info[i].CanImport = varinfo[i].CanImport;
				c->var_info[i].CanExport = varinfo[i].CanExport;
				TRACE2(printf("%d) %d  %d\n",i,
				  varinfo[i].CanImport, varinfo[i].CanExport));
				}
			p = ProcFor(PgmLookupAtom("main$"),0);
			MkAtom(c->lit_position[0],p);
			c->query_size = RopmGetQuerySize(c);
	{int i; for(i = 1; i < c->num_arcs+1; i++)
		c->builtin_index[i] = PgmBuiltinIndex(c->lit_position[i]); 
	}
			c->next = NULL /*procedures[p].clauses */;
			procedures[p].clauses = c;
			TopNames = CopyTopLevelVars();
			NextClause();
			TRACE(PrintClause(c));
		}
		else {
		        Malloc_Clause(c);
			palloc(SLOT *,1,c->lit_position);
			palloc(SHORT,1,c->builtin_index);
			c->lit_position[0] = term;
			c->num_nodes = 1;
			c->num_arcs = 0;
			c->num_vars = NextOffset-1;
			c->builtin_index[0] = 0;
			p = XtrFunc(*(c->lit_position[0]));
			c->query_size = RopmGetQuerySize(c);
	{int i; for(i = 1; i < c->num_arcs+1; i++)
		c->builtin_index[i] = PgmBuiltinIndex(c->lit_position[i]); 
	}
			c->next = procedures[p].clauses;
			procedures[p].clauses = c;
			TRACE(PrintClause(c));
			NextClause();
		}
	}
	TRACE(OsPrintf("node %d: Consulted program (%d tokens read)\n",
			OsMyPeNum(), tokens));
	UserSetDet();
	return TRUE;
}



isdigit(c)
char c;
{
    return((c == '0') || (c == '1') || (c == '2') || (c == '3') || (c == '4') 
	|| (c == '5') || (c == '6') || (c == '7') || (c == '8') || (c == '9'));
}



int PgmMainFunctor()
{
	return ProcFor(PgmLookupAtom("main$"),0);
}

char *PgmVarName(index)
int index;
{
	return TopNames[index];
}



