/*
 * graph.c
 * Generates arbitrary graphs
 *
 * AUTHOR:  David Roch
 *
 * CONTRIBUTORS:
 *	DAR - David Roch
 *
 * HISTORY:
 *	5/27/88 - Created DAR
 *	6/20/88 - Converts sets to sequential
 *	8/01/88 - Moved things not directly related to graph
 *		  generation to set_state.c
 *	8/08/88 - Started implementing ``freezer'' graph gen. algorithm
 *
 */

/* headers */

#include "typedefs.h"
#include "pgm_typedefs.h"
#include "state.h"
#include "lattice.h"
#include "graph.h"


/* macros */
#include "macros.c"
#include "pgm_macros.c"
#include "bit_macros.c"
#include "gpmacros.c"

/* function prototypes */
#include "protos/callpat.p"
#include "protos/state.p"
#include "protos/lattice.p"

/* debugging flags */
#define HIBUG	1		/* high level debugging */
#define LOBUG	1 		/* low level debugging */

/* externals */
extern NameEntry 	*NmTable[], Names[];
extern Procedure 	procedures[];
extern SHORT 		from[], to[], litcount, QueryFunc;
extern SLOT 		*pos[], ListFunc, NilAtom, ProcFor();
extern SLOT		*tuple;
extern char		cmperrmsg[];

/* globals */
SHORT	first_graph_index,
  	next_graph_index;

/* local definitions */
/* BitBlkLen - length of block neeeded for bit vectors in LONGs
 * Add 1 LONG for each bit vector as extra space is needed
 * for internal fragmentation and bookkeeping.
 * sets used in process_set():
 *	MaxLits - frozen, thawed, fire, cooked
 *	MaxVarTbl - new_variables
 *	MaxNodes - frontier, candidate, needed, used
 */
#define	BitBlkLen	((4*MaxLits + MaxVarTbl + 4*MaxNodes) \
			 /sizeof(LONG)/8 + 9)
#define	BitNodeLen	((MaxLits) / sizeof(LONG)/8 + 1)

/* SA_VectorAlloc(VECTOR *vector, SHORT length)
 * Stand Alone Vector Allocate
 * Allocates a vector of size length, but unlike the macro
 * VectorAlloc, we are not obligated to have a block of
 * memory available.
 */
#define SA_VectorAlloc(vector, length) \
  (vector) = (LONG *) calloc((SHORT) (length) / 8 + 1, sizeof(LONG)); \
  if ((vector) == NULL) \
     cmperror(5, "Out of memory when allocating a stand alone vector"); \
  *(vector) += (SHORT) (length) / 8 + 1


/* void init_index(SHORT first_lit)
 * Initializes the the get_graph_index routine.
 * first_lit is the index of the literal that contains
 * the set notation.
 */
void init_index(first_lit)
     SHORT	first_lit;
{
  next_graph_index = first_lit;
  first_graph_index = first_lit;
}

/* SHORT get_graph_index(CLAUSE *clause)
 * Determines what the next index into the from/to/pos table
 * should be.  If it is the first index for the current set
 * being processed, then we use the index that already exists,
 * otherwise a new one is returned
 */
SHORT get_graph_index(clause)
     CLAUSE	*clause;
{
  SHORT	index;
  
  if (first_graph_index != next_graph_index) {
    index = next_graph_index;		/* new arc */
    next_graph_index = clause->num_arcs++ + 1;
  } else {
    index = first_graph_index;		/* reuse arc holding set */
    next_graph_index = clause->num_arcs + 1;
  } /* end if (first_graph_index.... else ... */
  
  if (clause->num_arcs > MaxLits)
    cmperror(10, "SYS ERROR - Too many literals (possibly null arcs) for graph generation");
  return index;
} /* end get_graph_index() */
  
  
/*
 * SHORT init_lits(SLOT *list, SLOT *lit_ary, STATETBL *lit_states)
 * Convert the list of literals to an array stored
 * at lit_ary.  Return the number of literals in
 * the list
 */

SHORT init_lits(list, lit_ary, lit_states)
     SLOT	*list, *lit_ary[];
     STATETBL	*lit_states[];
{
  SHORT	count = 0, i;
  SLOT	*tail, *head;
  
  tail = list;
  DEREFPTR(tail);
  while (CONS_CELL_PTR(tail)) {
    head = tail + 1;
    DEREFPTR(head);
    NEXT_CELL_PTR(tail);
    DEREFPTR(tail);
    if (CONS_CELL_PTR(head)) {
      PrintTerm(head, tuple);
      OsPrint("\n");
      cmperror(0, "Warning:  Set within a set - not processing");
    } else {
      lit_ary[count++] = head;
    } /* end if (CONS_CELL_PTR(head)) then/else */
  } /* end while (CONS_CELL_PTR(tail)... */
  if (! NIL_ATOM_PTR(tail)) {
    lit_ary[count++] = head;
  }

# ifdef LOBUG
  printf("init_lits: %d Literals to be processed\n", count);
# endif
  for (i=0; i< count; i++) {
    lit_states[i] = NULL;
#   ifdef LOBUG
    PrintTerm(lit_ary[i], tuple);
    OsPrint("\n");
#    endif
  }
  return count;
} /* end init_lits */

/*
 * SHORT interesting(STATETBL *after, STATETBL *before, VECTOR *new_vars)
 * Given the state of a literal before and after it executes,
 * see if any of its accessed variables are in the new variable set.
 * If so, return non zero, otherwise zero.
 *
 * Currently, an interesting variables is:
 *	- at least one of the literals variables appears in new_variables
 *
 */
SHORT interesting(after, before, new_vars)
     STATETBL	*after, *before;
     VECTOR	*new_vars;
{
  SHORT	i, j;

# ifdef LOBUG
  printf("interesting():  ");
# endif

  /* loop through all variables in after */
  for (i=0; i < after->vars; i++) {
    if (IN_FLAG(after->st[i].flags, ACCESSED)) { /* if literal accesses var */
      j = varcheck(after->st[i].var, before); /* find index into before */
      if (BitInVector(new_vars, j)) { /* if accessed var in new variables */
#	ifdef LOBUG
	printf("yes\n");
#	endif
	return 1;	/* interesting */
      } /* end if (BitInVector(new_vars... */
    } /* end if (IN_FLAG(... */
  }
# ifdef LOBUG
  printf("no\n");
# endif
  return 0;	/* boring */
} /* end interesting() */

/*
 * SHORT independent(SHORT lit1, SHORT lit2, 
 *		     STATETBL *lit_states[], STATETBL *state)
 * Given the indices of two literals in the lits array along
 * with their states after execution (lit_states) from a known
 * state (state), determine if the literals are independent
 *
 * Returns:
 *	INDPENDENT if literals are independent
 *	DEPENDENT if literals aren't
 * Algorithm:
 *	For each of the two literals, determine which variables
 *	  may have been modified.  As they are determined, mark
 *	  the root of their alias group as modified.
 *	Perform a second pass upon the variables.  This time,
 *	  examine the root of the alias groups for each
 *	  variable.  If the root has the possibility of being
 *	  modified, mark that variable as modified.
 *	Choose the literal with the smallest number of variables.
 *	  For each variable in this literal with a modified
 *	  flag, see if it exists in the other literal.  If
 *	  it has been modified, the literals are not independent.
 */
SHORT independent(lit1, lit2, lit_states, state)
     SHORT	lit1, lit2;
     STATETBL	*lit_states[];
     STATETBL	*state;
{

  {
    /* Determine which variables have the possibility of having
     * been modified in the execution of the two literals
     */
    SHORT	i, j, lit, after_exec, before_ind;
    
    for (i=0; i < 2; i++) {
      if (i)	/* Must perform for lit1 and lit2 */
	lit = lit2;
      else
	lit = lit1;
      /* loop through all variables of lit */
      for (j=0; j < lit_states[lit]->vars; j++) {

	/* determine variable type after execution */
	after_exec = lit_states[lit]->st[j].type;
	
	/* find the literals index in the before state */
	before_ind = varcheck(lit_states[lit]->st[j].var, state);
	
	/* This test determines whether or not a variable may
	 * have been modified.  It does not take aliasing into
	 * account, that is handled later.  The test is true
	 * if a variables has not been modified
	 */
	if (/* if the variable has not been used */
	    ! IN_FLAG(lit_states[lit]->st[j].flags, ACCESSED)
	    ||
	    /* if we get this far, the variable has been used, but
	     * we still don't know if it is modified
	     */
	    (
	     before_ind != DOES_NOT_EXIST /* make sure not a new var */
	     && (	/* if the variable is unbound before and
			 * after execution, then it is unmodified */
		(after_exec == FREEVAR &&
		 state->st[before_ind].type) == FREEVAR)
		||	/* OR, if the variable was fully instantiated
			 * before execution, it cannot have been
			 * modified */
		(lub_instantiation(state->st[before_ind].type,
				   GROUND) == GROUND)
	     ))
	  CLEAR_FLAG( lit_states[lit]->st[j].flags, MODIFIED);
	else {
	  /* mark the var as modified */
	  SET_FLAG( lit_states[lit]->st[j].flags, MODIFIED);
	  /* mark the alias group to which it belongs as modified */
	  SET_FLAG( lit_states[lit]->st[lit_states[lit]->st[j].alias].flags,
		    MODIFIED);
	}
      } /* end j loop */
    } /* end i loop */
  } /* end determine modified variables block */

  {
    /* Second pass */
    SHORT	i, alias_root;

    for (i=0; i < lit_states[lit1]->vars; i++) {
      alias_root = lit_states[lit1]->st[i].alias;
      if (IN_FLAG(lit_states[lit1]->st[alias_root].flags, MODIFIED))
	SET_FLAG(lit_states[lit1]->st[i].flags, MODIFIED);
    }
    for (i=0; i < lit_states[lit2]->vars; i++) {
      alias_root = lit_states[lit2]->st[i].alias;
      if (IN_FLAG(lit_states[lit2]->st[alias_root].flags, MODIFIED))
	SET_FLAG(lit_states[lit2]->st[i].flags, MODIFIED);
    }
  } /* end second pass */
  { /* final pass */

    SHORT	i, min, min_lit, max_lit, var;

    if (lit1 < lit2) {
      min_lit = lit1;
      max_lit = lit2;
    } else {
      min_lit = lit2;
      max_lit = lit1;
    }
    min = lit_states[min_lit]->vars;
    for (i=0; i < min; i++) {
      if (/* if variable modified and */
	  IN_FLAG(lit_states[min_lit]->st[i].flags, MODIFIED) &&
	  /* if variable exists in other literal and */
	  (var = varcheck(lit_states[min_lit]->st[i].var,
			  lit_states[max_lit])) != DOES_NOT_EXIST &&
	  /* if variable in other literal is modified */
	  IN_FLAG(lit_states[max_lit]->st[var].flags, MODIFIED)
	  )
	/* then these literals are not independent */
	return (DEPENDENT);
    } /* end i */
    /* looks like we survived, let everyone know it... */
    return (INDEPENDENT);
  }
} /* end independent() */

/*
 * void max_independents(VECTOR *set, STATETBL *state,
 *			 SLOT **literals, SHORT num_lits)
 * Examines the sets of literals that are ready for execution
 * and determines which ones are AND independent.  The largest
 * independent overwrites variables set.
 */
void max_independents(set, state, literals, num_lits)
     VECTOR	*set;
     STATETBL	*state;
     SLOT	**literals;
     SHORT	num_lits;
{
  SETLINK	*independents, *current, *ptr;
  SHORT		i, j, k, max;

  /* WARNING:  be sure to check that the set handed to max_inds
   * is of size num_lits.  If not, we may have to allocate
   * vectors of the maximim size.  VectorCopy only copies vectors
   * of the same size.
   */
  independents = NULL;
  for (i=0; i < num_lits; i++)
    if (BitInVector(set, i)) {
      /* create new SETLINK with i */
      Malloc_Block(current, sizeof(SETLINK), SETLINK);
      SA_VectorAlloc(current->set, num_lits);
      InsInVector(current->set, i);	/* set i */
      current->next = independents;	/* link into list */
      independents = current;		/*	""	*/
      
      for (j=i; i < num_lits; j++)
	if (BitInVector(set, j) && independent(i, j, literals, state)) {

	  /* create new SETLINK w/ {i, j} */
	  Malloc_Block(current, sizeof(SETLINK), SETLINK);
	  SA_VectorAlloc(current->set, num_lits);
	  InsInVector(current->set, i);	/* set i */
	  InsInVector(current->set, j);	/* set j */
	  current->next = independents;	/* link into list */
	  independents = current;	/*	""	*/

	  /* check if this pair can be added to any other set */
	  for (ptr=independents; ptr != NULL; ptr=ptr->next)
	    if (! (BitInVector(ptr->set, i) && BitInVector(ptr->set, j))) {
	      SHORT indep = 1;
	      /* i & j not in this set, check if they are independent
	       * of every element of the set
	       */
	      for (k=0; k < num_lits && indep; k++)
		/* if in set and not i or j */
		if (BitInVector(ptr->set, k) && i != k && j != k)
		  indep = independent(i, k, literals, state) &&
		    independent(j, k, literals, state);
	      /* if indep then i, j should be added */
	      if (indep) {
		InsInVector(ptr->set, i);
		InsInVector(ptr->set, j);
	      }
	    } /* end if (! (BitInVector....)) */
	} /* end if (BitInVector(ptr->set, j)...) */
    } /* end if (BitInVector(ptr->set, i)...) */

  /* select largest */
  max = -1;
  for (ptr=independents; ptr=ptr->next; ptr != NULL) 
    if (VectorCardinality(ptr->set) > max) {
      max = VectorCardinality(ptr->set);
      current = ptr;
    }
  CopyVector(current->set, set);	/* record best set */
  
# ifdef HIBUG
  printf ("Maximal independent set:  ");
  WriteBitsVector(set);
# endif
      
  /* deallocate memory */
  for (ptr = independents; ptr != NULL; ) {
#   ifdef LOBUG
    /* Dump each set */
    WriteBitsVector(ptr->set);
#   endif
    
    /* free the bit map associated with this SETLINK */
    cfree( ptr->set, VectorLength(ptr->set), sizeof(LONG));
    
    current = ptr;	/* save current one */
    ptr = ptr->next;	/* move to next link */
    free(current);	/* free SETLINK */
  } /* end for (ptr = independents; ptr != NULL; ) */
} /* end max_independents() */
	      
/* void thaw_literals(VECTOR *frozen, VECTOR *thawed, 
 *		      VECTOR *new_variables, STATETBL *frontier_state,
 *		      SLOT **literals, SHORT num_vars, SHORT num_lits
 *
 * Place all literals that are eligible to fire and are considered
 * interesting in the thawed set.  They are not yet removed from
 * frozen.
 * For the current definition of interesting, see procedure interesting()
 */

void thaw_literals(frozen, thawed, new_variables,
		   frontier_state, lit_states,
		   literals, num_vars, num_lits)
     VECTOR	*frozen, *thawed, *new_variables;
     STATETBL	*frontier_state, *lit_states[];
     SLOT	**literals;
     SHORT	num_vars, num_lits;
{
  SHORT		i;
  VECTOR	*possible, *worth_while;
  LONG		bit_space[BitNodeLen];
  SHORT		bit_p = 0;
    
# ifdef LOBUG
  printf("thaw_literals():\n");
  printf("  New variable set (indices into frontier state)\n");
  WriteBitsVector(new_variables);
# endif

  VectorAlloc(bit_space, bit_p, BitNodeLen, possible, MaxLits);
  EmptyVector(thawed);	/* thawed <- null set */
  worth_while = thawed;	/* alias interesting to thawed */

  /* compute literals that are eligible when instantiated to frontier_state */
  for (i=0; i < num_lits; i++) {
    if (BitInVector(frozen, i)) {	/* for each frozen literal */

      if (lit_states[i] != NULL) {	/* old literal state - release */
	if (num_vars)
	  Free_Block(lit_states[i]->st, sizeof(STATE)*num_vars);
	Free_Block(lit_states[i], sizeof(STATETBL));
	lit_states[i] = NULL;
      } /* end if (lit_states[i] != NULL) */
      
#     ifdef LOBUG
      printf("  Considering to thaw - ");
      PrintTerm(literals[i], tuple);
      OsPrint("\n");
#     endif

      /* determine what state will be when executed with frontier_state */
      lit_states[i] = copy_statetbl(frontier_state, num_vars);
      clear_access(lit_states[i]);
      if (arcmap(literals[i], lit_states[i]))
	InsInVector(possible, i);	/* lit is eligible */
    } /* end if (BitInVector(frozen, i)) */
  } /* end for (i=0; i < num_lits; i++) */

# ifdef LOBUG
  printf("Eligible Literals:  ");
  WriteBitsVector(possible);
# endif

  for (i=0; i < num_lits; i++) /* Find interesting variables */
    if (BitInVector(possible, i)) /* test eligible literals */
      if (interesting(lit_states[i], frontier_state, new_variables))
	InsInVector(worth_while, i);

# ifdef LOBUG
  printf("Eligible and interesting:  ");
  WriteBitsVector(worth_while);
  WriteBitsVector(thawed);
# endif

  if (IsVectorEmpty(worth_while))	/* no interesting literals? */
    if (! IsVectorEmpty(possible)) {	/* any eligible literals? */
      /* When there are no interesting variables, we pick one
       * from the set of eligible ones.  Currently, we just choose
       * the first one.  However, we should check for meta-logical
       * predicates first
       */
      i=0;
      while (i < MaxLits && ! BitInVector(possible, i))
	i++;
      InsInVector(thawed, i);
#     ifdef LOBUG
      printf("No interesting lits - thawed %d\n", i);
#     endif
    } /* end if (! IsVectoryEmpty...) */
} /* end thaw_literals() */



/* void find_needed(SLOT *literals[], SHORT literal,
 *		    STATETBL *states[], STATETBL *frontier_state,
 *		    VECTOR *frontier, VECTOR *needed)
 * Given a literal state, determine which nodes are needed to
 * generate a node capable of executing the arc that produced
 * state.
 */
void find_needed(literals, literal, states, frontier_state, frontier, needed)
     SLOT	*literals[];	/* this arg only for debug info */
     SHORT	literal;
     STATETBL	*states[], *frontier_state;
     VECTOR	*frontier, *needed;
{
  SHORT		numvars[MaxNodes], i, j, varcount, 
  		maxvars, maxnode, index1, index2;

  /* initialize the count mechanism */
  varcount = 0;
  for (i=0; i < states[literal]->vars; i++) {
    /* We must find a node that satisfies each variable that
     * will be accessed when literal executes and exists in
     * the frontier state.  Variables not in the frontier state
     * are those which are first referenced in this literal */
    if (IN_FLAG(states[literal]->st[i].flags, ACCESSED) &&
	varcheck(states[literal]->st[i].var, frontier_state) > -1) {
      varcount++;
    }
  } /* end for (i=0 ... */
  EmptyVector(needed);	/* initialize needed */
  
# ifdef HIBUG
  printf("find_needed():\n  Number of variables needed %d\n", varcount);
  printf("  literal finding needed nodes for:  ");
  PrintTerm(literals[literal], tuple);
  OsPrint("\n");
# endif

  /* determine number of variables needed by literal that each
   * node is capable of satisfying
   */
  maxvars = 0;
  for (i=0; i < MaxNodes; i++) {
    numvars[i] = 0;	/* initialize */
    if (BitInVector(frontier, i)) {	/* for each node i in frontier */
      for (j=0; j < states[literal]->vars; j++) { /* check each variable j */
	/* if the var j in literal has been accessed, it is
	 * considered neeed */
	if (IN_FLAG(states[literal]->st[j].flags, ACCESSED)) {
	  /* find out whether var j exists in frontier node i
	   * and/or in the frontier_state */
	  index1 = varcheck(states[literal]->st[j].var, states[i]);
	  index2 = varcheck(states[literal]->st[j].var, frontier_state);
	  if (index2 > -1 &&	/* j exists? */
	      index1 > -1 &&
	      /* var j must have the same type in node i as in the
	       * frontier state for this node to possibly suffice the
	       * need of literal */
	      states[i]->st[index1].type == frontier_state->st[index2].type)
	    numvars[i]++;
	} /* end if (IN_FLAG(states[... */
      } /* end for (j=0; ... */
#     ifdef LOBUG
      printf("  Node %d has %d eligible variable(s) ", i, numvars[i]);
#     endif
      if (numvars[i] > maxvars) {	/* if a new maximum */
#       ifdef LOBUG
	printf("new maximum");
#	endif
	maxvars = numvars[i];
	maxnode = i;
      } /* end if (numvars[i] > maxnode) */
#     ifdef LOBUG
      printf("\n");
#     endif
    } /* end if (BitInVector(frontier, i)) */
  } /* end for (i=0; i < MaxNodes; i++) */
  
  while (varcount) {	/* until all vars accounted for */
    /* choose the the node with the maximum number of variables */
    if (maxvars) {	/* if a maximum found */
      InsInVector(needed, maxnode);
      for (i=0; i < states[literal]->vars; i++) {
	if (IN_FLAG(states[literal]->st[i].flags, ACCESSED)) {
	  /* for each accessed variable in literal, check if
	   * it can be satisfied by node maxnode */
	  index1 = varcheck(states[literal]->st[i].var, states[maxnode]);
	  index2 = varcheck(states[literal]->st[i].var, frontier_state);
	  if (index1 > -1 &&	/* var must exist in states[maxnode] */
	      index2 > -1 &&	/*     and in frontier_state */
	      states[maxnode]->st[index1].type	/* and be of the same type */
	       == frontier_state->st[index2].type) {
	    varcount--;		/* one less variable to be satisfied */
	    numvars[maxnode]--;
#	    ifdef LOBUG
	    printf("  Variable %d has been satisified\n",
		   states[literal]->st[i].var);
#	    endif
	    for (j=0; j < MaxNodes; j++) {
	      if (BitInVector(frontier, j) && j != maxnode) {
		/* decrement the variable count for each frontier node
		 * that uses this var as it is no longer grounds for 
		 * including it in the needed set */
		index1 = varcheck(states[literal]->st[i].var, states[j]);
		if (index1 > -1 &&	/* if var i in state j */
		    states[j]->st[index1].type == /* candidate? */
		      frontier_state->st[index2].type) {
#		  ifdef LOBUG
		  printf("  Decrementing count of node %d\n", j);
#		  endif
		  numvars[j]--;
		} /* end if candidate */
	      } /* end if (BitInVector(frontier, j)) */
	    } /* end for (j=0; j < MaxNodes... */
	  } /* end if (index1 > -1 && index2...  */
	} /* end if (IN_FLAG(states[... */
      } /* end for (i=0; ... */
      /* algorithm integrity check */
      if (numvars[maxnode]) {
	sprintf(cmperrmsg, "SYSTEM ERR - node %d's numvar count not 0 (%d) in find_needed()", maxnode, numvars[maxnode]);
	cmperror(0, cmperrmsg);
      } /* end if (numvars[maxnode]) */
#     ifdef LOBUG
      printf("  %d vars left to satisfy\n", varcount);
#     endif
      /* find new maxnode */
      maxvars = 0;
      for (i = 0; i < MaxNodes; i++)
	if (numvars[i] > maxvars) {
	  maxvars = numvars[i];
	  maxnode = i;
	} /* end if (numvars[i]... */
    } else
	cmperror(5, "Fatal error in find_needed() - can't satisfy all vars");
  } /* end while (varcount) */
} /* end find_needed() */



/*
 * SHORT find_create_node(VECTOR *needed, VECTOR *frontier,
 *			  CLAUSE *clause)
 * Given a set of needed nodes, find or create a node
 * which has arcs leading from all needed nodes to it.
 * Place that node in the frontier state and return it.
 */
SHORT find_create_node(needed, frontier, clause)
     VECTOR	*needed;
     VECTOR	*frontier;
     CLAUSE	*clause;
{
  SHORT		i = 0, node = -1, index;

# ifdef LOBUG
  printf("find_create_node():  ");
# endif
  if (VectorCardinality(needed) == 1) {
    /* if only 1 node needed, just use it */
    while (node == -1)
      if BitInVector(needed, i)
	node = i;
      else
	i++;
#   ifdef LOBUG
    printf("Only node needed:  %d\n", node);
#   endif
  } else {
    /* more than one node */
    node = clause->num_nodes++;
    if (clause->num_nodes >= MaxNodes)
      cmperror(10, "MaxNodes insufficient for graph generation needs");
    for (i=0; i < MaxNodes; i++)
      if (BitInVector(needed, i)) {
	index = get_graph_index(clause);
	from[index] = i;
	to[index] = node;
	/* until we have defined a NullArc, we assume that
	 * the user has defined the fact null/0.  This serves
	 * as the literal associated with null arcs
	 */
	pos[index] = procedures[ProcFor(PgmLookupAtom("null"), 0)
				].clauses->lit_position[0];
	/* lit_position may not be the correct choice - NEED TO FIX THIS XX */
      } /* end if (BitInVector... */
  } /* end if (VectorCardinality...) ... else... */
  return node;
} /* end find_create_node() */

#ifdef HIBUG
/*
 * print_attached_lits(LIT_LIST *list[])
 * A debugging routine, prints out all literals attached to candidate
 * nodes in the list passed to it.
 */
print_attached_lits(list)
     LIT_LIST	*list[];
{
  SHORT		i, not_empty;
  LIT_LIST	*traverse;
  
  printf("Candidate node - literal list\n");
  for (i=0; i < MaxNodes; i++) {
    if ((traverse = list[i]) != NULL) {
      printf("Node %d - [", i);
      not_empty = 1;
    }
    else
      not_empty = 0;
    while (traverse != NULL) {
      printf("%d ", traverse->lit_index);
      traverse = traverse->next;
    }
    if (not_empty)
      printf("]\n");
  } /* end for i */
}
#endif HIBUG

/*
 * void select_fire_lits(VECTOR *fire, LIT_LIST *attached)
 * We wish to select which literals that are attached to
 * a candidate node should be fired.  This is done by a
 * pluggable heuristic (this function).  Currently, we just
 * select all current literals, but eventually we could allow
 * multiple heuristics (strategy chosen by compiler or programmer).
 * Fire is the set that will contain all literals to be fired.
 */
void select_fire_lits(fire, attached)
     VECTOR	*fire;
     LIT_LIST	*attached;
{
  EmptyVector(fire);	/* fire <- empty set */
  while (attached != NULL) {
    InsInVector(fire, attached->lit_index);
    attached = attached->next;
  }
} /* end select_fire_lits */
    
/*
 * void process_set(STATETBL *state, *states, CLAUSE *clause, SHORT arc)
 * Given:
 *	The state at one of the literals (pos[arc]) which
 *		consists of a set of goals with no imposed
 *		ordering,
 *	an array of states containing room for any new states
 *		that may be derived during the generation of
 *		a data join graph for a particular set of goals,
 *	The clause structure for this particular clause,
 *	An index (arc) into the global arc structures:
 *		pos - containing literals
 *		from - Tells what node this arc comes from
 *		to - Tells which node this arc leads to
 * We assume that the literal contained at pos[arc] is a set
 * structure.  We will break up this set and create individual
 * literals connected by a data join graph.  As a result of this,
 * pos, from, and to will be expanded.  We will also perform state,
 * analysis along for these arcs as they are generated.
 */

void process_set(state, states, clause, arc)
     STATETBL	*state, *states[];
     CLAUSE	*clause;
     SHORT	arc;
{
  LIT_LIST	*attached_lits [MaxNodes];
  LONG		bit_space[BitBlkLen];
  NODES		*needed_nodes[MaxNodes];
  SHORT		bit_p = 0, num_lits, num_vars, num_candidates, i, j,
  		source, dest, candidate_node, inplace = 1, first = 1;
  SLOT		*literals[MaxLits], *set;
  STATETBL	*frontier_state, *literal_states[MaxLits];
  VECTOR	*frontier, *frozen, *thawed, *new_variables,
		*needed, *candidate, *used, *fire, *cooked;

  
  /* These bitmaps will be used as sets */
  VectorAlloc(bit_space, bit_p, BitBlkLen, frontier, MaxNodes);
  VectorAlloc(bit_space, bit_p, BitBlkLen, candidate, MaxNodes);
  VectorAlloc(bit_space, bit_p, BitBlkLen, needed, MaxNodes);
  VectorAlloc(bit_space, bit_p, BitBlkLen, used, MaxNodes);
  VectorAlloc(bit_space, bit_p, BitBlkLen, new_variables, MaxVarTbl);
  VectorAlloc(bit_space, bit_p, BitBlkLen, frozen, MaxLits);
  VectorAlloc(bit_space, bit_p, BitBlkLen, thawed, MaxLits);
  VectorAlloc(bit_space, bit_p, BitBlkLen, fire, MaxLits);
  VectorAlloc(bit_space, bit_p, BitBlkLen, cooked, MaxLits);
  
  /* initialize */
  set = pos[arc];	/* this is the literal set to be broken up */
  num_vars = clause->num_vars;	/* # variables in clause */
  num_lits = init_lits(set, literals, literal_states);	/* # of lits in set */
  frontier_state = copy_statetbl(state, num_vars);
  clear_access(frontier_state);	/* indicate no variables have been accessed */
  for (i=0; i < MaxNodes; i++) {
    needed_nodes[i] = NULL;
  }
  source = from[arc];
  dest = to[arc];
  init_index();
  InsInVector(frontier, source);	/* frontier <- source node */

  for (i = 0; i < MaxNodes; i++) /* initialize the candidate node/literal */
    attached_lits[i] = NULL;	 /* structure */

  /* put all literals in freezer */
  for (i=0; i < num_lits; i++) {
    InsInVector(frozen, i);
  } /* end for (i=0... */

  /* We will consider all non FREEVAR variables to be initially interesting */
  for (i=0; i < frontier_state->vars; i++) {
    if (frontier_state->st[i].type != FREEVAR) {
      InsInVector(new_variables, i);
    } /* end if (frontier_state->st[i].type != FREEVAR) */
  } /* end for (i=0... */

  /* main loop of algorithm */
  while (! IsVectorEmpty(frozen)) {	/* while frozen not empty */
#   ifdef HIBUG
    printf("  frozen literals:  ");
    WriteBitsVector(frozen);
    printf("  frontier_state\n");
    print_state(frontier_state);
#   endif
    thaw_literals(frozen, thawed, new_variables,
		  frontier_state, literal_states,
		  literals, num_vars, num_lits);
    if (IsVectorEmpty(thawed))
      cmperror(5, "Unable to resolve graph, no literals ready to be fired");
#   ifdef HIBUG
    printf("  thawed literals:  ");
    WriteBitsVector(thawed);
#   endif
      
    /* determine what nodes are needed for each literal that is eligible
     * to fire */
    EmptyVector(used);	/* used <- null set */
    num_candidates = 0;	/* reset candidate count */
    for (i=0; i < num_lits; i++) 
      if (BitInVector(thawed, i)) {
	find_needed(literals, i, literal_states, frontier_state,
		    frontier, needed);
#	ifdef HIBUG
	printf("Needed nodes:  ");
	WriteBitsVector(needed);
#	endif
	/* find or create a node from which literal i can fire */
	candidate_node = find_create_node(needed, frontier, clause);
	/* used <- used U needed -- keep track of the nodes that
	 * we've used so we can take them out of the frontier later
	 */
	VectorUnion(used, used, needed);
	attach(i, candidate_node);	/* save this for later */
      } /* end if (BitInVector(thawed, i)) */
    
#   ifdef HIBUG
    print_attached_lits(attached_lits);	/* dump candidate node/lit relation */
#   endif

    EmptyVector(new_variables);		/* new_variables <- empty */
    /* Insure that none of the candidate nodes has been included
     * in the used set.  (Although they may be justly included, we
     * don't wish them to be in there.  This is just a gut feeling
     * I don't have the time to verify this).
     */
    for (i=0; i < MaxNodes; i++)
      if (BitInVector(candidate, i)) { /* if a candidate node */
	DelFromVector(used, i);	/* fast to just delete rather than checking */
      } /* end if */
	
    /* remove the used nodes from the frontier */
    VectorDifference(frontier, frontier, used);
    
    /* for each node in candidate_nodes */
    for (i=0; i < MaxNodes; i++)
      if (BitInVector(candidate, i)) {
	/* determine which literals to fire by heuristic */
	select_fire_lits(fire, attached_lits[i]);
	VectorDifference(frozen, frozen, fire);	/* frozen <- frozen - fire */
	VectorUnion(cooked, cooked, fire);	/* cooked <- cooked U fire */
	DelFromVector(frontier, i);	/* drop candidate node from frontier */

	/* for each literal in the fire set */
	for (j=0; j < MaxLits; j++)
	  if (BitInVector(fire, j)) {
	    SHORT	new_node, new_arc;
	    
	    /* create a new node */
	    if ((new_node = ClauseNumNodes(clause)++) > MaxNodes)
	      cmperror(5, "Exceeded max # of nodes during graph generation");
	    if (first) {
	      /* if first arc generated, reuse the arc where the set was */
	      new_arc = arc;
	      first = 0;
	    } else {
	      /* create a new arc */
	      if ((new_arc = ClauseNumArcs(clause)++) > MaxLits)
		cmperror(5, "Exceeded max # of nodes during graph generation");
	    } /* end if first */
	    from [new_arc] = i;
	    to [new_arc] = new_node;
	    pos [new_arc] = literals[j];
	    /* find all vars j might modify */

	  } /* end if BitInVector(fire, j) */
      } /* end if BitInVector(candidate, i) */
    
    break;	/* JUST FOR NOW */
  } /* end while (! IsVectorEmpty(thawed)) */
    
} /* end process_set() */
