%{
/*
 * PCN Abstract Machine Emulator
 * Authors:     Steve Tuecke and Ian Foster
 *              Argonne National Laboratory
 *
 * Please see the DISCLAIMER file in the top level directory of the
 * distribution regarding the provisions under which this software
 * is distributed.
 *
 * tok_pdb.lex	- lex based tokenizer for PDB
 */

#define HandleToken() \
    if (yyleng >= MAX_PARSE_STRING_LENGTH) \
	return (PDBT_ERROR); \
    else \
    { \
	strcpy(yylval.sval, yytext); \
	return (PDBT_TOKEN); \
    }

#define CommandOrToken(the_token) \
    if (yyp_start_new_command == TRUE) \
    { \
        yyp_start_new_command = FALSE; \
	yylval.ival = the_token; \
        return (the_token); \
    } \
    else \
    { \
        HandleToken() \
    }

#define HelpCommandOrToken(the_token) \
    if (yyp_start_new_command == TRUE) \
    { \
	yylval.ival = the_token; \
        return (the_token); \
    } \
    else \
    { \
        HandleToken() \
    }


#ifdef FLEX_SCANNER

#undef YY_INPUT
#define YY_INPUT(buf, result, max_size) \
	buf[0] = yyp_parse_inputline[yyp_parse_nextchar++]; \
	result = (buf[0] == '\0') ? YY_NULL : 1;

#else /* FLEX_SCANNER */

#undef input

#define input() (((yytchar=yysptr>yysbuf?U(*--yysptr):((int) yyp_parse_inputline[yyp_parse_nextchar++]))==10?(yylineno++,yytchar):yytchar)==EOF?0:yytchar)

/* This is what lex defines as input()
#define input() (((yytchar=yysptr>yysbuf?U(*--yysptr):getc(yyin))==10?(yylineno++,yytchar):yytchar)==EOF?0:yytchar)
*/

#endif /* FLEX_SCANNER */

%}

%p 5000

VC	[a-zA-Z0-9_\.]
VFC	[a-zA-Z_]
SPECIAL	[:,\^\(\)\{\}\-\+\/\*\%\<\>]

%%

\"[^"]*		{
		    if (yytext[yyleng-1] == '\\')
		    {
			yymore();
		    }
		    else
		    {
			yytext[yyleng] = input(); /* Pick up trailing " */
			yytext[yyleng] = '\0';    /* NULL terminate over " */
			if (yyleng-1 >= MAX_PARSE_STRING_LENGTH)
			    return (PDBT_ERROR);
			else
			{
			    strcpy(yylval.sval, yytext+1);
			    return (PDBT_STRING);
			}
		    }
		}

"#"[0-9]?[0-9]*	{	yylval.ival = atoi(yytext+1);
			return (PDBT_INSTANCE);
		}

"^"[0-9]?[0-9]*	{	yylval.ival = atoi(yytext+1);
			return (PDBT_REDUCTION);
		}

"_"[uU]?[0-9a-fA-F]?[0-9a-fA-F]*	{
			sscanf(yytext+2,"%x", &(yylval.ival));
			return (PDBT_UNDEF);
		}

"_"[uU]?"*"[0-9a-fA-F]?[0-9a-fA-F]*	{
			sscanf(yytext+3,"%x", &(yylval.ival));
			return (PDBT_UNDEF);
		}

"$"{VFC}?{VC}*	{	if (yyleng-1 >= MAX_PARSE_STRING_LENGTH)
			    return (PDBT_ERROR);
			else
			{
			    strcpy(yylval.sval, yytext+1);
			    return (PDBT_VARIABLE);
			}
		}

[0-9]?[0-9]*	{	yylval.ival = atoi(yytext);
			return (PDBT_INTEGER);
		}

[0-9]*\.[0-9]?[0-9]*	{	yylval.dval = atof(yytext);
				return (PDBT_DOUBLE);
			}

[ \n\t]		;

{SPECIAL}?	{	yylval.ival = (int) yytext[0];
			return (yylval.ival);
		}

"="		{	yyp_start_new_command = FALSE;
			return ('=');
		}

abort		{ CommandOrToken (PDBT_ABORT) }
abor		{ CommandOrToken (PDBT_ABORT) }
abo		{ CommandOrToken (PDBT_ABORT) }
ab		{ CommandOrToken (PDBT_ABORT) }
a		{ CommandOrToken (PDBT_ABORT) }

break		{ CommandOrToken (PDBT_BREAK) }
brea		{ CommandOrToken (PDBT_BREAK) }
bre		{ CommandOrToken (PDBT_BREAK) }
br		{ CommandOrToken (PDBT_BREAK) }
b		{ CommandOrToken (PDBT_BREAK) }

continue	{ CommandOrToken (PDBT_CONTINUE) }
continu		{ CommandOrToken (PDBT_CONTINUE) }
contin		{ CommandOrToken (PDBT_CONTINUE) }
conti		{ CommandOrToken (PDBT_CONTINUE) }
cont		{ CommandOrToken (PDBT_CONTINUE) }
con		{ CommandOrToken (PDBT_CONTINUE) }
co		{ CommandOrToken (PDBT_CONTINUE) }
c		{ CommandOrToken (PDBT_CONTINUE) }

debug		{ CommandOrToken (PDBT_DEBUG) }
debu		{ CommandOrToken (PDBT_DEBUG) }
deb		{ CommandOrToken (PDBT_DEBUG) }

delete		{ CommandOrToken (PDBT_DELETE) }
delet		{ CommandOrToken (PDBT_DELETE) }
dele		{ CommandOrToken (PDBT_DELETE) }
del		{ CommandOrToken (PDBT_DELETE) }

disable		{ CommandOrToken (PDBT_DISABLE) }
disabl		{ CommandOrToken (PDBT_DISABLE) }
disab		{ CommandOrToken (PDBT_DISABLE) }
disa		{ CommandOrToken (PDBT_DISABLE) }
dis		{ CommandOrToken (PDBT_DISABLE) }
di		{ CommandOrToken (PDBT_DISABLE) }

enable		{ CommandOrToken (PDBT_ENABLE) }
enabl		{ CommandOrToken (PDBT_ENABLE) }
enab		{ CommandOrToken (PDBT_ENABLE) }
ena		{ CommandOrToken (PDBT_ENABLE) }
en		{ CommandOrToken (PDBT_ENABLE) }
e		{ CommandOrToken (PDBT_ENABLE) }

help		{ HelpCommandOrToken(PDBT_HELP) }
hel		{ HelpCommandOrToken(PDBT_HELP) }
he		{ HelpCommandOrToken(PDBT_HELP) }
h		{ HelpCommandOrToken(PDBT_HELP) }
"?"		{ HelpCommandOrToken(PDBT_HELP) }

list		{ CommandOrToken (PDBT_LIST) }
lis		{ CommandOrToken (PDBT_LIST) }
li		{ CommandOrToken (PDBT_LIST) }
l		{ CommandOrToken (PDBT_LIST) }

load		{ CommandOrToken (PDBT_LOAD) }
loa		{ CommandOrToken (PDBT_LOAD) }
lo		{ CommandOrToken (PDBT_LOAD) }

modules		{ CommandOrToken (PDBT_MODULES) }
module		{ CommandOrToken (PDBT_MODULES) }
modul		{ CommandOrToken (PDBT_MODULES) }
modu		{ CommandOrToken (PDBT_MODULES) }
mod		{ CommandOrToken (PDBT_MODULES) }

move		{ CommandOrToken (PDBT_MOVE) }
mov		{ CommandOrToken (PDBT_MOVE) }

next		{ CommandOrToken (PDBT_NEXT) }
nex		{ CommandOrToken (PDBT_NEXT) }
ne		{ CommandOrToken (PDBT_NEXT) }
n		{ CommandOrToken (PDBT_NEXT) }

nodebug		{ CommandOrToken (PDBT_NODEBUG) }
nodebu		{ CommandOrToken (PDBT_NODEBUG) }
nodeb		{ CommandOrToken (PDBT_NODEBUG) }
node		{ CommandOrToken (PDBT_NODEBUG) }
nod		{ CommandOrToken (PDBT_NODEBUG) }
no		{ CommandOrToken (PDBT_NODEBUG) }

print		{ CommandOrToken (PDBT_PRINT) }
prin		{ CommandOrToken (PDBT_PRINT) }
pri		{ CommandOrToken (PDBT_PRINT) }

procedures	{ CommandOrToken (PDBT_PROCEDURES) }
procedure	{ CommandOrToken (PDBT_PROCEDURES) }
procedur	{ CommandOrToken (PDBT_PROCEDURES) }
procedu		{ CommandOrToken (PDBT_PROCEDURES) }
proced		{ CommandOrToken (PDBT_PROCEDURES) }
proce		{ CommandOrToken (PDBT_PROCEDURES) }
proc		{ CommandOrToken (PDBT_PROCEDURES) }
pro		{ CommandOrToken (PDBT_PROCEDURES) }
procs		{ CommandOrToken (PDBT_PROCEDURES) }

quit		{ CommandOrToken (PDBT_QUIT) }
qui		{ CommandOrToken (PDBT_QUIT) }
qu		{ CommandOrToken (PDBT_QUIT) }
q		{ CommandOrToken (PDBT_QUIT) }

show		{ CommandOrToken (PDBT_SHOW) }
sho		{ CommandOrToken (PDBT_SHOW) }
sh		{ CommandOrToken (PDBT_SHOW) }
s		{ CommandOrToken (PDBT_SHOW) }

source		{ CommandOrToken (PDBT_SOURCE) }
sourc		{ CommandOrToken (PDBT_SOURCE) }
sour		{ CommandOrToken (PDBT_SOURCE) }
sou		{ CommandOrToken (PDBT_SOURCE) }
so		{ CommandOrToken (PDBT_SOURCE) }

status		{ CommandOrToken (PDBT_STATUS) }
statu		{ CommandOrToken (PDBT_STATUS) }
stat		{ CommandOrToken (PDBT_STATUS) }
sta		{ CommandOrToken (PDBT_STATUS) }
st		{ CommandOrToken (PDBT_STATUS) }

summary		{ CommandOrToken (PDBT_SUMMARY) }
summar		{ CommandOrToken (PDBT_SUMMARY) }
summa		{ CommandOrToken (PDBT_SUMMARY) }
summ		{ CommandOrToken (PDBT_SUMMARY) }
sum		{ CommandOrToken (PDBT_SUMMARY) }
su		{ CommandOrToken (PDBT_SUMMARY) }

switch		{ CommandOrToken (PDBT_SWITCH) }
switc		{ CommandOrToken (PDBT_SWITCH) }
swit		{ CommandOrToken (PDBT_SWITCH) }
swi		{ CommandOrToken (PDBT_SWITCH) }
sw		{ CommandOrToken (PDBT_SWITCH) }

vars		{ CommandOrToken (PDBT_VARS) }
var		{ CommandOrToken (PDBT_VARS) }
va		{ CommandOrToken (PDBT_VARS) }
v		{ CommandOrToken (PDBT_VARS) }


{VFC}?{VC}*"*"	{	if (yyleng-1 >= MAX_PARSE_STRING_LENGTH)
			    return (PDBT_ERROR);
			else
			{
			    yytext[yyleng-1] = '\0';
			    strcpy(yylval.sval, yytext);
			    return (PDBT_WILDCARD);
			}
		}

{VFC}?{VC}*"*.wrap" {	if (yyleng-1 >= MAX_PARSE_STRING_LENGTH)
			    return (PDBT_ERROR);
			else
			{
			    yytext[yyleng-6] = '\0';
			    strcpy(yylval.sval, yytext);
			    return (PDBT_WILDCARD_WRAP);
			}
		}

{VFC}?{VC}*	{	HandleToken()  }

.?		{	yylval.ival = (int) yytext[0];
			return (PDBT_ERROR);
		}

%%

#ifdef FLEX_SCANNER

static void yyp_clear_input()
{
    yyrestart(yyin);
}

#else /* FLEX_SCANNER */

static void yyp_clear_input()
{
    yysptr=yysbuf;	
}

int yywrap()
{
    return (1);
}

#endif /* FLEX_SCANNER */

