/****************************************************************
Copyright (C) The University of Melbourne 1993
All Rights Reserved

Permission to use, copy, modify, and distribute this software and
its documentation for any purpose and without fee is hereby
granted, provided that the above copyright notice appear in all
copies and that both that the copyright notice and this
permission notice and warranty disclaimer appear in supporting
documentation, and that the name of The University of Melbourne 
or any of its entities not be used in advertising or publicity
pertaining to distribution of the software without specific,
written prior permission.

THE UNIVERSITY OF MELBOURNE DISCLAIMS ALL WARRANTIES WITH REGARD
TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE UNIVERSITY
OF MELBOURNE OR ANY OF ITS ENTITIES BE LIABLE FOR ANY
SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF
THIS SOFTWARE.

AUTHORS : Jason Lee (jasonl@cs.mu.oz.au)
	  Andrew Davison (ad@cs.mu.oz.au)

COMMENTS : This is the main function for bebop.
**************************************************************/

/* Include all the header files needed */
#include "bebop.h"

/* Extern declarations */
extern FILE *yyin;
extern int yydebug;
extern int nerrors;

/* Prototypes (forward declarations) */
void init_lex(void);
void init_symbol(void);
void init_parse(void);
int  yyparse(void);
void print_warnings(void);
void dump_tables(void);
void dump_tree(TREEPTR root);
void free_tables(void);
void free_tree(TREEPTR root);
char compile(void);
void check_iam(void);
void check_const(void);
static void dump_usage(char *program_name);

/* Global variables that can be accessed and are set in this file */
char *classname = NULL;		/* The class name of current program */
TREEPTR prog_root = NULL;	/* Root of parse tree produced */
char *file_nl = NULL;		/* Name of end file produced */
char *basefile = NULL;		/* Name of file being compiled */
char *fstr = NULL;		/* Name of file only no path  */
char pnpf = FALSE;		/* Do they wish a raw pnp file  */
char  warnings = FALSE;		/* Whether or not to issue certain warnings */

#ifndef USE_FUNCTIONS
void *tmp_pointer = NULL;	/* Used in all memory allocation */
#endif


/* Main : Reads command line options if any, and then compiles
	  each file given on command line to file.nl. May run
	  nc (Prolog Compiler) to farther compile file.nl.
	  See NU-Prolog Reference Manual, The University of
	  Melbourne for more information.
*/
int main(int argc, char **argv)
{

	/* Exit status after files are compiled
	   if any file caused errors then status is set
	   to FAILURE otherwise to SUCCESS
	*/
	char exitstatus = EXIT_SUCCESS;

	/* Some options which may be set, given default setting first */
	char bbdebug = FALSE;	/* Whether or not debugging is set */

	char run_nc = TRUE;	/* Whether or not to farther compile file.nl
				   via running nc on file.nl
				*/

	/* Program name */
	char *program_name;

	/* Suffix to add to file once compiled */
	const char *nl = NL;

	/* Command will end up being command to pass to system in order to
	   farther compile file.nl via running nc.
	*/
	char *command = NULL;

	/* Some general purpose variables */
	int i, j, k;
	int currfile;

	/* Get program name minus path stuff */
	if ((fstr = strrchr(argv[0], DIR)))
	{
		++fstr;
		program_name = fstr;
	}
	else
	{
		program_name = argv[0];
	}

	/* Make sure yydebug is FALSE */
	yydebug = FALSE;



	/* Process options and files, illegal option dump usage and exit
	   Defaults which are set	yydebug = FALSE
					bbdebug = FALSE
					run_nc = TRUE
					warnings = FALSE
					pnpf = FALSE
	   Thus the default is just to compile file --> file.nl and
	   then compile file.nl with nc.
	*/
	/* No files given */
	if (argc == 1)
	{
		dump_usage(program_name);
		return EXIT_FAILURE;
	}
	/* Start at second argument, first character */
	currfile = 1;
	j = 0;

	/* Look and process options, if any */
	if (argv[currfile][j] == DASH)
	{
		fstr = &argv[currfile++][++j];
		while(*fstr != EOS)
		{
			switch (*fstr)
			{
				case 'd' : yydebug = TRUE;
					   break;
				
				case 'b' : bbdebug = TRUE;
					   break;

				case 'n' : run_nc = FALSE;
					   break;

				case 'w' : warnings = TRUE;
					   break;

				case 'p' : pnpf = TRUE;
					   break;

				default  : dump_usage(program_name);
					   return EXIT_FAILURE;
			}
			++fstr;
		}
	}

	/* Check if any files after processing options */
	if (argc == currfile)
	{
		dump_usage(program_name);
		return EXIT_FAILURE;
	}

	/* Now processed options and there are files so compile each file
	*/
	for(; currfile < argc; ++currfile)
	{
		/* Open file to compile setting basefile*/
		basefile = argv[currfile];
		i =  strlen(basefile);

		/* Check that it is a bebop file, i.e look for ".bp"
		   at end if none present add one (i.e assume meaning
		   file.bp given file).
		*/
		file_nl = &basefile[i];
		if (i > LEN_SUFFIX)
		{
			file_nl -= (LEN_SUFFIX);
			if (strncmp(file_nl, SUFFIX, LEN_SUFFIX))
			{
				basefile = copy_string(basefile, (i + LEN_SUFFIX));
				basefile = strcat(basefile, SUFFIX);
			}
			else
			{
				i -= LEN_SUFFIX;
			}
		}
		else
		{
			basefile = copy_string(basefile, (i + LEN_SUFFIX));
			basefile = strcat(basefile, SUFFIX);
		}

		/* Now open up file for reading, if it fails skip it */
		if (!(yyin = fopen(basefile, "r")))
		{
			fprintf(stderr, "\n%s: cannot open file '%s", program_name, basefile);
			perror("'");
			fprintf(stderr, "\n\n");
			exitstatus = EXIT_FAILURE;
			continue;

		}
		basefile[i] = EOS;

		/* Now just extract the file name */
		if (( fstr = strrchr(basefile, DIR)))
		{
			++fstr;
		}
		else
		{
			fstr = basefile;
		}
		i = strlen(fstr);

		/* Make current final file name i.e file.nl */
		file_nl = copy_string(fstr, (i + LEN_NL));
		for(j = 0; j < LEN_NL; i++,j++)
		{
			file_nl[i] = nl[j];
		}
		file_nl[i] = EOS;

		/* Tell user what we are doing */
		fprintf(stderr,"\n\n******** Processing file %s.bp ********\n",basefile);
		/* Initialize compiler */
		init_symbol();
		init_lex();
		init_parse();

		/*Parse and create parse tree */
		yyparse();

		/* Do some after error checking*/
		check_iam();
		check_const();

		/* If errors forget about compiling */
		if (nerrors)
		{
			fprintf(stderr, "\n\nError%s: %d\n", ((nerrors > 1) ? "s " : " " ), nerrors);
			fprintf(stderr, "******** Compilation failed for %s.bp ********\n\n", basefile);
			exitstatus = EXIT_FAILURE;
		}
		else
		{
			/* Print out warnings if user wishes*/
			if (warnings)
			{
				print_warnings();
			}

			/*Time to compile and produce file.nl
			*/
			if (!compile())
			{
				fprintf(stderr, "******** Compilation failed for %s.bp ********\n\n", basefile);
				exitstatus = EXIT_FAILURE;
			}
			else
			{

				fprintf(stderr,"\n******** Compilation finished %s produced *********\n\n", file_nl);


				/* Do they wish to run nc on file produced */
				if (run_nc)
				{
					fprintf(stderr, "\n******** Calling nc to compile %s ********\n", file_nl);

					/* command will be "nc -c file.nl" */
					i = strlen(file_nl);
					command = copy_string(NC, (LEN_NC + i));
					for(j = 0, k = LEN_NC; j < i; k++, j++)
					{
						command[k] = file_nl[j];
					}
					command[k] = EOS;
	
					/* Call system to carry out command
				   	Note : From what i can gather if system
				   	returns 1 it failed, otherwise it
					succeeded
					*/
					if (system(command))
					{
						fprintf(stderr, "\n******** Compilation using nc failed for file %s ********\n", file_nl);
						exitstatus = EXIT_FAILURE;
					}
	
					else
					{
		
						fprintf(stderr, "\n******** Compilation using nc finished  ********\n");
					}
				
				}
			}
		}

		/* Debugging, not all that usefull but there for user */
		/* NOTE : Use at own risk, may core dump .... */
		/* This debugging stuff isn't very stable. */
		if (bbdebug)
		{
			fprintf(stderr,"\n\n******** Dumping internal tables ********\n");
			dump_tables();
			fprintf(stderr,"\n\n******** Dumping parse tree produced ********\n");
			dump_tree(prog_root);

		}

		/*Free memory */
		free_tree(prog_root);
		free_tables();
		fclose(yyin);
		free(command);
		free(file_nl);
		file_nl = NULL;
	}

	return exitstatus;
}



/*
   Dump_usage : just dumps the correct usage of the program 
   given the program name.
*/
static void dump_usage(char *program_name)
{

	fprintf(stderr, "Usage: %s [-nwph] files \n", program_name);
	fprintf(stderr, "Options : n = do not run nc (NU-Prolog Compiler)\n");
	fprintf(stderr, "          h = help, what you are reading now\n");
	fprintf(stderr, "          w = warn about possible predicates that are called and not defined\n");
	fprintf(stderr, "              within the class. Also about declarations with arities different\n              from defined predicates\n");
	fprintf(stderr, "          p = produce raw pnp file `file.pnp' \n");
	fprintf(stderr, "\nNOTE : 1. default is to compile bebop class to NU-Prolog, file -> file.nl and then compile with nc\n");

}

