%Copyright (C) The University of Melbourne 1993
%All Rights Reserved

%Permission to use, copy, modify, and distribute this software and
%its documentation for any purpose and without fee is hereby
%granted, provided that the above copyright notice appear in all
%copies and that both that the copyright notice and this
%permission notice and warranty disclaimer appear in supporting
%documentation, and that the name of The University of Melbourne 
%or any of its entities not be used in advertising or publicity
%pertaining to distribution of the software without specific,
%written prior permission.

%THE UNIVERSITY OF MELBOURNE DISCLAIMS ALL WARRANTIES WITH REGARD
%TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
%MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE UNIVERSITY
%OF MELBOURNE OR ANY OF ITS ENTITIES BE LIABLE FOR ANY
%SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
%WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
%IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
%ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF
%THIS SOFTWARE.

%AUTHORS : Jason Lee (jasonl@cs.mu.oz.au)
%	   Andrew Davison (ad@cs.mu.oz.au)


% Operators needed for bebop
% These are to do with sending messages.
% And saving/restoring history lists.
% Also with editing a previous goal that is hist
?- op(990, xfx, '::').
?- op(980, xfx, '!').
?- op(900, fx, r).
?- op(900, fx, s).
?- op(900, fx, e).


% the main entry point, any command line arguments are taken as bebop
% files to load, that is to compile and then load.
main([_ |As]) :-
	% do a load of the files given on the line
	$loadfiles(As),
	flushOutput(user),
	$bebop.
	

% Top level entry point.
$bebop :-
	% we want an interrupt to cause exit from bebop.
	signal(sigint, $exit),
	nl,
	writeln('Welcome to bebop version 1.0.'),
	flushOutput(user),
	$bebop_loop(1, hist(20, (Hist - Hist)), [], MesgList).

% $bebop_loop is the main looping stuff which writes the prompt
% and reads in the goals (queries) and executes the queries.
% Here we also keep the current command number, history list,
% the history list grows no greater than the past 20 queries
% and mapping of variable names to actual variables used also
% the global message list variable.

$bebop_loop(Num, Hist, Maps, MesgList) :-
	% Print out prompt
	format(user, "~d --> ", [Num]),
	flushOutput(user),
	!,

	% read in and tokenise Query and check it is valid term
	((getTokenList(TokenLt), tread(TokenLt, _)) ->

		% if user wishes to exit bebop via  or halt do so
		((streamEof(user) ; TokenLt = [[halt | atom]]) ->
			nl,
			writeln('Exiting bebop, bye.'),
			nl,
			flushOutput(user),
			% To force a reset of file errors, i.e end_of_file
			close(user_input),
			% This is needed to force a quick exit from bp.
			% due to the possible background suspension of
			% goals.
			exit(1)
		;

		% Otherwise, if the query was not a top level goal
		% (h, r, s, l, e or number) rename the query add to hist
		% list update query number then produce the new term
		% (goal/query) to execute. If top level goal then handle it.
		% Before executing renamed query add/unify variables in
		% query with ones in the Maps.

			($bebop_rename(TokenLt, Maps, Num, Hist, NewNum, NewHist, NewTokenLt) ->
				( NewTokenLt ~= $no_execute ->

					% Turn back into a term extracting
					% variables and there names. Now do
					% a match (mapping) of variable names
					% to ones actually used, unify any
					% variables with previous ones etc
					treadTerm(NewTokenLt, Goal, NameList, VarList),
					intToString(Num, StrNum),
					append("$", StrNum, IdNum),
					$varmapping(NameList, VarList, Maps, NewMaps, MesgList, IdNum, Num),

					$do_query(Goal, NameList, VarList)
				 ;
					NewMaps = Maps
				),
				$bebop_loop(NewNum, NewHist, NewMaps, MesgList)
			 ;
				% Renaming failed or not a "true" goal 
				% (i.e was h, r etc), the reason why
				% renaming failed should have been
				% reported to user
				$bebop_loop(Num, Hist, Maps, MesgList)
			)
		)
	 ;
		% Syntax error with query, don't add to history list 
		% or update command number.
		$bebop_loop(Num, Hist, Maps, MesgList)
	).


% This is where all Goals are renamed to use bebop class predicates
% Here we also take care of top level goals to do with history
% listing, saving, and restoring and listing of variable renamings
% also history editing

% History list requested.
$bebop_rename([[h | atom]], _, Num, Hist, Num, Hist, $no_execute) :-
	!,
	Hist = hist(_, (Hhd - _)),
	(nonvar(Hhd) ->
		nl,
		$print_history(Hhd),
		flushOutput(user)
	).

% Edit some history 
$bebop_rename([[e  | atom] | Es], Maps, Num, Hist, NewNum, NewHist, NewTokLt) :-
	!,
	% First get history number, if none then assume mean last query
	% i.e Num - 1.
	(Es = [[N | number]] ->
		NN = N
	 ;
		NN is Num - 1
	),
	Hist = hist(Max, (Hhd - _)),

	% Only edit if requested history edit number is valid
	% in the range (Num - Max) < N < Num and there is some history
	((nonvar(Hhd), $rtest(NN, Max, Num)) ->
		% Get the history to edit
		$get_hist(NN, Hhd, ToEhist),

		% Now what we do is write out the history to a tmp file
		% and then vi that file, read back in the editing file
		% and execute the new query after removing tmp file.
		Tmp = "/tmp/bebopedit",
		getppid(PPID),
		intToString(PPID, STRPPID),
		append(Tmp, STRPPID, Fileedit),
		% assume vi is in users path
		ViCommandb = "vi ",
		append(ViCommandb, Fileedit, ViCommande),

		% If we cannot create tmp file forget about editing
		(open(Fileedit, write, Stream) ->

			% write out history
			$writeTokLt(Stream, ToEhist),
			write(Stream, '.'),
			nl(Stream),
			close(Stream),

			% call the system to vi the file
			% if this fails tell user.
			(system(ViCommande, 0) ->

				% Once file has been edited read back in
				% if we can not tell user.
				(open(Fileedit, read, Stream2) ->
					$readfile(hist(Num, (H - H)), Stream2, _, Num, _),
					unlink(Fileedit),
					((nonvar(H), H = [h(_, Editedhist) | _]) ->
						% echo edited query to user
						nl,nl,
						$writeTokLt(user, Editedhist),
						nl,
						flushOutput(user),
						$bebop_rename(Editedhist, Maps, Num, Hist, NewNum, NewHist, NewTokLt)
					 ;
						NewNum = Num,
						NewHist = Hist,
						NewTokLt = $no_execute
					)
				 ;
					NewNum = Num,
					NewHist = Hist,
					NewTokLt = $no_execute,
					unlink(Fileedit),
					writeln(user_error, 'Unable to read in edited history'),
					nl(user_error),
					flushOutput(user_error)
				)
			 ;
				NewNum = Num,
				NewHist = Hist,
				NewTokLt = $no_execute,
				unlink(Fileedit),
				writeln(user_error, 'Unable to edit the history'),
				nl(user_error),
				flushOutput(user_error)
			)
		 ;
			NewNum = Num,
			NewHist = Hist,
			NewTokLt = $no_execute,
			writeln(user_error, 'Unable to create file tmp file to edit history'),
			nl(user_error),
			flushOutput(user_error)
		)
	 ;
		NewNum = Num,
		NewHist = Hist,
		NewTokLt = $no_execute
	).

% read in a history and add it to current one
$bebop_rename([[r | atom], [File | Type]], _, Num, Hist, NewNum, NewHist, $no_execute) :-
	!,
	(Type = atom ->
		(open(File, read, Stream) ->
			$readfile(Hist, Stream, NewHist, Num, NewNum),
			close(Stream)
		 ;
			write(user_error, 'Unable to open file : '),
			writeln(user_error, File),
			nl(user_error),
			flushOutput(user_error),
			NewNum = Num,
			NewHist = Hist
		)
	 ;
		writeln(user_error, 'Usage: r Filename, where Filename is an atom'),
		nl(user_error),
		flushOutput(user_error),
		NewNum = Num,
		NewHist = Hist
	).

% save current history to file
$bebop_rename([[s | atom], [File | Type]], _, Num,Hist,Num,Hist,$no_execute) :-
	!,
	(Type = atom ->
		(open(File, write, Stream) ->
			Hist = hist(_, (Hhd - _)),
			$print_h_file(Stream, Hhd),
			close(Stream)
		 ;
			write(user_error, 'Unable to create file : '),
			writeln(user_error, File),
			nl(user_error),
			flushOutput(user_error)
		)
	 ;
		writeln(user_error, 'Usage: s Filename, where Filename is an atom'),
		nl(user_error),
		flushOutput(user_error)
	 ).

% list variable mappings
$bebop_rename([[l | atom]], Maps, Num, Hist, Num, Hist, $no_execute) :-
	!,
	(Maps ~= [] ->
		nl,
		$arrange_maps(Maps, Maps2),
		$print_maps(Maps2),
		flushOutput(user)
	).

% Do a previous goal from history.
$bebop_rename([[G | number]], Maps, Num, Hist, NewNum, NewHist, NewTokLt) :-
	integer(G),
	!,
	Hist = hist(Max, (Hhd - _)),
	((nonvar(Hhd), $rtest(G, Max, Num)) ->
		$get_hist(G, Hhd, HistGoal),
		$writeTokLt(user, HistGoal),
		nl,
		flushOutput(user),
		$bebop_rename(HistGoal, Maps, Num, Hist, NewNum, NewHist, NewTokLt)
	 ;
		NewNum = Num,
		NewHist = Hist,
		NewTokLt = $no_execute
	).

% ordinary goal/query to rename
$bebop_rename(TokenLt, _, Num, Hist, NewNum, NewHist, NewTokenLt) :-
	!,
	$add_history(TokenLt, Hist, NewHist, Num, NewNum),
	$parse_rename(TokenLt, NewTokenLt).

% test if number is in a range, that is is a valid history number.
$rtest(G, Max, Num) :-
	NNum is Num - 1,
	(if (NNum =< Max) then
		Low = 0
	 else
		Low is NNum - Max
	),
	(if ((G =< Low) or (G >= Num)) then
		write(user_error, 'Goal is not in history list, only '),
		(if Low = 0 then
			CurNum is Num - 1
		 else
			CurNum = Max
		),
		write(user_error, CurNum),
		(if (CurNum =< 1) then
			writeln(user_error, ' goal currently in list.')
	 	else
			writeln(user_error, ' goals currently in list.')
		),
		nl(user_error),
		flushOutput(user_error),
		fail
	).


% parse and rename the query given by user to produce new tokenlist
$parse_rename([], []).

$parse_rename([[Var | var], ['::' | atom] | Rest], NewTokenLt) :-
	!,
	(treadTerm(Rest, Term, NameL, VarL) ->
		$varlisttostrin(VarL, SVarL),
		$extract_mc(Term, MesgTok, ConstTok, SepTok,OutTok,NameL,SVarL),
		NTT = [[$bb_send | atom], ['(' | atom], ["BB$_Mesg" | var], [',' | atom], [Var | var], [',' | atom] | Args],
		$noendappend(MesgTok, Args, NewEndM),
		$noendappend([[',' | atom]], NewEndM, NewEndM2),
		$noendappend(ConstTok, NewEndM2, CC),
		$noendappend([[')' | atom]], CC, NewTokenRest),
		(SepTok \= [] ->
			$noendappend(SepTok, NewTokenRest, NewTokR2),
			$parse_rename(OutTok, NewTokR2)
	 	;
			$parse_rename(OutTok, NewTokenRest)
		),
		NewTokenLt = NTT
	 ;
		writeln(user_error, 'The syntax error may be caused because the bebop message operator is being used illegally'),
		nl(user_error),
		flushOutput(user_error),
		NewTokenLt = $no_execute
	).

$parse_rename([[Name | atom], ['(' | atom] | Rest], NewTokenLt) :-
	!,
	$extract_par([['(' | atom] | Rest], Parext, LeftOver),
	Parext = [['(' |atom] | Args],
	Pred = [[Name | atom] | Parext],
	tread(Pred, Term),
	functor(Term, F, A),
	($pred_test(Pred, F, A) ->
		$noendappend(Pred, NewTokenLt, NewEnd)
	 ;
		append([['(' | atom], ["BB$_Mesg" | var], [',' | atom]], Args, NewArgs),
		NewPred = [[Name | atom] | NewArgs],
		$noendappend(NewPred, NewTokenLt, NewEnd)
	),
	$parse_rename(LeftOver, NewEnd).

$parse_rename([['(' | atom] | Rest], NewTokenLt) :-
	!,
	$extract_par([['(' | atom] | Rest], ExtPar, LeftOver),
	$parse_rename(ExtPar, NewTokenHead),
	$noendappend(NewTokenHead, NewTokenLt, NewTokenEnd),
	$parse_rename(LeftOver, NewTokenEnd).

$parse_rename([['[' | atom] | Rest], NewTokenLt) :-
	!,
	$extract_lst([['[' | atom] | Rest], ExtPar, LeftOver),
	$noendappend(ExtPar, NewTokenLt, NewTokenTail),
	$parse_rename(LeftOver, NewTokenTail).

$parse_rename([A | Rest], NewTokenLt) :-
	NewTokenLt = [A | NewTokR],
	$parse_rename(Rest, NewTokR).

% take a list of variables and convert to strings
$varlisttostrin([], []).

$varlisttostrin([V |VarL], SVarL) :-
	termToString(V, SV),
	SVarL = [SV | SM],
	$varlisttostrin(VarL, SM).

% pred test, test predicate to see if it is a system or loaded predicate
% and not a class
$pred_test(_, F, _) :-
	member(F, ['if', 'then', 'else', '->']).

$pred_test(Pred, F, A) :-
	predicateProperty(F, A, _),
	A2 is A + 1,
	((predicateProperty(F, A2, dynamic); predicateProperty(F, A2,compiled)),
	 \+ predicateProperty(F, A2, system),
	 \+ predicateProperty(F, A2, library),
	 \+ predicateProperty(F, A2, database)
	->
		write(user_error, 'Warning : goal query '),
		$writeTokLt(user, Pred),
		writeln(user_error, ' has been interpreted as a normal predicate due to a possible clash of definition, it may also be interpreted as a class'),
		nl(user_error),
		flushOutput(user_error)
	).

% extract paranthesisied or list terms
$extract_par([[')' | atom] | Rest], [[')' | atom]], Rest) :-
	!.

$extract_par([['(' | atom] | Rest], ExtPar, Left) :-
	!,
	ExtPar = [['(' | atom] | ExtR],
	$extract_par(Rest, ExtR, Left).

$extract_par([A | Rest], ExtPar, Left) :-
	ExtPar = [A | ExtR],
	$extract_par(Rest, ExtR, Left).

$extract_lst([[']' | atom] | Rest], [[']' | atom]], Rest) :-
	!.

$extract_lst([['[' | atom] | Rest], ExtPar, Left) :-
	!,
	ExtPar = [['[' | atom] | ExtR],
	$extract_lst(Rest, ExtR, Left).

$extract_lst([A | Rest], ExtPar, Left) :-
	ExtPar = [A | ExtR],
	$extract_lst(Rest, ExtR, Left).

% noendappend appends a list to a variable and creates a new list with a 
% variable end
$noendappend([], A, A).

$noendappend([A | As], Var, End) :-
	Var = [A | Var2],
	$noendappend(As, Var2, End).


% extract the message and any constraints given for a message command
$extract_mc(((Mesg ! Const), OutTerm), MesgTok, ConstTok, SepTok, OutTok,NameL,VarL) :-
	!,
	termToString(Mesg, SMesg),
	termToString(Const, SConst),
	termToString(OutTerm, SOutTerm),
	tokenize(SMesg, MesgTok1),
	$rejunk(MesgTok1, NameL, VarL, MesgTok),
	tokenize(SConst, ConstTok1),
	$rejunk(ConstTok1, NameL, VarL, ConstTok),
	tokenize(SOutTerm, OutTok1),
	$rejunk(OutTok1, NameL, VarL, OutTok),
	SepTok = [[',' | atom]].

$extract_mc(((Mesg ! Const); OutTerm), MesgTok, ConstTok, SepTok, OutTok,NameL,VarL) :-
	!,
	termToString(Mesg, SMesg),
	termToString(Const, SConst),
	termToString(OutTerm, SOutTerm),
	tokenize(SMesg, MesgTok1),
	$rejunk(MesgTok1, NameL, VarL, MesgTok),
	tokenize(SConst, ConstTok1),
	$rejunk(ConstTok1, NameL, VarL, ConstTok),
	tokenize(SOutTerm, OutTok1),
	$rejunk(OutTok1, NameL, VarL, OutTok),
	SepTok = [';' | atom].

$extract_mc((Mesg, OutTerm), MesgTok, ConstTok, SepTok, OutTok,NameL,VarL) :-
	termToString(Mesg, SMesg),
	termToString(OutTerm, SOutTerm),
	tokenize(SMesg, MesgTok1),
	$rejunk(MesgTok1, NameL, VarL, MesgTok),
	tokenize(SOutTerm, OutTok1),
	$rejunk(OutTok1, NameL, VarL, OutTok),
	ConstTok = [['[' | atom], [']' | atom]],
	SepTok = [[',' | atom]].

$extract_mc((Mesg; OutTerm), MesgTok, ConstTok, SepTok, OutTok,NameL,VarL) :-
	termToString(Mesg, SMesg),
	termToString(OutTerm, SOutTerm),
	tokenize(SMesg, MesgTok1),
	$rejunk(MesgTok1, NameL, VarL, MesgTok),
	tokenize(SOutTerm, OutTok1),
	$rejunk(OutTok1, NameL, VarL, OutTok),
	ConstTok = [['[' | atom], [']' | atom]],
	SepTok = [';' | atom].

$extract_mc((Mesg ! Const), MesgTok, ConstTok, [], [],NameL,VarL) :-
	!,
	termToString(Mesg, SMesg),
	termToString(Const, SConst),
	tokenize(SMesg, MesgTok1),
	$rejunk(MesgTok1, NameL, VarL, MesgTok),
	tokenize(SConst, ConstTok1),
	$rejunk(ConstTok1, NameL, VarL, ConstTok).

$extract_mc(Mesg, MesgTok, ConstTok, [], [],NameL,VarL) :-
	!,
	termToString(Mesg, SMesg),
	tokenize(SMesg, MesgTok1),
	$rejunk(MesgTok1, NameL, VarL, MesgTok),
	ConstTok = [['[' | atom], [']' | atom]].

% Rename junk variable names back to original names
$rejunk([], _, _, []).

$rejunk([[S | var] | Rst], RealN, Jvar, Res) :-
	!,
	$matchj(S, Jvar, RealN, NewS),
	Res = [[NewS | var] | NRes],
	$rejunk(Rst, RealN, Jvar, NRes).

$rejunk([A | As], Re, J, R) :-
	R = [A | Rs],
	$rejunk(As, Re, J, Rs).

% match junk name with real name to use
$matchj(S, [], [], S).

$matchj(S, [Jv | Js], [Re | Rs], NS) :-
	(if S = Jv then
		NS = Re 
	 else
		$matchj(S, Js, Rs, NS)
	).

% read in a file of saved history queries
$readfile(Hist, Stream, NewHist, Num, NewNum) :-
	(if (not streamEof(Stream)) then
		((getTokenList(Stream, Token), tread(Token, _)) ->
			$readl2(Hist, Stream, Token, NewHist, Num, NewNum)
		  ;
			writeln(user_error, 'Skipping error in file'),
			nl(user_error),
			flushOutput(user_error),
			$readfile(Hist, Stream, NewHist, Num, NewNum)
		)
	else
		NewHist = Hist,
		NewNum = Num
	).

$readl2(Hist, Stream, Tok, NewHist, Num, NewNum) :-
	(if (Tok = [[[] | end_of_file]]) then
		$readfile(Hist, Stream, NewHist, Num, NewNum)
	 else
		$add_history(Tok, Hist, NHist, Num, NN),
		$readfile(NHist, Stream, NewHist, NN, NewNum)
	).

% get goal associated with number
$get_hist(Num, Hist, NG) :-
	(nonvar(Hist) ->
		Hist = [h(N, G) | As],
		(if (Num = N) then
			NG = G
		else
			$get_hist(Num, As, NG)
		)
	).

% print_history justs prints out history in nice format
$print_history(Hist) :-
	(nonvar(Hist) ->
		Hist = [h(Num, G) | Rs],
		write('\t'),
		write(Num),
		write('\t'),
		$writeTokLt(user, G),
		write('.'),
		nl,
		flushOutput(user),
		$print_history(Rs)
	).


% print_h_file justs prints out history to file
$print_h_file(Stream, Hist) :-
	(nonvar(Hist) ->
		Hist = [h(_, G) | Rs],
		$writeTokLt(Stream, G),
		write(Stream, '.'),
		nl(Stream),
		flushOutput(Stream),
		$print_h_file(Stream, Rs)
	).

% print out varaible mappings to user
$print_maps([]).

$print_maps([p(N, Lmaps) | Rs]) :-
	write('\t'),
	write(N),
	write('\t'),
	$printlmaps(Lmaps),
	flushOutput(user),
	nl,
	$print_maps(Rs).

% printlmaps prints out mappings of variable names.
$printlmaps([]).

$printlmaps([(ON, NN) | Rs]) :-
	atomToString(AO, ON),
	atomToString(AN, NN),
	write(AO),
	write(' -> '),
	write(AN),
	write('  '),
	flushOutput(user),
	$printlmaps(Rs).

% arrange maps to a structure that is suitable and easier to print out
$arrange_maps(Maps, Maps2) :-
	Maps = [v(_, _, _, N) | _],
	N2 is N + 1,
	$arrange_maps(Maps, N2, [], Maps2, CL).

$arrange_maps([], _, M, M, []).

$arrange_maps([v(NewName, OldName, _, N) | Rs], CNum, Map, Map2, CL) :-
	(if (N = CNum) then
		CL = [(OldName, NewName) | NCL],
		$arrange_maps(Rs, CNum, Map, Map2, NCL)
	 else
		CL = [],
		NMap = [p(N, NCL) | Map],
		NCL = [(OldName, NewName) | RCL],
		$arrange_maps(Rs, N, NMap, Map2, RCL)
	).


% writeTokLt prints out the list of token, type pairs to stream
$writeTokLt(_, []) :- !.

$writeTokLt(Stream, [[A | var] | Rest])  :-
	!,
	atomToString(Atom, A),
	write(Stream, Atom),
	$writeTokLt(Stream, Rest).

$writeTokLt(Stream, [[A | quoted] | Rest]) :-
	!,
	writev(Stream, [quoteall], A),
	$writeTokLt(Stream, Rest).

$writeTokLt(Stream, [[A | _] | Rest]) :-
	!,
	write(Stream, A),
	$writeTokLt(Stream, Rest).

% write answer writes out the bindings for the goal the executed
?- $write_answer2(_, A) when ground(A).

$write_answer2([], []).

$write_answer2([A | As], [B | Bs]) :-
	atomToString(At, A),
	nl,
	write(At),
	write(' = '),
	write(B),
	flushOutput(user),
	$write_answer2(As, Bs).

% write answer writes out bindings but first removes BB$_Mesg form list
$write_answer([], []).

$write_answer([A | As], [B |Bs]) :-
	( if (A = "BB$_Mesg") then
		$write_answer2(As, Bs)
	  else
		$write_answer1([A | As], [B | Bs], NewA, NewB),
		$write_answer2(NewA, NewB)
	).
		
% Remove BB$_Mesg from list for now
$write_answer1([], [], [], []).

$write_answer1([A | As], [B | Bs], NewA, NewB) :-
	( if (A = "BB$_Mesg") then
		NewA = As,
		NewB = Bs
	  else
		NewA = [A | NAs],
		NewB = [B | NBs],
		$write_answer1(As, Bs, NAs, NBs)
	).

% do query handles failures and interrupts
% thanks to Jeff Schultz for the help with the code below
$do_query(Q, NameList, VarList) :-
	signal(sigint, $sighandle),
	catch($run_query(Q, NameList, VarList), _),
	signal(sigint, $exit).

$run_query(Q, NameList, VarList) :-
	call(Q),
	$write_answer(NameList, VarList),
	nl,
	write('Again ? (<return> = no or `;\' = yes) '),
	flushOutput(user),
	$my_get0(T),
	$readmore(T).

$run_query(_, _, _) :-
	writeln('fail').

% get chars until newline or comma is read
$my_get0(T) :-
	getl(Line),
	(if streamEof(user) then
		close(user_input),
		T = end
	 else
		$what(Line, T)
	).

% what looks at the line to see if a comma ';' is before the newline
$what([0'\n | _], end) :- !.

$what([0'; | _], more) :- !.

$what([_ | Rs], T) :-
	$what(Rs, T).

%readmore either succeeds or fail depending on term passed to it
$readmore(end).

$readmore(more) :-
	fail.

% handle interrupt signals
$sighandle(_, _, _) :-
        nl,
        writeln(user_error, 'Query interrupted, terminating query.'),
	nl(user_error),
	flushOutput(user_error),
        throw(interrupt).

% varmapping maps variable names to new ones that can be used later if
% desired. Thus if the name has a unique id "$n" as it's suffix then
% it is bound to the earlier variable with same name minus id
$varmapping([], [], M, M, _, _, _).

$varmapping(["BB$_Mesg" | Ns], [Var | Vs], Maps,NewMaps,MesgList,StrNum,Num) :-
	!,
	Var = MesgList,
	$varmapping(Ns, Vs, Maps, NewMaps, MesgList, StrNum,  Num).

$varmapping([Name | Ns], [Var | Vs], Maps, NewMaps, MesgList, StrNum, Num) :-
	( member(v(Name, _, ActV, _), Maps) ->
		Var = ActV,
		$varmapping(Ns, Vs, Maps, NewMaps, MesgList, StrNum,  Num)
	 ;
		append(Name, StrNum, NewName),
		Maps2 = [v(NewName, Name, Var, Num) | Maps],
		$varmapping(Ns, Vs, Maps2, NewMaps, MesgList, StrNum, Num)
	).


% Add goal to history list
$add_history(Tok, Hist, NewHist, Num, NewNum) :-
	Hist = hist(Max, (Hhd - Htl)),
	Num > Max,
	!,
	Hhd = [_ | NHhd],
	Htl = [h(Num, Tok) | NHtl],
	NewNum is Num + 1,
	NewHist = hist(Max, (NHhd - NHtl)).

$add_history(Tok, hist(M, (Hhd - Htl)), NewHist, Num, NewNum) :-
	NewNum is Num + 1,
	Htl = [h(Num, Tok) | NHtl],
	NewHist = hist(M, (Hhd - NHtl)).
	
