/* $Id: inout.c,v 1.47 1993/10/14 18:46:05 bd Exp $ */ 

#include "include.h"
#include "term.h"
#include "tree.h"
#include "predicate.h"
#include "names.h"
#include "exstate.h"
#include "engine.h"
#include "storage.h"
#include "unify.h"
#include "copy.h"
#include "gc.h"
#include "initial.h"
#include "config.h"
#include "display.h"
#include "foreign.h"
#include "error.h"
#include "inout.h"

#ifdef macintosh
#include <unix.h>
#endif

typedef struct stream {
  struct gvamethod	*method;
  envid 	*env;
  FILE 		*file;
} stream;

Gvainfo newstream();
bool 	unifystream();
int 	printstream();
Gvainfo	copystream();
Gvainfo	gcstream();
int     uncopiedstream();
envid	*deallocatestream();
int	killstream();
bool    sendstream();

static gvamethod streammethod = {
  newstream,
  unifystream,
  printstream,
  copystream,
  gcstream,
  uncopiedstream,
  deallocatestream,
  killstream,
  NULL,
  sendstream,
  NULL,
  NULL
};


Term term_eof;
Term term_stream;

Functor functor_fclose_1;
Functor functor_typeof;
Functor functor_fflush_1;
Functor functor_putc_1;
Functor functor_putc_2;
Functor functor_getc_1;
Functor functor_getc_2;
Functor functor_puts_1;
Functor functor_puts_2;
Functor functor_putd_1;
Functor functor_putf_1;
Functor functor_write_generic;
Functor functor_write_constraint;


Atom atom_fclose;
Atom atom_fflush;


/*
 * METHOD DEFINITIONS
 *
 */

Gvainfo newstream(old)
     stream *old;
{
  stream *new;
  NEW(new,stream);
  return (Gvainfo) new;
}

Gvainfo copystream(old,new)
     stream *old, *new;
{
  new->file = old->file;
  return (Gvainfo) new;
}

Gvainfo gcstream(old,new,gcst)
     stream *old, *new;
     gcstatep gcst;
{
  new->file  = old->file;
  return (Gvainfo) new;
}


void externstream()
{
  return;
}


/* deallocatestream() is called by the gc if the generic object
 * is unreferenced after a gc. If the stream has becomed garbage
 * it can be closed immediately. 0 is returned to indicate that
 * no further actions need to be taken by the gc.
 */

envid *deallocatestream(strm)
     stream *strm;
{
  if(strm->file != NULL)
    fclose(strm->file);

  return NULL;
}

/* uncopiedstream() is called by the copier if the stream has not been
 * copied. There is no need to add an extra object on the close list.
 */

int uncopiedstream(strm)
     stream *strm;
{
  return 0;
}


bool unifystream(strm,y,andb,exs)
     Term strm;
     Term y;
     andbox *andb;
     exstate *exs;
{
  if(Eq(strm, y))
    return TRUE;
  return FALSE;
}


int killstream(strm)
     stream *strm;
{
  if(strm->file != NULL)
    fclose(strm->file);

  return 1;
}



int printstream(file,strm,tsiz)
     FILE *file;
     stream *strm;
     int tsiz;
{
  if(strm->file == stdin)
    fprintf(file, "{stream: stdin}");
  else if(strm->file == stdout)
    fprintf(file, "{stream: stdout}");
  else if(strm->file == stderr)
    fprintf(file, "{stream: stderr}");
  else
    fprintf(file, "{stream: %#lx}", (unsigned long)strm);
  return 1;
}


bool sendstream(message, self, exs)
     Term message, self;
     exstate *exs;
{
  stream *streamself = (stream *)(RefGva(Ref(self)));

  bool method_fclose_1();
  bool method_typeof();
  bool method_fflush_1();
  bool method_putc_1();
  bool method_putc_2();
  bool method_getc_1();
  bool method_getc_2();
  bool method_write_generic();
  bool method_write_constraint();
  bool method_puts_1();
  bool method_puts_2();
  bool method_putd_1();
  bool method_putf_1();

  if(IsATM(message)) {
    Atom op = Atm(message);

    if(op == atom_fclose) {
      fclose(streamself->file);
      streamself->file = NULL;
    } else if(op == atom_fflush)
      fflush(streamself->file);
    else
      return FALSE;

    return TRUE;
  }

  if(IsSTR(message)) {
    Functor op = StrFunctor(Str(message));
    int arity = op->arity;
    Term X0, X1;

    if(arity > 0) {
      GetStrArg(X0, Str(message), 0);
      Deref(X0, X0);
    }
    if(arity > 1) {
      GetStrArg(X1, Str(message), 1);
      Deref(X1, X1);
    }
    
    if(op == functor_fclose_1)
      return method_fclose_1(streamself, X0, exs);
    else if(op == functor_typeof)
      return method_typeof(X0, exs);
    else if(op == functor_fflush_1)
      return method_fflush_1(streamself, X0, exs);
    else if(op == functor_putc_1)
      return method_putc_1(streamself, X0, exs);
    else if(op == functor_putc_2)
      return method_putc_2(streamself, X0, X1, exs);
    else if(op == functor_getc_1)
      return method_getc_1(streamself, X0, exs);
    else if(op == functor_getc_2)
      return method_getc_2(streamself, X0, X1, exs);
    else if(op == functor_write_generic)
      return method_write_generic(streamself, X0, exs);
    else if(op == functor_write_constraint)
      return method_write_constraint(streamself, X0, exs);
    else if(op == functor_puts_1)
      return method_puts_1(streamself, X0, exs);
    else if(op == functor_puts_2)
      return method_puts_2(streamself, X0, X1, exs);
    else if(op == functor_putd_1)
      return method_putd_1(streamself, X0, exs);
    else if(op == functor_putf_1)
      return method_putf_1(streamself, X0, exs);
    else
      return FALSE;
  }

  IfVarSuspend(message);
  return FALSE;
}


/*
 * SUPPORT
 *
 */

stream *open_stream(name,mode,exs,andb)
     Term name;
     Term mode;
     exstate *exs;
     andbox *andb;
{
  stream *strm;
  FILE *filep;
  char pathBuf[MAXPATHLEN+1];

  if (IsATM(name)) {
    if(!expand_file_name(AtmPname(Atm(name)),pathBuf))
      return NULL;

    filep = fopen(pathBuf, AtmPname(Atm(mode)));
  } else
    filep = fdopen(IntVal(Int(name)), AtmPname(Atm(mode))); /* file number */
    
  if(filep == NULL)
    return  NULL;

  MakeGvainfo(strm,stream,&streammethod,andb);

  add_gvainfo_to_close((Gvainfo)strm,exs);

  strm->file = filep;
  
  return strm;
}


/*
 * MAKING NEW STREAMS
 *
 */

/* fopen(FileName, Mode, Stream) */

bool akl_fopen_3(Arg)
     Argdecl;
{
  Term filename, mode, streamt, tmpv;
  stream *strm;
  
  Deref(filename, A(0));
  Deref(mode,     A(1));
  Deref(streamt,   A(2));

  IfVarSuspend(filename);
  if(!IsATM(filename) && !IsINT(filename))
    USAGE_FAULT("fopen/3: 1st argument not an atom or an integer");

  IfVarSuspend(mode);
  if(!IsATM(mode))
    USAGE_FAULT("fopen/3: 2nd argument not an atom");

  strm = open_stream(filename, mode, exs, exs->andb);
  if(strm == NULL) {
    WARNING("fopen/3: fopen returned NULL - failing");
    return FALSE;
  }

  MakeCvaTerm(tmpv, (Gvainfo)strm);

  return unify(streamt, tmpv, exs->andb, exs);
} 

/* fopen(FileName, Mode, Stream, Result) */

bool akl_fopen_4(Arg)
     Argdecl;
{
  Term filename, mode, streamt, result, tmps, tmpr;
  stream *strm;
  
  Deref(filename, A(0));
  Deref(mode,     A(1));
  Deref(streamt,  A(2));
  Deref(result,   A(3));

  IfVarSuspend(filename);
  if(!IsATM(filename) && !IsINT(filename))
    USAGE_FAULT("fopen/3: 1st argument not an atom or an integer");

  IfVarSuspend(mode);
  if(!IsATM(mode))
    USAGE_FAULT("fopen/3: 2nd argument not an atom");

  strm = open_stream(filename, mode, exs, exs->andb);

  if(strm == NULL) {
    tmps = term_nil;
    EXCEPTION(tmpr, ERRNO);
  } else {
    MakeCvaTerm(tmps, (Gvainfo)strm);
    tmpr = term_true;
  }
  
  return unify(streamt, tmps, exs->andb, exs) &&
	 unify(result, tmpr, exs->andb, exs);
} 

/* stdin(StdIn) */

bool akl_stdin(Arg)
     Argdecl;
{
  Term streamt, tmpv;
  stream *strm;

  Deref(streamt, A(0));

  MakeGvainfo(strm, stream, &streammethod, exs->andb);
  strm->file = stdin;

  MakeCvaTerm(tmpv, (Gvainfo)strm);
  return unify(streamt, tmpv, exs->andb, exs);
}


/* stdout(StdOut) */

bool akl_stdout(Arg)
     Argdecl;
{
  Term streamt, tmpv;
  stream *strm;
  
  Deref(streamt, A(0));

  MakeGvainfo(strm, stream, &streammethod, exs->andb);
  strm->file = stdout;

  MakeCvaTerm(tmpv, (Gvainfo)strm);
  return unify(streamt, tmpv, exs->andb, exs);
}


/* stderr(StdErr) */

bool akl_stderr(Arg)
     Argdecl;
{
  Term streamt, tmpv;
  stream *strm;
  
  Deref(streamt, A(0));

  MakeGvainfo(strm, stream, &streammethod, exs->andb);
  strm->file = stderr;

  MakeCvaTerm(tmpv, (Gvainfo)strm);
  return unify(streamt, tmpv, exs->andb, exs);
}


/*
 * METHODS
 *
 */


/* fclose(Result)@Stream */

bool method_fclose_1(self, result, exs)
     stream *self;
     Term result;
     exstate *exs;     
{
  Term tmp;

  if(fclose(self->file) == EOF)
    EXCEPTION(tmp, ERRNO)
  else {
    self->file = NULL;
    tmp = term_true;
  }
    
  return unify(result, tmp, exs->andb, exs);
}  


/* fclose@Stream */
/* - in sendstream */

/* typeof(Type)@Stream */

bool method_typeof(type, exs)
     Term type;
     exstate *exs;     
{
  return unify(type, term_stream, exs->andb, exs);
}


/* fflush(Result)@Stream */

bool method_fflush_1(self, result, exs)
     stream *self;
     Term result;
     exstate *exs;     
{
  Term tmp;

  if(fflush(self->file) == EOF)
    EXCEPTION(tmp, ERRNO)
  else
    tmp = term_true;

  return unify(result, tmp, exs->andb, exs);
}


/* fflush@Stream */
/* - in sendstream */


/* putc(Char,Result)@Stream */

bool method_putc_2(self, ch, result, exs)
     stream *self;
     Term ch;
     Term result;
     exstate *exs;
{
  Term tmp;

  IfVarSuspend(ch);

  if(!IsINT(ch))
    USAGE_FAULT("putc/2@stream: 1st argument not an integer");

  if(putc((char)IntVal(Int(ch)), self->file) == EOF)
    EXCEPTION(tmp, ERRNO)
  else
    tmp = term_true;
  
  return unify(result, tmp, exs->andb, exs);
}

/* putc(Char)@Stream */

bool method_putc_1(self, ch, exs)
     stream *self;
     Term ch;
     exstate *exs;
{
  IfVarSuspend(ch);

  if(!IsINT(ch))
    USAGE_FAULT("putc/1@stream: 1st argument not an integer");

  putc((char)IntVal(Int(ch)), self->file);
  return TRUE;
}


/* getc(Char,Result)@Stream */

bool method_getc_2(self, ch, result, exs)
     stream *self;
     Term ch;
     Term result;
     exstate *exs;
{
  int e;
  Term tmp1, tmp2;
  
  e =  getc(self->file);
  
  if(e == EOF) {
    if(!feof(self->file))
      EXCEPTION(tmp2, ERRNO)
    else {
      EXCEPTION(tmp2, term_eof)
      e = -1;
    }
  } else {
    tmp2 = term_true;
  }
  MakeIntegerTerm(tmp1, e);

  return unify(ch, tmp1, exs->andb, exs) &&
         unify(result, tmp2, exs->andb, exs);
}


/* getc(Char)@Stream */

bool method_getc_1(self, ch, exs)
     stream *self;
     Term ch;
     exstate *exs;
{
  int e;
  Term tmp;
  
  e =  getc(self->file);
  
  if(e == EOF) {
    e = -1;
  }
  MakeIntegerTerm(tmp, e);

  return unify(ch, tmp, exs->andb, exs);
}


bool method_puts_1(self, string, exs)
     stream *self;
     Term string;
     exstate *exs;
{
  IfVarSuspend(string);

  if(!IsATM(string))
    USAGE_FAULT("puts/1@stream: argument not an atom");

  fprintf(self->file, "%s", AtmPname(Atm(string)));
  return TRUE;
}

bool method_puts_2(self, string, ch, exs)
     stream *self;
     Term string;
     Term ch;
     exstate *exs;
{
  register char *p;
  register char c, cq;
  register FILE *f = self->file;

  IfVarSuspend(string);
  IfVarSuspend(ch);

  if(!IsATM(string))
    USAGE_FAULT("puts/2@stream: 1st argument not an atom");

  p = AtmPname(Atm(string));

  if(!IsINT(ch))
    USAGE_FAULT("puts/2@stream: 1st argument not an integer");

  cq = IntVal(Int(ch));

  putc(cq, f);
  while((c = *(p++))) {
    if(c == cq)
      putc(c, f);
    putc(c, f);
  }
  putc(cq, f);

  return TRUE;
}

bool method_putd_1(self, i, exs)
     stream *self;
     Term i;
     exstate *exs;
{
  IfVarSuspend(i);

  if(!IsINT(i))
    USAGE_FAULT("putd/1@stream: argument not an integer");

  fprintf(self->file, "%d", IntVal(Int(i)));
  return TRUE;
}



bool method_putf_1(self, flt, exs)
     stream *self;
     Term flt;
     exstate *exs;
{
  char temp[MAXNUMLEN];

  extern void ftoa();

  IfVarSuspend(flt);

  if(!IsFLT(flt))
    USAGE_FAULT("putf/1@stream: argument not a float");

  ftoa(FltVal(Flt(flt)), temp);
  fprintf(self->file, "%s", temp);

  return TRUE;
}




/*
 * WRITING TERMS
 *
 */


/* write_generic(Gen)@Stream */

bool method_write_generic(self, x, exs)
     stream *self;
     Term x;
     exstate *exs;
{
  IfVarSuspend(x);
  if(!IsGEN(x))
    return FALSE;

  Gen(x)->method->print(self->file, Gen(x), -1);
  return TRUE;
}


/* write_constraint(Var)@Stream */

bool method_write_constraint(self, x, exs)
     stream *self;
     Term x;
     exstate *exs;
{
  if(IsCvaTerm(x)) {
    RefCvaMethod(Ref(x))->print(self->file, RefGva(Ref(x)), -1);
  }

  if(!IsVar(x)) {
    WARNING("Constrained variable became determined during printing!");
  }

  return TRUE;
}


/*
 * LOW-LEVEL
 *
 */

/* $display(Term) */

bool akl_display(Arg)
     Argdecl;
{
  display_term(A(0),-1);
  return TRUE;
}


void initialize_inout() {
  
  term_eof = TagAtm(store_atom("eof"));
  term_stream = TagAtm(store_atom("stream"));

  functor_fclose_1 = store_functor(store_atom("fclose"),1);
  functor_typeof = store_functor(store_atom("typeof"),1);
  functor_fflush_1 = store_functor(store_atom("fflush"),1);
  functor_putc_1 = store_functor(store_atom("putc"),1);
  functor_putc_2 = store_functor(store_atom("putc"),2);
  functor_getc_1 = store_functor(store_atom("getc"),1);
  functor_getc_2 = store_functor(store_atom("getc"),2);
  functor_puts_1 = store_functor(store_atom("puts"),1);
  functor_puts_2 = store_functor(store_atom("puts"),2);
  functor_putd_1 = store_functor(store_atom("putd"),1);
  functor_putf_1 = store_functor(store_atom("putf"),1);
  functor_write_generic = store_functor(store_atom("write_generic"),1);
  functor_write_constraint = store_functor(store_atom("write_constraint"),1);

  atom_fclose = store_atom("fclose");
  atom_fflush = store_atom("fflush");

  define("$display",akl_display,1);

  define("fopen", akl_fopen_3, 3);
  define("fopen", akl_fopen_4, 4);

  define("stdout", akl_stdout, 1);
  define("stdin", akl_stdin, 1);
  define("stderr", akl_stderr, 1);
}
