/*-------------------------------------------------------------------------*/
/* Prolog to Wam Compiler               INRIA Rocquencourt - ChLoE Project */
/* C Run-time                                           Daniel Diaz - 1991 */
/*                                                                         */
/* Wam Implementation                                                      */
/*                                                                         */
/* wam_engine.c                                                            */
/*-------------------------------------------------------------------------*/
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
int sscanf();

#define WAM_ENGINE

#include "wam_engine.h"
#include "fd_engine.h"



/*---------------------------------*/
/* Constants                       */
/*---------------------------------*/

#define ERR_UNKNOWN_STACK          "Error: Unknown stack <%s>"




/*---------------------------------*/
/* Type Definitions                */
/*---------------------------------*/

/*---------------------------------*/
/* Global Variables                */
/*---------------------------------*/

#ifdef WAM_PROFILE

static int max_local_used_size      =0;
static int global_max_used_size     =0;
static int max_trail_used_size      =0;

static int nb_of_create_choice_point=0;
static int nb_of_update_choice_point=0;
static int nb_of_delete_choice_point=0;

#endif




/*---------------------------------*/
/* Function Prototypes             */
/*---------------------------------*/




/*-------------------------------------------------------------------------*/
/* CREATE_CHOICE_POINT                                                     */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Create_Choice_Point(CodePtr codep_alt,int arity)

{
 WamWord *adr;
 int      i;

#ifdef WAM_PROFILE

 nb_of_create_choice_point++;

#endif

 adr=B;
 B=Local_Top+CHOICE_STATIC_SIZE+arity;

 ALTB(B)=(WamWord) codep_alt;
 CPB(B) =(WamWord) CP;
 EB(B)  =(WamWord) E;
 BB(B)  =(WamWord) adr;
 BCB(B) =(WamWord) BC;
 HB(B)  =(WamWord) H;
 TRB(B) =(WamWord) TR;

 for(i=0;i<arity;i++)
     AB(B,i)=A(i);

 STAMP++;
}




/*-------------------------------------------------------------------------*/
/* UPDATE_CHOICE_POINT                                                     */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Update_Choice_Point(CodePtr codep_alt,int arity)

{
 WamWord *adr;
 int      i;

#ifdef WAM_PROFILE

 nb_of_update_choice_point++;

#endif

 ALTB(B)=(WamWord) codep_alt;

 Untrail(TRB(B),TR)

 CP=(WamCont)   CPB(B);
 E =(WamWord *) EB(B);
 BC=(WamWord *) BCB(B);
 H =(WamWord *) HB(B);

 for(i=0;i<arity;i++)
     A(i)=AB(B,i);
}




/*-------------------------------------------------------------------------*/
/* DELETE_CHOICE_POINT                                                     */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Delete_Choice_Point(int arity)

{
 WamWord *adr;
 int      i;

#ifdef WAM_PROFILE

 nb_of_delete_choice_point++;

#endif

 Untrail(TRB(B),TR)

 CP=(WamCont)   CPB(B);
 E =(WamWord *) EB(B);
 BC=(WamWord *) BCB(B);
 H =(WamWord *) HB(B);

 for(i=0;i<arity;i++)
     A(i)=AB(B,i);

 B =(WamWord *) BB(B);      /* warning B must be the last element restored */

 STAMP--;
}




/*-------------------------------------------------------------------------*/
/* GET_NIL                                                                 */
/*                                                                         */
/*-------------------------------------------------------------------------*/
Bool Get_Nil(int a)

{
 WamWord word,tag,*adr;

 Deref(&A(a),word,tag,adr)
 if (tag==REF)
    {
     Bind_UV(adr,word_nil)
     return TRUE;
    }

 return (word==word_nil);
}




/*-------------------------------------------------------------------------*/
/* GET_CONSTANT                                                            */
/*                                                                         */
/*-------------------------------------------------------------------------*/
Bool Get_Constant(AtomInf *atom,int a)

{
 WamWord word,tag,*adr;

 Deref(&A(a),word,tag,adr)
 switch(tag)
    {
     case REF:
         Bind_UV(adr,Tag_Value(CST,atom))
         return TRUE;

     case CST:
         return (UnTag_CST(word)==atom);
    }

 return FALSE;
}




/*-------------------------------------------------------------------------*/
/* GET_INTEGER                                                             */
/*                                                                         */
/*-------------------------------------------------------------------------*/
Bool Get_Integer(int n,int a)

{
 WamWord word,tag,*adr;

 Deref(&A(a),word,tag,adr)
 switch(tag)
    {
     case REF:
         Bind_UV(adr,Tag_Value(INT,n))
         return TRUE;

     case INT:
         return (UnTag_INT(word)==n);

     case FDV:
         return Unify_FDV_And_INT(UnTag_FDV(word),n);
    }

 return FALSE;
}




/*-------------------------------------------------------------------------*/
/* GET_LIST                                                                */
/*                                                                         */
/*-------------------------------------------------------------------------*/
Bool Get_List(int r)

{
 WamWord word,tag,*adr;

 Deref(&A(r),word,tag,adr)
 switch(tag)
    {
     case REF:
         Bind_UV(adr,Tag_Value(LST,H))
         S=WRITE_MODE; 
         return TRUE;

     case LST:                                   /* init S, i.e. MODE=READ */
         S=(WamWord *) UnTag_LST(word)+OFFSET_CAR;
         return TRUE;	 
    }

 return FALSE;
}




/*-------------------------------------------------------------------------*/
/* GET_STRUCTURE                                                           */
/*                                                                         */
/*-------------------------------------------------------------------------*/
Bool Get_Structure(AtomInf *atom,int n,int r)

{
 WamWord word,tag,*adr;

 Deref(&A(r),word,tag,adr)
 switch(tag)
    {
     case REF:
         Bind_UV(adr,Tag_Value(STC,H))
         Global_Push(Functor_Arity(atom,n))
         S=WRITE_MODE;
	 return TRUE;

     case STC:                                   /* init S, i.e. MODE=READ */
         adr=UnTag_STC(word);
         if (Functor_And_Arity(adr)!=Functor_Arity(atom,n))
             return FALSE;

         S=adr+OFFSET_ARG;
	 return TRUE;
    }

 return FALSE;
}




/*-------------------------------------------------------------------------*/
/* PUT_Y_UNSAFE_VALUE                                                      */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Put_Y_Unsafe_Value(int y,int a)

{
 WamWord word,tag,*adr;

 Deref(&Y(E,y),word,tag,adr)

 if (tag==REF && adr>=(WamWord *) EE(E))
    {
     A(a)=Tag_Value(REF,H);
     Globalize_Local_Unbound_Var(adr)
    }
  else
     A(a)=(Global_UnMove(tag)) ? Tag_Value(REF,adr) : word;
}





/*-------------------------------------------------------------------------*/
/* UNIFY_X_Y_VARIABLE                                                      */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Unify_X_Y_Variable(WamWord *x_y_adr)

{
 WamWord word,tag,*adr;

 if (S!=WRITE_MODE)
    {
     tag=Tag_Of(word= *S);
     *x_y_adr=(Global_UnMove(tag)) ? Tag_Value(REF,S) : word;
     S++;
    }
  else
    {
     *x_y_adr=word=Make_Self_Ref(H);
     Global_Push(word)
    }
}




/*-------------------------------------------------------------------------*/
/* UNIFY_X_Y_VALUE                                                         */
/*                                                                         */
/*-------------------------------------------------------------------------*/
Bool Unify_X_Y_Value(WamWord *x_y_adr)

{
 WamWord word,tag,*adr;

 if (S!=WRITE_MODE)
    {
     if (!Unify(x_y_adr,S))
         return FALSE;
     S++;
     return TRUE;
    }

 Global_Push(*x_y_adr)
 return TRUE;
}




/*-------------------------------------------------------------------------*/
/* UNIFY_X_Y_LOCAL_VALUE                                                   */
/*                                                                         */
/*-------------------------------------------------------------------------*/
Bool Unify_X_Y_Local_Value(WamWord *x_y_adr)

{
 WamWord word,tag,*adr;

 if (S!=WRITE_MODE)
     return Unify(x_y_adr,S++);

 Deref(x_y_adr,word,tag,adr)

 if (tag==REF && Is_A_Local_Adr(adr))
     Globalize_Local_Unbound_Var(adr)
  else
     Global_Push((Global_UnMove(tag)) ? Tag_Value(REF,adr) : word)

 return TRUE;
}




/*-------------------------------------------------------------------------*/
/* UNIFY_VOID                                                              */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Unify_Void(int n)

{
 WamWord *end_adr;

 if (S!=WRITE_MODE)
     S+=n;
  else
     for(end_adr=H+(n);H<end_adr;++H)
         *H=Make_Self_Ref(H);
}




/*-------------------------------------------------------------------------*/
/* UNIFY_CONSTANT                                                          */
/*                                                                         */
/*-------------------------------------------------------------------------*/
Bool Unify_Constant(AtomInf *atom)

{
 WamWord word,tag,*adr;

 if (S!=WRITE_MODE)
    {
     Deref(S,word,tag,adr)
     S++;
     switch(tag)
        {
         case REF:
             Bind_UV(adr,Tag_Value(CST,atom))
	     return TRUE;

         case CST:
	     return (UnTag_CST(word)==atom);
        }

     return FALSE;
    }

 Global_Push(Tag_Value(CST,atom))

 return TRUE;
}




/*-------------------------------------------------------------------------*/
/* UNIFY_INTEGER                                                           */
/*                                                                         */
/*-------------------------------------------------------------------------*/
Bool Unify_Integer(int n)

{
 WamWord word,tag,*adr;

 if (S!=WRITE_MODE)
    {
     Deref(S,word,tag,adr)
     S++;     
     switch(tag)
        {
         case REF:
             Bind_UV(adr,Tag_Value(INT,n))
             return TRUE;

         case INT:
             return (UnTag_INT(word)==n);

         case FDV:
             return Unify_FDV_And_INT(UnTag_FDV(word),n);
        }

     return FALSE;
    }

 Global_Push(Tag_Value(INT,n))

 return TRUE;
}




/*-------------------------------------------------------------------------*/
/* UNIFY_NIL                                                               */
/*                                                                         */
/*-------------------------------------------------------------------------*/
Bool Unify_Nil(void)

{
 WamWord word,tag,*adr;

 if (S!=WRITE_MODE)
    {
     Deref(S,word,tag,adr)
     S++;
     if (tag==REF)
        {
         Bind_UV(adr,word_nil)
         return TRUE;
        }
      else
         return (word==word_nil);
    }

 Global_Push(word_nil)

 return TRUE;
}




/*-------------------------------------------------------------------------*/
/* SET_STACK_DEFAULTS                                                      */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Set_Stack_Defaults(char *name,char *env_var_name,int default_size_kb)

{
 int i;

 for(i=0;i<NB_OF_STACKS;i++)
     if (Lib2(strcmp,name,stk_tbl[i].name)==0)
         break;

 if (i==NB_OF_STACKS)
     Fatal_Error(ERR_UNKNOWN_STACK,name);

 stk_tbl[i].env_var_name=env_var_name;
 stk_tbl[i].default_size=KBytes_To_Wam_Words(default_size_kb);
}




/*-------------------------------------------------------------------------*/
/* INIT_WAM_ENGINE                                                         */
/*                                                                         */
/* the top of stack are initialized and space for feint first choice point */
/* must be reserved (see Call_Prolog). It's achieved by set B with a value */
/* greater than E (see Local_Top in wam_engine.h). The first real choice   */
/* point (try / try_me_else) or the first environment (allocate) will not  */
/* use words 0..CHOICE_STATIC_SIZE in local stack. So the space for a feint*/
/* choice point is preserved.                                              */
/*-------------------------------------------------------------------------*/
void Init_Wam_Engine(void)

{
 int   i,x;
 char *p;

 for(i=0;i<NB_OF_STACKS;i++)
     if (*stk_tbl[i].env_var_name)
        {
         p=(char *) Lib1(getenv,stk_tbl[i].env_var_name);
         if (p && *p)
            {
             Lib3(sscanf,p,"%d",&x);
             stk_tbl[i].size=KBytes_To_Wam_Words(x);
             continue;
            }

         stk_tbl[i].size=stk_tbl[i].default_size;
        }

 M_Allocate_Stacks();

 reg_bank=Global_Stack;        /* allocated X regs +  other non alloc regs */
 Global_Stack+=REG_BANK_SIZE;  /* at the beginning of the heap             */
 Global_Size-=REG_BANK_SIZE;

 Init_Atom_Pred();
 word_nil=Tag_Value(CST,atom_nil);     /* defined as a reg (see archi.def) */

 E=B=BC=Local_Stack;
 H=Global_Stack;
 TR=Trail_Stack;
 CP=NULL;

 Create_Choice_Point(NULL,0);                        /* dummy choice point */

 Init_Fd_Engine();
}




/*-------------------------------------------------------------------------*/
/* TERM_WAM_ENGINE                                                         */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Term_Wam_Engine(void)

{
#ifdef WAM_PROFILE
 int      used,max_used;
 WamWord *w;
 int      i;

#endif


#ifdef WAM_PROFILE

 Lib1(printf,"\n");
 Lib1(printf,"WAM Profile informations\n\n");
 Lib1(printf,"Stacks:\n");

 for(i=0;i<NB_OF_STACKS;i++)
    {
     used=Stack_Top(i)-stk_tbl[i].stack;

     w=stk_tbl[i].stack+stk_tbl[i].size-1;

     while(w >= stk_tbl[i].stack && (*w==0 || *w==M_MAGIC))
         w--;

     max_used=w-stk_tbl[i].stack+1;

     used*=sizeof(WamWord);
     max_used*=sizeof(WamWord);

     Lib5(printf,"   %-6s stack:%10d bytes   %10d end use   %10d max use\n",
        stk_tbl[i].name,stk_tbl[i].size*sizeof(WamWord),used,max_used);
    }

 Lib1(printf,"\n");

 Lib1(printf,"\n");
 Lib1(printf,"Choice points:\n");
 Lib2(printf,"   create      :%10d (try)\n",  nb_of_create_choice_point);
 Lib2(printf,"   update      :%10d (retry)\n",nb_of_update_choice_point);
 Lib2(printf,"   delete      :%10d (trust)\n",nb_of_delete_choice_point);

 Lib1(printf,"\n");


#endif
 Term_Fd_Engine();
}




/*-------------------------------------------------------------------------*/
/* SWITCH_REG_BANK                                                         */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Switch_Reg_Bank(WamWord *new_reg_bank)

{
 int i;
 WamWord *p1,*p2;

 if (reg_bank!=new_reg_bank)
    {
     p1=reg_bank+NB_OF_X_REGS;
     p2=new_reg_bank+NB_OF_X_REGS;

     for(i=0;i<NB_OF_NOT_ALLOC_REGS;i++)
         *p2++=*p1++;

     reg_bank=new_reg_bank;
    }
}




/*-------------------------------------------------------------------------*/
/* CALL_PROLOG                                                             */
/*                                                                         */
/* Call_Prolog runs the execution of one prolog goal.                      */
/* The current choice point is updated to set ALTB to Call_Prolog_Fail and */
/* CP is set to Call_Prolog_Success. At the end ALTB and CP are restored.  */
/* To ensure that a choice point always exists before invoking Call_Prolog,*/
/* Init_Wam_Engine reserve the space for a feint choice point, i.e ALTB can*/
/* be modified safely.                                                     */
/* The intermediate call to Call_Next allocates on the C stack enough space*/
/* for local variables declared in the called (i.e. with goto) functions.  */
/*                                                                         */
/* Call_Prolog returns TRUE if the predicate has succeed, FALSE otherwise. */
/*-------------------------------------------------------------------------*/
Bool Call_Prolog(CodePtr codep)

{
 Prototype(Prefix(Call_Prolog_Success))
 Prototype(Prefix(Call_Prolog_Fail))
 Bool     Call_Next();
 WamWord *cur_chc_pt=B;
 WamCont  save_CP   =CP;
 WamWord  save_ALTB =ALTB(cur_chc_pt);
 Bool     ok;

 ALTB(cur_chc_pt)=(WamWord) Prefix(Call_Prolog_Fail); /* modify choice pnt */
 Call_Execute_Prefix
 CP=(WamCont) Prefix(Call_Prolog_Success);

 ok=Call_Next(codep);

 CP=save_CP;                                       /* restore continuation */
 ALTB(cur_chc_pt)=save_ALTB;                       /* restore choice point */

 return ok;
}



Bool Call_Next(CodePtr codep)

{
 int t[1024];

 M_Indirect_Goto(codep)
 Foo(t);
}


int Foo(int *t)
{
}


Begin_Static_Wam_Code(Call_Prolog_Success)

 Foo(NULL);
 return TRUE;

End_Pred




Begin_Static_Wam_Code(Call_Prolog_Fail)

 Foo(NULL);
 return FALSE;

End_Pred




/*-------------------------------------------------------------------------*/
/* UNIFY                                                                   */
/*                                                                         */
/*-------------------------------------------------------------------------*/
Bool Unify(WamWord *start_u_adr,WamWord *start_v_adr)

{
 WamWord u_word,u_tag,*u_adr;
 WamWord v_word,v_tag,*v_adr;
 int     i;

 Deref(start_u_adr,u_word,u_tag,u_adr)
 Deref(start_v_adr,v_word,v_tag,v_adr)


 if (u_tag==REF)
    {
     if (v_tag==REF)
        {
         if (u_adr>v_adr)
             Bind_UV(u_adr,Tag_Value(REF,v_adr))
          else
             if (v_adr>u_adr)
                 Bind_UV(v_adr,Tag_Value(REF,u_adr))
        }
      else
         if (Global_UnMove(v_tag))
             Bind_UV(u_adr,Tag_Value(REF,v_adr))
          else
             Bind_UV(u_adr,v_word)

     return TRUE;
    }

 switch(v_tag)
    {
     case REF:
         if (Global_UnMove(u_tag))
             Bind_UV(v_adr,Tag_Value(REF,u_adr))
          else
             Bind_UV(v_adr,u_word)

         return TRUE;


     case INT:
         if (u_tag==FDV)
             return Unify_FDV_And_INT(u_adr,UnTag_INT(v_word));
     case CST:
         return (u_word==v_word);                    /* test tag and value */


     case FDV:
         if (u_tag==INT)
             return Unify_FDV_And_INT(v_adr,UnTag_INT(u_word));

         if (u_tag!=FDV)
             return(FALSE);

         return Unify_FDV_And_FDV(u_adr,v_adr);


     case LST:
         if (u_tag!=LST)
             return FALSE;

         u_adr=UnTag_LST(u_word);
         v_adr=UnTag_LST(v_word);

         if (u_adr==v_adr)
             return TRUE;

         return Unify(&Car(u_adr),&Car(v_adr)) &&
                Unify(&Cdr(u_adr),&Cdr(v_adr));


     default:                                                /* v_tag==STC */
         if (u_tag!=STC)
             return FALSE;
         
         u_adr=UnTag_STC(u_word);
         v_adr=UnTag_STC(v_word);

         if (u_adr==v_adr)
             return TRUE;

         if (Functor_And_Arity(u_adr) != Functor_And_Arity(v_adr))
             return FALSE;

         i=Arity(u_adr);
         do
            {
             --i;
             if (!Unify(&Arg(u_adr,i),&Arg(v_adr,i)))
                 return FALSE;
            }
         while(i);

         return TRUE;
    }
}




/*-------------------------------------------------------------------------*/
/* FATAL_ERROR                                                             */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Fatal_Error(char *format,...)

{
 va_list arg_ptr;


 printf("\nFatal Error: ");
 va_start(arg_ptr,format);
 vprintf(format,arg_ptr);
 va_end(arg_ptr);

 printf("\n");
 exit(1);
}
