/*-------------------------------------------------------------------------*/
/* Prolog to Wam Compiler               INRIA Rocquencourt - ChLoE Project */
/* Version 1.0  -  C Run-time                           Daniel Diaz - 1994 */
/* Extended to FD Constraints (July 1992)                                  */
/*                                                                         */
/* FD Range Implementation                                                 */
/*                                                                         */
/* fd_range.c                                                              */
/*-------------------------------------------------------------------------*/
#include <stdio.h>
int fprintf();

#include "bool.h"

#define FD_RANGE

#include "fd_range.h"




/*-------------------------------------------------------------------------*/
/* The file fd_range_hook.h must contains the definition of:               */
/*                                                                         */
/*   INTERVAL_INFINITY: an integer constant corresponding to the greatest  */
/*                      value for intervals (i.e. 0..INTERVAL_INFINITY).   */
/*                                                                         */
/*   vec_infinity     : an integer variable corresponding to the greatest  */
/*                      value for vectors   (i.e. 0..vec_infinity).        */
/*   vec_size         : an integer variable corresponding to the size of a */
/*                      vector in words (i.e. vec_infinity/32)             */
/*                      (see Define_Vector_Size() function).               */
/*                                                                         */
/*   RANGE_TOP_STACK  : (a pointer to) an integer variable corresponding to*/
/*                      the top the stack where are allocated the vectors. */
/*                      The user must handle the (re)initialization of     */
/*                      this pointer to a valid (read/write) memory area.  */
/*                      Allocated vectors are never recovered, so the user */
/*                      should take care of reinitializations (GC) of the  */
/*                      top of stack if needed.                            */
/*                                                                         */
/* The following macros can be redefined:                                  */
/*                                                                         */
/*   M_Mul(x,y)       : a macro returning an integer corresponding to x * y*/
/*   M_Div(x,y)       : a macro returning an integer corresponding to x / y*/
/*   M_Mod(x,y)       : a macro returning an integer corresponding to x % y*/
/*                      (x and y are integers)                             */
/*                                                                         */
/*   Libn(f,a1,...,an): a macro calling a C library function f with n args */
/*                      (n in 0..5) and returning                          */
/*-------------------------------------------------------------------------*/




/*---------------------------------*/
/* Constants                       */
/*---------------------------------*/

#define ALL_1                      ((unsigned) -1)

#define WRITE_BEGIN_RANGE          ""
#define WRITE_END_RANGE            ""
#define WRITE_LIMITS_SEPARATOR     ".."
#define WRITE_INTERVALS_SEPARATOR  ":"
#define WRITE_EXTRA_CSTR_SYMBOL    "@"




/*---------------------------------*/
/* Type Definitions                */
/*---------------------------------*/

/*---------------------------------*/
/* Global Variables                */
/*---------------------------------*/

/*---------------------------------*/
/* Function Prototypes             */
/*---------------------------------*/

/*---------------------------------*/
/* Auxiliary macros                */
/*---------------------------------*/

#define math_min(x,y)              ((x) <= (y) ? (x) : (y))
#define math_max(x,y)              ((x) >= (y) ? (x) : (y))




          /* Compute the Less Significant Bit (0-31) of a word != 0 */

#define Compute_LSB(word,bit)                                               \
    {                                                                       \
     unsigned word1=word;                                                   \
     bit= 0;                                                                \
                                                                            \
     if (!(word1 << 16))                                                    \
         word1 >>= 16, bit += 16;                                           \
                                                                            \
     if (!(word1 << 24))                                                    \
         word1 >>= 8, bit += 8;                                             \
                                                                            \
     if (!(word1 << 28))                                                    \
         word1 >>= 4, bit += 4;                                             \
                                                                            \
     if (!(word1 << 30))                                                    \
         word1 >>= 2, bit += 2;                                             \
                                                                            \
     if (!(word1 << 31))                                                    \
         bit++;                                                             \
    }




          /* Compute the Most Significant Bit (0-31) of a word != 0 */

#define Compute_MSB(word,bit)                                               \
    {                                                                       \
     unsigned word1=word;                                                   \
     bit= 31;                                                               \
                                                                            \
     if (!(word1 >> 16))                                                    \
         word1 <<= 16, bit -= 16;                                           \
                                                                            \
     if (!(word1 >> 24))                                                    \
         word1 <<= 8, bit -= 8;                                             \
                                                                            \
     if (!(word1 >> 28))                                                    \
         word1 <<= 4, bit -= 4;                                             \
                                                                            \
     if (!(word1 >> 30))                                                    \
         word1 <<= 2, bit -= 2;                                             \
                                                                            \
     if (!(word1 >> 31))                                                    \
         bit--;                                                             \
    }




/*-------------------------------------------------------------------------*/
/* DEFINE_VECTOR_SIZE                                                      */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Define_Vector_Size(int max_val)

{
 vec_size    =max_val/32+1;
 vec_infinity=vec_size*32-1;
}




/*-------------------------------------------------------------------------*/
/* VECTOR_FROM_INTERVAL                                                    */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Vector_From_Interval(Vector vec,int min,int max)

{
 Vector w_min=vec+Word_Nb(min);
 Vector w_max=vec+Word_Nb(max);
 Vector end  =vec+vec_size;

 for(;;)
     if (vec==w_min)
         break;
      else
         *vec++=0;

 for(;;)
     if (vec>w_max)
         break;
      else
         *vec++=ALL_1;

 for(;;)
     if (vec==end)
         break;
      else
         *vec++=0;

 *w_min &= ALL_1 << Bit_Nb(min);
 *w_max &= ALL_1 >> (31-Bit_Nb(max));
}




/*-------------------------------------------------------------------------*/
/* VECTOR_NB_ELEM                                                          */
/*                                                                         */
/*-------------------------------------------------------------------------*/
int Vector_Nb_Elem(Vector vec)

{
 static   int     nb_bits_in_byte[256]=
         {0,1,1,2,1,2,2,3,1,2,2,3,2,3,3,4,1,2,2,3,2,3,3,4,2,3,3,4,3,4,4,5,
          1,2,2,3,2,3,3,4,2,3,3,4,3,4,4,5,2,3,3,4,3,4,4,5,3,4,4,5,4,5,5,6,
          1,2,2,3,2,3,3,4,2,3,3,4,3,4,4,5,2,3,3,4,3,4,4,5,3,4,4,5,4,5,5,6,
          2,3,3,4,3,4,4,5,3,4,4,5,4,5,5,6,3,4,4,5,4,5,5,6,4,5,5,6,5,6,6,7,
          1,2,2,3,2,3,3,4,2,3,3,4,3,4,4,5,2,3,3,4,3,4,4,5,3,4,4,5,4,5,5,6,
          2,3,3,4,3,4,4,5,3,4,4,5,4,5,5,6,3,4,4,5,4,5,5,6,4,5,5,6,5,6,6,7,
          2,3,3,4,3,4,4,5,3,4,4,5,4,5,5,6,3,4,4,5,4,5,5,6,4,5,5,6,5,6,6,7,
          3,4,4,5,4,5,5,6,4,5,5,6,5,6,6,7,4,5,5,6,5,6,6,7,5,6,6,7,6,7,7,8};
 register Vector  end=vec+vec_size;
 register VecWord vec_word;
 register int     nb_elem=0;

 do
    {
     vec_word=*vec;
     nb_elem+=nb_bits_in_byte[ vec_word      & 0xFF];
     nb_elem+=nb_bits_in_byte[(vec_word>> 8) & 0xFF];
     nb_elem+=nb_bits_in_byte[(vec_word>>16) & 0xFF];
     nb_elem+=nb_bits_in_byte[(vec_word>>24) & 0xFF];
     vec++;
    }
 while(vec<end);

 return nb_elem;
}




/*-------------------------------------------------------------------------*/
/* VECTOR_EMPTY                                                            */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Vector_Empty(Vector vec)

{
 Vector end=vec+vec_size;

 do
     *vec++ = 0;
 while(vec<end);
}




/*-------------------------------------------------------------------------*/
/* VECTOR_FULL                                                             */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Vector_Full(Vector vec)

{
 Vector end=vec+vec_size;

 do
     *vec++ = ALL_1;
 while(vec<end);
}




/*-------------------------------------------------------------------------*/
/* VECTOR_COPY                                                             */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Vector_Copy(Vector vec,Vector vec1)

{
 Vector end=vec+vec_size;

 do
     *vec++ = *vec1++;
 while(vec<end);
}




/*-------------------------------------------------------------------------*/
/* VECTOR_UNION                                                            */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Vector_Union(Vector vec,Vector vec1)

{
 Vector end=vec+vec_size;

 do
     *vec++ |= *vec1++;
 while(vec<end);
}




/*-------------------------------------------------------------------------*/
/* VECTOR_INTER                                                            */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Vector_Inter(Vector vec,Vector vec1)

{
 Vector end=vec+vec_size;

 do
     *vec++ &= *vec1++;
 while(vec<end);
}




/*-------------------------------------------------------------------------*/
/* VECTOR_COMPL                                                            */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Vector_Compl(Vector vec)

{
 Vector end=vec+vec_size;

 do
     *vec = ~(*vec), vec++;
 while(vec<end);
}




/*-------------------------------------------------------------------------*/
/* VECTOR_ADD_VECTOR                                                       */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Vector_Add_Vector(Vector vec,Vector vec1)

{
 Vector aux_vec;
 int    elem,x;

 Vector_Allocate(aux_vec);
 Vector_Copy(aux_vec,vec);
 Vector_Empty(vec);
 
 BEGIN_ENUM_VECTOR(aux_vec)

     elem=vec_elem;

     BEGIN_ENUM_VECTOR(vec1)

         x=elem+vec_elem;

         if ((unsigned) x>vec_infinity)
             BREAK_ENUM_VECTOR

         Set_Int_In_Vector(vec,x);

     END_ENUM_VECTOR

 END_ENUM_VECTOR
}




/*-------------------------------------------------------------------------*/
/* VECTOR_SUB_VECTOR                                                       */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Vector_Sub_Vector(Vector vec,Vector vec1)

{
 Vector aux_vec;
 int    elem,x;

 Vector_Allocate(aux_vec);
 Vector_Copy(aux_vec,vec);
 Vector_Empty(vec);
 
 BEGIN_ENUM_VECTOR(aux_vec)

     elem=vec_elem;

     BEGIN_ENUM_VECTOR(vec1)

         x=elem-vec_elem;

         if ((unsigned) x<=vec_infinity)
             Set_Int_In_Vector(vec,x);

     END_ENUM_VECTOR

 END_ENUM_VECTOR
}




/*-------------------------------------------------------------------------*/
/* VECTOR_MUL_VECTOR                                                       */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Vector_Mul_Vector(Vector vec,Vector vec1)

{
 Vector aux_vec;
 int    elem,x;

 Vector_Allocate(aux_vec);
 Vector_Copy(aux_vec,vec);
 Vector_Empty(vec);
 
 BEGIN_ENUM_VECTOR(aux_vec)

     elem=vec_elem;

     BEGIN_ENUM_VECTOR(vec1)

         x=M_Mul(elem,vec_elem);

         if ((unsigned) x>vec_infinity)
             BREAK_ENUM_VECTOR	   

         Set_Int_In_Vector(vec,x);

     END_ENUM_VECTOR

 END_ENUM_VECTOR
}




/*-------------------------------------------------------------------------*/
/* VECTOR_DIV_VECTOR                                                       */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Vector_Div_Vector(Vector vec,Vector vec1)

{
 Vector aux_vec;
 int    elem,x;

 Vector_Allocate(aux_vec);
 Vector_Copy(aux_vec,vec);
 Vector_Empty(vec);
 
 BEGIN_ENUM_VECTOR(aux_vec)

     elem=vec_elem;

     BEGIN_ENUM_VECTOR(vec1)

         if (M_Mod(elem,vec_elem)==0)
            {
             x=M_Div(elem,vec_elem);

             if ((unsigned) x>vec_infinity)
                 BREAK_ENUM_VECTOR

             Set_Int_In_Vector(vec,x);
            }

     END_ENUM_VECTOR

 END_ENUM_VECTOR
}




/*-------------------------------------------------------------------------*/
/* VECTOR_MOD_VECTOR                                                       */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Vector_Mod_Vector(Vector vec,Vector vec1)

{
 Vector aux_vec;
 int    elem,x;

 Vector_Allocate(aux_vec);
 Vector_Copy(aux_vec,vec);
 Vector_Empty(vec);
 
 BEGIN_ENUM_VECTOR(aux_vec)

     elem=vec_elem;

     BEGIN_ENUM_VECTOR(vec1)

         x=M_Mod(elem,vec_elem);
         Set_Int_In_Vector(vec,x);

     END_ENUM_VECTOR

 END_ENUM_VECTOR
}




/*-------------------------------------------------------------------------*/
/* VECTOR_ADD_TERM                                                         */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Vector_Add_Term(Vector vec,int n)

{
 int nb_of_word;
 int nb_of_bit;
 int rem,rem1;
 int i,j;

 if (n>=0)
    {
     nb_of_word=Word_Nb(n);
     nb_of_bit =Bit_Nb(n);

     if (nb_of_word)
        {
         i=vec_size-1;
         j=vec_size-1-nb_of_word;

         while(j>=0)
             vec[i--]=vec[j--];

         while(i>=0)
             vec[i--]=0;
        }

     if (nb_of_bit)
        {
         rem=0;
         for(i=nb_of_word;i<vec_size;i++)
            {
             rem1=vec[i] >> (32-nb_of_bit);
             vec[i] = (vec[i] << nb_of_bit) | rem;
             rem=rem1;
            }
        }
    }
  else
    {
     nb_of_word=Word_Nb(-n);
     nb_of_bit =Bit_Nb(-n);

     if (nb_of_word)
        {
         i=0;
         j=nb_of_word;

         while(j<vec_size)
             vec[i++]=vec[j++];

         while(i<vec_size)
             vec[i++]=0;
        }

     if (nb_of_bit)
        {
         rem=0;
         for(i=vec_size-1-nb_of_word;i>=0;i--)
            {
             rem1=vec[i] << (32-nb_of_bit);
             vec[i] = (vec[i] >> nb_of_bit) | rem;
             rem=rem1;
            }
        }
    }
}




/*-------------------------------------------------------------------------*/
/* VECTOR_MUL_TERM                                                         */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Vector_Mul_Term(Vector vec,int n)

{
 Vector aux_vec;
 int    x;

 Vector_Allocate(aux_vec);
 Vector_Copy(aux_vec,vec);
 Vector_Empty(vec);
 
 BEGIN_ENUM_VECTOR(aux_vec)

     x=M_Mul(vec_elem,n);

     if ((unsigned) x>vec_infinity)
         BREAK_ENUM_VECTOR

     Set_Int_In_Vector(vec,x);

 END_ENUM_VECTOR
}




/*-------------------------------------------------------------------------*/
/* VECTOR_DIV_TERM                                                         */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Vector_Div_Term(Vector vec,int n)

{
 Vector aux_vec;
 int    x;

 Vector_Allocate(aux_vec);
 Vector_Copy(aux_vec,vec);
 Vector_Empty(vec);
 
 BEGIN_ENUM_VECTOR(aux_vec)

     if (M_Mod(vec_elem,n)==0)
        {
         x=M_Div(vec_elem,n);

         if ((unsigned) x>vec_infinity)
             BREAK_ENUM_VECTOR

         Set_Int_In_Vector(vec,x);
        }

 END_ENUM_VECTOR
}




/*-------------------------------------------------------------------------*/
/* VECTOR_MOD_TERM                                                         */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Vector_Mod_Term(Vector vec,int n)

{
 Vector aux_vec;
 int    x;

 Vector_Allocate(aux_vec);
 Vector_Copy(aux_vec,vec);
 Vector_Empty(vec);
 
 BEGIN_ENUM_VECTOR(aux_vec)

     x=M_Mod(vec_elem,n);
     if ((unsigned) x<=vec_infinity)
         Set_Int_In_Vector(vec,x);

 END_ENUM_VECTOR
}




/*-------------------------------------------------------------------------*/
/* IS_INT_IN_RANGE                                                         */
/*                                                                         */
/*-------------------------------------------------------------------------*/
Bool Is_Int_In_Range(Range *range,int n)

{
 return n>=range->min && n<=range->max &&
        (Is_Interval(range) || Is_Int_In_Vector(range->vec,n));
}




/*-------------------------------------------------------------------------*/
/* RANGE_COPY                                                              */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Range_Copy(Range *range,Range *range1)

{
 range->extra_cstr=range1->extra_cstr;
 range->min       =range1->min;
 range->max       =range1->max;

 if (Is_Interval(range1))
     range->vec=NULL;
  else
    {
     Vector_Allocate_If_Necessary(range->vec);
     Vector_Copy(range->vec,range1->vec);
    }
}




/*-------------------------------------------------------------------------*/
/* RANGE_NB_ELEM                                                           */
/*                                                                         */
/*-------------------------------------------------------------------------*/
int Range_Nb_Elem(Range *range)

{
 if (Is_Interval(range))                        /* here range is not empty */
     return range->max - range->min+1;


 return Vector_Nb_Elem(range->vec);
}




/*-------------------------------------------------------------------------*/
/* RANGE_BECOMES_SPARSE                                                    */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Range_Becomes_Sparse(Range *range)

{
 Vector_Allocate_If_Necessary(range->vec);

 range->min=math_max(range->min,0);

 if (range->extra_cstr=(range->max > vec_infinity))
     range->max=vec_infinity;

 if (Is_Not_Empty(range))
     Vector_From_Interval(range->vec,range->min,range->max);
}




/*-------------------------------------------------------------------------*/
/* RANGE_FROM_VECTOR                                                       */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Range_From_Vector(Range *range)

{
 Vector start;
 Vector end;
 int bit;

 start=range->vec-1;
 for(;;)
     if (*++start)
         break;
      else
         if (start>=range->vec+vec_size)
            {
             Set_To_Empty(range);
             return;
            }

 end=range->vec+vec_size;
 for(;;)
     if (*--end)
         break;


 Compute_LSB(*start,bit);
 range->min=Word_Nb_And_Bit_Nb(start-range->vec,bit);

 Compute_MSB(*end,bit);
 range->max=Word_Nb_And_Bit_Nb(end-range->vec,bit);
}




/*-------------------------------------------------------------------------*/
/* RANGE_UNION                                                             */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Range_Union(Range *range,Range *range1)

{
 int   swt=(Is_Sparse(range)<<1)+Is_Sparse(range1);
 Range r;
 Bool  extra_cstr;

 if (swt==0)                                     /* Interval with Interval */
    {
     if (Is_Not_Empty(range) && Is_Not_Empty(range1) &&
         range1->min <= range->max+1 && range->min <= range1->max+1)
        {
/*       range->extra_cstr=FALSE; */
         range->min=math_min(range->min,range1->min);
         range->max=math_max(range->max,range1->max);
         return;
        }

     Range_Becomes_Sparse(range);
     Range_Copy(&r,range1);                     /* we cannot modify range1 */
     range1=&r;
     Range_Becomes_Sparse(range1);
    }
 else
     if (swt==1)                                   /* Interval with Sparse */
         Range_Becomes_Sparse(range);
      else
         if (swt==2)                               /* Sparse with Interval */
            {
             Range_Copy(&r,range1);             /* we cannot modify range1 */
             range1=&r;
             Range_Becomes_Sparse(range1);
            }

                                                     /* Sparse with Sparse */

 extra_cstr=range->extra_cstr | range1->extra_cstr;

 if (Is_Empty(range))
    {
     Range_Copy(range,range1);
     range->extra_cstr=extra_cstr;
     return;
    }

 range->extra_cstr=extra_cstr;

 if (Is_Empty(range1))
     return;

 range->min=math_min(range->min,range1->min);
 range->max=math_max(range->max,range1->max);
 Vector_Union(range->vec,range1->vec);
}




/*-------------------------------------------------------------------------*/
/* RANGE_INTER                                                             */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Range_Inter(Range *range,Range *range1)

{
 int   swt=(Is_Sparse(range)<<1)+Is_Sparse(range1);
 Range r;

 if (swt==0)                                     /* Interval with Interval */
    {
/*   range->extra_cstr=FALSE; */
     range->min=math_max(range->min,range1->min);
     range->max=math_min(range->max,range1->max);
     return;
    }

 if (swt==1)                                       /* Interval with Sparse */
     Range_Becomes_Sparse(range);
  else
     if (swt==2)                                   /* Sparse with Interval */
        {
         Range_Copy(&r,range1);                 /* we cannot modify range1 */
         range1=&r;
         Range_Becomes_Sparse(range1);
        }
                                                     /* Sparse with Sparse */

 range->extra_cstr &= range1->extra_cstr;

 if (Is_Empty(range))
     return;


 if (Is_Empty(range1))
    {
     Set_To_Empty(range);
     return;
    }

 Vector_Inter(range->vec,range1->vec);
 Range_From_Vector(range);                           /* adjust min and max */
}




/*-------------------------------------------------------------------------*/
/* RANGE_COMPL                                                             */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Range_Compl(Range *range)

{
 if (range->min==range->max)
    {
     Vector save_vec=range->vec;

     Range_Compl_Singleton(range,range->min);
     if (save_vec && !range->vec)                  /* if range was sparse  */
        {                                          /* it must remain sparse*/
         range->vec=save_vec;
         Range_Becomes_Sparse(range);
        }

     return;
    }

 if (Is_Interval(range))                                       /* Interval */
    {
     if (Is_Empty(range))
        {
         range->min=0;
         range->max=INTERVAL_INFINITY;
         return;
        }

     if (range->min<=0)
        {
         if (range->max>=INTERVAL_INFINITY)
             Set_To_Empty(range);
          else
            {
             range->min=range->max+1;
             range->max=INTERVAL_INFINITY;
            }

         return;
        }

     if (range->max>=INTERVAL_INFINITY)
        {
         range->max=range->min-1;
         range->min=0;

         return;
        }

     Range_Becomes_Sparse(range);
    }
                                                                 /* Sparse */
 range->extra_cstr=TRUE;

 if (Is_Empty(range))
    {
     range->min=0;
     range->max=vec_infinity;
     Vector_Full(range->vec);
    }
  else
    {
     Vector_Compl(range->vec);
     Range_From_Vector(range);
    }
}




/*-------------------------------------------------------------------------*/
/* RANGE_COMPL_SINGLETON                                                   */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Range_Compl_Singleton(Range *range,int n)

{
 if (n<=0)
    {
     Init_Interval_Range(range,n+1,INTERVAL_INFINITY);
     return;
    }

 if (n>=INTERVAL_INFINITY)
    {
     Init_Interval_Range(range,0,n-1);
     return;
    }

 range->extra_cstr=TRUE;
 range->min=0;
 range->max=vec_infinity;

 Vector_Allocate_If_Necessary(range->vec);
 Vector_Full(range->vec);

 if ((unsigned) n <= vec_infinity)
     Reset_Int_In_Vector(range->vec,n);
}




/*-------------------------------------------------------------------------*/
/* RANGE_ADD_RANGE                                                         */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Range_Add_Range(Range *range,Range *range1)

{
 int   swt=(Is_Sparse(range)<<1)+Is_Sparse(range1);
 Range r;

 if (Is_Empty(range))
     return;

 if (Is_Empty(range1))
    {
     Set_To_Empty(range);
     return;
    }

 if (swt==0)                                     /* Interval with Interval */
    {
/*   range->extra_cstr=FALSE; */
     range->min+=range1->min;
     range->max+=range1->max;
     return;
    }
 else
     if (swt==1)                                   /* Interval with Sparse */
         Range_Becomes_Sparse(range);
      else
         if (swt==2)                               /* Sparse with Interval */
            {
             Range_Copy(&r,range1);             /* we cannot modify range1 */
             range1=&r;
             Range_Becomes_Sparse(range1);
            }

                                                     /* Sparse with Sparse */
 Vector_Add_Vector(range->vec,range1->vec);

 range->min+=range1->min;
 range->max+=range1->max;
 range->extra_cstr|=(range1->extra_cstr | range->max>vec_infinity);

 if (range->extra_cstr || range->min<0)
     Range_From_Vector(range);
}




/*-------------------------------------------------------------------------*/
/* RANGE_SUB_RANGE                                                         */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Range_Sub_Range(Range *range,Range *range1)

{
 int   swt=(Is_Sparse(range)<<1)+Is_Sparse(range1);
 Range r;

 if (Is_Empty(range))
     return;

 if (Is_Empty(range1))
    {
     Set_To_Empty(range);
     return;
    }

 if (swt==0)                                     /* Interval with Interval */
    {
/*   range->extra_cstr=FALSE; */
     range->min-=range1->max;
     range->max-=range1->min;
     return;
    }
 else
     if (swt==1)                                   /* Interval with Sparse */
         Range_Becomes_Sparse(range);
      else
         if (swt==2)                               /* Sparse with Interval */
            {
             Range_Copy(&r,range1);             /* we cannot modify range1 */
             range1=&r;
             Range_Becomes_Sparse(range1);
            }

                                                     /* Sparse with Sparse */
 Vector_Sub_Vector(range->vec,range1->vec);

 range->min-=range1->max;
 range->max-=range1->min;
 range->extra_cstr|=range1->extra_cstr;

 if (range->extra_cstr || range->min<0)
     Range_From_Vector(range);
}




/*-------------------------------------------------------------------------*/
/* RANGE_MUL_RANGE                                                         */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Range_Mul_Range(Range *range,Range *range1)

{
 Range r;

 if (Is_Empty(range))
     return;

 if (Is_Empty(range1))
    {
     Set_To_Empty(range);
     return;
    }

 if (Is_Interval(range))
     Range_Becomes_Sparse(range);

 if (Is_Interval(range1))
    {
     r.vec=NULL;
     Range_Copy(&r,range1);                     /* we cannot modify range1 */
     range1=&r;
     Range_Becomes_Sparse(range1);
    }
                                                     /* Sparse with Sparse */
 Vector_Mul_Vector(range->vec,range1->vec);

 range->extra_cstr|=range1->extra_cstr;
 Range_From_Vector(range);
}




/*-------------------------------------------------------------------------*/
/* RANGE_DIV_RANGE                                                         */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Range_Div_Range(Range *range,Range *range1)

{
 Range r;

 if (Is_Empty(range))
     return;

 if (Is_Empty(range1))
    {
     Set_To_Empty(range);
     return;
    }

 if (Is_Interval(range))
     Range_Becomes_Sparse(range);

 if (Is_Interval(range1))
    {
     r.vec=NULL;
     Range_Copy(&r,range1);                     /* we cannot modify range1 */
     range1=&r;
     Range_Becomes_Sparse(range1);
    }
                                                     /* Sparse with Sparse */
 Vector_Div_Vector(range->vec,range1->vec);

 range->extra_cstr|=range1->extra_cstr;
 Range_From_Vector(range);
}




/*-------------------------------------------------------------------------*/
/* RANGE_MOD_RANGE                                                         */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Range_Mod_Range(Range *range,Range *range1)

{
 Range r;

 if (Is_Empty(range))
     return;

 if (Is_Empty(range1))
    {
     Set_To_Empty(range);
     return;
    }

 if (Is_Interval(range))
     Range_Becomes_Sparse(range);

 if (Is_Interval(range1))
    {
     r.vec=NULL;
     Range_Copy(&r,range1);                     /* we cannot modify range1 */
     range1=&r;
     Range_Becomes_Sparse(range1);
    }

                                                     /* Sparse with Sparse */
 Vector_Mod_Vector(range->vec,range1->vec);

 range->extra_cstr|=range1->extra_cstr;
 Range_From_Vector(range);
}




/*-------------------------------------------------------------------------*/
/* RANGE_ADD_TERM                                                          */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Range_Add_Term(Range *range,int n)

{
 if (n==0 || Is_Empty(range))
     return;

 if (Is_Interval(range))                                       /* Interval */
    {
     range->min+=n;
     range->max+=n;

     return;
    }
                                                                 /* Sparse */
 Vector_Add_Term(range->vec,n);

 range->min+=n;
 range->max+=n;

 range->extra_cstr|=(range->max>vec_infinity);
 if (range->extra_cstr || range->min<0)
     Range_From_Vector(range);
}



/*-------------------------------------------------------------------------*/
/* RANGE_MUL_TERM                                                          */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Range_Mul_Term(Range *range,int n)

{
 if (n==1 || Is_Empty(range))
     return;

 if (Is_Interval(range))                                       /* Interval */
     Range_Becomes_Sparse(range);
                                                                 /* Sparse */
 Vector_Mul_Term(range->vec,n);

 range->min=M_Mul(range->min,n);
 range->max=M_Mul(range->max,n);

 range->extra_cstr|=(range->max>vec_infinity);
 if (range->extra_cstr)
     Range_From_Vector(range);
}




/*-------------------------------------------------------------------------*/
/* RANGE_DIV_TERM                                                          */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Range_Div_Term(Range *range,int n)

{
 if (n==1 || Is_Empty(range))
     return;

 if (Is_Interval(range))                                       /* Interval */
     Range_Becomes_Sparse(range);
                                                                 /* Sparse */
 Vector_Div_Term(range->vec,n);

 range->min=M_Div(range->min+n-1,n);
 range->max=M_Div(range->max,n);

 range->extra_cstr|=(range->max>vec_infinity);
 if (range->extra_cstr)
     Range_From_Vector(range);
}




/*-------------------------------------------------------------------------*/
/* RANGE_MOD_TERM                                                          */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Range_Mod_Term(Range *range,int n)

{
 Range aux;

 if (Is_Empty(range))
     return;

 if (n<0)
     n= -n;

 if (Is_Interval(range))                                       /* Interval */
    {
     if (range->min >= 0)
        {
         if (range->max - range->min + 1 >= n)
            {
             range->min=0;
             range->max=n-1;

             return;
            }

         range->min=M_Mod(range->min,n);
         range->max=M_Mod(range->max,n);

         if (range->min > range->max)
            {
             Init_Interval_Range(&aux,0,range->max);
             range->max=n-1;

             Range_Union(range,&aux);
            }

         return;
        }

     if (range->max <= 0)
        {
         if (range->max - range->min + 1 >= n)
            {
             range->min= -(n-1);
             range->max=0;

             return;
            }

         range->min=M_Mod(range->min,n);
         range->max=M_Mod(range->max,n);

         if (range->min > range->max)       /* Only 0 will remain in the   */
            {                               /* range due to the changeover */
                                            /* from Interval to Sparse     */
             Init_Interval_Range(&aux,-(n-1),range->max);
             range->max=0;

             Range_Union(range,&aux);
            }

         return;
        }

                                 /* Here range->min < 0 and range->max > 0 */
     range->min=math_max(range->min,-n+1);
     range->max=math_min(range->max, n-1);
     return;
    }
                                                                 /* Sparse */
 Vector_Mod_Term(range->vec,n);

 Range_From_Vector(range);
}




/*-------------------------------------------------------------------------*/
/* RANGE_WRITE                                                             */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Range_Write(FILE *out,Range *range)

{
 int limit1= -1;
 int limit2;

 if (Is_Empty(range))
    {
     Lib2(fprintf,out,"<empty range>");
     return;
    }

 if (range->min==range->max)
    {
     Lib3(fprintf,out,"{%d}",range->min);
     return;
    }

 if (Is_Interval(range))
    {
     Lib3(fprintf,out,"%s",WRITE_BEGIN_RANGE);
     Lib5(fprintf,out,"%d%s%d",range->min,WRITE_LIMITS_SEPARATOR,
                               range->max);
     Lib3(fprintf,out,"%s",WRITE_END_RANGE);
     return;
    }

 Lib3(fprintf,out,"%s",WRITE_BEGIN_RANGE);

 BEGIN_ENUM_VECTOR(range->vec)
     if (limit1== -1)
         limit1=limit2=vec_elem;
      else
         if (vec_elem==limit2+1)
             limit2=vec_elem;
          else
            {
             Lib3(fprintf,out,"%d",limit1);
             if (limit2!=limit1)
                 Lib4(fprintf,out,"%s%d%",WRITE_LIMITS_SEPARATOR,limit2);

             Lib3(fprintf,out,"%s",WRITE_INTERVALS_SEPARATOR);

             limit1=limit2=vec_elem;
            }
 END_ENUM_VECTOR

 if (limit1!= -1)
    {
     Lib3(fprintf,out,"%d",limit1);
     if (limit2!=limit1)
         Lib4(fprintf,out,"%s%d%",WRITE_LIMITS_SEPARATOR,limit2);
    }

 Lib3(fprintf,out,"%s",WRITE_END_RANGE);

 if (range->extra_cstr)
     Lib3(fprintf,out,"%s",WRITE_EXTRA_CSTR_SYMBOL);
}
