;;  $Id: host-lispworks.lisp,v 1.4 1993/01/25 14:07:10 anjo Exp $
;;  
;;  File	host-lispworks.lisp
;;  Part of	PCE/Lisp interface
;;  Author	Anjo Anjewierden, anjo@swi.psy.uva.nl
;;  Purpose	Find function name based on memory address
;;  Works with	PCE 4.5,  LispWorks 3.1
;;  
;;  Notice	Copyright (c) 1992  University of Amsterdam
;;  
;;  History:	31/03/92  (Created)
;;  		05/01/93  (Last Modified)


;;  ------------------------------------------------------
;;  Directives
;;  ------------------------------------------------------

(in-package "PCE")

(export '(address-to-c-function		; Address -> FunctionName
	  pce-reset
	  ))


;; Define the functions necessary for communication between LISP and 'C'.
;; Define a foreign callable for 'C' to call when XtMainLoop has input for the
;; LISP process.


;;  ------------------------------------------------------
;;  Initialise PCE/LW communication
;;  ------------------------------------------------------

(defun pce-lispworks-main-loop ()
  ;; tell Xt about interesting fds
  (sys::map-noticed-fds 'add-lisp-input) ;; add to xwn-display-manager context
  ;; ensure that Xt is notified about all future interesting fds
  (setq sys::*notice-fd-func* 'add-lisp-input)
  (setq sys::*unnotice-fd-func* 'remove-lisp-input) ; -lisp added ANJO
  ;; run lisp-input when LispWorks sleeps
  (setf (symbol-function 'mp::x-sleep-until-signal-or-timeout)
        #'mp::sleep-until-signal-or-timeout)
  (setf (symbol-function 'mp::sleep-until-signal-or-timeout) 'lisp-input)
  )


;;  ------------------------------------------------------
;;  XtMainLoop in Lisp
;;  ------------------------------------------------------
;;
;;  Process all events pending in Xt.  This function could be
;;  written in C, but the function which calls the alternative
;;  input handlers (xt-app-next-event in this example) should be
;;  written to expect them to 'throw' down the C stack.  LispWorks
;;  will take care of the C stack pointer, but the function should
;;  rely on the input handler returning.
;; 

(define-foreign-function pce-lispworks-initialise ())

(define-foreign-function (pce-redraw |pceRedraw|) ())

(defun lisp-xt-main-loop ()
  (let ((event (make-x-event))
	(context (pce-context)))
    (loop (pce-redraw)
	  (xt-app-next-event context event)
	  (xt-dispatch-event event)
	  (unless (xt-app-pending context) (return)))))


(define-foreign-type x-event
    (:union
     (type :int)
     ;; .. set of all x-event types.
     (pad (:array :long 24))
     ))


(define-foreign-function (xt-pending |XtPending|) ()
                         :result-type :boolean) 

(define-foreign-function (xt-next-event |XtNextEvent|)
                         ((event-return :alien))) 

(define-foreign-function (xt-dispatch-event |XtDispatchEvent|)
                         ((event :alien))) 


;; -----------------------------------------------
;; Alternative (ie LispWorks) input initialization
;; -----------------------------------------------

;; -----------------------------------------
;; Alternative (ie LispWorks) input handling
;; -----------------------------------------

(defvar *waiting-for-input* nil)


;; Define the LISP code to be called by the above handshake.
;; LISP has no need for the args passed.
(defun input-callback (client-data source id)
  (declare (ignore client-data source id))
  (if *waiting-for-input*
      (throw 'capi-main-loop nil)
    (error "Input Callback called outside lisp-input")))

(ffi::foreign-callable input-callback
		       (
			:uinteger;; caddr-t   client-data
			:uinteger;; int       *source
			:uinteger;; XtInputId *id
		        )
		       :result-type :integer)


;; This function will be called whenever LISP goes to sleep
;; and will return as soon as there is input for LISP.
(defun lisp-input (time)
  (declare (ignore time))
  (let ((*waiting-for-input* t))
    (catch 'capi-main-loop (lisp-xt-main-loop)))
  1)

(defconstant xt-input-read-mask (ash 1 0))

(define-foreign-type xt-input-id :uinteger)

(define-foreign-function (xt-app-next-event |XtAppNextEvent|)
  ((context (:alien :integer))
   (event (:alien x-event))))

(define-foreign-function (xt-app-pending |XtAppPending|)
  ((context (:alien :integer))))

(define-foreign-function (xt-app-add-input |XtAppAddInput|)
  ((context (:alien :integer))
   (source :integer)
   (condition :integer)
   (callback :integer)
   (client-data :integer))
  :result-type (:alien xt-input-id))

(define-foreign-function (xt-remove-input |XtRemoveInput|)
    ((id xt-input-id)))


(defparameter *input-callback-address*
  (foreign-symbol-address 'input-callback))


(defvar *fd-to-id-array* (make-array 128)) ;length is maxfd


(define-foreign-variable (pce-context context)
    :result-type (:alien :integer))


(defun add-lisp-input (fd)
  (unless (svref *fd-to-id-array* fd) ;; prevent multiple addition
    (setf
     (svref *fd-to-id-array* fd)
     (xt-app-add-input (pce-context) fd xt-input-read-mask
                       *input-callback-address* 0))))


(defun remove-lisp-input (fd)
  (when-let (id (svref *fd-to-id-array* fd))
    (xt-remove-input id)
    (setf (svref *fd-to-id-array* fd) nil)))


;;  ------------------------------------------------------
;;  Foreign function to symbol address mapping
;;  ------------------------------------------------------

;;! address-to-c-function address ==> { name | NIL }
;;
;;  Returns the name (as a string) of the function of which the code
;;  falls in the given (memory) address.
;;
;;  This function is called when PCE wants to print a C stack trace.

(defun address-to-c-function (address)
  (let ((closest 0)
	(name nil))
    (maphash #'(lambda (key ffi-struct)
		 (let ((value (foreign-symbol-address
			       (coff-symbol-name ffi-struct)
			       :errorp nil)))
		   (when (and value	; Could be NIL for undefined functions?
			      (<= value address)
			      (> value closest))
			 (setq name key
			       closest value))))
	     *foreign-symbols-table*)
    name))


(defun print-all-foreign-functions ()
  (maphash #'(lambda (key ffi-struct)
	       (declare (ignore key))
	       (format t "~A = ~A = ~A~%"
		       (coff-symbol-name ffi-struct)
		       (coff-symbol-value ffi-struct)
		       (foreign-symbol-address
			(coff-symbol-name ffi-struct :error nil))))
	     *foreign-symbols-table*))


(foreign-callable address-to-c-function (:integer)
		  :result-type :string)


;;  ------------------------------------------------------
;;  Foreign structure pcelisp-status
;;  ------------------------------------------------------

(define-foreign-type pcelisp-status
  (:structure
   (arguments :long)
   (error :long)
   (name :long)
   (return-type :long)
   (result :long)
   (return-as :long)))


;;  The syntax for setting these may differ -- so:

(defun set-pcelisp-status-arguments (ps value)
  (setf (pcelisp-status->arguments ps) value))

(defun set-pcelisp-status-error (ps value)
  (setf (pcelisp-status->error ps) value))

(defun set-pcelisp-status-name (ps value)
  (setf (pcelisp-status->name ps) value))


(defun get-pcelisp-status-arguments (ps)
  (pcelisp-status->arguments ps))

(defun get-pcelisp-status-error (ps)
  (pcelisp-status->error ps))

(defun get-pcelisp-status-name (ps)
  (pcelisp-status->name ps))

(defun get-pcelisp-status-return-type (ps)
  (pcelisp-status->return-type ps))

(defun get-pcelisp-status-result (ps)
  (pcelisp-status->result ps))


;;  ------------------------------------------------------
;;  Foreign Function Declarations
;;  ------------------------------------------------------

(define-foreign-function pcelisp-c-initialise ()
  :result-type :boolean)


;;  ------------------------------------------------------
;;  Low-level Memory Management
;;  ------------------------------------------------------

(define-foreign-function (pcelisp-c-mark "_pcelisp_mark") ()
  :result-type :uinteger)


(define-foreign-function (pcelisp-c-rewind "_pcelisp_rewind")
  ((mark :uinteger)))


;;  ------------------------------------------------------
;;  Allocation of new symbols in the interface
;;  ------------------------------------------------------

(define-foreign-function pcelisp-c-new-name
  ((name :simple-string)
#+STATIC-LISP-SYMBOLS (value :as-is)
#-STATIC-LISP-SYMBOLS (value :uinteger))
  :result-type :boolean)


(define-foreign-function pcelisp-c-new-symbol
  ((name :simple-string)
   (package :simple-string))
  :result-type :boolean)


(define-foreign-function pcelisp-c-new-assoc
  ((name :simple-string)
   (value :as-is))
  :result-type :boolean)


;;  ------------------------------------------------------
;;  Argument pushing
;;  ------------------------------------------------------

(define-foreign-function pcelisp-c-int
  ((value :fixnum))
  :result-type :boolean)


(define-foreign-function pcelisp-c-real
  ((value :float))
  :result-type :boolean)


(define-foreign-function pcelisp-c-string
  ((value :simple-string))
  :result-type :boolean)


(define-foreign-function pcelisp-c-name
#+STATIC-LISP-SYMBOLS ((value :as-is))
#-STATIC-LISP-SYMBOLS ((value :uinteger))
  :result-type :boolean)


(define-foreign-function pcelisp-c-assoc
  ((value :as-is))
  :result-type :boolean)


(define-foreign-function pcelisp-c-ref
  ((value :fixnum))
  :result-type :boolean)


(define-foreign-function pcelisp-c-symbol
#+STATIC-LISP-SYMBOLS ((value :as-is))
#-STATIC-LISP-SYMBOLS ((value :uinteger))
  :result-type :boolean)


;;  ------------------------------------------------------
;;  Existence checks
;;  ------------------------------------------------------

(define-foreign-function pcelisp-c-is-assoc
  ((value :as-is))
  :result-type :boolean)


(define-foreign-function pcelisp-c-is-ref
  ((value :fixnum))
  :result-type :boolean)


;;  ------------------------------------------------------
;;  PCE virtual machine instructions
;;  ------------------------------------------------------

(define-foreign-function pcelisp-c-send
  ((status (:alien pcelisp-status)))
  :result-type :boolean)


(define-foreign-function pcelisp-c-new
  ((status (:alien pcelisp-status)))
  :result-type :boolean)


(define-foreign-function pcelisp-c-new-named
  ((name :simple-string))
  :result-type :boolean)


(define-foreign-function pcelisp-c-get
  ((status (:alien pcelisp-status)))
  :result-type :boolean)


;;  ------------------------------------------------------
;;  Returning values
;;  ------------------------------------------------------

(define-foreign-function pcelisp-c-pull
  ((status (:alien pcelisp-status))
   (vector :integer)
   (argc :fixnum))
  :result-type :boolean)


(define-foreign-function pcelisp-c-fetch-integer
  ((status (:alien pcelisp-status)))
  :result-type :fixnum)


(define-foreign-function pcelisp-c-fetch-real
  ((status (:alien pcelisp-status)))
  :result-type :single-float)


(define-foreign-function pcelisp-c-fetch-native
  ((status (:alien pcelisp-status)))
  :result-type :as-is)


(define-foreign-function pcelisp-c-fetch-name
  ((status (:alien pcelisp-status)))
  :result-type :as-is)


(define-foreign-function pcelisp-c-fetch-assoc
  ((status (:alien pcelisp-status)))
  :result-type :as-is)


(define-foreign-function pcelisp-c-fetch-string
  ((status (:alien pcelisp-status)))
  :result-type :string)


(define-foreign-function pcelisp-c-fetch-assoc-name
  ((status (:alien pcelisp-status)))
  :result-type :string)


;;  ------------------------------------------------------
;;  Special functions
;;  ------------------------------------------------------

(define-foreign-function (pce-reset "_pcelisp_reset") ())


;;  ------------------------------------------------------
;;  Management of PCELISP-STATUS pointers
;;  ------------------------------------------------------

;;  Some of the data between Lisp and PCE is passed directly through a
;;  foreign structure called pcelisp-status.  This saves the overhead
;;  of a foreign function call for trivial data (e.g. number of
;;  arguments, error status) and also makes it possible to call PCE
;;  recursively.
;;
;;  Unfortunately, allocating and freeing a pcelisp-status pointer
;;  takes a lot of time (0.5 milli seconds on a SparcStation II),
;;  so we have to set up a small manager for them.  [NB.  I am always
;;  amazed to see how Lisp implementors succeed in making the most
;;  critical operation (memory allocation) as slow as possible.]

(defvar *pcelisp-status-free-list* nil)

(defun alloc-pcelisp-status ()
  (or (pop *pcelisp-status-free-list*)
      (make-pcelisp-status)))

(defun free-pcelisp-status (ps)
  (push ps *pcelisp-status-free-list*))


;;  ------------------------------------------------------
;;  Call-back from PCE (messages to @lisp)
;;  ------------------------------------------------------

(defun pce-call-back (argc function vector)
  (let ((argv nil)
	(rval 0)
	(real-function (if (symbolp function)
			   function
			 (pcelisp-find-symbol function)))
	(ps (alloc-pcelisp-status)))
    (dotimes (i argc)
	     (pcelisp-c-pull ps vector (- argc (1+ i)))
	     (push (pcelisp-answer ps) argv))
    (if (fboundp real-function)
	(if (apply real-function argv)
	    (setf rval 1)
	  (setf rval 0))
      (progn
	(warn ";;; PCE-CALL-BACK (WARNING): Function ~A (~A) not found"
	      real-function argv)
	(setf rval 0)))
    (free-pcelisp-status ps)
    rval))


(foreign-callable pce-call-back
		  (:integer
#+STATIC-LISP-SYMBOLS :as-is
#-STATIC-LISP-SYMBOLS :integer
		   :integer)
		  :foreign-name "_pce_call_back"
		  :result-type :integer)


;;! pcelisp-call-back
;;
;;  This function is called when the user requests information about
;;  the host language (from the PCE tracer).  The argument is a
;;  keyword indicating the kind of request.
;;
;;  The implementation uses the functions defined in CLtL2 (p. 913).

(defun pcelisp-call-back (request-index)
  (let ((request (pcelisp-find-symbol request-index)))
    (case request
	  (:ABORT
	   (pce-reset)			; Try to reset PCE
	   (abort))			; Abort to Lisp top-level
	  (:BACKTRACE
	   (break "Backtrace request from PCE -- Use :C [ONTINUE] to return"))
	  (:BREAK
	   (break "Break request from PCE -- Use :C [ONTINUE] to return"))
	  (:HALT
	   (quit))
	  (:FATAL
	   (format t ";;; Fatal error in PCE -- aborting to Lisp top-level~%")
	   (pce-reset)
	   (abort)))
    1))


(foreign-callable pcelisp-call-back
		  (:uinteger)
		  :foreign-name "_pcelisp_call_back"
		  :result-type :integer)


;;! pcelisp-define-symbol symbol-name package-name ==> symbol
;;
;;  Defines a Lisp symbol from ^symbol-name^ and ^package-name^.  This
;;  function is called from the C interface when an unknown Lisp
;;  symbol is encountered.  Returns the symbol created.

(defun pcelisp-define-symbol (symbol-name package-name)
  (let ((symbol (intern symbol-name (or (find-package package-name)
					(make-package package-name)))))
    (or (pcelisp-symbol-index symbol)
	(pcelisp-add-symbol symbol))))


(foreign-callable pcelisp-define-symbol
		  (:string :string)
		  :foreign-name "_pcelisp_define_symbol"
		  :result-type :integer)


;;  ------------------------------------------------------
;;  Backwards compatibility with Lucid
;;  ------------------------------------------------------

(export '(pwd
	  quit
	  string-append
	  shell))


(defun pwd (&optional (dir "./"))
  (get-working-directory dir))


(defun quit ()
  (bye))


(defun string-append (&rest strings)
  (apply #'concatenate (cons 'string strings)))


(defun shell (command)
  (system::call-system command))
