/* This is make.pl
**
** Purpose: ``Update algorithm'' of PROM
**
** (c) T.Kielmann, 92-04-22
**
*/



/*
** **********************************************************************
** Main target: make/1
** make list of targets
**
*/

/*
** 1. remove temporal (help) predicates from knowledge base,
** 2. build source trees for all targets,
** 3. and update finally files out of date
*/

make(Targets) :-
	remove_tmp_predicate(topleveltarget),
        remove_tmp_predicate(isok),
        remove_tmp_predicate(oldexp),
	getsrctrees(Targets,SrcTrees),
	update(SrcTrees).

remove_tmp_predicate(Predicate) :-
        recorded(Predicate,_,Key),
        erase(Key),
        fail.
remove_tmp_predicate(_) :-
        !.



/*
** **********************************************************************
** collect source trees for targets
**
*/


/*
** construct a source tree for a target:
** 1. mark target as ``toplevel target''
** 2. create tree of sources
**
** all source trees form a list of triples:
** (Target,SourceTree,RuleIdentification)
**
*/

getsrctrees([],[]).
getsrctrees([Target1|TargetList],[(Target1,SrcTree1,Rule1)|SrcTreeList]) :-
	recordz(topleveltarget,topleveltarget(Target1)),
	remove_tmp_predicate(missing_source),
	makedeptree(Target1,SrcTree1,Rule1),
	getsrctrees(TargetList,SrcTreeList).


/*
** make dependence tree:
** 1. collect all sources
** 2. choose appropriate rule
*/

makedeptree(Target,SrcList,RuleNo) :-
	alldependencies(Target,Dependencies),
	deponkind(Target,SrcList,Dependencies,RuleNo).

/*
** in case of missing source files...
*/

makedeptree(Target,_,_) :-
	recorded(topleveltarget,topleveltarget(Target)),
	recorded(missing_source,_),
	write('prom: '),
        write(Target),
        write(' is missing (some of) the following source files:'),nl,
	findall(Source,recorded(missing_source,Source),Sources),
	print_missing(Sources),
	!,
	fail.

print_missing([]) :-
	nl.
print_missing([M|Tail]) :-
	write(M),write(' '),
	print_missing(Tail).


/*
** collect all sources:
** 1. get bag of sources
** 2. flatten the result to a simple list
*/

alldependencies(Target,Dependencies) :-
	findall(SourceList,recorded(dep,dep(Target,SourceList)),DepsByEntries),
	!,
	flatten(DepsByEntries,Dependencies),!.


/*
** **********************************************************************
** the list of sources depends on the kind of the target file
*/

/*
** ordinary file with a rule to make it
*/
deponkind(Target,SrcList,Sdep,RuleNo) :-
	recorded(rule,rule(Target,Srule,_,RuleNo)),
	append(Sdep,Srule,Sall),
	unifyvarterms(Sall,Sall,InstantiatedSources),
	sort(InstantiatedSources,NoDoubleSources),
	makesrctree(SrcList,NoDoubleSources),
	!.

/*
** term which denotes several files...
*/
deponkind(Target,SrcList,Sdep,expand) :-
	not recorded(rule,rule(Target,_,_,_)),
	expandtostructs(Target,TargetStructs),
	append(Sdep,TargetStructs,Sall),
	unifyvarterms(Sall,Sall,InstantiatedSources),
        sort(InstantiatedSources,NoDoubleSources),
        makesrctree(SrcList,NoDoubleSources),
	!.

/*
** basefile (source without predecessors...)
*/
deponkind(Target,[],[],norule) :-
/*	not recorded(rule,rule(Target,_,_,_)),
*/	expand(Target,[Filename]),
	not structure(_,[Filename],[]),
	exists_file(Filename),!.

/*
** missing sourcefile...
*/
deponkind(Target,_,_,_) :-
	not expandtostructs(Target,_),
	expand(Target,[Filename]),
	not structure(_,[Filename],[]),
	not exists_file(Filename),!,
	recordz(missing_source,Target),
	fail.

/*
** unresolvable structure...
*/
deponkind(Target,_,_,_) :-
	not expandtostructs(Target,_),
	expand(Target,[Filename]),
	structure(_,[Filename],[]),
	write('prom: '),
	write(Filename),
	write(' cannot be resolved'),
	nl,
	abort.



/*
** unify ``ungrounded'' terms with actual literals
*/

unifyvarterms([],_,[]).
unifyvarterms([T1|TL],All,TLU) :-
	isvarterm(T1),!,
	unifyvarterm(T1,All),
	unifyvarterms(TL,All,TLU).
unifyvarterms([T1|TL],All,[T1|TLU]) :-
	unifyvarterms(TL,All,TLU).

unifyvarterm(T1,All) :-
	member(T1,All),
	not isvarterm(T1).

isvarterm(T) :- atom(T),!,fail.
isvarterm(T) :- var(T).
isvarterm(T) :- T=..L,
		member(M,L),
		isvarterm(M).



/*
** replace term by other terms if it can be replaced by structures
*/

expandtostructs(Target,Expansion) :-
	recorded(def,def(Target,Expansion)),
	allstructs(Expansion).

allstructs([]).
allstructs([H|T]) :-
	structure(S,[H],[]),
	not S = +(_,_),
	allstructs(T).


/*
** build recursively dependence tree for the source files
*/

makesrctree([],[]).
makesrctree([(S1,S1List,RuleNo)|SList],[S1|SOther]) :-
	makedeptree(S1,S1List,RuleNo),
	makesrctree(SList,SOther).



/*
** **********************************************************************
** update process
*/


/*
** update list of targets
*/

update([]).

/*
** basefiles are always ``up to date''
** warn, if a ``top-level'' target is a basefile
*/
update([(Target,[],norule)|TargetList]) :-
	!,
	warniftoplevel(Target),
	update(TargetList).
/*
** simple term expansion is an empty update step...
*/
update([(_,SrcList,expand)|TargetList]) :-
	!,
	update(SrcList),
	update(TargetList).
/*
** ``real'' target:
** ensure actuality
*/
update([(Target,SrcList,RuleNo)|TargetList]) :-
	update(SrcList),
	recorded(rule,rule(Target,_,ActList,RuleNo)),
	makeactuality(Target,SrcList,ActList),
	update(TargetList).

/*
** print warning message
*/
warniftoplevel(Target) :-
	recorded(topleveltarget,topleveltarget(Target)),
	write('prom: don''t know how to make '),
	write(Target),nl,
	!.
warniftoplevel(_).	

/*
** ensure actuality
*/
/*
** a target already marked as ok needs no further actions
*/
makeactuality(Target,_,_) :-
	recorded(isok,isok(Target)),
	!.
/*
** if the target is ``out of date'' with respect to its sources,
** we have to perform actions
** finally, we mark the target as ``isok''
*/
makeactuality(Target,SrcList,ActList) :-
	outofdate(Target,SrcList),
	!,
	doacts(ActList),!,
	test_exist(Target),
	recordz(isok,isok(Target)).
/*
** if a ``top level'' target is up to date, we print this fact
*/
makeactuality(Target,_,_) :-
	recorded(topleveltarget,topleveltarget(Target)),
	write(Target),write(' is up to date.'),nl.
/*
** every other target must be a basefile and is ok per se
*/
makeactuality(Target,_,_) :-
	not recorded(topleveltarget,topleveltarget(Target)).

/*
** test for ``out of date'' with respect to all sources
*/
outofdate(T,List):-member((M,_,_),List),
		   expand(T,[TE]),time(TE,TT),
		   expand(M,[ME]),time(ME,TM),
		   TM @>= TT.

time(File,Stamp) :- exists_file(File),
                    time_file(File,Stamp),
                    !.
time(_,0).



/*
** **********************************************************************
** performing of actions
*/

doacts([]).
doacts([A|List]) :-
	A =.. [call|ActTokenList],
        create_action(ActTokenList,Action),
	write(Action),nl,
	!,
	perform(Action),
	doacts(List).



/*
** create a single atom describing the action to be performed
** out of the token list
*/
create_action(ActTokenList,Action) :-
	expandact(ActTokenList,AExp),
	flatten(AExp,AFlat),
        insert_blanks(AFlat,AWithBlanks),
	concat_atom(AWithBlanks,Action).


/*
** expand all elements of an action
*/

expandact([],[]).
expandact([T1|TL],[T1E|TLE]) :- expand(T1,T1E),
				expandact(TL,TLE).

/*
** we insert blanks after each argument (except the last one)
*/
insert_blanks([Arg],[Arg]).
insert_blanks([Arg1|ArgList],[Arg1,' ' | ListWithBlanks]) :-
         insert_blanks(ArgList,ListWithBlanks).



/*
** if the ``no action'' switch is set, we only print the actions
*/
perform(_) :- recorded(noaction,noaction),!.
perform(Action) :-
	shell(Action,Status),
        check_status(Action,Status),
        !.

/*
** if an action did not return a zero status,
** we print a message and abort updating
*/

check_status(_,0) :-
	!.
check_status(Action,Status) :-
	write('prom: action'),nl,
        write('       '),
	write(Action),nl,
	write('returned status '),
        write(Status),nl,
        abort.


/*
** check if a target exists after creation
** (only if the ``noaction'' switch is not set)
*/

test_exist(_) :-
        recorded(noaction,noaction).
test_exist(Target) :-
	expand(Target,[TargExp]),
	exists_file(TargExp),!.
test_exist(Target) :-
	expand(Target,[TargExp]),
	write('prom: action did not create '),
	write(TargExp),nl,
	abort.

