%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Copyright (C) 1989 Regents of the University of California.
% All rights reserved.  This program may be freely used and modified for
% non-commercial purposes provided this copyright notice is kept unchanged.
% Written by Peter Van Roy
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Example of the use of hidden arguments:
% The compilation of unification into an intermediate language

% For an explanation of what this program does see "An Intermediate Language
% to Support Prolog's Unification", Proceedings of the 1989 North American
% Conference on Logic Programming.

% Some sample executions:
%
% | ?- u(X, a, [X], Out).
% | ?- u(X, [Y,Z], [X,Y,Z], Out).
% | ?- u(X, t(1,2,s(3)), [], Out).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Declarations:

% Accumulators:

acc_info(code,   T, Out, In, (Out=[T|In])).   % Generated code.
acc_info(vars,   V, In, Out, incl(V,In,Out)). % Set of initialized variables.
acc_info(offset, I, In, Out, (Out is I+In)).  % Offset into writemode term.
acc_info(size,   I, In, Out, (Out is I+In)).  % Size of a term.

% Predicates:

pred_info(u,          2, [       vars     ]).
pred_info(init_var,   3, [            code]).
pred_info(Name,   Arity, [       vars,code]) :-
    member(Name/Arity,
	[unify/2, uninit/2, init/4, unify_var/2, unify_block/4, make_slots/5,
	 unify_writemode/4, unify_readmode/3, unify_args/6, unify_arg/6]).
pred_info(block,      2, [offset,vars,code]).
pred_info(block_args, 5, [offset,vars,code]).
pred_info(size,       1, [size]).
pred_info(size_args,  3, [size]).

member(X, [X|_]).
member(X, [_|L]) :- member(X, L).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% u(X, T, In, Out)
% Unify variable X with term T and write the result:
u(X, T) -->> unify(X, T):code(Code,[]), write_code(Code, 4).
 
% unify(X, T, In, Out, Code, Link)
% Unify the variable X with the term T, given the set In of
% variables initialized before the unification.
% Return the intermediate code generated in the accumulator 'code'
% and the set Out of variables initialized after the unification.
unify(X, T) -->> In/vars, \+in(X, In), !, uninit(X, T).
unify(X, T) -->> In/vars,   in(X, In), !,   init(X, T, nonlast, _).

%**** Uninit assumes X has not yet been initialized:
uninit(X, T) -->> compound(T), !, [move(Tag^h, X)]:code,
        termtag(T, Tag), unify_block(nonlast, T, _, _), [X]:vars.
uninit(X, T) -->>   atomic(T), !, [move(tatm^T, X)]:code, [X]:vars.
uninit(X, T) -->>      var(T), !, unify_var(X, T).

%**** Init assumes X has already been initialized:
init(X, T, Last, LLbls) -->> nonvar(T), !,
        termtag(T,Tag),
	[deref(X), switch(Tag,X,[trail(X)|Write],Read,fail)]:code,
	insert(In,Out):vars,
        {unify_writemode(X, T, Last, LLbls, In, _, Write, [])},
        {unify_readmode(X, T, LLbls, In, Out, Read, [])}.
init(X, T,    _,     _) -->> var(T), !, unify_var(X, T).

%**** Unifying two variables together:
unify_var(X, Y) -->> In/vars, in(X, In),   in(Y, In), !, [unify(X,Y,fail)]:code.
unify_var(X, Y) -->> In/vars, in(X, In), \+in(Y, In), !, [move(X,Y)]:code,
	[Y]:vars.
unify_var(X, Y) -->> In/vars, \+in(X, In), in(Y, In), !, [move(Y,X)]:code,
	[X]:vars.
unify_var(X, Y) -->> In/vars, \+in(X, In), \+in(Y, In), !,
        [move(tvar^h,X), move(tvar^h,Y), add(1,h), move(Y,[h-1])]:code,
	[X,Y]:vars.

%**** Unify_readmode assumes X is a dereferenced nonvariable
% at run-time and T is a nonvariable at compile-time.
unify_readmode(X, T, LLbls) -->> structure(T), !,
	[equal([X],tatm^(F/N),fail)]:code,
        functor(T, F, N),
	unify_args(1, N, T, 0, X, LLbls).
unify_readmode(X, T, LLbls) -->> cons(T), !,
        unify_args(1, 2, T, -1, X, LLbls).
unify_readmode(X, T,     _) -->> atomic(T), !,
	[equal(X,tatm^T,fail)]:code.

unify_args(I, N, _, _, _,         _) -->> I>N, !.
unify_args(I, N, T, D, X, [_|LLbls]) -->> I=N, !,
        unify_arg(I, T, D, X, last, LLbls). 
unify_args(I, N, T, D, X,     LLbls) -->> I<N, !,
        unify_arg(I, T, D, X, nonlast, _),
        I1 is I+1, unify_args(I1, N, T, D, X, LLbls).

unify_arg(I, T, D, X, Last, LLbls) -->>
	[move([X+ID],Y)]:code,
        ID is I+D, arg(I, T, A),
	[Y]:vars,
        init(Y, A, Last, LLbls).

%**** Unify_writemode assumes X is a dereferenced unbound
% variable at run-time and T is a nonvariable at compile-time.
unify_writemode(X, T, Last, LLbls) -->> compound(T), !,
	[move(Tag^h,[X])]:code,
        termtag(T, Tag),
	unify_block(Last, T, _, LLbls).
unify_writemode(X, T,    _,     _) -->> atomic(T), !,
	[move(tatm^T,[X])]:code.

%**** Generate a minimal sequence of moves to create T on the heap:
unify_block(   last, T, Size,   [Lbl|_]) -->> !,
	size(T):size(0,Size),
	[add(Size,h),jump(Lbl)]:code.
unify_block(nonlast, T, Size, [_|LLbls]) -->> !,
        size(T):size(0,Size), Offset is -Size,
	[add(Size,h)]:code,
	block(T, LLbls):offset(Offset,0).

block(T, LLbls) -->> structure(T), !, D/offset,
	[move(tatm^(F/N),[h+D])]:code,
        functor(T, F, N),
	[N]:offset, [1]:offset,
	D1 is D+1,
        make_slots(1, N, T, D1, Offsets),
        block_args(1, N, T, Offsets, LLbls).
block(T, LLbls) -->> cons(T), !, D/offset,
	[2]:offset,
        make_slots(1, 2, T, D, Offsets),
        block_args(1, 2, T, Offsets, LLbls).
block(T,    []) -->> atomic(T), !.
block(T,    []) -->> var(T), !.

block_args(I, N, _,  [],            []) -->> I>N, !.
block_args(I, N, T, [D],   [Lbl|LLbls]) -->> I=N, !, D/offset,
	[label(Lbl)]:code,
        arg(I, T, A), block(A, LLbls).
block_args(I, N, T, [D|Offsets], LLbls) -->> I<N, !, D/offset,
        arg(I, T, A), block(A, _), I1 is I+1,
        block_args(I1, N, T, Offsets, LLbls).

make_slots(I, N, _, _,            []) -->> I>N, !.
make_slots(I, N, T, D, [Off|Offsets]) -->> I=<N, !,
        arg(I, T, A),
	In/vars, init_var(A, D, In),
        make_word(A, Off, Word),
	[move(Word,[h+D])]:code,
	[A]:vars,
        D1 is D+1, I1 is I+1,
        make_slots(I1, N, T, D1, Offsets).

% Initialize first-time variables in write mode:
init_var(V, I, In) -->> var(V), \+in(V, In), !, [move(tvar^(h+I),V)]:code.
init_var(V, _, In) -->> var(V),   in(V, In), !.
init_var(V, _,  _) -->> nonvar(V), !.

make_word(C, Off, Tag^(h+Off)) :- compound(C), !, termtag(C, Tag).
make_word(V,   _, V)           :- var(V), !.
make_word(A,   _, tatm^A)      :- atomic(A), !.

% Calculate the size of T on the heap:
size(T) -->> structure(T), !,
	functor(T, _, N),
	[1]:size, [N]:size,
	size_args(1, N, T).
size(T) -->> cons(T), !,
	[2]:size,
	size_args(1, 2, T).
size(T) -->> atomic(T), !.
size(T) -->> var(T), !.

size_args(I, N, _) -->>  I>N, !.
size_args(I, N, T) -->> I=<N, !,
	arg(I, T, A), size(A), I1 is I+1, size_args(I1, N, T).

%**** Utility routines:

in(X, [Y|_]) :- X==Y, !.
in(X, [Y|L]) :- X\==Y, in(X, L).

incl(A, [B|S], [A,B|S]) :- var(A), A@<B, !.
incl(A, [B|S], [B|S])   :- var(A), A==B, !.
incl(A, [B|S], [B|S1])  :- var(A), A@>B, !, incl(A, S, S1).
incl(A, [], [A]) :- var(A), !.
incl(A, S, S) :- nonvar(A), !.

compound(X)  :- nonvar(X), functor(X,_,A), A>0.
cons(X)      :- compound(X),   X=[_|_].
structure(X) :- compound(X), \+X=[_|_].

termtag(T, tstr) :- structure(T).
termtag(T, tlst) :- cons(T).
termtag(T, tatm) :- atomic(T).
termtag(T, tvar) :- var(T).

write_code([], _).
write_code([I|L], N) :- write_code(I, L, N).
 
write_code(switch(Tag,V,Wbr,Rbr,Fail), L, N) :- !, N1 is N+4,
        tab(N), write('switch('), write(V), write(') {'), nl,
        tab(N), write(tvar), write(':'), nl, write_code(Wbr, N1),
        tab(N), write(Tag), write(':'), nl, write_code(Rbr, N1),
        tab(N), write('else: '), write(Fail), nl,
        tab(N), write('}'), nl, write_code(L, N).
write_code(label(Lbl), L, N) :- !, N1 is N-4,
        tab(N1), write(Lbl), write(':'), nl, write_code(L, N).
write_code(Instr, L, N) :-
        \+(Instr=label(_)), \+(Instr=switch(_,_,_,_,_)),
        tab(N), write(Instr), nl, write_code(L, N).
