#include <stdlib.h>
#include "token.h"

#define INPUTBUFFERSIZE 10000

unsigned char inputbuffer[SLACK+INPUTBUFFERSIZE];
unsigned char *bufferend=&(inputbuffer[SLACK+INPUTBUFFERSIZE]);
unsigned char *bufferptr=&(inputbuffer[SLACK+INPUTBUFFERSIZE]);
int inputhandle = 0;

unsigned char tokenstring[MAXTOKENLENGTH+1];
unsigned char *tokenptr;
int integertoken;
double floattoken;
unsigned char tokenchar;
int parseerror;
int errorline;
int linenumber = 1;
unsigned char chararr[256];
unsigned char charconv[256];


#define CHAR_CONVERSION()  tokenchar = charconv[tokenchar]

#define RESETTOKEN tokenptr = tokenstring
#define ADDTOKENCHAR(c) { if (tokenptr < tokenstring+MAXTOKENLENGTH) { \
        *tokenptr++ = (c); } else seterr(ERR_TOKEN_TOO_LONG); }
#define TOKENLENGTH (tokenptr - tokenstring)

/* return next char and skip it ; update line count */
#define SKIPCHAR() (new_line_char(*bufferptr) ? \
        (linenumber++, *bufferptr++) : *bufferptr++)

void inittokenizer()
{
  register unsigned int i;
  for (i=0 ; i<=255; i++) {
    chararr[i] = 0;
    if (PROLOGCHAR(i)) chararr[i] |= _PROLOGCHAR;
    if (GRAPHIC_CHAR(i)) chararr[i] |= _GRAPHIC_CHAR;
    if (GRAPHIC_TOKEN_CHAR(i)) chararr[i] |= _GRAPHIC_TOKEN_CHAR;
    if (ALPHA_NUMERIC_CHAR(i)) chararr[i] |= _ALPHA_NUMERIC_CHAR;
    if (HEXADECIMAL_DIGIT_CHAR(i)) chararr[i] |= _HEXADECIMAL_DIGIT_CHAR;
    if (SOLO_CHAR(i)) chararr[i] |= _SOLO_CHAR;
    if (SYMBOLIC_CONTROL_CHAR(i)) chararr[i] |= _SYMBOLIC_CONTROL_CHAR;
    if (LAYOUT_CHAR(i)) chararr[i] |= _LAYOUT_CHAR;
    printf("");
  }
  for (i=0 ; i<=255; i++) {
    charconv[i] = i;
  }
}

void seterr(err)
register int err;
{
  if (parseerror == ERR_NOERROR) {
    parseerror = err;
    errorline = linenumber;
  }
}

void showerror()
{
  register unsigned char *s;

  switch (parseerror) {
    case ERR_NOERROR: s = "Clause correctly parsed"; break;
    case ERR_MISSING_QUOTE: s = "Missing quote"; break;
    case ERR_TOKEN_TOO_LONG: s = "Token too long"; break;
    case ERR_UNEXPECTED_EOF: s = "Unexpected eof"; break;
    case ERR_BACK_QUOTED_STRING: s = "Back quoted strings not supported"; break;
    case ERR_NONDET: s = "Non deterministic code"; break;
    case ERR_INCOMPLETE_REDUCTION: s = "Incomplete reduction"; break;
    default: s = "?";
  }
  printf("syntax_error on line %d: %s\n", errorline, s);
}

int lookahead_char2()
/* return current char and refill buffer is empty  */
{
  if (bufferptr >= bufferend) {
    int count = fileread(&(inputbuffer[SLACK]), INPUTBUFFERSIZE);
    if (!count) { /* end-of-file ; return 0 */
      count++;
      inputbuffer[SLACK] = 0;
    }
    bufferend = &(inputbuffer[SLACK+count]);
    bufferptr = &(inputbuffer[SLACK]);
  }
  return *bufferptr;
}

 
void returnchar(c)
int c;
/* return char to buffer */
/* SLACK is used in case the buffer would have been refilled */
{
  if (new_line_char(c)) linenumber--;
  *--bufferptr = c;
}

void single_line_comment()  
/*
 *   single line comment (* 6.4.1 *)
 *      = end line comment char (* 6.5.3 *),
 *        comment txt (* 6.4.1 *),
 *        new line char (* 6.5.4 *) ;
 */
{
  /* end_line_comment_char(LOOKAHEAD_CHAR) is true */
  SKIPCHAR();
  while (prologchar(LOOKAHEAD_CHAR) && !new_line_char(LOOKAHEAD_CHAR)) 
    SKIPCHAR();
  SKIPCHAR();
}

int bracketed_comment()
/*
 *  bracketed comment (* 6.4.1 *)
 *     = comment open (* 6.4.1 *),
 *       comment text (* 6.4.1 *),
 *       comment close (* 6.4.1 *) ;

 *  comment open (* 6.4.1 *)
 *    = comment 1 char (* 6.4.1 *),
 *      comment 2 char (* 6.4.1 *) ;

 *  comment close (* 6.4.1 *)
 *    = comment 2 char (* 6.4.1 *),
 *      comment 1 char (* 6.4.1 *) ;

 *  comment text (* 6.4.1 *)
 *    = { char (* 6.5 *) } ;
 */
{
  if (comment_1_char(LOOKAHEAD_CHAR)) { 
    register unsigned char c = SKIPCHAR();
    if (comment_2_char(LOOKAHEAD_CHAR)) {
      SKIPCHAR();
      do {
        while (prologchar(LOOKAHEAD_CHAR) && !comment_2_char(LOOKAHEAD_CHAR)) 
          SKIPCHAR();
        SKIPCHAR();
      } while (prologchar(LOOKAHEAD_CHAR) && !comment_1_char(LOOKAHEAD_CHAR));
      SKIPCHAR();
      return 1;
    } else {
      returnchar(c);
      return 0;
    }
  } else
    return 0;
}

int char_code_list_token()
/*
 * char code list token (* 6.4.6 *)
 *    = double quote char (* 6.5.5 *),
 *      { double quoted item } (* 6.5.6 *),
 *      double quote char (* 6.5.5 *) ;
 */
{
  /* double_quote_token(LOOKAHEAD_CHAR) is true */
  SKIPCHAR();
  RESETTOKEN;
  while (get_double_quoted_item()) ;
  if (double_quote_char(LOOKAHEAD_CHAR)) {
    SKIPCHAR();
    ADDTOKENCHAR(0);  /* terminator */
    return 1;
  } else
    return 0;
}

int number_token()
/*
 *  float number token (* 6.4.5 *)
 *    = integer constant (* 6.4.4 *),
 *      fraction (* 6.4.5 *),
 *      [ exponent (* 6.4.5 *) ] ;

 *  fraction (* 6.4.5 *)
 *    = decimal point char (* 6.4.5 *),
 *      decimal digit char (* 6.4.5 *),
 *      { decimal digit char (* 6.5.2 *) ;
 
 *  exponent (* 6.4.5 *)
 *    = exponent char (* 6.4.5 *),
 *      sign (* 6.4.5 *),
 *      integer constant (* 6.4.4 *) ;

 *  sign (* 6.4.5 *)
 *    = negative sign char (* 6.4.5 *)
 *    | [ positive sign char (* 6.4.5 *) ] ;


 *  integer token (* 6.4.4 *)
 *    = integer constant (* 6.4.4 *)
 *    | character code constant (* 6.4.4 *)
 *    | binary constant (* 6.4.4 *)
 *    | octal constant (* 6.4.4 *)
 *    | hexadecimal constant (* 6.4.4 *) ;
 
 *  integer constant (* 6.4.4 *)
 *    = decimal digit char (* 6.4.4 *),
 *      { decimal digit char (* 6.4.4 *) } ;

 *  character code constant (* 6.4.4 *)
 *    = "0", single quote char (* 6.5.5 *),
 *           single quoted char (* 6.4.2.1 *) ;
 */
{
  RESETTOKEN;
  ADDTOKENCHAR(SKIPCHAR());
  while (decimal_digit_char(LOOKAHEAD_CHAR)) 
    ADDTOKENCHAR(SKIPCHAR());
  if (decimal_point_char(LOOKAHEAD_CHAR)) {
    SKIPCHAR();
    if (decimal_digit_char(LOOKAHEAD_CHAR)) {
      ADDTOKENCHAR('.');
      ADDTOKENCHAR(SKIPCHAR());
      while (decimal_digit_char(LOOKAHEAD_CHAR)) 
        ADDTOKENCHAR(SKIPCHAR());
      if (exponent_char(LOOKAHEAD_CHAR)) {  
        ADDTOKENCHAR(SKIPCHAR());
        if (sign_char(LOOKAHEAD_CHAR)) 
          ADDTOKENCHAR(SKIPCHAR());
        if (decimal_digit_char(LOOKAHEAD_CHAR)) {
          ADDTOKENCHAR(SKIPCHAR());
          while (decimal_digit_char(LOOKAHEAD_CHAR)) 
            ADDTOKENCHAR(SKIPCHAR());
          ADDTOKENCHAR(0); /* terminator */
          floattoken = atof(tokenstring);
          return FLOAT_NUMBER_TOKEN;
        } else 
          return 0;
      } else {
        ADDTOKENCHAR(0); /* terminator */
        floattoken = atof(tokenstring);
        return FLOAT_NUMBER_TOKEN;
      }
    } else {
      returnchar('.');
      ADDTOKENCHAR(0); /* terminator */
      integertoken = atoi(tokenstring);
      return INTEGER_TOKEN;
    }
  } else { 
    /* up to here we have read an integer */
    if (tokenstring[0] == '0' && TOKENLENGTH == 1) {
      register unsigned char c = LOOKAHEAD_CHAR;
      if (c == 'x' &&
         (integertoken = hexadecimal_constant()) != -1) 
           return INTEGER_TOKEN;
      if (c == 'o' &&
         (integertoken = octal_constant()) != -1)
           return INTEGER_TOKEN;
      if (c == 'b' &&
         (integertoken = binary_constant()) != -1)
           return INTEGER_TOKEN;
      if (single_quote_char(c)) {
        SKIPCHAR();
        if (get_single_quoted_char()) {
          integertoken = tokenchar;
          return INTEGER_TOKEN;
        } else 
          return 0;
      }
      integertoken = 0; /* read number 0 */
      return INTEGER_TOKEN;
    } else {
      ADDTOKENCHAR(0); /* terminator */
      integertoken = atoi(tokenstring);
      return INTEGER_TOKEN;
    }
  }
}

int hexadecimal_constant()
/*
 *   hexadecimal constant (* 6.4.4 *)
 *     = hexadecimal constant indicator (* 6.4.4 *),
 *       hexadecimal digit char (* 6.5.2 *),
 *       { hexadecimal digit char (* 6.5.2 *) } ;
 *
 *   hexadecimal constant indicator (* 6.4.4 *) 
 *     = "0x" ;   ---> 0 already read by caller 
 */
{
  register unsigned n = 0;
  register unsigned char c;

  /* LOOKAHEAD == x is true */
  SKIPCHAR();
  if (hexadecimal_digit_char(LOOKAHEAD_CHAR)) {
    while (hexadecimal_digit_char(c = LOOKAHEAD_CHAR)) {
      if (decimal_digit_char(c))
        n = n * 16 + SKIPCHAR() - '0';
      else if (capital_letter_char(c))
        n = n * 16 + SKIPCHAR() - 'A' + 10;
      else
        n = n * 16 + SKIPCHAR() - 'a' + 10;
    }
    return n;
  } else {
    returnchar('x');
    return -1;
  }
}

int octal_constant()
/*
 *   octal constant (* 6.4.4 *)
 *     = octal constant indicator (* 6.4.4 *),
 *       octal digit char (* 6.5.2 *),
 *       { octal digit char (* 6.5.2 *) } ;
 *
 *   octal constant indicator (* 6.4.4 *) 
 *     = "0o" ;   ---> 0 already read by caller 
 */
{
  register unsigned n = 0;
  
  /* LOOKAHEAD == o is true */
  SKIPCHAR();
  if (octal_digit_char(LOOKAHEAD_CHAR)) {
    while (octal_digit_char(LOOKAHEAD_CHAR)) {
      n = n * 8 + SKIPCHAR() - '0';
    }
    return n;
  } else {
    returnchar('o'); 
    return -1;
  }
}

int binary_constant()
/*
 *   binary constant (* 6.4.4 *)
 *     = binary constant indicator (* 6.4.4 *),
 *       binary digit char (* 6.5.2 *),
 *       { binary digit char (* 6.5.2 *) } ;
 *
 *   binary constant indicator (* 6.4.4 *) 
 *     = "0b" ;   ---> 0 already read by caller 
 */
{
  register unsigned n = 0;
  
  /* LOOKAHEAD == b is true */
  SKIPCHAR();
  if (binary_digit_char(LOOKAHEAD_CHAR)) {
    while (binary_digit_char(LOOKAHEAD_CHAR)) {
      n = n * 2 + SKIPCHAR() - '0';
    }
    return n;
  } else {
    returnchar('b');
    return -1;
  }
}

int variable_token()
/*
 * variable token (* 6.4.3 *)
 *    = anonymous variable (* 6.4.3 *)
 *    | named variable (* 6.4.3 *)

 * anonymous variable (* 6.4.3 *)
 *    = variable indicator char (* 6.4.3 *)

 * named variable (* 6.4.3 *)
 *    = variable indicator char (* 6.4.3 *), 
 *      alpha numeric char (* 6.5.2 *), 
 *      { alpha numeric char (* 6.5.2 *) }
 *    | capital letter char (* 6.5.2 *),
 *      { alpha numeric char (* 6.5.2 *) } ;
 */
{
  /* variable_indicator_char(LOOKAHEAD_CHAR) || 
     capital_letter_char(LOOKAHEAD_CHAR) is true */
  RESETTOKEN;
  ADDTOKENCHAR(SKIPCHAR());
  while (alpha_numeric_char(LOOKAHEAD_CHAR))
    ADDTOKENCHAR(SKIPCHAR());
  ADDTOKENCHAR(0);  /* terminator */
  return VARIABLE_TOKEN;
}

int get_meta_escape_sequence()
/*
 *  meta escape sequence (* 6.4.2.1 *)
 *    = backslash char (* 6.5.5 *),
 *      meta char (* 6.5.5 *)
 */
{
  if (backslash_char(LOOKAHEAD_CHAR)) {
    register int c = SKIPCHAR();
    if (meta_char(LOOKAHEAD_CHAR)) {
      tokenchar = SKIPCHAR();
      return 1;
    } else {
      returnchar(c);
      return 0;
    }
  } else
    return 0;
}

int get_control_escape_sequence()
/*
 * control escape sequence (* 6.4.2.1 *)
 *   = backslash char (* 6.5.5 *),
 *     symbolic control char (* 6.4.2.1 *)
 
 * symbolic control char (* 6.4.2.1 *)
 *   = symbolic alert char (* 6.4.2.1 *)               "a"
 *   = symbolic vertical tab char (* 6.4.2.1 *)        "v"
 *   = symbolic horizontal tab char (* 6.4.2.1 *)      "t"
 *   = symbolic backspace char (* 6.4.2.1 *)           "b"
 *   = symbolic form feed char (* 6.4.2.1 *)           "f"
 *   = symbolic new line char (* 6.4.2.1 *)            "n"
 *   = symbolic carriage return char (* 6.4.2.1 *)     "r"
 */
{
  if (backslash_char(LOOKAHEAD_CHAR)) {
    register int c = SKIPCHAR();
    if (symbolic_control_char(LOOKAHEAD_CHAR)) {
      switch (SKIPCHAR()) {
        case 'a': tokenchar = 7;  break; /* alert char, bell */ 
        case 'b': tokenchar = 8;  break; /* backspace */
        case 'f': tokenchar = 12; break; /* formfeed */
        case 'n': tokenchar = 10; break; /* newline */
        case 'r': tokenchar = 13; break; /* carriage return */
        case 't': tokenchar = 9;  break; /* horizontal tab */
        case 'v': tokenchar = 11; break; /* vertical tab */
      }
      return 1;
    } else {
      returnchar(c);
      return 0;
    }
  } else
    return 0;
}

int get_octal_escape_sequence()
/*
 *   octal escape sequence (* 6.4.2.1 *)
 *     = backslash char (* 6.5.5 *),
 *       octal digit char (* 6.5.2 *),
 *       { octal digit char (* 6.5.2 *) },
 *       backslash char (* 6.5.5 *) ;
 */
{
  if (backslash_char(LOOKAHEAD_CHAR)) {
    register int c = SKIPCHAR();
    if (octal_digit_char(LOOKAHEAD_CHAR)) {
      register unsigned n = SKIPCHAR() - '0';
      while (octal_digit_char(LOOKAHEAD_CHAR)) {
        n = n * 8 + SKIPCHAR() - '0';
      }
      if (backslash_char(LOOKAHEAD_CHAR)) {
        SKIPCHAR();
        tokenchar = n;
        return 1;
      } else {
        printf("Error");
        return 0;
      }
    } else {
      returnchar(c);
      return 0;
    }
  } else
    return 0;
}

int get_hexadecimal_escape_sequence()
/*
 *   hexadecimal escape sequence (* 6.4.2.1 *)
 *      = backslash char (* 6.5.5 *),
 *        symbolic hexadecimal char (* 6.4.2.1 *),
 *        hexadecimal digit char (* 6.5.2 *),
 *        { hexadecimal digit char (* 6.5.2 *) },
 *        backslash char (* 6.5.5 *) ;
 */
{
  if (backslash_char(LOOKAHEAD_CHAR)) {
    register int c = SKIPCHAR();
    if (symbolic_hexadecimal_char(LOOKAHEAD_CHAR)) {
      register unsigned char lac;
      register unsigned n = 0;

      SKIPCHAR();
      while (hexadecimal_digit_char(lac = LOOKAHEAD_CHAR)) {
        if (decimal_digit_char(lac))
          n = n * 16 + SKIPCHAR() - '0';
        else if (capital_letter_char(lac))
          n = n * 16 + SKIPCHAR() - 'A' + 10;
        else
          n = n * 16 + SKIPCHAR() - 'a' + 10;
      }
      if (backslash_char(lac)) {
        tokenchar = n;
        SKIPCHAR();
        return 1;
      } else {
        printf("Error");
        return 0;
      }
    } else {
      returnchar(c);
      return 0;
    }
  } else
    return 0;
}

int get_non_quote_char()
/*
 *   non quote char (* 6.4.2.1 *)
 *     = graphic char (* 6.5.1 *)
 *     | alpha numeric char (* 6.5.2 *)
 *     | solo char (* 6.5.3 *)
 *     | space char (* 6.5.4 *)
 *     | meta escape sequence (* 6.4.2.1 *)
 *     | control escape sequence (* 6.4.2.1 *)
 *     | octal escape sequence (* 6.4.2.1 *)
 *     | hexadecimal escape sequence (* 6.4.2.1 *)
 */
{
  if (graphic_char(LOOKAHEAD_CHAR) ||
      alpha_numeric_char(LOOKAHEAD_CHAR) ||
      solo_char(LOOKAHEAD_CHAR) ||
      space_char(LOOKAHEAD_CHAR)) {
    tokenchar = SKIPCHAR();
    return 1;
  } else {
    if (get_meta_escape_sequence()) return 1;
    if (get_control_escape_sequence()) return 1;
    if (get_octal_escape_sequence()) return 1;
    if (get_hexadecimal_escape_sequence()) return 1;
    return 0;
  }
}

int get_single_quoted_char()
/*
 *  single quoted char (* 6.4.2.1 *)
 *    = non quote char (* 6.4.2.1 *)
 *    | single quote char (* 6.5.5 *),
 *      single quote char (* 6.5.5 *)
 *    | double quote char (* 6.5.5 *)
 *    | back quote char (* 6.5.5 *) ;
 */
{
  if (single_quote_char(LOOKAHEAD_CHAR)) {
    register char c = SKIPCHAR();
    if (!single_quote_char(LOOKAHEAD_CHAR)) {
      returnchar(c);
      return 0;
    } else {
      tokenchar = SKIPCHAR();
      return 1;
    }
  }
  if (double_quote_char(LOOKAHEAD_CHAR)) {
    tokenchar = SKIPCHAR();
    return 1;
  }
  if (back_quote_char(LOOKAHEAD_CHAR)) {
    tokenchar = SKIPCHAR();
    return 1;
  }
  if (get_non_quote_char()) {
    CHAR_CONVERSION();
    return 1;
  }
  return 0;
}

int get_double_quoted_char()
/*
 *  double quoted char (* 6.4.2.1 *)
 *    = non quote char (* 6.4.2.1 *)
 *    | single quote char (* 6.5.5 *)
 *    | double quote char (* 6.5.5 *),
 *      double quote char (* 6.5.5 *)
 *    | back quote char (* 6.5.5 *) ;
 */
{
  if (double_quote_char(LOOKAHEAD_CHAR)) {
    register char c = SKIPCHAR();
    if (!double_quote_char(LOOKAHEAD_CHAR)) {
      returnchar(c);
      return 0;
    } else {
      tokenchar = SKIPCHAR();
      return 1;
    }
  }
  if (single_quote_char(LOOKAHEAD_CHAR)) {
    tokenchar = SKIPCHAR();
    return 1;
  }
  if (back_quote_char(LOOKAHEAD_CHAR)) {
    tokenchar = SKIPCHAR();
    return 1;
  }
  if (get_non_quote_char()) {
    CHAR_CONVERSION();
    return 1;
  }
  return 0;
}

int get_back_quoted_char()
/*
 *  back quoted char (* 6.4.2.1 *)
 *    = non quote char (* 6.4.2.1 *)
 *    | single quote char (* 6.5.5 *)
 *    | double quote char (* 6.5.5 *)
 *    | back quote char (* 6.5.5 *), 
 *      back quote char (* 6.5.5 *) ;
 */
{
  if (back_quote_char(LOOKAHEAD_CHAR)) {
    register char c = SKIPCHAR();
    if (!back_quote_char(LOOKAHEAD_CHAR)) {
      returnchar(c);
      return 0;
    } else {
      tokenchar = SKIPCHAR();
      return 1;
    }
  }
  if (single_quote_char(LOOKAHEAD_CHAR)) {
    tokenchar = SKIPCHAR();
    return 1;
  }
  if (single_quote_char(LOOKAHEAD_CHAR)) {
    tokenchar = SKIPCHAR();
    return 1;
  }
  if (single_quote_char(LOOKAHEAD_CHAR)) {
    tokenchar = SKIPCHAR();
    return 1;
  }
  if (get_non_quote_char()) {
    CHAR_CONVERSION();
    return 1;
  }
  return 0;
}

int get_single_quoted_item()
/*
 *   single quoted item (* 6.4.2 *)
 *      = single quoted char (* 6.4.2.1 *)
 *      | continuation escape sequence (* 6.4.2.1 *) ;

 *   continuation escape sequence (* 6.4.2.1 *)
 *      = backslash char (* 6.5.5 *),
 *        new line char (* 6.5.4 *);
 */
{
  if (backslash_char(LOOKAHEAD_CHAR)) {
    register int c = SKIPCHAR();
    if (new_line_char(LOOKAHEAD_CHAR)) {
      SKIPCHAR();
      return 1;
    } else 
      returnchar(c);
  } 
  if (get_single_quoted_char()) {
    ADDTOKENCHAR(tokenchar);
    return 1;
  } else
    return 0;
}

int get_double_quoted_item()
/*
 *   double quoted item (*6.4.6 *)
 *      = double quoted char (* 6.4.2.1 *)
 *      | continuation escape sequence (* 6.4.2 *)
 */
{
  if (backslash_char(LOOKAHEAD_CHAR)) {
    register int c = SKIPCHAR();
    if (new_line_char(LOOKAHEAD_CHAR)) {
      SKIPCHAR();
      return 1;
    } else 
      returnchar(c);
  } 
  if (get_double_quoted_char()) {
    ADDTOKENCHAR(tokenchar);
    return 1;
  } else
    return 0;
}

int get_back_quoted_item()
/*
 *   back quoted item (*6.4.6 *)
 *      = back quoted char (* 6.4.2.1 *)
 *      | continuation escape sequence (* 6.4.2 *)
 */
{
  if (backslash_char(LOOKAHEAD_CHAR)) {
    register int c = SKIPCHAR();
    if (new_line_char(LOOKAHEAD_CHAR)) {
      SKIPCHAR();
      ADDTOKENCHAR(0);
      return 1;
    } else 
      returnchar(c);
  } 
  if (get_back_quoted_char()) {
    return 1;
  } else
    return 0;
}

int get_back_quoted_string()
/*
 *   back quoted string (* 6.4.2 *)
 *     = back quote char (* 6.5.5 *),
 *       { back quoted item (* 6.4.2 *) },
 *       back quote char (* 6.5.5 *)
 */
{   
  if (back_quote_char(LOOKAHEAD_CHAR)) {
    SKIPCHAR();
    while (get_back_quoted_item());
    if (back_quote_char(LOOKAHEAD_CHAR)) {
      SKIPCHAR();
      ADDTOKENCHAR(tokenchar);
      return 1;
    } else {
      seterr(ERR_MISSING_QUOTE);
      return 0;
    }
  } else  
    return 0;
}

int get_name_token()
/*
 *   name token (* 6.4.2 *)
 *     = identifier token (* 6.4.2 *) 
 *     | graphic token (* 6.4.2 *)
 *     | quoted token (* 6.4.2 *)
 *     | semicolon token (* 6.4.2 *)
 *     | cut token (* 6.4.2 *)
 
 *   identifier token (* 6.4.2 *)
 *      = small letter char (* 6.5.2 *),
 *        { alpha numeric char (* 6.5.2 *) } ;

 *   graphic token (* 6.4.2 *)
 *      = graphic token char (* 6.4.2 *)
 *        { graphic token char (* 6.4.2 *) } ; 

 *   quoted token (* 6.4.2 *)
 *     = single quote char (* 6.5.5 *),
 *       { single quoted item (* 6.4.2 *) },
 *       single quote char (* 6.5.5 *)
 */
{
   register unsigned char c = LOOKAHEAD_CHAR;
   RESETTOKEN;
   if (small_letter_char(c)) {
     ADDTOKENCHAR(SKIPCHAR());
     while (alpha_numeric_char(LOOKAHEAD_CHAR)) 
       ADDTOKENCHAR(SKIPCHAR());
     ADDTOKENCHAR(0); /* term */
     return 1;
   }
   if (graphic_token_char(c)) {
     ADDTOKENCHAR(SKIPCHAR());
     while (graphic_token_char(LOOKAHEAD_CHAR)) 
       ADDTOKENCHAR(SKIPCHAR());
     ADDTOKENCHAR(0); /* term */
     return 1;
   }
   if (cut_token(c)) {
     ADDTOKENCHAR(SKIPCHAR());
     ADDTOKENCHAR(0); /* term */
     return 1;
   }
   if (semicolon_token(c)) {
     ADDTOKENCHAR(SKIPCHAR());
     ADDTOKENCHAR(0); /* term */
     return 1;
   }
   if (single_quote_char(c)) {
     SKIPCHAR();
     while (get_single_quoted_item()) ;
     if (single_quote_char(LOOKAHEAD_CHAR)) {
       SKIPCHAR();
       ADDTOKENCHAR(0); /* term */
       return 1;
     } else {
       seterr(ERR_MISSING_QUOTE);
       return 0;
     } 
   }
   if (get_back_quoted_string()) seterr(ERR_BACK_QUOTED_STRING);
   return 0;
}

int token()
/*
 *   token (* 6.4 *)
 *     = name token (* 6.4.2 *)
 *     | variable token (* 6.4.3 *)
 *     | integer token (* 6.4.4 *)
 *     | float token (* 6.4.5 *)
 *     | char code list token (* 6.4.6 *)
 *     | open token (* 6.4.8 *)
 *     | close token (* 6.4.8 *)
 *     | open list token (* 6.4.8 *)
 *     | close list token (* 6.4.8 *)
 *     | open curly token (* 6.4.8 *)
 *     | close curly token (* 6.4.8 *)
 *     | head tail separator token (* 6.4.8 *)
 *     | comma token (* 6.4.8 *)
 *     | end token (* 6.4.8 *)

 *  layout text sequence (* 6.4.1 *)
 *     = layout text (* 6.4.1 *),
 *       { layout text (* 6.4.1 *) } ;

 *  layout text (* 6.4.1 *)
 *     = layout char (* 6.5.4 *)
 *     | comment (* 6.4.1 *)

 *   comment (* 6.4.1 *)
 *     = single line commment (* 6.4.1 *)
 *     | bracketed comment (* 6.4.1 *)
 */
{
  register unsigned char c;
  register int layout_inserted = 0;
  register int more_layout = 1;

  do {
    if (layout_char(LOOKAHEAD_CHAR)) {
      SKIPCHAR();
      layout_inserted = 1;
    } else if (end_line_comment_char(LOOKAHEAD_CHAR)) {
      single_line_comment();
      layout_inserted = 1;
    } else if (comment_1_char(LOOKAHEAD_CHAR)) {
      if (bracketed_comment()) 
        layout_inserted = 1;
      else
        more_layout = 0;
    } else 
      more_layout = 0;
  } while (more_layout);

  c = LOOKAHEAD_CHAR;
  if (capital_letter_char(c) || variable_indicator_char(c)) 
    return variable_token(); 
  if (comma_token(c)) {
    SKIPCHAR(); 
    return COMMA_TOKEN;
  }
  if (close_token(c)) {
    SKIPCHAR(); 
    return CLOSE_TOKEN;
  }
  if (open_token(c)) {
    SKIPCHAR(); 
    return layout_inserted ? OPEN_TOKEN : OPEN_CT_TOKEN;
  }
  if (end_token(c)) {
    SKIPCHAR(); 
    if (layout_char(LOOKAHEAD_CHAR) || end_line_comment_char(LOOKAHEAD_CHAR))
      return END_TOKEN;
    else 
      returnchar('.');
  }
  if (decimal_digit_char(c)) return number_token();
  if (close_list_token(c)) {
    SKIPCHAR(); 
    return CLOSE_LIST_TOKEN;
  }
  if (open_list_token(c)) {
    SKIPCHAR(); 
    return OPEN_LIST_TOKEN;
  }
  if (head_tail_separator_token(c)) {
    SKIPCHAR(); 
    return HEAD_TAIL_SEPARATOR_TOKEN;
  }
  if (open_curly_token(c)) {
    SKIPCHAR(); 
    return OPEN_CURLY_TOKEN;
  }
  if (close_curly_token(c)) {
    SKIPCHAR(); 
    return CLOSE_CURLY_TOKEN;
  }
  if (double_quote_char(LOOKAHEAD_CHAR)) {
    if (char_code_list_token()) 
       return CHAR_CODE_LIST_TOKEN;
  }
  if (get_name_token()) 
    return NAME_TOKEN;
  if (!LOOKAHEAD_CHAR) {
    SKIPCHAR();
    return EOFFILE;
  }
  return 0;
}

int char_conversion(c1,c2)
register unsigned char c1,c2;
{
  charconv[c1] = c2;
  return 1;
}
