
/*----------------------------------------------------------

     AUTO-C  Demonstration Compiler Generator System
     -----------------------------------------------

     Expert Systems Limited. (c) 1983.


     AUTO-C is a simple compiler system for simple small
     programming languages. The package consists of:-

     [1] A control harness.
     [2] A grammar rule translation scheme.
     [3] A simple lexical analyser.
     [4] A simple table-driven code generator.
     [5] Run-time support routines.

     Using the AUTO-C description language the user is able 
     to define his own languages; both at the syntactic level
     and at the code-generation level.

     The system is simple and yet provides insight into some
     of the tools and techniques involved in the PROLOG-
     approach to compiler-writing.

-----------------------------------------------------------*/


?-op(250,xfx, --} ).
?-op(105,xfx, &  ).
?-op(100,fx,  @  ).
?-op(100,xf,  :  ).
?-op(100,fx,  ?  ).
?-op(90, xfx, ~  ).
?-op(251,fx,{).
?-op(250,xf,}).
?-op(255,xfx,-->).

/*
.pa
A Brief Description Of The AUTO-C Specification Language
--------------------------------------------------------

[1] Syntax Specification - PROLOG Grammar Rules Notation
--------------------------------------------------------

The  standard  PROLOG grammar rule notation is a powerful methodology for  the 
specification of Definite Clause Grammars. The full description of the grammar 
rule notation,  its use and its translation into standard PROLOG,  is left  to 
the standard work: 'Programming in PROLOG - Clocksin & Mellish'.

Grammar  rules represent productions of a grammar and are read in the same way 
as BNF. For example:-

     a --> b,c,d.

is read as: "'a' may take the form of 'b' followed by 'c' followed by 'd'".

Terminal   symbols,   i.e.   those   appearing  in  the   source   program   ( 
keywords,brackets etc ) are denoted by symbols enclosed in square brackets.
Goals  to be executed directly without grammar rule annotation are enclosed in 
the special 'curly brackets'. Thus the following grammar rule:-

     a --> [if], b, [then], c, { g1 } .

may  be throught of as:  
'To  recognise  symbol  'a'  we must recognise terminal 'if' (  a  'token'  ), 
recognise  non-terminal 'b',  recognise terminal 'then' and finally  recognise 
non-terminal 'c'. We then execute the PROLOG goal 'g1'.

Arguments  may  be  given to grammar symbols and thus used  to  construct  and 
return a syntax tree during the parse. For example, we may write:-

     expression( expval(Value) ) --> term1( V1 ), ['+'], term2( V2 ),
                                     { Value is V1 + V2 } .

The reader is referred to the supplied grammar for the small language TINY  in 
order to gain familiarity with the notation.


.pa
[2] The Code Generation Template Specification Language
-------------------------------------------------------

The  AUTO-C demonstration system provides a simple notation for expressing the 
mapping of the syntax trees returned from the parsing process onto  equivalent 
sequences of instructions for the target machine language. Thus the process of 
code  generation  is  reduced to an automatic translation  scheme  based  upon 
pattern-matching with the defined 'code generation templates'.

For each syntax tree structure defined in the language grammar rules,  we must 
define  a  'template' which maps the tree onto the target machine  code.  each 
template definition has the form:

     Syntax Tree   --}   Machine Code.

Note that the operator '--}' is different to the grammar rule operator '-->'.
The  form  of  'Syntax Tree' must be identical to the form  expressed  in  the 
grammar rules.  The 'Machine Code' expression a set of target instructions  of 
the form :

     Instr1 & Instr2 & ... & InstrN .

Each target instruction may take one of the following forms:-

@Term     :    This represents the target code generated by the translation
               of the syntax tree 'Term'. In general, 'Term' will be an
               argument of the LHS Syntax Tree of the template.

LABEL(L)  :    L is a variable wich becomes instantiated to the number of
               the next free machine code label.

L:        :    L is a variable which has already appeared as argument to
               a 'LABEL' instruction. A label with number L will appear
               at this point in the output code.

?Goal     :    'Goal' is a normal PROLOG goal which is executed at this
               point in the translation. Goal will not normally generate
               any code but instantiates variables for later use.

Term      :    An un-adorned PROLOG term will be emitted as it appears
               in the template.

For example, consider the following template:-

     if_expr( Test, Truepart, Falsepart ) --}  @Test & 
                                               'JFALSE' & 'LABEL'(L1) &
                                               @Truepart &
                                               'JUMP' & 'LABEL'(L2) &
                                        L1: &  @Falsepart &
                                        L2:  .
.pa
This will be translated as:-
     "Generate the code for the tree 'Test'.
      Emit the instructions: 'JFALSE' L1 .
      Generate the code for the tree 'Truepart'.
      Emit the instructions: 'JUMP' L2.
      Declare the label L1 at this point.
      Generate the code for the tree 'Falsepart'.
      Declare the label L2 at this point. "

Again,  the  reader  is  referred to the sample templates given for  the  TINY 
language .




[3] Lexical Declarations - Keywords and Special Tokens
------------------------------------------------------

To a certain extent, the Lexical Analyser is table driven; with such tokens as 
keywords  and  tokens  consisting entirely of 'punctuation  characters'  being 
declared  in the language definition file.  The definition file for the  small 
language TINY contains PROLOG facts which define the TINY keywords and 'symbol 
tokens'.

The  user is invited to modify the lexical analyser to 'tokenise' a  different 
language.
.pa
*/
/*----------------------------------------------------------------*/
/*     [1] AUTO-C Control Harness Procedures                      */
/*----------------------------------------------------------------*/


auto_c:-
     auto_c_menu,
     repeat,
       perform_option,
     fail.

auto_c_menu:-
     writef(
       / & 'AUTO-C Compiler Demonstration Options' &
       / & '-------------------------------------' &
       / & 'NEW:     Create a new grammar for AUTO-C.' &
       / & 'OLD:     Load AUTO-C grammar from file.' &
       / & 'EDIT:    Enter a new AUTO-C program.' &
       / & 'PARSE:   Parse the AUTO-C program.' &
       / & 'CODE:    Generate code for program.' &
       / & 'HALT:    Exit to VMS.' & / & / ),
     !.

perform_option:-
     repeat,
     get_option( X ),
     valid_option(X),
     nl,
     !, 
     call(X), !.

get_option( X ):-
     writef( / & 
     'Enter AUTO-C Option or ''h'' for help ( terminate with ''.'' ): '),
     read(X), !.

valid_option( new ).
valid_option( old ).
valid_option( edit ).
valid_option( parse ).
valid_option( code ).
valid_option( halt ).
valid_option( h ):- auto_c_menu, !, fail.
valid_option( X ):-
     writef( X & ' is not a valid option !' & / ), !, fail.

/*
.pa
*/
/*--------------------------------------------------------------*/
/*    [2] AUTO-C Options Command Service Procedures             */
/*--------------------------------------------------------------*/
/*
NEW:

Prompts for the name of the file containing the PROLOG grammar
rules for AUTO-C and invokes the 'grammar rule translator' package
to convert the grammar rules into PROLOG clauses.
The translated grammar rules constitute a parser for the language
defined by the grammar rules.
The translated grammar is written out to file 'autoc.grm' for
use by any subsequent 'old' commands.
*/

new:-
     writef('Enter name of file containing grammar rules' &
            / & '( Terminate with a ''.'' ) : '),
     read( File ), 
     writef( / & 'Translating grammar from file: ' & File & / & /),
     translate_file( File ),
     writef( / & 'Done.' & / ).



/*
OLD:
The command simply re-consults the file 'autoc.grm' which should
contain a complete parser/code-generator from a previous call
to 'new'.
*/

old:- 
     reconsult( 'autoc.grm' ).


/*
EDIT:
The 'edit' command allows the user to type in a complete
source program for subsequent compilation by the 'compile'
command.
The program text is written to a file 'autoc.txt' .
*/

edit:-
     writef('Enter a source program and terminate with CTL/Z <CR>'& / & /),
     tell( 'autoc.txt' ),
     repeat,
         get0(C),
         (
            ( C = 26 ; C=end_of_file ),
            told, tell(user),
            writef( 'OK' & / )
         ;
            put(C),fail
         ), 
    !.
/*
.pa
*/

/*
PARSE:

The 'parse' command performs the first phase of compilation upon
the source program in file 'autoc.txt'.
The lexical analyser is used to return a list of tokens representing
the program source.
The parser created using 'translate' is accessed via the 'phrase'
predicate and used to parse the token list.
During the course of the syntax analysis, the run-time support
predicates invoked from the parser create a line numbered listing
on the terminal.
A syntax tree for the entire program is returned by the parser
provided that no syntax errors were detected. This tree is used
by the 'code' command in generating object code for the program.
NOTE: The error recovery mechanism has been left deliberately
crude - its development being left as an exercise for the user.
*/

parse:-
     initialisation( C0 ),
     lexical_analysis( C0, Token_list ),
     phrase(  program(Tree) , Token_list ),
     asserta( syntax_tree(Tree)),
     seen,see(user),
     writef( / & / & 'Parsing Successful - NO ERRORS.' & / & / ).
parse:-
     seen, see(user),
     write(' <HERE>'),nl,
     writef(/ & 'SYNTAX ERROR at point marked <HERE>' & / & /).


initialisation( C0 ):-
     retractall( labelroot(_) ),
     retractall( line_number(_)),
     retractall( syntax_tree(_)),     
     assert( labelroot(1)),
     assert( line_number(2)),
     see( 'autoc.txt' ),
     nl,
     write('1'),put(9),
     get0(C0),
     !.
     
/*
.pa
*/
/*
CODE:

The 'code' command takes the syntax tree generated by the
'compile' command and uses the code generation '--}' productions
to generate object code for the defined target machine.
For demonstration purposes, the code is written to the screen.
The syntax tree is mapped onto its corresponding code fragment
using the '--}' operator.
The code fragment is then 'emitted' using the recursive 'emit'
predicate.
*/

code:-
     nl,
     retract( syntax_tree(Tree)),
     Tree --} Code_fragment,
     emit( Code_fragment ),
     nl,
     trimcore.
code :-
     nl,
     write('Cannot generate code for an erroneous program.'),
     nl.

emit( T1 & T2 ) :-  
     emit(T1), emit(T2).
emit( @T ) :- 
     T --} F, emit(F).
emit( L: ):-
     label_check(L),
     encode( labeldef(L) ).
emit( L: ):-
     encode( labeldef(L) ).
emit( ?Goal ):-
     call( Goal ).
emit( Instruction ):-
     encode( Instruction ).

encode( 'LABEL'(L) ):-
     label_check(L),
     encode( 'LABEL'(L) ).
encode( labeldef(L) ):-
     writef( L & ':' & /).
encode( Instruction ):-
     writef( '    ' & Instruction & / ).

label_check(L):-
     var(L),
     retract( labelroot(L)),
     N is L+1,
     asserta( labelroot(N) ).
/*
.pa
*/
/*-------------------------------------------------------------*/
/*      [4] PROLOG - Grammar Rule Translation Scheme           */
/*-------------------------------------------------------------*/

/*
The following section consists of the standard PROLOG grammar rule
translation scheme as detailed in 'Clocksin & Mellish'.
The major predicates are :-

PHRASE( Tree,Sentence ):
Succeeds if a correct parsing of Sentence returns the syntax
tree Tree. Sentence is a list of tokens.

TRANSLATE_FILE( File ):
Reads in a file of grammar rules, translates each rule into a 
PROLOG clause and asserts the rules into the database.
The translated clauses are also written to file 'autoc.grm'.
*/

phrase( TopGoal,Tokens ) :-
	TopGoal =..[ Pred|Args ],
        append(Args,[Tokens,_],Newargs),
	Goal =.. [Pred|Newargs],
	call(Goal).


translate_file(File) :-
	see(File),
        repeat,
          read(G),
          tell( user ),
          write(G),nl,
          tell( 'autoc.grm' ),
          grammar_rule( G ),
        G=end_of_file,
        told,
        tell( user ),
        seen,
        see( user ).

grammar_rule( G ) :-
	translate( G, P ), 
        assert( P ),
        writeq( P ),
        writef( ' . ' & / ), !.


translate((P0-->Q0),(P:-Q)) :-
	'$l_h_s'(P0,S0,S,P),
	'$r_h_s'(Q0,S0,S,Q1),
	'$flatten'(Q1,Q).
translate(Ordinaryclause,Ordinaryclause).
/*
.pa
*/

'$l_h_s'( NT, S0, S, P ) :-
	not var(NT),
	'$tag'(NT,S0,S,P).

'$r_h_s'( (X1,X2), S0, S, P ) :- !,
	'$r_h_s'(X1,S0,S1,P1),
	'$r_h_s'(X2,S1,S,P2),
	'$and'(P1,P2,P).
'$r_h_s'( (X1;X2), S0, S, (P1;P2) ) :- !,
	'$or'(X1,S0,S,P1),
	'$or'(X2,S0,S,P2).
'$r_h_s'( {P}, S, S, P ) :- !.
'$r_h_s'( !, S, S, ! ) :- !.
'$r_h_s'( [T], S0,S1, '$terminal'(T,S0,S1) ):- !.
'$r_h_s'( [T1|T2], S0,S, ('$terminal'(T1,S0,S1),Ts) ):-
     '$r_h_s'( T2, S1,S,Ts ).
'$r_h_s'( X, S0, S, P ) :-
	'$tag'(X,S0,S,P).

'$or'( X, S0, S, P ) :-
	'$r_h_s'(X,S0a,S,Pa),
	( var(S0a), S0a == S, !, S0 = S0a, P = Pa;
	P = (S0 = Sa,Pa) ).

'$tag'( X, S0, S, P ) :-
	X =.. [F|A],
	append(A,[S0,S],AX),
	P =.. [F|AX].

'$and'( true, P, P ) :- !.
'$and'( P, true, P ) :- !.
'$and'( P, Q, (P,Q) ).

'$flatten'( A, A ) :- var(A), !.
'$flatten'( (A,B), C ) :- !,
	'$flatten1'(A,C,R),
	'$flatten'(B,R).
'$flatten'( A, A ).

'$flatten1'( A, (A,R), R ) :- var(A), !.
'$flatten1'( (A,B), C, R ) :- !,
	'$flatten1'(A,C,R1),
	'$flatten1'(B,R1,R).
'$flatten1'( A, (A,R), R ).

islist([]) :- !.
islist([_|_]).
/*
.pa
*/
/*----------------------------------------------------------------*/
/*     [5] Lexical Analysis Procedures ( tokenisation )           */
/*----------------------------------------------------------------*/

/*
LEXICAL_ANALYSIS( C0, Tokens ):

Succeeds if C0 is the next character from the input file and
Tokens is a list of generated tokens representing the rest
of the source file.
Lexical analysis is the process of grouping together related 
characters of the input stream to form special symbols, called
'tokens'. These tokens are then analysed by the parser, or syntax
analyser, to yield a syntax tree representing the grammatical
structure of the source program.
The following lexical analyser is a simple 'ad hoc' system which
groups characters together according to their 'character class'
( letter, digit, etc. ) and uses a 1-character look-ahead to
judge the end of a token. All whitespace characters, except 
carriage return, are discarded by the scanner. The <CR> is
returned as the special token 'nl', and is used to format the
listing output during the parsing process. The 'nl' token
is however syntactically transparent.

*/

lexical_analysis( end_of_file, [] ):- !.
lexical_analysis( C0, [T1|T2] ):-
     read_token( C0, T1/C1 ),
     lexical_analysis( C1, T2 ),
     !.



/*
READ_TOKEN( C0, T/C ).

Constructs the next token from the input character stream.
The classification is based upon the classification of the
first character C0 of the coming token ; 
C0   is the first character of the coming token.
T    is the completed token.
C    is the first character following token T.
*/

/*
.pa
*/

read_token( T , nl/C ):-                /* New Line */
     ( T = 10 ; T = 13 ),
     get0(C),
     !.
read_token( C0, T/C ):-                 /* Whitespace  */
     whitespace( C0 ),
     get0( C1 ),
     read_token( C1, T/C ),
     !.
read_token( C0, T/C ):-                 /* Identifiers */
     alphameric( C0, C1 ),
     get0( C2 ),
     read_identifier( C2, Letters, C ),
     name( Identifier, [C1|Letters] ),
     classify_identifier( Identifier, [C1|Letters], T ),
     !.
read_token( C0, const( Value )/C ):-    /* Constants  */
     is_numeric( C0 ),
     get0( C1 ),
     read_constant( C1, Digits, C ),
     evaluate_constant( [C0|Digits], Value, 0 ),
     !.
read_token( C0, T/C ):-                     /* Symbol Tokens */
     symbol_char( [C0] ),
     get0( C ),
     name( T, [C0] ),
     !.
read_token( end_of_file, keyword(stop)/end_of_file ):-!.      /* End of File */
read_token( C, illegal_char(T)/C1 ):-
     name( T,[C] ),
     get0( C1 ).



/*
READ_IDENTIFIER( C0, I, C1 ).

Reads in an alphanumeric identifier string as a list
of characters in I.
Arguments:
C0       is the last character read in.
I	 is the completed identifier as a character list.
C1       is the first character following the identifier.
*/

read_identifier( C0, [Ihead|Itail], C1 ):-
     alphanumeric( C0, Ihead ),
     get0( C2 ),
     read_identifier( C2, Itail, C1 ).
read_identifier( C0, [], C0 ).

/*
.pa
*/
/*
CLASSIFY_IDENTIFIER( Identifier, Idnamelist, Classified_Id ).

Given an identifier and a list of its characters, the 
procedure classifies the name as one of:-
[1] A reserved TINY keyword.
[2] A user-supplied identifier.
Arguments:-
Identifier:    Atom name of the identifier.
Idnamelist:    List of characters in the name.
Classified_Id: The classified form of Identifier.
*/


classify_identifier( Identifier,Idnamelist,keyword(Identifier) ):-
     keyword(Identifier).
classify_identifier( Identifier, Idnamelist, id(Identifier) ).





/*
READ_CONSTANT( C0, Digits, C1 ).

Given the first digit, the procedure reads in a decimal
integer constant and returns a list of its digits.
Arguments:-
C0     : The first digit ( ASCII 48-57 ) of the constant.
Digits : A list of the digits in the constant in ASCII.
C1     : The first character following the constant.
*/

read_constant( C0, [C0|Digits], C1 ):-
     is_numeric( C0 ),
     get0( C2 ),
     read_constant( C2, Digits, C1 ).
read_constant( C0, [], C0 ).



/*
.pa
*/
/*
EVALUATE_CONSTANT( Digits, Value, Accumulator ).

Given a list of the ASCII digit characters '0'..'9' 
which comprise a decimal constant, the procedure 
returns the decimal integer value of the constant.
Arguments:-
Digits      : Is a list of the ASCII digit codes.
Value       : Is the integer value of the constant.
Accumulator : Is the value of the digits preceeding 'Digits'.
*/

evaluate_constant( [], Value, Value ).
evaluate_constant( [D0|D1], Value, Accumulator ):-
     V is Accumulator*10 + D0-48,
     evaluate_constant( D1, Value, V ).






/* CHARACTER CLASSIFICATION RULES */

alphameric( Letter, Letter ):-
     Letter >= 97, Letter =< 122, !.
alphameric( Letter, LowerCase ):-
     Letter >= 65, Letter =< 90,
     LowerCase is Letter + 32, !.


is_numeric( Digit ):-
     Digit >= 48, Digit =< 57, !.

alphanumeric( A1, A2 ):-
     alphameric( A1, A2 ).
alphanumeric( A,A ):-
     is_numeric( A ).

whitespace( 32 ):- !.
whitespace( 10 ):- !.
whitespace( 13 ):- !.
whitespace( 9 ):- !.

/*
.pa
*/
/*----------------------------------------------------------------*/
/*     [6] Run-Time Compiler Support Predicates                   */
/*----------------------------------------------------------------*/

/*
$TERMINAL( T, S1,S2 ):

Succeeds if terminal token T is at the head of the current
input sentence S1.
T is removed from the head of S1 and traced to the listing output
stream, leaving the output sentence S2.
If the next token is the special 'new line' token 'nl', then
it is simply traced and removed. The next token is then checked
for equality with T.
Keyword tokens are treated differently in that the first argument
of keyword(X) is unified with T.
*/



'$terminal'( T, [nl,keyword(T)|Ts], Ts ):-
     '$trace'( nl ),
     '$trace'( T ), 
     !.
'$terminal'( T, [nl,T|Ts], Ts ):-
     '$trace'( nl ),
     '$trace'( T ),
     !.
'$terminal'( T, [keyword(T)|Ts], Ts ):-
     '$trace'( T ),
     !.
'$terminal'( T, [T|Ts], Ts ):-
     '$trace'( T ),
     !.


/*
.pa
*/
/*
$TRACE( T ):

Traces the token T to the current output stream.
If T is 'nl' then a new-line is thrown and the line number is
incremented.
If T is a compound term of the form 'f(X)', then the first
argument X is traced.
*/

'$trace'( nl ):-
     retract( line_number(LN) ),
     nl,
     write(LN),
     put(9),
     LN2 is LN+1,
     asserta( line_number(LN2) ).
'$trace'( T ):-
     atomic(T),
     write(T),
     put(32).
'$trace'( T ):-
     arg( 1,T,A ),
     write(A),
     put(32).



/* UTILITIES */

append( [],X,X ).
append( [X|Y], L, [X|Z] ):-  append( Y,L,Z ).


writef( T1 & T2 ):-
     writef( T1 ),
     writef( T2 ).
writef( / ):-
     nl.
writef( @T ):-
     call(T).
writef( T ):-
     write(T).


?-seeing(F),see(user),auto_c,see(F).


