/**************************************************************/
/*                                                            */
/*  E. Y. Shapiro's Algorithmic Program Debugger              */
/*                                                            */
/*  Algorithmic Program Debugging, MIT Press, 1982            */
/*  Appendix II, pp. 185-213                                  */
/*                                                            */
/**************************************************************/

     
%%% PDSDC

%% The Diagnosis Components

     
% A Depth-Bounded Interpreter
     
solve(P,X) :-
      solve(P,25,X), (X \== true, !; true).
     
solve(true,D,true) :- !.
solve(A,0,(overflow,[])) :- !.
solve((A,B),D,S) :- !,
      solve(A,D,Sa),
      (Sa = true -> solve(B,D,Sb), S = Sb; S = Sa).
solve(A,D,Sa) :-
      system(A) -> A, Sa = true;
      D1 is D - 1,
      clause(A,B), solve(B,D1,Sb),
      (Sb = true -> Sa = true;
       Sb = (overflow,S) -> Sa = (overflow,[(A :- B)|S])).

     
% Tracing an Incorrect Procedure by "Divide-and-Query"
     
false_solution(A) :-
      writel(['Error: wrong solution ',A,'.diagnosing...']), nl,
      fpm((A,W),_,0), % just to find W, the length of the computation
      fp(A,W,X) -> handle_error('false clause',X);
      write('!Illegal call to fp'), nl.
     
fp(A,Wa,X) :-
     fpm((A,Wa),((P :- Q),Wm),Wa),
     (Wa = 1 -> X = (P :- Q);
      query(forall,P,true) -> Wa1 is Wa - Wm, fp(A,Wa1,X);
      fp(P,Wm,X)).
 
    
% An Interpreter that Computes the Middle Point of a Computation
     
% fpm((A,Wa),(M,Wm),W) :- solve A whose weight is Wa. Find
% a goal M in the computation whose weight, Wm, is less than 
% W/2, and is the heaviest son of a node whose weight exceeds 
% (W+1)/2

fpm(((A,B),Wab),M,W) :- !,
      fpm((A,Wa),(Ma,Wma),W), fpm((B,Wb),(Mb,Wmb),W),
      Wab is Wa + Wb,
      (Wma >= Wmb -> M = (Ma,Wma); M = (Mb,Wmb)).
fpm((A,0),(true,0),W) :-
      system(A), !, A;
      fact(A,true).
fpm((A,Wa),M,W) :-
      clause(A,B), fpm((B,Wb),Mb,W),
      Wa is Wb + 1,
      (Wa > (W + 1)/2 -> M = Mb; M = ((A :- B),Wa)).
     

% Tracing an Incomplete Procedure (improved)

missing_solution(A) :-
      writel(['Error. missing solution ',A,'.diagnosing...']),                  nl, query(e
xists,A,true),
      \+solve(A,true) -> ip(A,X), 
                    handle_error('uncovered atom',X);
      write('!Illegal call to ip'), nl.
     
ip(A,X) :-
      clause(A,B), ip1(B,X) -> true; X = A.
ip1((A,B),X) :- !,
      (query(exists,A,true), (A, ip1(B,X); \+A, ip(A,X))).
      % cannot use -> because need to check all solutions
      % in case of a non-deterministic procedure
ip1(A,X) :-
      query(exists,A,true), (A -> break(ip1(A,X)); ip(A,X)).

     
% Tracing a Stack Overflow

stack_overflow(P,S) :-
      writel(['Error: stack overflow on ',P,'.diagnosing...']), nl,
      (find_loop(S,S1) -> check_segment(S1);
       check_segment(S)).
     
find_loop([(P :- Q)|S],Sloop) :-
      looping_segment((P :- Q),S,S1) -> Sloop = [(P :- Q)|S1];
      find_loop(S,Sloop).
     
looping_segment((P :- Q),[(P1 :- Q1)|S],[(P1 :- Q1)|Sl]) :-
      same_goal(P,P1) -> writel([P,' is looping.']), nl, 
                    Sl = [];
      looping_segment((P :- Q),S,Sl).
     
check_segment([(P :- Q),(P1 :- Q1)|S]) :-
      query(legal_call,(P,P1),true) ->
      check_segment([(P1 :- Q1)|S]);
      false_subgoal(P,Q,P1,C) -> false_solution(C);
      handle_error('diverging clause',(P :- Q)).
     
false_subgoal(P,(Q1,Q2),P1,Q) :-
      % search for a subgoal Q of P to the left of P1 that returned 
              % a false solution
      Q1 \== P1,
      (query(forall,Q1,false) -> Q = Q1;
      false_subgoal(P,Q2,P1,Q)).
   
  
% An Interpreter that Monitors Errors

msolve(P,X) :-
      msolve(P,25,X), (X \== true, !; true).
     
msolve(A,0,(overflow,[])) :- ! .
msolve((A,B),D,S) :- !,
      msolve(A,D,Sa),
      (Sa = true -> msolve(B,D,Sb), S = Sb; S= Sa).
msolve(A,D,Sa) :-
      system(A) -> A, Sa = true;
      D1 is D - 1,
      setof0((A,B,Sb),(clause(A,B),msolve(B,D1,Sb)),R),
      result(R,A,Sa).
     
result(R,A,(overflow,[(A :- B)|St])) :-
      member((A,B,(overflow,St)),R), !.
result(R,A,false) :-
      member((A,_,false),R), !;
      member((A,B,true),R), fact(A,false), !,
      false_solution(A);
      fact(A,true), \+(member((A,_,true),R)), !,
      missing_solution(A).
result([],A,false) :-
      attribute(A,total), !,
      writel(['Error trapped: no solution to ',A]), nl,
      missing_solution(A).
result([A1,A2|R],A,false) :-
      attribute(A,determinate), !,
      writel(['Error trapped: too many solutions to ',A]), nl,
      member((A,_,_), [A1,A2|R]), query(forall,A,false), !,
      false_solution(A).
result(R,A,true) :-
      member((A,_,true),R).
   

   
%%% PDS5.

%% The Diagnosis System
   
:- initialized -> true;
            % these files are required to run pds
            ['xref.def'], [pdsdc,pdsdb,dsutil], [pdsini],
                assert(initialized).
     
pds :-
      nl, read('@',P), (P = exit; solve_and_check(P), pds).
     
solve_and_check(P) :-
      bagof0((P,X),solve(P,X),S), check_solutions(P,S).
     
check_solutions(P,S) :-
      member((P1,(overflow,X)),S) -> stack_overflow(P1,X);
      member((P1,true),S), fact(P1,false) -> false_solution(P1);
      fact(P,true), \+member((P,true),S) -> missing_solution(P);
      confirm_solutions(P,S).
     
confirm_solutions(P,[(P1,X)|S]) :-
      writel(['solution: ',P1,';']),
      ((system(P1); fact(P1,true)) -> nl,
      confirm_solutions(P,S);
      confirm(' ok') -> assert_fact(P1,true),
      confirm_solutions(P,S);
      assert_fact(P1,false), false_solution(P1)).
confirm_solutions(P,[]) :-
      write('no (more) solutions.'),
      (system(P) -> nl;
      confirm(' ok') -> true;
      ask_for_solution(P), assert_fact(P,true),
      missing_solution(P)).
     
handle_error('false clause',X) :- !,
      writel(['error diagnosed: ',X,' is false.']),
      nl,plisting(X).
handle_error('uncovered atom',X) :- !,
      writel(['error diagnosed: ',X,' is uncovered.']), nl,
      plisting(X).
handle_error('diverging clause',X) :- !,
      writel(['error diagnosed: ',X,' is diverging.']), nl,
      plisting(X).
 

 
%%% MIS
    
%% The Model Inference System
    
:- initialized -> true;
      ['xref.def'], [misrg,dsutil,pdsref,type],
      [pdsdc,pdsdb,missrc,pdsini],
      assert(initialized).
     
mis :- nl, ask_for('Next fact',Fact),
      (Fact = check -> check_fact(_);
       Fact = (P,V), (V = true; V = false) ->
       assert_fact(P,V), check_fact(P);
       write('!Illegal input'), nl),
       !, mis.
     
check_fact(P) :-
      write('Checking fact(s)...'), ttyflush,
      (fact(P,true), \+solve(P) ->
       nl, missing_solution(P), check_fact(_);
       fact(P,false), solve(P) ->
       nl, false_solution(P), check_fact(_);
       write('no error found.'), nl).
     
solve(P) :-
      solve(P,X),
      (X = (overflow,S) -> nl, stack_overflow(P,S), solve(P);
       true).
     
handle_error('false clause',X) :-
      writel(['Error diagnosed: ',X,' is false.']), nl,
      retract(X), plisting(X).
handle_error('uncovered atom',X) :-
      writel(['Error diagnosed: ',X,' is uncovered.']), nl,
      search_for_cover(X,C),
      assert(C), plisting(X).
handle_error('diverging clause',X) :-
      writel(['Error diagnosed: ',X,' is diverging.']), nl,
      retract(X), plisting(X).
     
:- assert(value(search_strategy,adaptive)).

 
    
%%% PDSREF

%% A General Refinement Operator for pds
     
refinement(((P :- Q),(Vi,Vo,Vf)),((P :- Q),(Vi,[],[]))) :-
      % Close a Clause
      Vo \== [],
      unisubset(Vo, Vi),
      eqdifset(Vf, Vo,[]),
      noduplicate_atom(P,Q).
     
refinement(((P :- true),(Vi, Vo, [])), ((P :- true),(Vi2, Vo, []))) :-
      % instantiate head and inputs
      dmember(Var, Vi, Vi1),
      term_to_vars(Var, NewVars),
      append(Vi1, NewVars, Vi2).
     
     
refinement(((P :- true),(Vi, Vo, [])), ((P :- true),(Vi, Vo2, []))) :-
      % instantiate head and outputs
      dmember(Var, Vo, Vo1),
      term_to_vars(Var, NewVars),
      append(Vo1, NewVars, Vo2).
     
refinement(((P :- true),(Vi, Vo, [])), ((P :- true),(Vi1, Vo, []))) :-
      % unify two input vars
      dmember(Var1, Vi, Vi1),
      member(Var2, Vi1),
      Var1 @< Var2, % not to create duplicates
      Var1 = Var2.
     
refinement(((P :- Q1),(Vi,Vo,Vf)),((P :- Q2),(Vi1,Vo,Vf1))) :-
      % add output producing goal
      Vo \== [],
      body_goal(P, Q, QVi, QVo),
      QVo \== [],
      unisubset(QVi, Vi),
      noduplicate_atom(Q, (P, Q1)),
      free_vars(Vf, QVi, QVo, Qf1),
      append(Vi, QVo, Vi1),
      qconc(Q, Q1, Q2).
     
refinement(((P :- Q1), (Vi,[],[])), ((P :- Q2),(Vi,[],[]))) :-
      % add test predicate
      body_goal(P, Q, QVi, []),
      unisubset(QVi, Vi),
      noduplicate_atom(Q, (P,Q1)),
      qconc(Q, Q1, Q2).
     
body_goal(P, Q, QVi, QVo) :-
      called(P,Q),
      input_vars(Q,QVi), output_vars(Q, QVo).
     
qconc(A, true, A) :- !.
qconc(A, (B,X), (B,Y)) :- !, qconc(A, X, Y).
qconc(A, B, (B,A)).
     
% unisubset(V1, V2) :- V1 is a subset of V2
unisubset([], _) :- !.
unisubset([X|V1], V2) :-
      dmember(X, V2, V3), unisubset(V1, V3).
     
% dmember(X, L1, L2) :- the difference between lists L1 and L2 is X
dmember(X, [X|L], L).
dmember(X, [Y|L1], [Y|L2]) :-
      dmember(X, L1, L2).
     
% check no goals with duplicate inputs
noduplicate_atom(P1, (P2,Q)) :- !,
      (same_goal(P1, P2), !, fail; noduplicate_atom(P1, Q)).
noduplicate_atom(P1, P2) :-
      same_goal(P1, P2), !, fail; true.
     
% eqdifset(V1, V2, V3) :- variable set V1 - V2 is V3
eqdifset(V, [], V) :- !.
eqdifset(V1, [X|V2], V3) :-
      eqdelmember(X, V1, V4), !, eqdifset(V4, V2, V3);
      writel(['type conflict in ',eqdifset(V1,[X|V2],V3)]),
      break.
     
% eqdelmember(X1, L1, L2) :- the difference between lists L1 and L2 is X
eqdelmember(X1, [], []) :- !.
eqdelmember(X1, [X2|L], L) :- X1 == X2, !.
eqdelmember(X, [Y|L1],[Y|L2]) :-
      eqdelmember(X, L1, L2).
     
% Create the input set Xi and output set Xo and free set Xf of  % variables of 
a clause P :- Q. Does also typechecking
     
create_io((P :- Q), (Xi,Xo,Xf)) :-
      atom_vartype(P, Vi, Vo),
      create_io1(Vi, Xi, Vo, Xo, [], Xf, Q).
     
% Create_io1(Vi, Xi, Vo, Xo, [], Xf, Q) :- if Vi, Vo, Vf are  
% given input variable set, output variable set and free 
% variable set, then together with Q, Xi, Xo and Xf are the
% input, output and free variable sets

 create_io1(Xi, Xi, Xo, Yo, Xf, Yf, true) :- !,
      eqdifset(Xo, Xi, Yo),
      eqdifset(Xo, Yo, Xf1),
      eqdifset(Xf, Xf1, Yf).
 create_io1(Xi, Yi, Xo, Yo, Xf, Yf, (P,Q)) :- !,
      atom_vartype(P, Vi, Vo),
      eqdifset(Vi, Xi, Vdif),
      (Vdif = [], !,
       append(Xi, Vo, Xi1),
       free_vars(Xf, Vi, Vo, Xf1),
       !, create_io1(Xi1, Yi, Xo, Yo, Xf1, Yf, Q);
       writel(['uninstantiated input variables ',Vdif,' in
       atom',P]),fail).
 create_io1(Xi, Yi, Xo, Yo, Xf, Yf, P) :-
      create_io1(Xi, Yi, Xo, Yo, Xf, Yf, (P,true)).
     
% free_vars(Vf, Vi, Vo, Vf1) :- remove from  Vf Vi, and add Vo,
% getting Vf1
free_vars(Vf, Vi, Vo, Vf2) :-
      eqdifset(Vf, Vi, Vf1),
      append(Vf1, Vo, Vf2).
     
 

%%% DCGREF
     
%% A Refinement Operator for Definite Clause Grammars
     
refinement(((P :- Q1),(Vi,Vi,Vo)),((P :- Q2),(Vi1,Vi1,Vo))) :-
      % add goal
      nonterminal(Q),
      Q =.. [F,Qi,Qo],
      Vi = [Qi], Vi1 = [Qo],
      \+((P =.. [F,Pi,_], Pi == Qi)),
      qconc(Q,Q1,Q2).
     
refinement(((P :- Q),(Vi,Vi,Vo)),((P :- Q),([],[],[]))) :-
      % Close a  clause
      Vi \== [], Vi = Vo.
     
refinement(((P :- Q),(Vi,Vi,Vo)),((P :- Q),(Vi1,Vi1,Vo))) :-
      % Instantiate
      Vi = [[X|Xs]],
      terminal(X),
      Vi1 = [Xs].
     
qconc(A,true,A) :- !.
qconc(A,(B,X),(B,Y)) :- !, qconc(A,X,Y).
qconc(A,B,(B,A)).
     
% Create the input set Xi, the output set Xo and the free set Xf
% of variables of a clause P :- Q. Does also typechecking
create_io((P :- true),([Xi],[Xi],[Xo])) :-
      P =..[_,Xi,Xo].
     
ntlisting :-
      nonterminal(X), \+system(X), plisting(X), fail; true.
clearnt :-
      nonterminal(X), \+system(X), X =..[F|_], abolish(F,2), fail; true.
   
 
 
%%% MISSRC

%% An Implementation of Search Strategies 
     
covers(C, P) :-
      (value(search_strategy,S), member(S,[eager,lazy,adaptive])         -> tru
e;
       break('Incorrect or missing search strategy')),
       covers(S,C,P).
     
% The eager covers test
covers(eager,((P :- Q), (Vi,Vf,Vo)), P1) :-
      (Q = true; Vo = []) -> verify((P = P1, satisfiable(Q)));
       verify(P = P1).
     
% The lazy covers test
covers(lazy,((P :- Q), _), P1) :-
      verify((P = P1, fact_satisfiable(Q))).
     
% The adaptive covers test
covers(adaptive, ((P :- Q), _), P1) :-
      verify((P = P1, fact_solve(Q))).
     
fact_satisfiable((P, Q)) :- !,
      fact_satisfiable(P), fact_satisfiable(Q).
fact_satisfiable(P) :-
      system(P) -> P; fact(P, true).
     
fact_solve(P) :-
      fact_solve(P, 25, X),
      (X = (overflow, S) -> stack_overflow(P,S), fact_solve(P);
      true).
     
fact_solve(A, 0, (overflow,[])) :- !.
fact_solve((A,B), D, S) :-
      fact_solve(A, D, Sa),
      (Sa = true -> fact_solve(B, D, Sb), S = Sb; S = Sa).
fact_solve(A, D, Sa) :-
      system(A) -> A, Sa = true;
      fact(A, true) -> Sa = true;
      D1 is D - 1,
      clause(A, B), fact_solve(B, D1, Sb),
      (Sb = true -> Sa = true;
        Sb = (overflow,S) -> Sa=(overflow,[(A :- B)|S])).
  
 
     
%%% MISRG

%% A Pruning Breadth-First Search of the Refinement Graph
     
search_for_cover(P,Clause) :-
      nl, writel(['Searching for a cover to ',P,'...']), nl,
      mgt(P,P1), create_io((P1 :- true), Vs),
      search_for_cover([((P1 :- true), Vs)|Xs], Xs, P,
      Clause,1).
     
% search_for_cover(head, Tail, Goal, Clause, Length) :-
% The list between Head and Tail is the current queue of
% clauses. Search it for a true clause that covers Goal.
% Whenever you take a clause from the Head of the queue, add
% all its refinements that cover Goal to Tail, setting it to
% the new Tail of the queue. Length is the number of clauses
% searched so far
     
search_for_cover(Qhead, Qtail, P, C, Qlength) :-
      Qhead == Qtail,
      writel(['Failed to find a cover for ',P,'. queue is
      empty']), nl,!, fail.
search_for_cover([X|Qhead], Qtail, P, Clause, Qlength) :-
      X = (Xclause, _),  writel(['Refining: ',Xclause]), nl,
      bagof0(Y, X^(refinement(X,Y), covers(Y,P)), Qnew),
      length(Qnew,Qnewlength),
      Qlength1 is Qlength + Qnewlength,
      writel(['New refinements:'|Qnew], v, nl), nl,
      check_refinements(Qnew, Qhead, Qtail, P, Clause,Qlength1).
     
check_refinements(Qnew, Qhead, Qtail, P, Clause, Qlength) :-
      member((Clause, Cv), Qnew), good_clause((Clause, Cv),
                    Qlength).
check_refinements(Qnew, Qhead, Qtail, P, Clause, Qlength) :-
      append(Qnew, Qnewtail, Qtail),
      search_for_cover(Qhead, Qnewtail, P, Clause, Qlength).
     
good_clause((X, (Xi,[],[])), L) :-
      writel(['Checking: ',X]), nl,
      (refuted(X), !, writel(['Refuted: ',X]), nl, fail;
        looping(X), !, writel(['Looping: ',X]), nl, fail;
        writel(['Found clause: ',X]), nl,
        writel(['   after searching ',L,' clauses.']), nl).
     
looping((P :- Q)) :-
      \+legal_calls(P,Q).
     
refuted((P :- Q)) :-
      fact(P, false), fact_satisfiable(Q).

     
     
%%% PDS6.

%% The Interactive Debugging System     

:- initialized -> true;
   % files required to run the interactive debugger
   ['xref.def'], [dsutil,type,misrg,pdsref],
   [pdsdc,pdsdb,pdsini],
   [missrc,pdsrg,pdsref],
   assert(initialized).
     
pds :-
      nl, read('@',P), (P = exit; solve_and_check(P), pds).
     
solve_and_check(P) :-
      writel(['Solving ',P,'...']), nl,
      bagof0((P,X),msolve(P,X),S), confirm_solutions(P,S).
     
confirm_solutions(P,[(P1,(overflow,S))]) :- !,
      stack_overflow(P1,S), solve_and_check(P).
confirm_solutions(P,[(P1,false)]) :- !,
      solve_and_check(P).
confirm_solutions(P,[(P1,X)|S]) :-
      writel(['solution: ',P1,';']),
      ((system(P1); fact(P1,true)) -> nl, confirm_solutions(P,S);
      confirm(' ok') -> assert_fact(P1,true), confirm_solutions(P,S);
      assert_fact(P1,false),
      false_solution(P1), solve_and_check(P)).
confirm_solutions(P,[]) :-
      write('no (more) solutions.'),
      (system(P) -> nl;
       confirm(' ok') -> true;
       missing_solution(P), solve_and_check(P)).
     
handle_error('false clause',X) :- !,
      writel(['Error diagnosed: ',X,' is false.']), nl,
      X = (P :- Q),
      ask_then_do(['retract(y), (m)odify, or (r)eplace it'],
      [(false,true), (true,retract(X)),(r,(ask_for(['withwhat'],
                    C), retract(X), assert(C))),
      (m,(mgt(P,P1), clause(P1,Q1,_), 
                    verify(((P :- Q) = (P1 :- Q1))),
      % can't use Ref because of a Prolog bug.
      modify((P1 :- Q1),Y), retract(X), assert(Y)))]),
      plisting(P), !.
handle_error('uncovered atom',P) :- !,
      writel(['Error diagnosed: ',P,' is uncovered.']), nl,
      ask_then_do(
      ['add(y) or(m)odify a clause'],
      [(false,true),
      (true, (ask_for('which',C), assert(C))),
      (m, (ask_for('which',C1),
      (C1 = (_ :- _), !, retract(C1), C = C1;
       C1 = any, !, mgt(P,P1), C = (P1 :- true);
       C = (C1 :- true), retract(C1)),
       modify(C,P,Y), assert(Y)))]),
       plisting(P), !.
     
     
handle_error('diverging clause',(P :-Q)) :- !,
      writel(['Error diagnosed: ',(P :- Q),' is diverging.']),
      nl, X = (P :- Q),
      ask_then_do(
      ['retract(y), (m)odify, or (r)eplace it'],
      [(false,true),
      (true,retract(X)),
      (r,(ask_for(['with what'],C),
       retract(X), assert(C))),
       (m,(mgt(P,P1), clause(P1,Q1,_), 
       verify(((P :- Q) = (P1 :- Q1))),
       % can't use Ref because of a Prolog bug.
       modify((P1 :- Q1),Y), retract(X), assert(Y)))]),
      plisting(P), !.
     
modify(X,Y) :-
      reason(P,X), modify(X,P,Y).
modify(X,P,Y) :-
      search_rg(X,P,Y), confirm(ok), !; break(modify(X,P,Y)).
     
reason(P,X) :-
      reason1(P,X) -> true;
      ask_for(['What is a reason for ',X],P) ->
      assert(reason1(X,P)).
     
:- assert(value(search_strategy,eager)).
     


%%% PDSRG

%% A Bug-Correction Algorithm
     
% Search for Y that covers P, starting from X
search_rg(X,P,Y) :-
      create_io(X,Vx), !,
      search_rg1((X,Vx),P,Y).
     
search_rg1(X,P,Y) :-
      covers(X,P), X = (Xc,_), \+looping(Xc) ->
      check_refinements([X],Xs,Xs,P,Y,1);
      derefine(X,X1,P), search_rg1(X1,P,Y).
     
% derefine(X,Y) :- Y is the result of derefining X which means,
% in the meantime, omitting the last condition from X
derefine((X,Vx),Y,_) :-
      writel(['Derefining ',X,'...']), nl, derefine1((X,Vx),Y).
derefine1(((X :- Xs),Vx),Y) :-
      deconc(Xs,Ys), create_io((X :- Ys),Vy), new(((X :- Ys),Vy),Y).
derefine1(((X :- true),Vx),((Y :- true), Vy)) :-
      mgt(X,Y), \+variants(X,Y), create_io((Y :- true),Vy).
     
% Delete the last conjunct
deconc((X1,(X2,Xs)),(X1,Ys)) :- !, deconc((X2,Xs),Ys).
deconc((X1,X2),X1) :- !.
deconc(X,true) :- X \== true.

     
     
%%% PDSDB

%% Data Base for PDS
     
% The base relation is solutions(P,S) which denotes that
% the solutions of goal P are exactly S. This relation stores
% results of existential queries.
% On top of it, we compute the relation fact(P,V) which says  that
% P is known to have the truth value V
     
fact(P,V) :-
      var(P) -> (solutions(_,S), member(P,S), V = true;
                  solutions(P,[]),V = false);
      solutions(P,S), (member(P,S), V = true; 
      \+member(P,S), V = false).
     
listfact(P) :-
      fact(P,V), write(fact(P,V)), nl, fail; true.
     
is_instance(P1,P2) :-
      % P1 is an instance of P2
      verify((numbervars(P1,0,_),P1 = P2)).
     
assert_fact(P,V) :-
      fact(P,V1) -> (V = V1, !, true; break(assert_fact(P,V)));
      \+ground(P) -> break(assert_fact(P,V));
      % writel(['Asserting: ',fact(P,V)]), nl,
      (V = true -> assert(solutions(P,[P]));
        V = false -> assert(solutions(P,[]));
        break(assert_fact(P,V))).
     
     
query(exists,P,V) :-
      system(P) -> (P -> V = true; V = false);
      mgt(P,P1), solutions(P1,S), is_instance(P,P1) ->
        (member(P,S), V = true; \+member(P,S), V = false);
        fact(P,true), V = true;
      ask_for_solutions(P,S) ->
        (S = [] -> V = false; member(P,S), V = true).
     
query(forall,P,V) :-
      ground(P) -> query(exists,P,V);
      break(query(forall,P,V)).
     
query(solvable,P,V) :-
      system(P) -> (P -> V = true; V = false);
      fact(P,V1) -> V = V1;
      ask_for(['Query: ',P],V1,
      (V1 = true; V1 = false)) -> V = V1.
     
ask_for_solutions(P,S) :-
      bagof0(P,ask_for_solution(P),S),
      % writel(['Asserting: ',solutions(P,S)]), nl,
      assert(solutions(P,S)).
     
ask_for_solution(P) :-
      nl, ask_for(['Query: ',P],V,(V = true; V = false)),
      (V = false -> fail;
       ground(P) -> true;
       varand(P,Pvars),
       repeat,
         writel(['Which ',Pvars,'? ']), ttyflush,
         reade(Answer),
         (Answer = false, !, fail;
           Answer = Pvars -> true;
           write('does not unify; try again'), nl),
         (attribute(P,determinate), !; true)).
     
query(legal_call,(P1,P2),V) :-
      same_goal(P1,P2), !, V = false;
      legal_call((Q1,Q2),V1), same_goal(P1,Q1), same_goal(P2,Q2), !,
             V = V1;
      confirm(['Is ',(P1,P2),'a legal call']), !,
             assert(legal_call((P1,P2),true)), V = true;
      assert(legal_call((P1,P2),false)), V = false .
     
known_illegal_call(P1,P2) :-
      same_goal(P1,P2), !, V = false;
      legal_call((Q1JL,Q2JL),false), mgt(Q1JL,Q1), mgt(Q2JL,Q2),
       same_goal(P1,Q1), same_goal(P2,Q2).
     
same_goal(P,Q) :-
      functor(P,F,N), functor(Q,F,N),
      input_vars(P,Pi), input_vars(Q,Qi), !, variants(Pi,Qi).
     
satisfiable((P,Q)) :- !,
      query(exists,P,true), satisfiable(Q).
satisfiable(P) :-
      query(exists,P,true).
     
legal_calls(P,true) :- !.
legal_calls(P,Q) :-
      (Q = (Q1,Q2), !, true; Q = Q1, Q2 = true),
      (known_illegal_call(P,Q1), !, fail; true),
      (fact(Q1,true), !, legal_calls(P,Q2); true).
      % for all true solutions to Q1, Q2 shouldn't loop
     
clear :-
      abolish(solutions,2),
      abolish(legal_call,2).
     
clear(P) :-
      (retract(solutions(P,_)), fail; true).
     
edit_facts :-
      solutions(P,S),
      confirm(['Retract ',solutions(P,S)]),
      retract(solutions(P,S)),
      fail;
      true.
     
/* Information about a procedure:
     
   :- declare(P,A), where P is, for example, qs(+[x],-[x]), 
                    and A is, for example,  [determinate,total]
     
This will create the resulting data:
   declared(P,InV,OutV,A),where Inv(OutV) are pairs of input (output)
   variables and their types, and A is the list of attributes
     
*/
     
declare(Pmode,Ps) :-
      mgt(Pmode,P),
      P =..[F|Pargs],
      Pmode =..[F|Fargs],
      varplusminus(Pargs,Fargs,InV,OutV),
      (retract(declared1(P,_,_,_)), fail; true),
      %  writel(['Declaring ',(P,InV,OutV,Ps)]), nl,
      assert(declared1(P,InV,OutV,Ps)).
     
varplusminus([V|Pargs],[+(T)|Fargs],[(V,T)|PlusV],MinusV) :- !,
      varplusminus(Pargs,Fargs,PlusV,MinusV).
varplusminus([V|Pargs],[-(T)|Fargs],PlusV,[(V,T)|MinusV]) :- !,
      varplusminus(Pargs,Fargs,PlusV,MinusV).
varplusminus([],[],[],[]) :- !.
varplusminus(Pargs,Fargs,PlusV,MinusV) :-
      break(varplusminus(Pargs,Fargs,PlusV,MinusV)).
     
declared(P,Pi,Po,[]) :-
      nonterminal(P), P =..[_,Pi,Po].
     
declared(P,Pi,Po,Pa) :-
      declared1(P,Pi1,Po1,Pa1), !, Pi1 = Pi, Po1 = Po, Pa1 = Pa;
      ask_for(['Declare ',P],declare(Pv,Pa)), declare(Pv,Pa),
      declared(P,Pi,Po,Pa).
     
attribute(X,Xa) :-
      declared(X,_,_,Xas), !, member(Xa,Xas).
     
input_vars(P,InV) :-
      declared(P,InV,_,_).
     
output_vars(P,OutV) :-
      declared(P,_,OutV,_).
     
atominfo(P,_,_,_) :- break(atominfo(P,_,_,_)).
     
declare_called(P,Ps) :-
      (retract(called1(P,_)), fail; true),
      assert(called1(P,Ps)).
     
called(P,Q) :-
      system(P), !, fail;
      called1(P,Qs), !, member(Q,Qs);
      ask_for(['Procedures called by ',P],Ps),
      and_to_list(Ps,Ps1),
      declare_called(P,Ps1).


     
%%% DSUTIL

%% Utilities Used in the Debugging System
     
member(X,[X|_]).
member(X,[_|L]) :- member(X,L).
     
append([],L,L).
append([X|L1],L2,[X|L3]) :- append(L1,L2,L3).
     
reverse(X,Y) :- rev(X,[],Y).
rev([X|Xs],Ys,Zs) :- rev(Xs,[X|Ys],Zs).
rev([],Xs,Xs).
     
set(P,V) :-
      retract(value(P,_)), !, set(P,V);
      assert(value(P,V)).
     
add1(P,V1) :-
      retract(value(P,V)), integer(V), !,
         V1 is V + 1, assert(value(P,V1));
      writel(['no value for ',P,', initializing it to 1']), nl,
         set(P,0), V1 = 1.
     
ask_for(Request,Answer,Test) :-
      repeat,
      ask_for(Request,Answer), Test, !.
     
ask_for(Request,Answer) :-
      repeat,
         writel(Request), write('? '), ttyflush,
         reade(X),
         (directive(X), !,
            (X, !; write('?'), nl),
            ask_for(Request,Answer);
            Answer = X), !.
     
confirm(P) :-
      ask_for(P,V),
      (V = true, !, true;
        V = false, !, fail;
        confirm(P)).
     
% writel(L,E,S) :- write list L with list element format E (w
% simple, v more elaborate) and separator S
writel(L,E,S) :-
      var(L), !, write(E,L);
      L = [], !, true;
      L = [X], !, write(E,X);
      L = [X|L1], !, writel(X,E,nil), write(s,S),
      writel(L1,E,S);
      write(E,L).
     
write(w,X) :- write(X).
write(v,X) :- writev(X).
write(s,S) :- % S are possible separators
      S = nil, !, true;
      S = nl, !, nl;
      S = bl, !, write(' ');
      S = comma, !, write(',');
      write(S).
     
% Output of list L, elaborate printing, no separation of
% elements 
writel(L) :-
      writel(L,v,nil).
     
% Read with prompt P (e.g. @)
read(P,X) :- prompt(P1,P), read(X), prompt(P,P1).
     
% Read and expand X
reade(X) :-
      read(X1),
      (expand(X1,X), !, true;  X = X1).
     
expand(t,true).
expand(yes,true).
expand(y,true).
expand(f,false).
expand(no,false).
expand(n,false).
expand(a,abort).
expand(b,break).
expand(push,exe).
     
directive(abort).
directive(trace).
directive(break).
directive(info).
directive(X) :-
      X = true, !, fail;
      X = (_=<_), !, fail;
      X = (_<_), !, fail;
      X = (_>_), !, fail;
      X = (_>=_), !, fail;
      system(X).
     
% Capitalize variables of term X
writev(X) :-
     lettervars(X), write(X), fail .
writev(X) .
     
lettervars(X) :-
     varlist(X,V1),
     %sort(V1,V2),
     V1 = V2,
     unify_vars(V2,
     ['X','Y','Z','U','V','W','X1','Y1','Z1','U1','V1','W1',
     'X2','Y2','Z2','U2','V2','W2','X3','Y3','Z3',
     'U3','V3','W3','X4','Y4','Z4','U4','V4','W4']),!.
     
% unify_vars(A,B) :- replace variables of A by variables of B
unify_vars([X|L1],[X|L2]) :- !,
     unify_vars(L1,L2).
unify_vars([_|L1],[_|L2]) :- !,
     unify_vars(L1,L2).
unify_vars(_,_).
     
break(P) :- portray(P), nl, call(break).
     
% varlist(T,L,[]) :- L is all occurrences of distinct variables
% in term T
varlist(X,L) :- varlist(X,L,[]), ! .
varlist(X,[X|L],L) :- var(X), ! .
varlist(T,L0,L) :- T =..[F|A], !, varlist1(A,L0,L).
varlist1([T|A],L0,L) :-  varlist(T,L0,L1), !, varlist1(A,L1,L).
varlist1([],L,L).
     
mgt(P,P0) :-
     functor(P,F,N),
     functor(P0,F,N).
     
verify(P) :- \+(\+(P)).
     
ground(P) :- numbervars(P,0,0).
     
variants(P,Q) :-
     verify((numbervars(P,0,N), numbervars(Q,0,N), P = Q)).
     
varand(P,Vs1) :-
     varlist(P,Vs),
     list_to_and(Vs,Vs1).
     
list_to_and([],true) :- !.
list_to_and([X],X) :- !.
list_to_and([X|Xs],(X,Ys)) :- !,
     list_to_and(Xs,Ys).
     
and_to_list((X,Y),[X|Z]) :- !,
     and_to_list(Y,Z).
and_to_list(true,[]) :- !.
and_to_list(X,[X]) :- !.
     
and_member(P,(P,Q)).
and_member(P,(P1,Q)) :- !, and_member(P,Q).
and_member(P,P).
     
forall(X,P,Y) :-
     setof(Y,X:P,S),forall1(S).
     
forall1([]).
forall1([X|S]) :- X, forall1(S).
     
portray(X) :-
      lettervars(X),
      portray1(X,6),
      fail.
portray(X).
     
portray1(X,N) :-
      N1 is N - 1,
     (var(X), !, write(X);
       atomic(X), !, write(X);
       N = 0, !, write('#');
       X = [_|_], !, write('['), portray_list(X,N1,5),
                        write(']');
       X = (_,_), !, write('('), portray_and(X,N1), write(')');
       X =..[F|A], !, portray_term(F,A,N1);
       break(portray1(X,N))).
     
portray_args(X,N) :-
       X = [], !, true;
       X = [Y], !, portray1(Y,N);
       X = [Y|Ys], !, portray1(Y,N), write(','), !,
       portray_args(Ys,N).
     
portray_list(X,N,D) :-
       var(X), !, portray1(Y,N);
       X = [], !, true;
       D = 0, !, write('..#');
       X = [Y1|Y2], Y2 == [], !, portray1(Y1,N);
       X = [Y1|Y2], var(Y2), !, portray1(Y1,N), write('|'), !,
       portray1(Y2,N);
       X = [Y1,Y2|Ys], !, portray1(Y1,N), write(','), 
       D1 is D - 1, !,
       portray_list([Y2|Ys],N,D1);
       X = [Y1|Y2], !, portray1(Y1,N), write('|'), !,
       portray1(Y2,N).
     
portray_and(X,N) :-
       var(X), !, portray1(X,N);
       X = (Y,Ys), !, portray1(Y,N), write(','), !, portray_and(Ys,N);
       portray1(X,N).
     
portray_term(F,[A],N) :-
       current_op(P,T,F), !,
       write(F), write(' '), portray1(A,N).
portray_term(F,[A,B],N) :-
       current_op(P,T,F), !,
       portray1(A,N), write(F), portray1(B,N).
portray_term(F,A,N) :-
       write(F), write('('), portray_args(A,N), write(')').
     
bagof0(X,P,S) :-
      bagof(X,P,S), !, true; S=[].
     
setof0(X,P,S) :-
      setof(X,P,S), !, true; S=[].
     
new(X,Y) :- % Y is a fresh copy of X (with new variables)
      abolish('gross hack',1),
      assert('gross hack'(X)),
      retract('gross hack'(Y)).
     
plisting([]) :- !.
plisting([P|Ps]) :- !,
      plisting(P), nl, !, plisting(Ps).
plisting(X) :-
      (X = (P :- _), !, mgt(P,P1); mgt(X,P1)),
      writel(['Listing of ',P1,':']), nl,
      (clause(P1,Q), tab(4), writev((P1 :- Q)), write('.'), 
       nl, fail; true), nl.
     
ask_then_do(Question,Responses) :-
      % Display the question. A response is a list of 
      % (Answer, Action)-pairs.
      % Verify that the user given answer is in a pair.
      % If so, perform the associated action.

      ask_for(Question,Answer),
      member((Answer,Action),Responses) -> Action;
      setof(Answer,Action:member((Answer,Action),Responses),Answers),
      writel('legal answers are',Answers), nl,
      ask_then_do(Question,Responses).
     
     

%%% PDSINI

%% Initialization Stuff
     
:-declare(qsort(+[x],-[x]),[determinate,total]).
:-declare_called(qsort(X,Y),
      [qsort(Z,U),partition(V,W,X1,Y1),append(Z1,U1,V1),c(W1,X2,Y2)]).
:-declare(partition(+[x],+x,-[x],-[x]),[determinate,total]).
:-declare_called(partition(X,Y,Z,U),
      [partition(V,W,X1,Y1),Z1 < U1,V1 =< W1]).
:-declare(append(+[x],+[x],-[x]),[determinate,total]).
:-declare_called(append(X,Y,Z),[append(U,V,W)]).
:-declare(le(+0,+0),[determinate]).
:-declare_called(le(_,_),[le(Z,U)]).
:-declare(insert(+x,+[x],-[x]),[determinate,total]).
:-declare_called(insert(X,Y,Z),[insert(U,V,W),X2 < Y2,
       X1 =< Y1]).
:-declare(isort(+[x],-[x]),[determinate,total]).
:-declare_called(isort(X,Y),[isort(Z,U),insert(V,W,X1)]).
:-declare(+x'=<'+x,[determinate]).
:-declare(+x'<'+x,[determinate]).
:-declare(pack(+[[x]],-[x]),[determinate,total]).
:-declare_called(pack(X,Y),[pack(U,V),append(W,X1,Y1)]).
     

     
%%% TYPE

%% Typing Utilities for pds
     
     
% type(Type,Name,Terms,TermsType).
type(x,object,[],[]).
type(0,integer,[0,s(_)],[s(0)]).
type(1,integer,[0],[]).
type(10,boolean,
     [0,1,not(_),and(_,_),or(_,_)],
     [not(01),and(01,01),or(01,01)]).
type(io,binary,[nil,o(_),i(_)],[o(io),i(io)]).
type([],list,[[],[_|_]],[[a|[]]]).
type([X],'list of',[[],[_|_]],[[X|[X]]]).
type(bt(L),'binary tree',
     [leaf(_),t(_,_)],[leaf(L),t(bt(L),bt(L))]).
type(lbt(X),'labeled binary tree',
     [nil,t(_,_,_)],[t(lbt(X),X,lbt(X))]).
type(ttt(L),'two-three tree',
     [leaf(_),t(_,_,_)],[leaf(L),t(L,L,[ttt(L)])]).
type(terminal,terminal,[X],[]) :-
      terminal(X).
     
% Input a variable and its type, instantiate it to a term and
% return a list of the variables in the term + their types
term_to_vars((Term,TermType),Vars) :-
      term_of(TermType,Term),
      setof_vartype((Term,TermType),Vars).
     
term_of(Type,Term) :-
      type(Type,_,TermList,_),
      member(Term,TermList).
     
typed_term(Type,Term) :-
      type(Type,_,_,TypedTerms),
      member(Term,TypedTerms).
     
% atom_vartype(P,Vi,Vo) :- get type of vars in P.
atom_vartype(P,Vi,Vo) :-
      input_vars(P,Pi),
      output_vars(P,Po),
      terms_to_vartype(Pi,Vi),
      terms_to_vartype(Po,Vo).
     
% terms_to_vartype(T,V) :- take a list of (Term,Type) and
% return a list of (Var,Type) for all vars in the terms
terms_to_vartype([],[]).
terms_to_vartype([(Term,Type)|T],Vs) :-
      setof_vartype((Term,Type),Vs1),
      terms_to_vartype(T,Vs2),
      append(Vs1,Vs2,Vs).
     
setof_vartype((Term,TermType),Vars) :-
      setof((Var,Type),vartype(Term,TermType,Var,Type),Vars), !.
setof_vartype((Term,TermType),[]).
     
% vartype(Term,TermType,Var,Vartype) :-
% The type of variable Var that occurs in term Term 
% of type TermType is VarType
% Reports on type violations?
vartype(Var,Type,Var1,Type1) :-
      var(Var), !,
      Var = Var1,
      Type = Type1.
vartype(Term,TermType,Var,VarType) :-
      Term =..[Functor|Args],
      typed_term(TermType,Term1), Term1 =..[Functor|ArgsType],
      vartype1(Args,ArgsType,Var,VarType).
     
vartype1([Term|_],[TermType|_],Var,VarType) :-
      vartype(Term,TermType,Var,VarType).
vartype1([_|Args],[_|ArgsType],Var,VarType) :-
      vartype1(Args,ArgsType,Var,VarType).
     
type_check(:-(P,Q)) :- !,
      type_check(P), type_check(Q).
type_check((P,Q)) :- !,
      type_check(P), type_check(Q).
type_check(Atom) :-
      type(Atom,_,_,[AtomType]),
      type_check(Atom,AtomType).
     
type_check(Term,TermType) :-
      term_of(TermType,Term),
      (atomic(Term);
        Term =..[Functor|Args],
          typed_term(TermType,Term1),
          Term1 =..[Functor|ArgsType],
          type_check1(Args,ArgsType)).
     
type_check1([],[]).
type_check1([Term|Args],[TermType|ArgsType]) :-
      type_check(Term,TermType),
      type_check1(Args,ArgsType).

==================
