/*  HELP.PL  */


:- module help.


:- public help_text/3,
          uppercase_first_element/2.


/*
SPECIFICATION
-------------

This module links diagnostic error codes for facts and questions with
the text of a short error message and a longer explanation:

PUBLIC help_text( Code+, Short-, Explanation- ):

Short and Explanation are the texts corresponding to Code. The possible
values of Code are defined in PARSE_LOGIC.PL and PARSE_PROLOG.PL.


PUBLIC upper_case_first_element( L0+, L- ):

If L0's first element is an atom, uppercase its initial letter. Used
in making messages ready for display, and exported for the benefit
of COMMAND_HELP.
*/


/*
IMPLEMENTATION
--------------

A lot of typing...
*/


:- needs
    char_to_uppercase/2,
    token_contents/2.


help_text(
empty_question,
'You have typed a question mark with nothing before it.',
''
).


help_text(
empty_fact,
'You have typed a dot with nothing before it.',
''
).


help_text(
and_if_in_definition,
'You have got a fact with "and if" in the right-hand side.',
'Logic only allows one "if", between the left and right hand parts.'~<>
'Perhaps you should replace the "and if" by just an "and".' 
).


help_text(
if_in_definition,
'You have got a fact with an "if" in the right-hand side.',
'Logic only allows one "if", between the left and right hand parts.'~<>
'Perhaps you have written something like "and ... if", and could'~<>
'replace it by just an "and".'
).


help_text(
fact_too_short,
'What you\'ve just typed was too short to be a fact, and I didn\'t'~<>
'recognise it as a command.',
'Perhaps you mis-typed (e.g.) the command'~<>
'    section 1'~<>
'as'~<>
'    section1'~<>
'Or perhaps you mis-typed a fact. In Logic mode, every fact is'~<>
'either three words, or three words followed by an "if" followed'~<>
'by more words.'~<>
'Examples:'~<>
'    john loves mary.'~<>
'    john loves X if X likes wine.' 
).


help_text(
uncond_fact_too_long([Tok1,Tok2,Tok3,Tok4|Rest]),
'Your sentence was not in the right form to be a fact,'~<>
'and I didn\'t recognise it as a command.',
'In Logic mode, every fact is either three words, or three words'~<>
'followed by an "if" followed by more words.'~<>
'Examples:'~<>
'    john loves mary.'~<>
'    john loves X if X likes wine.'~<>
''~<>
'Your sentence contained more than three words, but the fourth,'...W4<>','~<>
'was not "if".' 
) :-
    token_contents( Tok4, W4 ).


help_text(
cond_fact_too_long,
'Your sentence was not in the right form to be a fact.',
'It contained at least four words before the "if".'~<>
'Every fact is either three words, or three words followed by an'~<>
'"if" followed by more words. The first three are always a "noun" or'~<>
'argument, a "verb" or predicate, and another argument.'~<>
'Examples:'~<>
'    john loves mary.'~<>
'    john loves X if X likes wine.' 
).


help_text(
fact_and_fact,
'You have what looks like two facts joined by "and".',
'e.g.  john loves mary and john likes wine.'~<>
'Logic does not allow this, and you must enter each fact separately.' 
).


help_text(
if_in_question,
'You have got a question with "and if" in the right-hand side.',
'This isn\'t allowed. You may be trying to ask a question which depends'~<>
'on assumptions, and you\'re trying to put those in after the "if".'
).


help_text(
bad_comparator,
'You have got a question or fact containing an <= or => symbol.',
'Were you meaning to compare numbers? If so, the correct symbols are'~<>
'=< and >= .'~<>
'(these are the ones that don\'t look like arrows).'
).


help_text(
is_as_predicate_in_fact,
'You have got a fact about "is".',
'The predicate "is" is reserved for arithmetic. You can\'t redefine it'~<>
'or use it in other contexts.'
).


help_text(
comma_as_predicate_in_fact( (G1,G2) ),
'Your sentence looked like a Prolog fact with a comma between two goals.',
'Did you mean to type a question?'~<>
'If not, were you trying to assert the two facts'...G1...'and'...G2...'in one go?'~<>
'Prolog won\'t allow that.'
).


help_text(
system_predicate_in_fact/[Pred,Arity],
'You have got a fact about the system predicate'...q_(Pred)<>'.',
'Such predicates (">", "=", "write", etc) are reserved for use by the'~<>
'system. You can\'t add new clauses for them.'
).


help_text(
prolog_in_logic,
'You have a bracket, comma, semicolon, or ":-" in your fact or question.',
'Are you trying to type in Prolog? You may have forgotten to'~<>
'give the "prolog" command to switch mode from Logic.' 
).



help_text(
connective_for_predicate/W,
'You have a "logic word" ('<>q_(W)<>') as your "verb" or predicate.',
'You can\'t have facts or questions like:'~<>
'   john if mary'~<>
'or'~<>
'   romeo and juliet'~<>
'It is confusing, and doesn\'t make much sense anyway.' 
).


help_text(
variable_for_predicate/W,
'You have a variable ('<>q_(W)<>') as your "verb" or predicate.'~
'You can\'t have facts or questions like:'~<>
'   john What mary'~<>
'or'~<>
'   Who Somethings wine'~<>
'Although this would make some kind of sense, at least in questions,'~<>
'Prolog just will not permit it.'
).


help_text(
non_atom_for_predicate/W,
'You have something that isn\'t a name ('<>qtoken_(W)<>') as a'~<>
'"verb" or predicate in your fact or question.',
'You can\'t have something like:'~<>
'   john 2 mary'~<>
'because the "verb" or predicate must always be a name.'
).


help_text(
unrecognised_error_in_fact,
'You have got a grammatical error which I can\'t recognise.',
'Ask your teacher to provide me with more specific diagnostics.'
).


help_text(
unrecognised_error_in_question,
'You have got a grammatical error which I can\'t recognise.',
'Ask your teacher to provide me with more specific diagnostics.'
).


help_text(
bad_prolog_fact(Message,Input),
seplist_(Message_,' ')<>'.'~<>
'The error was detected near the point shown below:'~<>                   
seplist_(Input,' '),
''
) :-
    upper_case_first_element( Message, Message_ ).


help_text(
bad_prolog_question(Message,Input)
seplist_(Message_,' ')<>'.'~<>
'The error was detected near the point shown below:'~<>
seplist_(Input,' '),
''
) :-
    upper_case_first_element( Message, Message_ ).


help_text(
token_error(Message,Input,_)
seplist_(Message_,' ')<>'.'~<>
'The error was detected near the point shown below:'~<>
puts_(Input),
''
) :-
    upper_case_first_element( Message, Message_ ).


upper_case_first_element( [H|T], [UH|T] ) :-
    atom( H ),
    H \= '',
    !,
    name( H, [HC1|HT] ),
    char_to_uppercase( HC1, UHC1 ),
    name( UH,[UHC1|HT] ).

upper_case_first_element( L, L ).


:- endmodule.
