;; Eulisp Module
;; Author: pete broadbery
;; File: gen-code.em
;; Date: 30/jul/1991
;;
;; Project:
;; Description: 
;;  code generator pass of compiler.
;;  assume names pass is complete.
;;

(defmodule gen-code 
  (standard0
   list-fns

   syntx-utl
   props;; Should try to avoid this --- maybe an analysis module?
   syntx-env;; Should try to avoid this --- maybe an analysis module?
   generate
   pass
   ;; use Should try to avoid this --- maybe an analysis module?

   rshow
   stop
   )
  ()

  (defcondition Compile-Time-Error () 
    msg "" values ())
  (export Compile-Time-Error)

  (export generate-code)

  (deflocal *last* ())

  (defun code-gen (thing state)
    (let ((prev *last*))
      (setq *last* state)
      ;;(format t "(Generating code for: ~a~%state: ~a" thing state)
      ;;(print-props thing)
      (let ((xx (generic-code-gen thing state)))
	(setq *last* prev)
	;;(format t "--> ~a)" xx)
	xx)))
      
  (defun last ()
    *last*)
  (export last)

  (defun generate-code (module)
    (let ((final-state 
	   (code-gen module
		     (make-initial-compiler-state module))))
      (modify-compiler-state 
       final-state
       'state-code 
       (add-code-vectors (list (reify-code-stream 
				(complete-installation module final-state)))
			 final-state)
       ;; blast the old stream
       'state-stream (make-new-code-stream))))
  
  (defconstant generic-code-gen (make-compiler-pass 'code-gen))

  (defmethod generic-code-gen ((seq sequence) state)
    (let ((newstate (fold (lambda (obj state)
			    (do-pop 1 (code-gen obj state)))
			  (sequence-body seq)
			  state)))
      (code-gen (sequence-end seq) newstate)))

  
  (defun make-initial-compiler-state (module)
    (make-compiler-state (make-new-code-stream)
			 (make-stack)
			 (make-static-store)
			 (make-code-list)))

  (defun complete-installation (mod state)
    state)

  ;; Modules. Unfortunately they have a closure...
  ;; Makes them easy to init though
  (defmethod generic-code-gen ((mod module-block) state)
    (let ((env (make-lambda-environment mod)))
      ;; Idea is to make a nice, clean easy to call 
      ;; function... 
      ((setter real-lambda-env) mod env)
      (do-code-sequence
       (list (lambda (state)
	       (modify-compiler-state
		state
		'state-stack (stack-push (state-stack state) nil)))
	     (lambda (state)
	       (do-alloc-env env state))
	     (lambda (state)
	       (code-gen (module-body mod) state))
	     ;; the return from progn, and throw the env away
	     (lambda (state)
	       (do-pop 2 state))
	     (lambda (state)
	       (do-push-static t state)) ;; hack
	       (lambda (state)
		 (do-return state)))
       state)))

  (defmethod generic-code-gen ((cd condition-term) state)
    (let ((lab1 (make-label state))
	  (lab2 (make-label state)))
      (let ((stack (state-stack state)))
	;; test for spectacularly dumb conditions...
	(if (literal-p (cond-test cd))
	    (if (null (literal-content (cond-test cd)))
		(code-gen (cond-f-part cd) state)
	      (code-gen (cond-t-part cd) state))
	  (if (term-tail-call cd)
	      (do-code-sequence 
	       (list (lambda (state) (code-gen (cond-test cd) state))
		     (lambda (state) (do-branch-on-nil lab1 state))
		     (lambda (state) (code-gen (cond-t-part cd) state))
		     (lambda (state) (modify-compiler-state
				      state
				      'state-stack stack))
		     (lambda (state) (do-add-label lab1 state))
		     (lambda (state) (code-gen (cond-f-part cd) state)))
	       state)
	    (do-code-sequence 
	     (list (lambda (state) (code-gen (cond-test cd) state))
		   (lambda (state) (do-branch-on-nil lab1 state))
		   (lambda (state) (code-gen (cond-t-part cd) state))
		   (lambda (state) (do-branch lab2 state))
		   ;; Should unscrew state here...
		   ;; Hopefully only stack in wrong
		   (lambda (state)
		     (modify-compiler-state
		      state
		      'state-stack stack))
		   (lambda (state)
		     (do-add-label lab1 state))
		   (lambda (state) 
		     (code-gen (cond-f-part cd) state))
		   (lambda (state)
		     (do-add-label lab2 state)))
	     state))))))
	 

  (defmethod generic-code-gen ((id ident-term) state)
    ;; maybe ought to take a quick look at the stack here...
    (let ((state (value-ref (ident-decl id) (ident-block id) id state)))
      (if (term-tail-call id)
	  (add-tidy-code (enclosing-lambda id) state)
	state)))
			 
  (defgeneric value-ref (id loc orig state))
  
  (defmethod value-ref ((x object) loc id state)
    (do-push-global 'some-value state))

  (defmethod value-ref ((x module-definition) loc id state)
    (local-module-ref x state))
  
  (defmethod value-ref ((x definition) loc id state)
    (if (binding-closed x) 
	;; Generate env. ref
	(closed-value-ref x loc id state)
      (open-value-ref x loc state)))
  
  (defmethod value-ref ((x imported-definition) loc id state)
    (do-push-global (external-name x)
		    state))

  (defmethod value-ref ((x lambda-id) loc id state)
    (if (binding-closed x) 
	;; Generate env. ref
	(closed-value-ref x loc id state)
      (open-value-ref x loc state)))
    
  ;; Ahrrhgg. Assignments
  (defmethod generic-code-gen ((x assignment-term) state)
    (let ((id (assign-var x)))
      (let* ((state (code-gen (assign-body x) state))
	     (state2 (set-value-ref (ident-decl id) (ident-block id) id state)))
	(if (term-tail-call x)
	    (add-tidy-code (enclosing-lambda x) state2)
	  state2))))
  
  
  (defgeneric set-value-ref (id loc orig state))

  (defmethod set-value-ref ((x object) loc id state)
    (do-global-set 'some-value state))

  (defmethod set-value-ref ((x module-definition) loc id state)
    (set-local-module-ref x state))

  (defmethod set-value-ref ((x definition) loc id state)
    (if (binding-closed x) 
	;; Generate env. ref
	(set-closed-value-ref x loc id state)
      (set-open-value-ref x loc state)))
  
  (defmethod set-value-ref ((x lambda-id) loc id state)
    (prog1 (if (binding-closed x) 
	       ;; Generate env. ref
	       (set-closed-value-ref x loc id state)
	     (set-open-value-ref x loc state))
      nil))
  
  ;; here?
  (defun find-env-depth (env target)
    (if (null env)
	(format t "Could not env ~a in ~a~%" env target)
      (cond ((eq env target)
	     0)
	    ((= (env-object-size env) 0)
	     (find-env-depth (env-object-prev env) target))
	    (t 
	     (+ (find-env-depth (env-object-prev env) target) 1)))))

  (defun open-value-ref (x loc state)
    (do-stack-ref (scanq-stack (state-stack state) x)
		  state))

  (defun local-module-ref (x state)
    (let ((xx (do-push-global (external-name x) state)))
      xx))


  (defun closed-value-ref (binding loc id state)
    (let ((env (stacked-lambda-env (enclosing-lambda id)))
	  (posn (binding-posn binding))
	  (def-env (lambda-env (enclosing-lambda binding))))
      ;;(stop (list env posn def-env))
      (let ((depth (find-env-depth env def-env)))
	(do-code-sequence
	 (list (lambda (state) (do-stack-ref (scanq-stack (state-stack state) env) state))
	       (lambda (state) (do-env-ref depth posn state)))
	 state))))

  (defun set-open-value-ref (x loc state)
    (let ((state (do-stack-ref 0 state)))
      (do-set-stack-ref (scanq-stack (state-stack state) x)
			state)))

  (defun set-local-module-ref (x state)
    (do-code-sequence 
     (list (lambda (state)
	     (do-stack-ref 0 state))
	   (lambda (state)
	     (do-global-set (external-name x) state)))
     state))

  (defun set-closed-value-ref (binding loc id state)
    (let ((env (stacked-lambda-env (enclosing-lambda id)))
	  (posn (binding-posn binding))
	  (def-env (lambda-env (enclosing-lambda binding))))
      ;;(format t "set-closed-ref: ~a ~a ~a ~a~%" env def-env posn)
      ;;(stop (list env posn def-env))
      (let ((depth (find-env-depth env def-env)))
	(do-code-sequence
	 (list (lambda (state) (do-stack-ref (scanq-stack (state-stack state) env) state))
	       (lambda (state)
		 (do-stack-ref 1 state))
	       (lambda (state)
		 (do-setter-env-ref depth posn state))
	       (lambda (state)
		 (do-pop 1 state)))
	 state))))

  ;;
  ;;; this-context: grab this function from the stack
  ;;
  
  (defmethod generic-code-gen ((ob special-term) state)
    (cond ((eq (special-term-name ob) 'call-next-method-internal)
	   (do-code-sequence 
	    (list (lambda (state)
		    (do-stack-ref (+ (stack-depth (state-stack state)) 1) state))
		  ;; check here not needed --- apply methods ought to check though
		  (lambda (state)
		    (do-cdr state)))
	    state))
	  ((eq (special-term-name ob) 'inline-fn)
	   (make-inline-lambda (special-term-data ob) state))
	  (t (format t "Unknown special")
	     state)))
  
  (defun make-inline-lambda (args state)
    (let ((ns (modify-compiler-state state 
				     'state-stack (stack-push (stack-push (stack-push (make-stack) 'x) 'y) 'z)
				     'state-stream (make-new-code-stream)
				     'state-statics (state-statics state)
				     'state-code (make-code-list)))
	  (init-label (make-label state)))
      (let ((inlined-state (let* ((state (do-add-label init-label ns))
				  (state (do-pop 1 ns))
				  (state (do-inline-code (cdr args) (car args) state))
				  (state (do-return state)))
			     state)))
	(let ((new-state (modify-compiler-state
                         state
                         'state-code
                         (add-code-vectors (cons (reify-code-stream inlined-state)
						 nil)
                                           state))))
	  (let* ((state (do-push-label init-label new-state))
		 (state (do-push-static nil state))
		 (state (do-allocate-closure (if (< (car args) 0) 
						 (cons t (- (car args)))
					       (cons nil (car args)))
					     state)))
	    state)))))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ;; Function Applications
  ;;
  ;;
  
  (defmethod generic-code-gen ((applic applic-term) state)
    (let ((obj (find-fn (applic-fun applic)))
	  (tail-flag (term-tail-call applic)))
      (check-arguments applic obj)
      ;; do any strange side effect type stuff specified by obj
      (do-callbacks (function-fn obj) applic)
      ((find-app-fn (function-type obj)) obj applic tail-flag state)))


  (defun check-arguments (applic obj)
    (let ((nargs (list-length (applic-args applic)))
	  (reqd-nargs (function-nargs obj)))
      (if (or (eq (function-type obj) 'unknown)
	      (= (cdr reqd-nargs) 9999)
	      (= nargs (cdr reqd-nargs))
	      (and (car reqd-nargs)
		   (>= (+ nargs 1) (cdr reqd-nargs))))
	  t
	(error "Function called with wrong number of args" 
		Compile-Time-Error
		'values (list reqd-nargs applic)
		'msg "Function called with wrong number of args (should be ~a): ~%~a~%"))))

  (defconstant find-callback (mk-finder))

  (defgeneric do-callbacks (obj applic)
    methods ((((x imported-definition) applic)
	      (labels ((do-callback (l)
				    (cond ((null l) nil)
					  (t ((find-callback (car l)) applic)
					     (do-callback (cdr l))))))
		      (do-callback (import-prop-ref x 'callbacks))))
	     (((x syntax-obj) y)
	      nil)))
  
  ((setter find-callback) 'set-setter
   (lambda (applic)
     ;;(format t "set-setter: ~a~%" applic)
     (if (not (ident-p (cadr (applic-args applic))))
	 nil
       ((setter obj-setter-decl) 
	(ident-decl (car (applic-args applic)))
	(ident-decl (cadr (applic-args applic)))))))

  (defconstant find-app-fn (mk-finder))

  
  ;; local function. Called with branch
  (defun apply-local-function (obj applic tail state)
    (let ((branch-lab (read-init-label (car obj)) state))
      (if tail
	  (do-branch branch-lab
		     (push-local-env (car obj)
				     (tail-prepare-args obj applic state)))
	(let ((ret-lab (make-label nil)))
	  (do-code-sequence 
	   (list (lambda (state)
		  (std-prepare-args obj applic state ret-lab))
		 (lambda (state)
		   (push-local-env (car obj) state))
		 (lambda (state)
		   (do-branch branch-lab state))
		 (lambda (state)
		   (do-add-label ret-lab state))
		 (lambda (state)
		   (correct-stack applic obj state)))
	   state)))))

  (defun apply-nonlocal-function (obj applic tail state)
    (if tail
	(do-apply-bvf (mk-calltype applic obj)
		      (tail-prepare-args obj applic state))
      (let ((ret-lab (make-label nil)))
	(do-code-sequence 
	 (list (lambda (state)
		 (std-prepare-args obj applic state ret-lab))
	       (lambda (state)
		 (do-apply-bvf (mk-calltype applic obj) state))
	       (lambda (state)
		 (do-add-label ret-lab state))
	       (lambda (state)
		 (correct-stack applic obj state)))
	 state))))

  (defun apply-unknown-function (obj applic tail state)
    (if tail
	(do-apply-any (mk-calltype applic obj)
		      (tail-prepare-args obj applic state))
      (let ((ret-lab (make-label nil)))
	(do-code-sequence 
	 (list (lambda (state)
		 (std-prepare-args obj applic state ret-lab))
	       (lambda (state)
		 (do-apply-any (mk-calltype applic obj) state))
	       (lambda (state)
		 (do-add-label ret-lab state))
	       (lambda (state)
		 (correct-stack applic obj state)))
	 state))))

  ;; self call or labels call.   
  
  (defun apply-lexical-function (obj applic tail state)
    (let ((branch-lab (read-init-label (car obj)) state))
      (if tail 
	  (do-code-sequence 
	   ;; Should blast the function call position...
	   (list (lambda (state)
		   (push-args (applic-args applic) obj state))
		 (lambda (state)
		   (grab-environment applic
				     (car obj)
				     state))
		 (lambda (state)
		   (do-slide (stack-depth (state-stack state))
			     (+ (actual-args applic obj) 1)
			     state))
		 (lambda (state)
		   (do-branch branch-lab state)))
	   state)
	(let ((ret-lab (make-label nil)))
	  (let* ((state (do-push-static nil state)) ;; XXX should be an env description{or something}
		 (state (do-push-label ret-lab state))
		 (state (push-args (applic-args applic) obj state))
		 (state (grab-environment applic 
					  (car obj)
					  state))
		 (state (do-branch branch-lab state))
		 (state (do-add-label ret-lab state))
		 (state (correct-stack applic obj state)))
	    state)))))
	       
  (defun apply-inline-function (obj applic tail state)
    (if tail
	(do-code-sequence 
	 (list (lambda (state)
		 (push-args (applic-args applic) obj state))
	       (lambda (state)
		 (do-slide (stack-depth (state-stack state))
			   (actual-args applic obj)
			   state))
	       (lambda (state)
		 (do-inline-code (import-prop-ref (function-fn obj) 'code)
                                 (actual-args applic obj)
                                 state))
	       (lambda (state)
		 (do-return state)))
	 state)
      (let* ((state (push-args (applic-args applic) obj state))
	     (state (do-inline-code (import-prop-ref (function-fn obj) 'code)
				    (actual-args applic obj)
				    state)))
	state)))
	 
  (defun apply-special-function (obj applic tail state)
    (cond ((eq (special-term-name (car obj)) 'call-next-method-internal)
	   (if tail 
	       (let* ((state  (tail-prepare-args obj applic state))
		      (state (do-apply-methods (mk-calltype applic obj)
					       state)))
		 state)
	     (let* ((ret-lab (make-label nil))
		    (state (std-prepare-args obj applic state ret-lab))
		    (state (do-apply-methods (mk-calltype applic obj) state))
		    (state (do-add-label ret-lab state))
		    (state (correct-stack applic obj state)))
	       state)))
	  (t (error "Unknown special" Compile-Time-Error))))

	  
  
  (progn 
    ((setter find-app-fn) 'lexical apply-lexical-function)
    ((setter find-app-fn) 'local apply-local-function)
    ((setter find-app-fn) 'bytefunction apply-nonlocal-function)
    ((setter find-app-fn) 'inline apply-inline-function)
    ((setter find-app-fn) 'unknown apply-unknown-function)
    ((setter find-app-fn) 'function apply-unknown-function)
    ((setter find-app-fn) 'special apply-special-function)
    )
  ;; move the arguments onto the stack, together with a 
  ;; label in the right (TM) place, and move the function
  ;; to the top
  (defun std-prepare-args (obj applic state ret-lab)
    (do-code-sequence 
     (list (lambda (state)
	     (code-gen (applic-fun applic) state))
	   (lambda (state)
	     (do-push-label ret-lab state))
	   (lambda (state)
	     (push-args (applic-args applic) obj state))
	   (lambda (state)
	     (do-stack-ref (+ (actual-args applic obj) 1) state)))
     state))
  
  
  ;; Push the arguments on to the stack, and slide down to a position where 
  ;; the tail call can be done

  (defun tail-prepare-args (obj applic state)
    (do-code-sequence
     (list (lambda (state)
	     (code-gen (applic-fun applic) state))
	   (lambda (state)
	     (push-args (applic-args applic) obj state))
	   (lambda (state)
	     (do-stack-ref (actual-args applic obj) state))
	   (lambda (state)
	     (do-slide (stack-depth (state-stack state))
		       (+ (actual-args applic obj) 1)
		       state))
	   (lambda (state)
	     (blast-current-fn state)))
     state))

  ;; this way 'cos I want to see what the code looks like...
  (defun blast-current-fn (state)
    (let* ((state (do-stack-ref 0 state)))
      (do-set-stack-ref  (+ (stack-depth (state-stack state)) 1) state)))
  
  (defun correct-stack (applic obj state)
	(modify-compiler-state
	 state
	 'state-stack 
	 (let ((stack (state-stack state)))
	   (stack-push (stack-pop stack
				  (+ (actual-args applic obj) 3))
		       (make-stack-val)))))
     
  (defun push-args (args obj state)
    ;; should do nary-check here
    (if (car (function-nargs obj))
	(push-nary-args (cdr (function-nargs obj)) args state)
      (fold (lambda (arg state)
	      (let ((xx (code-gen arg state)))
		xx))
	    args
	    state)))

  (defun push-nary-args (nargs args state)
    ;; keep pushing till we get to optionals
    (if (= nargs 1)
	(push-optional-args args state)
      (push-nary-args (- nargs 1) (cdr args)
		      (code-gen (car args) state))))
  
  (defun push-optional-args (args state)
    (if (null args)
	(do-push-static nil state)
      (do-code-sequence 
       (list (lambda (state)
	       (code-gen (car args) state))
	     (lambda (state)
	       (do-push-static nil state))
	     (lambda (state)
	       (do-cons state))
	     (lambda (state)
	       (do-stack-ref 0 state))
	     (lambda (state)
	       (push-remaining-args (cdr args) state)))
       state)))
  
  (defun push-remaining-args (args state)
    (if (null args)
	(do-pop 1 state)
      (push-remaining-args (cdr args)
			   (do-code-sequence 
			    (list (lambda (state)
				    (code-gen (car args) state))
				  (lambda (state)
				    (do-push-static nil state))
				  (lambda (state)
				    (do-cons state))
				  (lambda (state)
				    (do-setter-cdr state)))
			    state))))

  ;; actual args pushed:
  (defun actual-args (applic obj)
    (if (car (function-nargs obj))
	(cdr (function-nargs obj))
      (list-length (applic-args applic))))

  (defun mk-calltype (applic obj)
    (if (car (function-nargs obj))
	(- (cdr (function-nargs obj)))
      (list-length (applic-args applic))))

  (defun grab-environment (applic fn state)
    (if (not (lambda-closed-p fn))
	(do-push-static nil state)
      (let ((env (stacked-lambda-env (enclosing-lambda fn)))
	    (enc-lambda (enclosing-lambda applic)))
	(format t "(Grab env: ~a ~a ~a~%" applic env state)
	(let ((local-env (stacked-lambda-env enc-lambda)))
	  (if (= (env-object-size env) 0)
	      (do-push-static nil state)
	    (let ((state (fetch-environment enc-lambda state)))
	      (prog1 (if (eq env local-env)
			 state
		       (do-pop-env (find-env-depth local-env env) state))
		(format t ")"))))))))

  (defun push-local-env (fn state)
    (let ((env (stacked-lambda-env fn)))
      (if (= (env-object-size env) 0)
	  (let* ((state (do-pop 1 state))
		 (state (do-push-static nil state)))
	    state)
	(do-slot-ref 0 state))))
		 
  ;; Calling sequence...
  ;; args are in the order
  ;; [fn]/[mds] [return address] a0 a1 a2 a3 <env> [fn]
  
  ;; code gen for lambda should assume args on stack, 
  ;; and that, if nec. an env will be placed on the stack 
  ;; by its own calling routine.

  ;; When a fn completes, its stack should be contain it's return value
  ;; Compiling tail calls: 
  ;; should be just a jump to the relavant routine,
  ;; preparing the arguments as low on the stack as possible
  
  ;; if inline,
  ;; args should (hem hem) be OK
  ;; just insert the relavant code
  ;; o/w create new code-vector and compile into that.
  ;;   add code to initialise the function
  ;; exit: restore the stack to the initial state
  
  (defmethod generic-code-gen ((lam lambda-term) state)
    ;;(format t "(In Lambda: ")
    (let ((new-state (code-gen (lambda-body lam)
			       ;; does env-construction, etc
			       (add-entry-code lam
					       (new-code-state lam state)))))
      (let ((next-state (modify-compiler-state 
			 state
			 'state-statics (state-statics new-state)
			 'state-code 
			 (add-code-vectors (cons (reify-code-stream new-state)
						 (state-code new-state))
					   state))))
	;; bung lambda onto stack
	(let ((state (do-allocate-function lam
					   next-state
					   )))
	  (if (term-tail-call lam)
	      (add-tidy-code (enclosing-lambda lam) state)
	    state)))))
  
  (defun new-code-state (lam state)
    ;;(format t "(New state: ")
    (let ((stack (fold (lambda (arg stack)
			 (stack-push stack arg))
		       (lambda-ids lam)
		       (make-stack)))
	  (out-stream (make-new-code-stream)))
      (modify-compiler-state state 
			     ;; should only push env if we have one
			     'state-stack (stack-push stack (stacked-lambda-env (enclosing-lambda lam)))
			     'state-stream out-stream
			     'state-statics (state-statics state)
			     'state-code (make-code-list))))

  ;; add the stuff the program will have to do on entry
  (defun add-entry-code (lam state)
    (let* ((env (lambda-env lam))
	   (init-label (read-init-label lam))
	   (new-state (do-code-sequence
		       (cons (lambda (state)
			       (do-add-label init-label state))
			     (if (> (env-object-size env) 0)
				 (list (lambda (state)
					 (do-alloc-env env state)))
			       ()))
		       state)))
      ;;(format t "Env: ~a~%" env)
      (if t ;;(lambda-closed-p lam)
	  ;; copy things into the closure
	  (fold (lambda (bind state)
		  (if (binding-closed bind)
		      (add-to-env bind state)
		    state))
		(lambda-ids lam)
		new-state)
	;; throw away parent environment
	(do-pop 1 state))))
  
  ;; Get the entry point right !
  (defun read-init-label (lam)
    (or (lambda-init-label lam)
	(let ((newlab (make-label nil)))
	  ((setter lambda-init-label) lam newlab)
	  newlab)))
  
  (defun add-to-env (bind state)
    (let ((posn (scanq-stack (state-stack state)
			     bind)))
      (do-code-sequence 
       (list (lambda (state)
	       (do-stack-ref posn state))
	     (lambda (state)
	       (do-setter-env-ref 0 (binding-posn bind) state)))
       state)))

  ;; Lazily calculate environments
  (defun lambda-env (lam)
    (let ((e (real-lambda-env lam)))
      (or e
	  (let ((xx (make-lambda-environment lam)))
	    ((setter real-lambda-env) lam xx)
	    xx))))

  ;; make an environment... 
  (defun make-lambda-environment (lam)
    (let ((closed-bindings (collect allocable-defn-p (find-closure lam))))
      ;; enumerate them
      (fold (lambda (bind n)
	      ((setter binding-posn) bind n)
	      (+ n 1))
	    closed-bindings
	    0)
      (make-env-object (list-length closed-bindings)
		       (convert closed-bindings vector)
		       (enclosing-env lam))))
  
  (defgeneric allocable-defn-p (defn))
  (defmethod allocable-defn-p ((x lambda-id)) 
    t)
  
  (defmethod allocable-defn-p ((defn definition))
    (if (binding-as-arg defn)
	t
      (not (inhibit-alloc (defn-body defn)))))
  
  (defun binding-needed-p (defn)
    (and (binding-used defn)
	 (not (lambda-inhibit-alloc (defn-body defn)))))

  ;; discover what needs to be placed in the environment
  
  (defgeneric find-closure (obj)
    methods ((((lam lambda-term))
	      (append (collect (lambda (x) (if (binding-closed x) x nil))
			       (lambda-ids lam))
		      (get-internal-closed-bindings (lambda-body lam))))
	     (((mod module-block)) 
	      (get-internal-closed-bindings (module-body mod)))))
  
  (defgeneric enclosing-env (obj)
    methods ((((lam lambda-term))
	      (lambda-env (enclosing-lambda lam)))
	     (((mod module-block))
	      nil)))

  ;; finally [This is called in many places]
  (defun add-tidy-code (lam state)
    (do-code-sequence
     (list (lambda (state)
	     (do-slide (stack-depth (state-stack state)) 1 state))
	   (lambda (state)
	     (do-dead-code (do-return state))))
     state))

  ;; other side of the fence
  
  (defun do-allocate-function (lam state)
    ;;(format t "Alloc: ~a ~a~%" (lambda-inhibit-alloc lam) lam)
    (if (lambda-inhibit-alloc lam)
	(progn ;;(format t "Inhibit: ~a~%" lam)
	       (do-push-static nil state))
      (let ((s1 (do-push-label (read-init-label lam)
			       state))
	    (init-ilist (if t;; do we need to shove an env on the stack?
			    (list (lambda (state)
				    (fetch-environment (enclosing-lambda lam) state))
				  )
			  (list (lambda (state)
				  (do-push-static nil state))))))
	(do-allocate-closure (lambda-nargs lam)
			     (do-code-sequence init-ilist s1))
	)))
    
  (defun fetch-environment (lam state)
    ;; Should find the env of this function or block...
    ;; and place it on the top of the stack (maybe registerise if we're
    ;; feeling keen)
    ;;(format t "(Fetch env: ~a ~a" lam state)
    ;;(print-props lam)
    (let ((posn (scanq-stack (state-stack state) (stacked-lambda-env lam))))
      ;;(format t "at ~a" posn)
      (do-stack-ref posn state)
      ))

  (defun stacked-lambda-env (lam)
    (cond ((module-p lam) (lambda-env lam))
	  ((> (env-object-size (lambda-env lam)) 0)
	   (lambda-env lam))
	  (t (stacked-lambda-env (enclosing-lambda lam)))))
  
  ;; Macro lambdas...
  ;; do what we normally do, then turn it into a macro
  ;; the last bit is really just for the benifit of
  ;; the interpreter.

  (defmethod generic-code-gen ((x macro-lambda-term) state)
    (let ((state (call-next-method)))
      (let ((s1 (do-push-static bc-macro-type state)))
	(do-inline-code '((i-set-type)) 2 s1))))

  ;; Blocks...
  ;; rely on code-gen-for-decl.
  ;; 
  (defgeneric generic-code-gen-for-decl (decl state))

  (defun code-gen-for-decl (decl state)
    ;;(format t "Generating code for decl: ~a~%" decl)
    (generic-code-gen-for-decl decl state))

  (defmethod generic-code-gen ((blk block-term) state)
    (let ((state-locs (code-gen-for-decl (block-decl blk) state)))
      (let ((state (cdr state-locs))
	    (posns (car state-locs)))
	(let ((state (code-gen (block-body blk) state)))
	  (if (term-tail-call blk)
	      state
	    (fold delete-decl posns state))))))
  
  (defun delete-decl (posn state)
    (let ((offset (- (stack-depth (state-stack state)) posn)))
      (do-slide (+ offset 1) offset state)))
  
  ;;   for normal lets, stuff each arg onto the stack in turn
  ;;   recursive lets: allocate space for the objects all at once
  
  (defmethod generic-code-gen-for-decl ((decl and-decl) state)
    ;; over simple. I could be real cunning here.
    (fold (lambda (decl state)
	    (let ((aa (code-gen-for-decl decl (cdr state))))
	      (cons (append (car aa) (car state))
		    (cdr aa))))
	  (and-decl-decls decl)
	  (cons nil state)))

  (defmethod generic-code-gen-for-decl ((rec rec-decl) state)
    (code-gen-for-decl (rec-decl-decl rec)
		       state))
    
  (defmethod generic-code-gen-for-decl ((defn definition) state)
    ;; XX should map this down as a post-pass to annotate
    (let ((state (code-gen (defn-body defn) state)))
      (if (not (binding-needed-p defn))
	  (let ((state (do-pop 1 state)))
	    (format t "defn: Thrown away: ~a~%" defn)
	    (cons nil state))
	(if (binding-closed defn)
	    (cons nil (put-defn defn state))
	  (let ((state (name-stack-top defn state)))
	    (cons (list (stack-depth (state-stack state)))
		  state))))))

  (defgeneric inhibit-alloc (x)
    methods ((((x lambda-term))
	      nil
	      (format t "inhibit-set: ~a~%" x)
	      ((setter lambda-inhibit-alloc) x t)
	      t
	      )
	     (((x syntax-obj))
	      nil)))

  (defun put-defn (defn state)
    (let ((posn (binding-posn defn)))
      (do-code-sequence 
       (list (lambda (state)
	       (fetch-environment (enclosing-lambda defn) state))
	     do-swap
	     (lambda (state)
	       (do-setter-env-ref 0 posn state))
	     (lambda (state) 
	       (do-pop 1 state)))
       state)))

  (defmethod generic-code-gen-for-decl ((defn module-definition) state)
    (format t "~a " (defn-ide defn))
    (cons nil (do-global-set (external-name defn) (code-gen (defn-body defn) state))))

  
  (defun name-stack-top (name state)
    ;;(format t "Name tos: ~a ~a~%" name state)
    (modify-compiler-state 
     state 
     'state-stack
     (stack-push (stack-pop (state-stack state) 1)
		 name)))


  ;; Statics.

  (defmethod generic-code-gen ((x literal-term) state)
    (let ((state (if (eq (class-of (literal-content x)) integer)
		     (do-push-fixnum (literal-content x) state)
		   (do-push-static (literal-content x) state))))
      (if (term-tail-call x)
	  (add-tidy-code (enclosing-lambda x) state)
	state)))
  

  ;; Exports (we just ignore them)
  (defmethod generic-code-gen ((x export-spec) state)
    (do-push-static nil state))


   ;; end module
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; OLD function call methods....
;;
;; Nasty

  (defmethod generic-code-gen ((applic applic-term) state)
    (let ((obj (find-fn (applic-fun applic)))
	  (lab (make-label state))
	  (tail-flag (term-tail-call applic)))
      (check-arguments applic obj)
      (do-callbacks (function-fn obj) applic)
      ;;(format t "*Found function: ~a-->~a~%" (applic-fun applic) obj)
      ;; make the code for the call
      (if tail-flag
	  (add-tail-call-code applic lab obj state)
	(add-std-call-code applic lab obj state))))
  
  ;; paranoia
  (defun check-arguments (applic obj)
    (let ((nargs (list-length (applic-args applic)))
	  (reqd-nargs (function-nargs obj)))
      (if (or (eq (function-type obj) 'unknown)
	      (= (cdr reqd-nargs) 9999)
	      (= nargs (cdr reqd-nargs))
	      (and (car reqd-nargs)
		   (>= (+ nargs 1) (cdr reqd-nargs))))
	  t
	(error "Function called with wrong number of args" 
		Compile-Time-Error
		'values (list reqd-nargs applic)
		'msg "Function called with wrong number of args (should be ~a): ~%~a~%"))))

  (defconstant find-callback (mk-finder))

  (defgeneric do-callbacks (obj applic)
    methods ((((x imported-definition) applic)
	      (labels ((do-callback (l)
				    (cond ((null l) nil)
					  (t ((find-callback (car l)) applic)
					     (do-callback (cdr l))))))
		      (do-callback (import-prop-ref x 'callbacks))))
	     (((x syntax-obj) y)
	      nil)))
  
  ((setter find-callback) 'set-setter
   (lambda (applic)
     ;;(format t "set-setter: ~a~%" applic)
     (if (not (ident-p (cadr (applic-args applic))))
	 nil
       ((setter obj-setter-decl) 
	(ident-decl (car (applic-args applic)))
	(ident-decl (cadr (applic-args applic)))))))

  (defun add-std-call-code (applic label obj state)
    (do-code-sequence 
     (list 
      ;; entry code
      (lambda (state)
	(do-compute-fn applic obj state))
      ;; ho hum
      (if (eq (function-type obj) 'inline)
	  (lambda (state) state)
	(lambda (state)
	    (do-push-label label state)))
      ;; args
      (lambda (state)
	(push-fn-args (applic-args applic) obj
		      state))
      (lambda (state)
	(if (eq (function-type obj) 'inline)
	    state
	  (do-stack-ref (+ (actual-args applic obj) 1) state)))
      (lambda (state)
	(do-apply-function applic obj state))
      (lambda (state)
	(do-add-label label state))
      (lambda (state)
	(modify-compiler-state
	 state
	 'state-stack 
	 (compute-final-stack applic obj state))))
     state))

  (defun add-tail-call-code (applic label obj state)
    (do-code-sequence 
     (list (lambda (state)
	     (do-compute-fn applic obj state))
	   (lambda (state)
	     (push-fn-args (applic-args applic)
			   obj
			   state))
	   (lambda (state)
	     (do-tidy-tail-call applic obj state))
	   (lambda (state)
	     (do-apply-function applic obj state))
	   )
     state))

  (defun std-compute-fn (applic obj state)
    (code-gen (applic-fun applic) state))

  (defun push-args (args obj state)
    ;; should do nary-check here
    (if (car (function-nargs obj))
	(push-nary-args (cdr (function-nargs obj)) args state)
      (fold (lambda (arg state)
	      (let ((xx (code-gen arg state)))
		xx))
	    args
	    state)))

  (defun push-nary-args (nargs args state)
    ;; keep pushing till we get to optionals
    (if (= nargs 1)
	(push-optional-args args state)
      (push-nary-args (- nargs 1) (cdr args)
		      (code-gen (car args) state))))
  
  (defun push-optional-args (args state)
    (if (null args)
	(do-push-static nil state)
      (do-code-sequence 
       (list (lambda (state)
	       (code-gen (car args) state))
	     (lambda (state)
	       (do-push-static nil state))
	     (lambda (state)
	       (do-cons state))
	     (lambda (state)
	       (do-stack-ref 0 state))
	     (lambda (state)
	       (push-remaining-args (cdr args) state)))
       state)))
  
  (defun push-remaining-args (args state)
    (if (null args)
	(do-pop 1 state)
      (push-remaining-args (cdr args)
			   (do-code-sequence 
			    (list (lambda (state)
				    (code-gen (car args) state))
				  (lambda (state)
				    (do-push-static nil state))
				  (lambda (state)
				    (do-cons state))
				  (lambda (state)
				    (do-setter-cdr state)))
			    state))))

  ;; actual args pushed:
  (defun actual-args (applic obj)
    (if (car (function-nargs obj))
	(cdr (function-nargs obj))
      (list-length (applic-args applic))))

  (defun mk-calltype (applic obj)
    (if (car (function-nargs obj))
	(- (cdr (function-nargs obj)))
      (list-length (applic-args applic))))
				
  ;; state is: fn args... 

  (defun tidy-std-tail-call (applic obj state)
    (do-code-sequence 
     (list
      ;; function at the top...
      (lambda (state)
	(do-stack-ref (actual-args applic obj) state))
      (lambda (state)
	(do-stack-ref 0 state))
      ;; trash the fn. frame of self
      (lambda (state)
	(do-set-stack-ref (+ (stack-depth (state-stack state)) 1) state))
      ;; slide down
      (lambda (state)
	(do-slide (stack-depth (state-stack state))
		  (+ (actual-args applic obj) 1)
		  state)))
      state))

  (defun apply-bytefunction (applic obj state)
    (do-apply-bvf (mk-calltype applic obj) state))

  (defun apply-any (applic obj state)
    (do-apply-any (mk-calltype applic obj) state))

  (defun apply-methods (applic obj state)
    ;;(format t "Apply methods: ~a~%" state)
    (do-apply-methods (mk-calltype applic obj)
		      state))

  (defun compute-std-final-stack (applic obj state)
    (let ((stack (state-stack state)))
      (stack-push (stack-pop stack
			     (+ (actual-args applic obj) 3))
		  (make-stack-val))))

  (defun apply-inline-call (applic obj state)
    (let ((state (do-inline-code (import-prop-ref (function-fn obj) 'code) 
				 (actual-args applic obj)
				 state)))
      (if (term-tail-call applic)
	  (add-tidy-code (enclosing-lambda applic) state)
	state)))

  ;; slide down+ return...
  (defun tidy-inline-call (applic obj state)
    (do-slide (stack-depth (state-stack state))
	      (actual-args applic obj)
	      state))

  ;; for the time being...
  (defun compute-no-function (applic obj state)
    (code-gen (applic-fun applic) state))

  (defun apply-local-fn (applic obj state)
    (do-code-sequence 
     ;; XXX 0 is the posn of the environment of a function
     ;; unfortunately, we ain't calculated its env yet, so just push it.
     (list (lambda (state)
	     (do-slot-ref 0 state))
	   (lambda (state)
	     (do-branch (read-init-label (car obj)) state)))
     state))

  ;; Apply-self: call by shoving env
  (defun apply-self (applic obj state)
    (let ((s1 (stack-enclosing-env applic state)))
      (do-branch (read-init-label (car obj)) s1)))

  (defun stack-enclosing-env (applic state)
    ;;(print (list applic state))
    (let ((enc (enclosing-lambda applic)))
      (let ((env (stacked-lambda-env enc))
	    ;; for other locals, could use (enclosing-lambda (car obj)).
	    (e2 (stacked-lambda-env (enclosing-lambda enc))))
	;; XXX really a check for non-existent module-environment
	(if (= (env-object-size e2) 0) 
	    (do-push-static nil (do-pop 1 state))
	  (do-code-sequence 
	   (if (term-tail-call applic) 
	       (list (lambda (state)
		       (name-stack-top env (do-slot-ref 0 state))))
	     (cons (lambda (state)
		     (fetch-environment enc (do-pop 1 state)))
		   (if (eq env e2) nil
		     (list 
		      (lambda (state)
			(do-pop-env (find-env-depth env e2) state))))))
	   state)))))
	  
      

  (defconstant find-fns (mk-finder))

  (defun find-fn-computer (obj)
    (car (find-fns (function-type obj))))

  (defun find-apply-fn (obj)
    (cadr (find-fns (function-type obj))))
  
  (defun find-tidy-fn (obj)
    (caddr (find-fns (function-type obj))))
  
  (defun find-arg-pusher (obj)
    (nth 3 (find-fns (function-type obj))))

  (defun find-stack-computer (obj)
    (nth 4 (find-fns (function-type obj))))
  
  (defun do-compute-fn (applic obj state)
    ((find-fn-computer obj) applic obj state))

  (defun push-fn-args (applic obj state)
    ((find-arg-pusher obj) applic obj state))

  ;; fns called by applic generators
  (defun compute-final-stack (applic obj state)
    ((find-stack-computer obj) applic obj state))

  (defun do-tidy-tail-call (applic obj state)
    ((find-tidy-fn obj) applic obj state))

  (defun do-apply-function (applic obj state)
    ((find-apply-fn obj) applic obj state))

  ;; description of the calling procedures 
  ((setter find-fns) 'bytefunction
   (list std-compute-fn 
	 apply-bytefunction tidy-std-tail-call
	 push-args  compute-std-final-stack))

  ((setter find-fns) 'unknown
   (list std-compute-fn 
	 apply-any tidy-std-tail-call
	 push-args  compute-std-final-stack))

  ((setter find-fns) 'function
   (list std-compute-fn 
	 apply-any tidy-std-tail-call
         push-args compute-std-final-stack))

  ((setter find-fns) 'inline
   (list (lambda (applic obj state) state)
	 apply-inline-call
	 tidy-inline-call
	 push-args
	 (lambda (applic obj x) (state-stack x))))
  
  ((setter find-fns) 'cnm
   (list std-compute-fn
	 apply-methods
	 tidy-std-tail-call
	 push-args
	 compute-std-final-stack))
  
  ((setter find-fns) 'local-defun 
   (list compute-no-function
	 apply-local-fn
	 tidy-std-tail-call
	 push-args
	 compute-std-final-stack))
  
  ((setter find-fns) 'self-call
   (list compute-no-function
	 apply-self
	 tidy-std-tail-call
	 push-args
	 compute-std-final-stack))

