/**************************************************************************/
/* kinematic.c                                                  /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/* Author: P. Patrick van der Smagt                          _  \/\/  _   */
/*         University of Amsterdam                          | |      | |  */
/*         Dept. of Computer Systems                        | | /\/\ | |  */
/*         Amsterdam                                        | | \  / | |  */
/*         THE NETHERLANDS                                  | | /  \ | |  */
/*         smagt@fwi.uva.nl                                 | | \/\/ | |  */
/*                                                          | \______/ |  */
/* This software has been written with financial             \________/   */
/* support of the Dutch Foundation for Neural Networks                    */
/* and is therefore owned by the mentioned foundation.          /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/*                                                              \/\/      */
/**************************************************************************/

/*
 * WARNING: This file contains lots of OSCAR-specific code.
 *
 * This code is situtated in:
 *
 *	1. arrays max_angle[], min_angle[];
 *	2. couple_joints() and decouple_joints(), which are OSCAR specific;
 *	3. move(), which uses CCD_CAMERA_DISTANCE and the last joint
 *	   to calculate the distance camera---object;
 *	4. newton_euler(), which assumes that no prismatic joints are used;
 *	5. independent_newton_euler(), which isn't used anyway.
 */

#include "config.h"

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include "global.h"
#include "global_simmel.h"
#include "kinematic.h"
#include "matrix.h"
#include "vector.h"
#include "main.h"
#include "communication.h"
#include "io.h"
#include "ieee.h"




unsigned short check_limits(joint_values q, REAL *min_joint, REAL *max_joint)
{
  static REAL max_angle[NR_LINKS+1] =
	{0.0,  180.0,   90.0, 180.0,  90.0, 100.0,  180.0};
  static REAL min_angle[NR_LINKS+1] =
	{0.0, -180.0, -180.0,   -1.0, -90.0, -90.0, -180.0};
  unsigned short j;

  for (j=1; j<=NR_LINKS; j++)
	if (cisneg(q[j] - min_angle[j]) || cispos(q[j] - max_angle[j]))
	{
		*min_joint = min_angle[j]; *max_joint = max_angle[j];
		return j;
	}

  return 0;
}



int check_end_effector(REAL x, REAL y, REAL z)
{
  /*
   * x, y, z limits are as follows:
   * if z > 57, all is safe; otherwise, if z > 24 and
   * x > 16, the position is legal.
   *
   * For really safe measures yet low computation cost,
   * we calculate the position of frame 5 (i.e., the end
   * of link 4) and will not allow any world object approaching
   * that within 21 cm, the length of link 6.  Btw., this
   * _should_ be a circle.
   */
  if (z < 57.0 && (z < 0.0 || Msqr(x) + Msqr(y) < Msqr(37.0)))
	return 1;

  return 0;
}








/*
 * forward_kinematics gets as parameters the number of links in the robot and
 * an array of doubles representing the joint variables.
 *
 *                     i-1
 * Calculates matrices    A  which is the location of coordinate frame i
 *                         i 
 *                                                     0
 * with respect to frame i-1; finally this leads up to  A  which is the 
 *                                                       i
 * location of frame i with respect to the base frame.
 */

void forward_kinematics(joint_values target_position)
{
  homo m1, m2, m3;
  REAL *p1, *p2, *p3, *res;
  REAL theta, d, a, alfa;
  int i;

  p1 = m1; p2 = m2; p3 = m3;
  init_matrix(p1, HOMO_DEG, HOMO_DEG);

  /*
   * In the neural network code and camera/robot simulator, joint
   * values are target_position[1]...target_position[6].  This
   * simulator expects target_position[0]...target_position[5].
   */
   target_position++;

   /*
    * Fill in the values...
    */
   substitute(target_position);

   /* loop through the links */
   for (i=1; i<=NR_LINKS ; i++)
   {
	theta = get_param(i, DH_THETA);
	d = get_param(i, DH_D);
	a = get_param(i, DH_A);
	alfa = get_param(i, DH_ALFA);
	
	dh_transformation(p2, theta, d, a, alfa);
	mul_matrix(p1, HOMO_DEG, HOMO_DEG, p2, HOMO_DEG, HOMO_DEG,
			p3, HOMO_DEG, HOMO_DEG);

	/*                 0
	 * joint_homo[i] =  A
	 *                   i
	 */
	copy_homo(joint_homo[i], p3);

	res = p1;
	p1 = p3;
	p3 = res;
  }
}


void display_robot(void)
{
  homo object_homo;
  unsigned short i;

  /*
   * A waste to do all this computation if no bemmel
   * is connected anyway.
   */
  if (!bemmels_connected)
	return;

  for (i=1; i<=NR_LINKS; i++)
	send_homo(i, joint_homo[i]);

  init_matrix(object_homo, HOMO_DEG, HOMO_DEG);

  object_homo[3] = object_position[0];
  object_homo[7] = object_position[1];
  object_homo[11] = object_position[2];

  send_homo(NR_LINKS+1, object_homo);
}



void init_forwardkinematics(char *robotdefinition)
{
  read_definition(robotdefinition);
}



/*
 * In OSCAR, joint 3 is coupled to joint 2, such that its
 * position is always measured relative to the world and
 * not to joint 2.  The simulator doesn't agree with this.
 * Before sending joint values to the neural network, couple
 * the joints (see def.h for a layout of the joint values).
 */
void couple_joints(joint_values q, joint_values dq, joint_values ddq,
		   joint_values u_q, joint_values u_dq, joint_values u_ddq)
{
  unsigned short i;
  for (i=1; i<=NR_LINKS; i++)
  {
	q[i] = u_q[i];
	dq[i] = u_dq[i];
	ddq[i] = u_ddq[i];
  }
  q[3] += u_q[2];
  dq[3] += u_dq[2];
  ddq[3] += u_ddq[2];
}


/* 
 * Perform an action opposite to that in couple_joints, after
 * having received a set of joint values from the neural
 * network.
 */
void decouple_joints(joint_values u_q, joint_values u_dq, joint_values u_ddq,
		joint_values q, joint_values dq, joint_values ddq)
{
  unsigned short i;
  for (i=1; i<=NR_LINKS; i++)
  {
	u_q[i] = q[i];
	u_dq[i] = dq[i];
	u_ddq[i] = ddq[i];
  }
  u_q[3] -= q[2];
  u_dq[3] -= dq[2];
  u_ddq[3] -= ddq[2];
}



/*
 * Move the robot along the specified trajectory.   Returns the number
 * of joints that are still in motion.
 */
unsigned short move(joint_values q_1, joint_values dq_1, joint_values ddq_1)
{
  unsigned short i, error_joint;
  unsigned short moving = 0;
  joint_values prev_q, prev_dq, tmp_q, tmp_dq, tmp_ddq;
  REAL err_up, err_low;
  static joint_values prev_ddq;

  for (i=1; i<=NR_LINKS; i++)
  {
	prev_q[i] = q[i];
	prev_dq[i] = dq[i];

	q[i] += dq[i] + 0.5*ddq[i];
	dq[i] += ddq[i];

	/*
	 * Check if target has been reached for this joint.
	 * This can be checked by looking at the current and
	 * previous position of the joint.
	 */
	if ((cispos(q[i]-q_1[i]) && cisneg(prev_q[i]-q_1[i])) ||
	    (cisneg(q[i]-q_1[i]) && cispos(prev_q[i]-q_1[i])))
	{
		dq[i] = dq_1[i] = ddq[i] = ddq_1[i] = 0.0;
		q[i] = q_1[i];
		continue;
	}

	/*
	 * If the velocity and acceleration for this
	 * joint are 0, we can never reach the target.
	 * Give up for this joint.
	 */
	if (iszero((REAL) dq_1[i]) && iszero((REAL) ddq_1[i]))
		continue;
	
	moving++;

	/*
	 * Check if target speed has been reached for this joint.
	 * Stop acceleration.
	 *
	 * This method will fail with small accelerations when
	 * the joint moves at constant velocity.  Therefore, 
	 * first check if the acceleration has just been
	 * given a non-zero value; if so, skip this test.
	 */
	if (!ciszero(prev_ddq[i]))
	if ((cispos(dq[i]-dq_1[i]) && cisneg(prev_dq[i]-dq_1[i])) ||
	    (cisneg(dq[i]-dq_1[i]) && cispos(prev_dq[i]-dq_1[i])))
	{
		ddq_1[i] = ddq[i] = 0.0;
		dq[i] = dq_1[i];
	}
	prev_ddq[i] = ddq[i];
  }

  /*
   * Check if boundaries are exceeded.  If so, immediately
   * halt the robot.
   */
  if (error_joint = check_limits(q, &err_up, &err_low))
  {
	printf("joint %d exceeded limits\n", error_joint);
	for (i=1; i<=NR_LINKS; i++)
	{
		dq[i] = dq_1[i] = ddq[i] = ddq_1[i] = 0.0;
		q[i] = prev_q[i];
	}
	moving = 0;
  }

  decouple_joints(tmp_q, tmp_dq, tmp_ddq, q, dq, ddq);
  forward_kinematics(tmp_q);

  if (check_end_effector(joint_homo[NR_LINKS][3],
	joint_homo[NR_LINKS][7],
	joint_homo[NR_LINKS][11]))
  {
	printf("illegal end-effector position\n");
	for (i=1; i<=NR_LINKS; i++)
	{
		tmp_dq[i] = dq[i] = dq_1[i] = 0.0;
		tmp_ddq[i] = ddq[i] = ddq_1[i] = 0.0;
		q[i] = prev_q[i];
	}

	/*
	 * Re-calculate the forward kinematics, which happens
	 * to be the same one as before the move, but what
	 * the heck.
	 */
	decouple_joints(tmp_q, tmp_dq, tmp_ddq, q, dq, ddq);
	forward_kinematics(tmp_q);
	moving = 0;
  }

  display_robot();
  newton_euler(tmp_dq, tmp_ddq);
  relative_object_motion();
  camera();

  return moving;
}




/*
 * Calculate all the camera views.
 */
void camera(void)
{
  vector eef_view, object_view, tip_position;
  REAL h1, h2;

  /*
   * Note that:
   *	cv[2] = d, the distance between camera and object.  Is positive
   *		when the camera is above the object;
   *	v[6][2]=d', the rate with which d increases.  Is negative
   *		when the camera increases its velocity towards the
   *		object and approaches from above;
   *	a[6][2]=d'', the rate with which v increases.  Is positive
   *		when the brakes are put on.
   *
   * The functions newton_euler() and relative_object_motion()
   * should be called before this function!
   */

  /*
   * Should this variable be called `effectuator?'
   *
   * Laziness prevents me from using #define's to define
   * positions 3, 7, and 11 as the translation vector.
   */
  effector[0] = joint_homo[NR_LINKS][3];
  effector[1] = joint_homo[NR_LINKS][7];
  effector[2] = joint_homo[NR_LINKS][11];
  /*
   * The camera CCD is not placed exactly at the place where we
   * set the distance to 0; it is somewhat higher.
   *
   * Consequently, the area of the object can be negative!
   */
  h1 = hand_object_position[2] + CCD_FINGER_DISTANCE;

  h2 = v[NR_LINKS][2] / h1;
  area[0] = E * Msqr(focus) / Msqr(h1);
  if (signbit(h1))
	area[0] = -area[0];
  area[1] = -2.0 * area[0] * h2;
  area[2] = 2.0 * area[0] * (3.0 * Msqr(h2) - a[NR_LINKS][2]/h1);



  /*
   * Next the view by the fixed cameras.  We define a position in the tip
   * of the end-effector---by default, at position (0,0,0) relative to the
   * end-effector coordinate frame.
   */ 
  tip_position[0] = 0.0;
  tip_position[1] = 0.0;
  tip_position[2] = 0.0;
  tip_position[3] = 1.0;

  homo_vec_mul(eef_view, joint_homo[NR_LINKS], tip_position);
  homo_vec_mul(object_view, joint_homo[NR_LINKS], hand_object_position);

  homo_vec_mul(cam_1_eef, cam_1_inv_homo, eef_view);
  homo_vec_mul(cam_2_eef, cam_2_inv_homo, eef_view);
  homo_vec_mul(cam_1_obj, cam_1_inv_homo, object_view);
  homo_vec_mul(cam_2_obj, cam_2_inv_homo, object_view);
}




/*
 * This is the Newton-Euler method for iterative numeric approximation
 * of the Jacobian and Hessian of the manipulator.  I chose for this
 * method since it solves both problems and works for each and
 * every DH matrix.  See Fu, Gonzalez, & Lee, pp. 554--555, pp. 108--124.
 *
 * Note: works only for rotary joints.  To adapt this for prismatic
 * joints, see Fu, Gonzalez, & Lee.
 *
 * The results are stored in the global variables p[][], v[][], and a[][].
 *
 * Other variables:
 *	p_star = the location of coordinate frame i with respect to i-1
 *	w_star = the angular velocity ...
 */
void newton_euler(joint_values dq, joint_values ddq)
{
  unsigned short i;
  vector z_0, z_i_1, v0, v1, v2, v3, v4, p_star;
  homo h0;

  for (i=0; i<=NR_LINKS; i++)
  {
	init_vector(omega[i]);
	init_vector(d_omega[i]);
	init_vector(v[i]);
	init_vector(a[i]);
  }
  init_matrix(joint_homo[0], HOMO_DEG, HOMO_DEG);


  for (i=1; i<=NR_LINKS; i++)
  {
	dh_transformation(h0,
		get_param(i, DH_THETA), get_param(i, DH_D),
		get_param(i, DH_A), get_param(i, DH_ALFA));

	/*
	 * Save position of the current link.
	 */
	v0[0] = h0[3];
	v0[1] = h0[7];
	v0[2] = h0[11];
	v0[3] = 0.0;
	homo_vec_mul(p_star, joint_homo[i-1], v0); p_star[3] = 0.0;

	/*
	 * Just for fun, save the position of each link
	 * in the global variable p.
	 */
	p[i][0] = joint_homo[i][3];
	p[i][1] = joint_homo[i][7];
	p[i][2] = joint_homo[i][11];
	p[i][3] = 0.0;

	z_0[0] = 0.0; z_0[1] = 0.0; z_0[2] = 1.0; z_0[3] = 0.0;
	homo_vec_mul(z_i_1, joint_homo[i-1], z_0);
	z_i_1[3] = 0.0;

	/*                 .                                     *
	 * w  = w    + z   q	(note that the last term equals w  !)
	 *  i    i-1    i-1 i                                    i
	 */
	v0[0] = M_PI/180.0*dq[i] * z_i_1[0];
	v0[1] = M_PI/180.0*dq[i] * z_i_1[1];
	v0[2] = M_PI/180.0*dq[i] * z_i_1[2];
	v0[3] = 0.0;
	omega[i][0] = v0[0] + omega[i-1][0];
	omega[i][1] = v0[1] + omega[i-1][1];
	omega[i][2] = v0[2] + omega[i-1][2];
	omega[i][3] = 0.0;

	/*                 * *
	 *           ..   d w             .     *
	 * Here, z   q  = ---i-  and  z   q  = w   (for rotational links):
	 *        i-1 i    dt          i-1 i    i
	 *
	 * .    .          ..              .
	 * w  = w    + z   q  + w   x (z   q )
	 *  i    i-1    i-1 i    i-1    i-1 i
	 */
	cross_product(v1, omega[i-1], v0); v1[3] = 0.0;
	d_omega[i][0] = d_omega[i-1][0] + z_i_1[0]*M_PI/180.0*ddq[i] + v1[0];
	d_omega[i][1] = d_omega[i-1][1] + z_i_1[1]*M_PI/180.0*ddq[i] + v1[1];
	d_omega[i][2] = d_omega[i-1][2] + z_i_1[2]*M_PI/180.0*ddq[i] + v1[2];
	d_omega[i][3] = 0.0;

	/*                *
	 * v  = v    + w p
	 *  i    i-1    i i
	 */
	cross_product(v2, omega[i], p_star); v2[3] = 0.0;
	v[i][0] = v[i-1][0] + v2[0];
	v[i][1] = v[i-1][1] + v2[1];
	v[i][2] = v[i-1][2] + v2[2];
	v[i][3] = 0.0;

	/* .    .      .    *             *
	 * v  = v    + w x p  + w x (w x p )
	 *  i    i-1    i   i    i    i   i
	 */
	cross_product(v3, omega[i], v2); v3[3] = 0.0;
	cross_product(v4, d_omega[i], p_star); v4[3] = 0.0;
	a[i][0] = v4[0] + v3[0] + a[i-1][0];
	a[i][1] = v4[1] + v3[1] + a[i-1][1];
	a[i][2] = v4[2] + v3[2] + a[i-1][2];
	a[i][3] = 0.0;
  }
}



/*
 * Given the Newton-Euler angular velocity and acceleration
 * computations, this function computes the position, velocity,
 * and acceleration of the object with respect to the hand
 * coordinate frame.
 *
 *            * *
 *           d r          *   i
 *   v(t) = ----- +  w x r  -  A dh/dt
 *            dt                0
 *
 * where v(t) is the object velocity (relative to frame 0),
 * r* is the object position (relative to the hand coordinate frame),
 * and h is the position of the hand frame relative to the
 * base frame.  We are interested in d*r* / dt.
 *
 * This is conform Fu et al., p. 107.  Note that in that text,
 * eq. (3.3-15) r = r* + h expresses all of those vectors
 * in coordinate frame 0, even though r* is relative to the
 * starred (moving) coordinate frame.  Hence the A matrix above.
 *
 * The code below does not include the velocity of the object,
 * i.e., v(t) = 0 is chosen.
 *
 *
 * Ditto for the acceleration:
 *
 *            *2 *           * *                                    2
 *           d  r           d r                 *      dw    *     d h
 *   a(t) = ----2- + 2 w x ------  +  w x (w x r )  +  -- x r  -  ---2-
 *            dt             dt                        dt          dt
 *
 * but again the A matrix has to be added to translate the h vector
 * to the hand coordinate frame.  Here, also, I take a(t)=0 and
 * I'm interested in the first term of the RHS.
 */
void relative_object_motion(void)
{
  homo hand;
  vector v0, v1, v2, v3, v4, v5;
  int i;
  
  /*
   * The velocity.
   */
  invert_homo(hand, joint_homo[NR_LINKS]);
  homo_vec_mul(hand_object_position, hand, object_position);
  homo_vec_mul(v4, hand, v[NR_LINKS]);
  cross_product(v0, omega[NR_LINKS], hand_object_position);
  
  for (i=0; i<=2; i++)
    hand_object_velocity[i] = v0[i] - v4[i];

  /*
   * The acceleration.
   */
  cross_product(v1, omega[NR_LINKS], hand_object_velocity);
  cross_product(v2, omega[NR_LINKS], v0);
  cross_product(v3, d_omega[NR_LINKS], hand_object_position);
  homo_vec_mul(v5, hand, a[NR_LINKS]);
  for (i=0; i<=2; i++)
    hand_object_acceleration[i] = 2.0 * v1[i] + v2[i] + v3[i] - v5[i];
}




/*
 * Check if the number x is near enough to 0.  Incoming data (via
 * the socket) have rather low precision; therefore < 0.001 is
 * interpreted as near enough to 0.
 */
int ciszero(REAL x)
{
  double rounded_x;

  rounded_x = 0.0001 * ((int) (0.5 + x*10000.0));

  return iszero(rounded_x);

}


/*
 * Check if the number x is near enough to 0 or positive.  Incoming data (via
 * the socket) have rather low precision; therefore < 0.001 is
 * interpreted as near enough to 0.
 */
int cispos(REAL x)
{
  double rounded_x;

  rounded_x = 0.0001 * ((int) (0.5 + x*10000.0));

  return (iszero(rounded_x) || !signbit(rounded_x));
}



/*
 * Check if the number x is near enough to 0 or negative.  Incoming data (via
 * the socket) have rather low precision; therefore < 0.001 is
 * interpreted as near enough to 0.
 */
int cisneg(REAL x)
{
  double rounded_x;

  rounded_x = 0.0001 * ((int) (0.5 + x*10000.0));

  return (iszero(rounded_x) || signbit(rounded_x));
}




/*
 * Graveyard of unused routines.  They are so much unused that
 * they are not compiled.
 */

#if 0
/* 
 * The same as above, but calculating the motion of a link
 * about its own coordinate frame.
 */
void independent_newton_euler(joint_values dq, joint_values ddq)
{
  unsigned short i;
  REAL theta, d, a, alfa;
  vector omega[7], d_omega[7], d_v[7];
  vector v0, v1, v2, v3, v4, v5, v6, v7, v8, v9, p_star;
  homo h0, h1;
  homo i_1_R_0, i_R_0, i_R_i_1;

  for (i=0; i<=NR_LINKS; i++)
  {
	init_vector(omega[i]);
	init_vector(d_omega[i]);
	init_vector(d_v[i]);
  }
  init_homo(&h0);
  init_homo(&h1);

  for (i=1; i<=NR_LINKS; i++)
  {
	/*                  i-1    i        i
	 * First, calculate    R ,  R , and  R   .
	 *                      0    0        i-1
	 */
	invert_homo(&i_1_R_0, &(joint_homo[i-1]));
	invert_homo(&i_R_0, &(joint_homo[i]));

	theta = get_param( i , DH_THETA );
	d = get_param( i , DH_D );
	a = get_param( i , DH_A );
	alfa = get_param( i , DH_ALFA );
	dh_transformation( &h1 , theta , d , a , alfa );
	invert_homo(&i_R_i_1, &h1);

	/*
	 * Save position of the current link.
	 */
	p_star[0] = im(i_R_i_1, 0, HOMO_DEG-1);
	p_star[1] = im(h1, 1, HOMO_DEG-1);
	p_star[2] = im(h1, 2, HOMO_DEG-1);
	p_star[3] = 0.0;

	/* Next calculate omega[i]:
	 * i       i     i-1           .
	 *  R w  =  R   (   R w    + z q )
	 *   0 i     i-1     0 i-1    0 i
	 */

	/*        .
	 * v0 = z q
	 *       0 i
	 */
	init_vector(v0);
	v0[2] = dq[i];
	v0[3] = 0.0;

	/*      i-1
	 * v1 =    R w
	 *          0 i-1
	 */
	homo_vec_mul(v1, &i_1_R_0, omega[i-1]);
	v1[3] = 0.0;

	/*      i-1           .
	 * v2 =    R w    + z q
	 *          0 i-1    0 i
	 */
	add_vector(v2, v1, v0);
	v2[3] = 0.0;

	/*      i     i-1           .
	 * v3 =  R   (   R w    + z q )
	 *        i-1     0 i-1    0 i
	 */
	homo_vec_mul(v3, &i_R_i_1, v2);
	v3[3] = 0.0;

	/*      0  i     i-1           .
	 * w  =  R  R   (   R w    + z q )
	 *  i     i  i-1     0 i-1    0 i
	 */
	homo_vec_mul(omega[i], &(joint_homo[i]), v3);
	omega[i][3] = 0.0;



	/* Next calculate d_omega[i]:
	 * i  .    i     i-1  .       ..    i-1            .
	 *  R w  =  R   [   R w   + z q  + (   R w   ) x z q ]
	 *   0 i     i-1     0 i-1   0 i        0 i-1     0 i
	 */
	/*        .
	 * v0 = z q
	 *       0 i
	 */
	init_vector(v0);
	v0[2] = dq[i];
	v0[3] = 0.0;

	/*      i-1
	 * v1 =    R w
	 *          0 i-1
	 */
	homo_vec_mul(v1, &i_1_R_0, omega[i-1]);
	v1[3] = 0.0;

	/*       i-1            .
	 * v2 = (   R w   ) x z q
	 *           0 i-1     0 i
	 */
	cross_product(v2, v1, v0);
	v2[3] = 0.0;

	/*        ..
	 * v3 = z q
	 *       0 i
	 */
	init_vector(v3);
	v3[2] = ddq[i];
	v3[3] = 0.0;

	/*        ..    i-1            .
	 * v4 = z q  + (   R w   ) x z q
	 *       0 i        0 i-1     0 i
	 */
	add_vector(v4, v3, v2);
	v4[3] = 0.0;

	/*      i-1  .
	 * v5 =    R w
	 *          0 i-1
	 */
	homo_vec_mul(v5, &i_1_R_0, d_omega[i-1]);
	v5[3] = 0.0;

	/*      i-1  .       ..    i-1            .
	 * v6 =    R w   + z q  + (   R w   ) x z q ]
	 *          0 i-1   0 i        0 i-1     0 i
	 */
	add_vector(v6, v5, v4);
	v6[3] = 0.0;

	/*      i     i-1  .       ..    i-1            .
	 * v7 =  R   [   R w   + z q  + (   R w   ) x z q ]
	 *        i-1     0 i-1   0 i        0 i-1     0 i
	 */
	homo_vec_mul(v7, &i_R_i_1, v6);
	v7[3] = 0.0;

	/* .    0  i     i-1  .       ..    i-1            .
	 * w  =  R  R   [   R w   + z q  + (   R w   ) x z q ]
	 *  i     i  i-1     0 i-1   0 i        0 i-1     0 i
	 */
	homo_vec_mul(d_omega[i], &(joint_homo[i]), v7);
	d_omega[i][3] = 0.0;



	/* Finally, d_v:
	 * i  .     i  .      i   *     i   
	 *  R v  = ( R w ) x ( R p ) + ( R w ) x
	 *   0 i      0 i       0 i       0 i
	 *
	 *                   i         i   *    i     i-1  .
	 *               x [( R w ) x ( R p )] + R   (   R v   )
	 *                     0 i       0 i      i-1     0 i-1
	 */
	
	/*       i-1  .
	 * v0 = (   R v   )
	 *           0 i-1
	 */
	homo_vec_mul(v0, &i_1_R_0, d_v[i-1]);
	v0[3] = 0.0;

	/*      i     i-1  .
	 * v1 =  R   (   R v   )
	 *        i-1     0 i-1
	 */
	homo_vec_mul(v1, &i_R_i_1, v0);
	v1[3] = 0.0;

	/*       i   *
	 * v2 = ( R p )
	 *         0 i
	 */
	homo_vec_mul(v2, &i_R_0, p_star);
	v2[3] = 0.0;

	/*       i
	 * v3 = ( R w )
	 *         0 i
	 */
	homo_vec_mul(v3, &i_R_0, omega[i]);
	v3[3] = 0.0;

	/*        i         i   *
	 * v4 = [( R w ) x ( R p )]
	 *          0 i       0 i
	 */
	cross_product(v4, v3, v2);
	v4[3] = 0.0;

	/*       i          i         i   *
	 * v5 = ( R w ) x [( R w ) x ( R p )]
	 *         0 i        0 i       0 i
	 */
	cross_product(v5, v3, v4);
	v5[3] = 0.0;

	/*       i          i         i   *     i     i-1  .
	 * v6 = ( R w ) x [( R w ) x ( R p )] +  R   (   R v   )
	 *         0 i        0 i       0 i       i-1     0 i-1
	 */
	add_vector(v6, v5, v1);
	v6[3] = 0.0;

	/*       i  .
	 * v7 = ( R w )
	 *         0 i
	 */
	homo_vec_mul(v7, &i_R_0, d_omega[i]);
	v7[3] = 0.0;

	/*       i  .      i   *
	 * v8 = ( R w ) x ( R p )
	 *         0 i       0 i
	 */
	cross_product(v8, v7, v2);
	v8[3] = 0.0;

	/*       i  .      i   *     i   
	 * v9 = ( R w ) x ( R p ) + ( R w ) x
	 *         0 i       0 i       0 i
	 *
	 *                   i         i   *    i     i-1  .
	 *               x [( R w ) x ( R p )] + R   (   R v   )
	 *                     0 i       0 i      i-1     0 i-1
	 */
	add_vector(v9, v8, v6);
	v9[3] = 0.0;

	/* .    0  {  i  .      i   *     i   
	 * v  =  R { ( R w ) x ( R p ) + ( R w ) x
	 *  i     i{    0 i       0 i       0 i
	 *
	 *                   i         i   *    i     i-1  .     }
	 *               x [( R w ) x ( R p )] + R   (   R v   ) }
	 *                     0 i       0 i      i-1     0 i-1  }
	 */
	homo_vec_mul(d_v[i], &(joint_homo[i]), v9);
	d_v[i][3] = 0.0;
  }
}


REAL t_1[NR_LINKS+1], t_m[NR_LINKS+1];

REAL plan_motion(joint_values target_q, joint_values max_dq,
						joint_values new_ddq)
{
  unsigned int link;
  REAL t_1_a[NR_LINKS+1], t_1_b[NR_LINKS+1];
  REAL t_max;

  t_max = 0.0;
  /*
   * Knowing the target joint values, the maximum velocities, and
   * the accelerations of the joints we can determine the motion
   * profile.  First it has to be determined how long the motion
   * takes.  We will calculate the time of motion for each joint.
   */
  for (link=1; link<=NR_LINKS; link++)
  {
	/*
	 * Take acceleration 0 as acceleration 0, i.e.,
	 * not even a zillion steps suffice to get there.
	 */
	if (iszero((double) new_ddq[link]))
		t_1[link] = -1.0;
	else
	{
		/*
		 *                  a   q'- q'
		 * First calculate t  = -1---0-
		 *                  1     q''
		 *                         1
		 */
		t_1_a[link] = (max_dq[link] - dq[link]) / new_ddq[link];

		/*
		 *                           ___________
		 *                  b       /   q - q
		 * Then, calculate t  = \  / 2 --1---0-
		 *                  1    \/       q''
		 *                                 1
		 */
		t_1_b[link] = 2.0 * (target_q[link]-q[link]) / new_ddq[link];
		if (t_1_b[link] >= 0.0)
			t_1_b[link] = sqrt(t_1_b[link]);

		/*                    a   b
		 * Finally, t  = min(t , t )
		 *           1        1   1      a
		 * with this exception that, if t  is zero, an infinite
		 *                               1
		 * maximum velocity is assumed, such that t  is neglected.
		 *                                         1
		 */
		if (t_1_a[link] < 0.0)
			t_1[link] = t_1_b[link];
		else
			t_1[link] = min(t_1_a[link], t_1_b[link]);
	}

	/*
	 * If the maximum velocity is set to 0, assume 0.
	 */
	if (iszero((double) max_dq[link]))
		t_m[link] = -1.0;
	else
		t_m[link] = t_1[link] + ((target_q[link]-q[link])
				- 0.5*Msqr(t_1[link])*new_ddq[link]) /
				max_dq[link];

	if (t_m[link] > t_max)
		t_max = t_m[link];
  }
  return t_max;
}
#endif
