/**************************************************************************/
/* matrix.c --- after a version by F. J. Jungen                 /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/* Author: P. Patrick van der Smagt                          _  \/\/  _   */
/*         University of Amsterdam                          | |      | |  */
/*         Dept. of Computer Systems                        | | /\/\ | |  */
/*         Amsterdam                                        | | \  / | |  */
/*         THE NETHERLANDS                                  | | /  \ | |  */
/*         smagt@fwi.uva.nl                                 | | \/\/ | |  */
/*                                                          | \______/ |  */
/* This software has been written with financial             \________/   */
/* support of the Dutch Foundation for Neural Networks                    */
/* and is therefore owned by the mentioned foundation.          /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/*                                                              \/\/      */
/**************************************************************************/

#include "config.h"

#include <stdio.h>
#include <math.h>
#include "matrix.h"




#define deg2rad(q)	((q) * M_PI / 180.0)

/*
 * Initialise the matrix with the identity matrix.
 */
void init_matrix(REAL *M, int rows, int columns)
{
  int r, c;

  if (rows != columns)
  {
	fprintf(stderr, "init_matrix() has illegal matrix\n");
	panic_exit();
  }

  for (r = 0; r < rows; r++)
	for (c = 0; c < columns; c++)
		*M++ = (r==c) ? 1.0 : 0.0;
}




void mul_matrix(matrix A, int a_rows, int a_columns,
		matrix B, int b_rows, int b_columns,
		matrix C, int c_rows, int c_columns)
{
  REAL *p_a, *p_b, *p_c;
  int r, c, i;

  if (a_columns != b_rows || a_rows != c_rows || b_columns != c_columns)
  {
	fprintf(stderr, "Illegal matrix multiply in mul_matrix()\n");
	panic_exit();
  }

  p_c = C;
  for (r=0; r<a_rows; r++)
  {
	for (c=0; c<b_columns; c++)
	{
		p_a = A + r*a_columns;
		p_b = B + c;
		*p_c = 0;
		for (i=0; i<a_columns; i++)
		{
			*p_c += *(p_a++) * *p_b;
			p_b += b_columns;
		}
		p_c++;
	}
  }
}




void perspective_transform(homo M, REAL focus)
{
  int i, j;

  for (i=0; i<HOMO_DEG; i++)
	for (j=0; j<HOMO_DEG; j++)
		if (i == j)
			*(M++) = focus;
		else
			*(M++) = 0.0;
  *(M-2) = -1.0;

  /*
   * Result:
   *	1  0  0  0
   *	0  1  0  0
   *	0  0  1  0
   *	0  0 -1  1
   */
}



/*
 * PRE : M is a 4x4 matrix and theta and angle are in degrees.
 * POST: M contains the perspective transform depending on theta, angle and 
 *       distance. The viewer is located at (distance*cos( angle )*cos( theta )
 *	 , distance*cos( angle )*sin( theta ) , distance*sin( angle )).
 */

void viewpoint_transform(homo V, REAL theta, REAL angle, REAL distance)
{
  /* computation of V, the viewpoint transformation. */

  theta = deg2rad(theta);
  angle = deg2rad(angle);

  *(V++) = sin(-theta);
	*(V++) = cos(theta);
		*(V++) = 0.0;
			*(V++) = 0.0;
  *(V++) = -sin(angle) * cos(theta);
	*(V++) = -sin(angle) * sin(theta);
		*(V++) = cos(angle);
			*(V++) = 0.0;
  *(V++) = cos(angle) * cos(theta);
	*(V++) = cos(angle) * sin(theta);
		*(V++) = sin(angle);
			*(V++) = -distance;
  *(V++) = 0.0;
	*(V++) = 0.0;
		*(V++) = 0.0;
			*(V++) = 1.0;
}



void dh_transformation(homo M, REAL theta, REAL d, REAL a, REAL alpha)
{
  theta = deg2rad(theta);
  alpha = deg2rad(alpha);

  *M++ = cos(theta);
	*M++ = -sin(theta) * cos(alpha);
		*M++ = sin(theta) * sin(alpha);
			*M++ = a * cos(theta);
  *M++ = sin(theta);
	*M++ = cos(theta) * cos(alpha);
		*M++ = -cos(theta) * sin(alpha);
			*M++ = a * sin(theta);
  *M++ = 0.0;
	*M++ = sin(alpha);
		*M++ = cos(alpha);
			*M++ = d;
  *M++ = 0.0;
	*M++ = 0.0;
		*M++ = 0.0;
			*M++ = 1.0;
}


void invert_homo(homo HI, homo H)
{
  /*
   * Instead of writing two for-loops to solve this problem,
   * I chose for writing this one out---it's faster.
   */

  HI[0] = H[0];
  HI[1] = H[4];
  HI[2] = H[8];

  HI[4] = H[1];
  HI[5] = H[5];
  HI[6] = H[9];

  HI[8] = H[2];
  HI[9] = H[6];
  HI[10] = H[10];

  HI[12] = HI[13] = HI[14] = 0.0;
  HI[15] = 1.0;

  HI[3] = -HI[0] * H[3] - HI[1] * H[7] - HI[2] * H[11];
  HI[7] = -HI[4] * H[3] - HI[5] * H[7] - HI[6] * H[11];
  HI[11] = -HI[8] * H[3] - HI[9] * H[7] - HI[10] * H[11];
}



void copy_homo(homo D, homo S)
{
  int i;

  for (i=0; i<HOMO_DEG*HOMO_DEG; i++)
	*D++ = *S++;
}



/*
 * Print a homogenous matrix.
 *      In: the (homogenous) matrix H.
 *      Out:
 */
void print_matrix(homo H)
{
  printf("      %.2f %.2f %.2f %.2f\n   %.2f %.2f %.2f %.2f\n\
        %.2f %.2f %.2f %.2f\n   %.2f %.2f %.2f %.2f\n",
        H[0], H[1], H[2], H[3],
        H[4], H[5], H[6], H[7],
        H[8], H[9], H[10], H[11],
        H[12], H[13], H[14], H[15]);
}

