/**************************************************************************/
/* moving.c                                                     /\/\      */
/* Version 2.2.1 --  August  1991                               \  /      */
/*                                                              /  \      */
/* Author: P. Patrick van der Smagt                          _  \/\/  _   */
/*         University of Amsterdam                          | |      | |  */
/*         Dept. of Computer Systems                        | | /\/\ | |  */
/*         Amsterdam                                        | | \  / | |  */
/*         THE NETHERLANDS                                  | | /  \ | |  */
/*         smagt@fwi.uva.nl                                 | | \/\/ | |  */
/*                                                          | \______/ |  */
/* This software has been written with financial             \________/   */
/* support of the Dutch Foundation for Neural Networks                    */
/* and is therefore owned by the mentioned foundation.          /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/*                                                              \/\/      */
/**************************************************************************/
#include <stdio.h>
#include <math.h>

#include "global_communication.h"
#include "communication.h"
#include "data_code.h"

#include "moving.h"

#include "setup.h"


void reset_robot(void)
{
  joint_values q, dq, ddq;
  unsigned short i, faulty_joint;

  /*
   * Wait till robot is ready with its move and
   * then read the current position.
   *
   * The `waiting' is real easy: stop robot.
   */
  stop_robot();

  /*
   * Go to the `home' position.
   */
  if (NR_LINKS != 6)
  {
	fprintf(stderr,
		"Reset only coded for a 6 DoF robot---change reset_robot()\n");
	close_down(NULL, NULL);
  }

  q[1] = 0.0; q[2] = -70.0; q[3] = 90.0;
  q[4] = 0.0; q[5] = 180.0-q[3]; q[6] = 0.0;

  if (faulty_joint = move_robot(NULL_MOVE, q, 
  	NULL_MOVE, NULL_MOVE, FAST_DQ, FAST_DDQ))
  {
	fprintf(stderr, "reset_robot(): robot won't accept home position\n");
	fprintf(stderr, "Error in joint %d\n", faulty_joint);
	close_down(PANIC, PANIC);
  }

  /*
   * Next, make sure that the RESET is really performed:
   * wait until the move is ready.
   */
  get_joint_values(0, 1, q, dq, ddq);
}




/*
 * Move the plant according to delta_joints.
 * Returns 0 if everything ok, else the number (1...6) of
 * the joint that gave up.
 *
 * Has two modes of operation: if delta_q = NULL, then sends
 * the values in absolute_q; otherwise sends current_joints+delta_q.
 *
 * When the velocity dq or the acceleration ddq is NULL, it
 * calculates its own speed using the extra parameter velocity.
 */
unsigned short move_robot(joint_values delta_q, joint_values absolute_q,
			joint_values dq, joint_values ddq, REAL v, REAL a)
{
  joint_values q, dummy_delta, dummy_dq, dummy_ddq, prev_q, prev_dq, prev_ddq;
  unsigned short success;
  int i, j;

  /*
   * In order to make the motion profile more precise,
   * send a WAIT command to the robot before calculating
   * the new joint speeds and accelerations.
   */
  send_robot(WAIT);

  if (absolute_q == NULL_MOVE)
	absolute_q = q;

  for (i=0; i<robots+simulators; i++)
  {
	get_joint_values(i, 0, prev_q, prev_dq, prev_ddq);

	/*
	 * Move on delta joint values only; for this we need the current
	 * position.
	 */
	if (delta_q != NULL_MOVE)
		for (j=1; j<=NR_LINKS; j++)
			q[j] = prev_q[j] + delta_q[j];
	else
	{
		delta_q = dummy_delta;
		for (j=1; j<=NR_LINKS; j++)
		{
			q[j] = absolute_q[j]; delta_q[j] = q[j] - prev_q[j];
		}
	}

	if (dq == NULL_MOVE || NULL_MOVE)
	{
		REAL max_delta_q, max_delta_dq,
		     joint_velocity, joint_acceleration;
		unsigned short j;
		joint_values delta_dq;

		dq = dummy_dq;
		ddq = dummy_ddq;

		max_delta_q = max_delta_dq = 0.0;
		for (j=1; j<=NR_LINKS; j++)
			if (max_delta_q < fabs(delta_q[j]))
				max_delta_q = fabs(delta_q[j]);

		/*
		 * Choose for a constant velocity per joint, and the
		 * same time of motion for all the joints.
		 */
		joint_velocity = max_delta_q / v;
		for (j=1; j<=NR_LINKS; j++)
		{
			dq[j] = delta_q[j] / joint_velocity;
			delta_dq[j] = dq[j] - prev_dq[j];
			if (max_delta_dq < fabs(delta_dq[j]))
				max_delta_dq = fabs(delta_dq[j]);
		}

		/*
		 * Next, do something similar for accelerations.
		 */
		joint_acceleration = max_delta_dq / a;
		for (j=1; j<=NR_LINKS; j++)
			ddq[j] = delta_dq[j] / joint_acceleration;
	}

	/*
	 * Send joint angles to robot.  If the position was invalid,
	 * send_joint_values() returns a value != 0.  The number
	 * then returned is the number of the joint that failed.
	 *
	 * Note: by giving send_joint_values NO_CHECK_REACH, send_joint_values()
	 * should never return with an illegal value for a joint.
	 */
	success = send_joint_values(i, q, dq, ddq, NO_CHECK_REACH);

	/*
	 * After the first robot, all other robots have to
	 * go to the same position in joint space.  This is
	 * ensured by setting delta_q = NULL.
	 */
	delta_q = NULL_MOVE;
	dq = ddq = NULL_MOVE;
  }

  /*
   * Let the robot move again.
   */
  send_robot(CONTINUE);

  return success;
}




void fix_target(void)
{
  vector target_pos;
  double x, y, z;

  if (simulators == 0)
  {
	/*
	 * This is useless if only real robots are connected.
	 */
	printf("No simulators connected---fix_target not executed.\n");
	return;
  }

  printf("enter x, y, and z: ");
  scanf("%lf %lf %lf", &x, &y, &z);
  
  target_pos[0] = (REAL) x;
  target_pos[1] = (REAL) y;
  target_pos[2] = (REAL) z;
  getchar();
  target_pos[3] = 1.0;

  send_object_position(target_pos);
  printf("ready\n");
}


void user_move_robot(void)
{
  joint_values q, dq, ddq;
  unsigned short i;
  int move;

  printf("(a)bsolute or (r)elative: ");
  while (1)
  {
	if ((move = getchar()) == EOF) return;
	if (move == 'a' || move == 'r') break;
  }

  get_joint_values(0, 0, q, dq, ddq);

  printf("current position: (%d", (int) (0.5 + q[1]));
  for (i=2; i<=NR_LINKS; i++)
	printf(" %d", (int) (0.5 + q[i]));

  if (move == 'r')
  {
	joint_values delta_joints;

	printf(")\ngive %d delta values: ", NR_LINKS);

	for (i=1; i<=NR_LINKS; i++)
	{
		double dummy;
		scanf("%lf", &dummy);
		delta_joints[i] = (REAL) dummy;
		getchar();
	}

	if (move_robot(delta_joints, NULL_MOVE,
		NULL_MOVE, NULL_MOVE, SLOW_DQ, SLOW_DDQ))
		printf("Illegal target position\n");
  }
  else
  {
	joint_values absolute_joints;

	printf(")\ngive %d absolute values: ", NR_LINKS);

	for (i=1; i<=NR_LINKS; i++)
	{
	  	double dummy;
		scanf("%lf", &dummy);
		absolute_joints[i] = (REAL) dummy;
		getchar();
	}

	if (move_robot(NULL_MOVE, absolute_joints,
		NULL_MOVE, NULL_MOVE, SLOW_DQ, SLOW_DDQ))
		printf("Illegal target position\n");
  }

  printf("\nReady\n");
}


REAL deg2rad(REAL angle)
{
  return M_PI * angle / 180.0;
}

REAL rad2deg(REAL angle)
{
  return 180.0 * angle / M_PI;
}
