/******************************************************************************\
********************************************************************************
********                                                                ********
******      Visual robot simulator                                        ******
****                                                                        ****
**          Author: Craig Dillon, Andrew Conway                               **
**                                                                            **
**          MAIN.C                                                            **
****                                                                        ****
******                                                                    ******
********                                                                ********
********************************************************************************
\******************************************************************************/

#include<stdio.h>
#include<math.h>
#include<string.h>
#ifdef R_SGI 
#include<gl/gl.h>
#include<gl/device.h>
#endif
#include"windowlib.h"
#include"rob.h"

JOINT *joint[MAX_JOINTS];
JOINT *pjoint[MAX_JOINTS];
JOINT *from_joint[MAX_JOINTS];
JOINT *to_joint[MAX_JOINTS];
int numjoints;

VARIABLE *variable[MAX_VARIABLES];
int numvariables;

GRAPH *graph[MAX_GRAPHS];
int numgraphs;
int graph_active,graph_width,graph_height;

double time_step,current_time;
double time_bound_min=0.0,time_bound_max=1.0;
int time_sid;		/* time_step slidebar id */
float actual_time[10000];

int step_count,step_num;				/* current and number of steps */

#ifdef R_SGI 
Matrix Identity = {1,0,0,0, 0,1,0,0, 0,0,1,0, 0,0,0,1};
#endif

long wid_visual,wid_control,wid_graph,wid_schematic;
float prev_y,prev_x;

int global_mode;
JOINT **global_joint;

int pause_id,from_id,to_id,run_id;

int robot_created,runmode,world_mode;
char *save_name;
char *schematic_print_name;

long control_menu;	/* control menu */
long visual_menu;	/* visual menu */
int visual_view;	/* flag, uses same defines as menu */
long schematic_menu;
long graph_menu;

int visual_mode;
int visual_tilt;
float visual_x,visual_y,visual_z;

int numboxes;
BOX *box[MAX_BOXES];

void redraw_visual();
void robot_plan();

void do_control_menu(val)
long val;
{
	switch(val) {
		case SAVE_FROM:
			save_from();
			break;

		case SAVE_TO:
			save_to();
			break;

		case SAVE_PATH:
			save_path();
			break;

		case RESET_PATH:
			reset_path();
			break;

		default:
			break;

	}
}

void make_control_menu()
{
	char men_string[256];

	sprintf(men_string,"Save From|Save To|Save Path|Reset Path");
#ifdef R_SGI 
	control_menu=defpup(men_string);
#endif
}

void do_visual_menu(val)
long val;
{
    switch(val) {
        case VIEW_WORLD:
            visual_view=VIEW_WORLD;
			redraw_visual();
            break;

        case VIEW_LOCAL:
			visual_view=VIEW_LOCAL;
			redraw_visual();
            break;

        default:
            break;

    }
}

void make_visual_menu()
{
    char men_string[256];

    sprintf(men_string,"View World|View Local");
#ifdef R_SGI 
    visual_menu=defpup(men_string);
#endif
}

void visual_toggle_mode()
{
	if(visual_mode==VISUAL_ORIGIN) {
		visual_mode=VISUAL_MOBILE;
	} else {
		visual_mode=VISUAL_ORIGIN;
	}
	redraw_visual();
}

void visual_left()
{
	visual_tilt+=10;
	redraw_visual();
}

void visual_right()
{
	visual_tilt-=10;
	redraw_visual();
}

void redraw_visual()
{
	float rx,ry;
	long xs,ys,xo,yo;
	long ow;
	JOINT *jptr,*pjptr;
	long mx,my;

#ifdef R_SGI 
	ow=winget();
	winset(wid_visual);
	getsize(&xs,&ys);
	perspective(400,(float)xs/(float)ys,.25,15.0);
	reshapeviewport();
	getorigin(&xo,&yo);
	mx=getvaluator(MOUSEX);
	my=getvaluator(MOUSEY);
	if((mx>=xo)&&(mx<=(xo+xs))&&(my>=yo)&&(my<=(yo+ys))&&(!getbutton(FKEY))) {
		rx=200*(2.0*(mx-xo)/xs-1.0);
		ry=(10.0*(my-(float)yo)/(float)ys);
		prev_x=rx;
		prev_y=ry;
	} else {
		rx=prev_x;
		ry=prev_y;
	}
	if(visual_view==VIEW_WORLD) {
	    loadmatrix(Identity);
   	 	perspective(400,(float)xs/(float)ys,.01,100.0);
		if(visual_mode==VISUAL_ORIGIN) {
	    	lookat(0.0,10.0,3.0,0.0,0.0,2.0,visual_tilt);
   	 		pushmatrix();
   	 		rot(rx,'z');
		} else {
			lookat(0.0,1.0,0.0,
				0.0,0.0,0.0,
				visual_tilt);
			if(getbutton(LEFTMOUSE)) {
				visual_x+=0.10*fsin(rx*M_PI/180.0);
				visual_y+=0.10*fcos(rx*M_PI/180.0);
			}
			visual_z=ry;
			pushmatrix();
			rot(rx,'z');
			translate(visual_x,visual_y,-visual_z);
		}
	    czclear(0x00205020,getgdesc(GD_ZMAX));
   	 	draw_world();
    	joints_to_links();
    	calculate_position();
    	draw_robot();
	}
	if(visual_view==VIEW_LOCAL) {
		jptr=global_joint[numjoints-1];
		pjptr=global_joint[numjoints-2];
		czclear(0x00205020,getgdesc(GD_ZMAX));
    	loadmatrix(Identity);
    	perspective(400,(float)xs/(float)ys,.25,15.0);
        lookat(jptr->x,jptr->y,jptr->z,
            jptr->x+pjptr->dx,jptr->y+pjptr->dy,jptr->z+pjptr->dz,
            0);
        pushmatrix();
        draw_world();
        joints_to_links();
        calculate_position();
        draw_robot();
	}
	popmatrix();
	swapbuffers();
	winset(ow);
#else
	joints_to_links();
    calculate_position();
#endif
}

#define SW_WIDTH	210
#define SW_HEIGHT	320
#define TW_WIDTH	190
#define TW_HEIGHT	90

#define TSS_WIDTH	210
#define TSS_HEIGHT	90
#define TST_WIDTH	190
#define TST_HEIGHT	30

#define TWS_WIDTH	210
#define TWS_HEIGHT	130
#define TWT_WIDTH	190
#define TWT_HEIGHT	70

long sw[SW_WIDTH*SW_HEIGHT];
long tw[TW_WIDTH*TW_HEIGHT];
long tssw[TSS_WIDTH*TSS_HEIGHT];
long tstw[TST_WIDTH*TST_HEIGHT];
long twsw[TWS_WIDTH*TWS_HEIGHT];
long twtw[TWT_WIDTH*TWT_HEIGHT];

int jnum;	/* global variable kludge */

void redraw_text()
{
    JOINT *jptr;
    char s[40];
	long ow;

#ifdef R_SGI 
	ow=winget();
    winset(wid_control);

    jptr=global_joint[jnum];
    lrectwrite(HOFF+20+220*jnum,20,HOFF+20+220*jnum+TW_WIDTH-1,20+TW_HEIGHT-1,tw);
    cpack(0x00ffffff);
  
    cmov2(HOFF+30+220*jnum,90);
    sprintf(s,"a: %lf",jptr->a);
    charstr(s);

    cmov2(HOFF+30+220*jnum,70);
    sprintf(s,"d: %lf",jptr->d);
    charstr(s);

    cmov2(HOFF+30+220*jnum,50);
    sprintf(s,"alpha: %lf",jptr->alpha);
    charstr(s);

    cmov2(HOFF+30+220*jnum,30);
    sprintf(s,"theta: %lf",jptr->theta);
    charstr(s);

	winset(ow);
#endif
}

void redraw_all_text()
{
	for(jnum=0;jnum<numjoints;jnum++) {
		redraw_text();
	}
}

void redraw_timestep()
{
	char s[40];
	long ow;

#ifdef R_SGI 
	ow=winget();
    winset(wid_control);

	lrectwrite(20,290,20+TST_WIDTH-1,290+TST_HEIGHT-1,tstw);
	cpack(0x00ffffff);
    sprintf(s,"Time Step: %lf",time_step);
    cmov2(30,300);
    charstr(s);
	
	winset(ow);
#endif
}

void redraw_time()
{
	long ow;
	char s[40];

#ifdef R_SGI 
	ow=winget();
    winset(wid_control);

	lrectwrite(20,130,20+TWT_WIDTH-1,130+TWT_HEIGHT-1,twtw);
    cpack(0x00ffffff);

    sprintf(s,"Number: %d",step_num);
    cmov2(30,180);
    charstr(s);

    sprintf(s,"Current: %d",step_count);
    cmov2(30,160);
    charstr(s);

    sprintf(s,"Time: %lf",current_time);
    cmov2(30,140);
    charstr(s);

	winset(ow);
#endif
}

void redraw_control()
{
	int i;
	JOINT *jptr;
	char s[40];
	long ow;

#ifdef R_SGI 
	ow=winget();
	winset(wid_control);
	cpack(0x00402080);
    clear();	

	fill_array(SW_WIDTH,SW_HEIGHT,sw,0x00402080);
	fill_invarray(TW_WIDTH,TW_HEIGHT,tw,0x00402080);
	fill_array(TSS_WIDTH,TSS_HEIGHT,tssw,0x00402080);
	fill_invarray(TST_WIDTH,TST_HEIGHT,tstw,0x00402080);
	fill_array(TWS_WIDTH,TWS_HEIGHT,twsw,0x00402080);
	fill_invarray(TWT_WIDTH,TWT_HEIGHT,twtw,0x00402080);

	lrectwrite(10,240,10+TSS_WIDTH-1,240+TSS_HEIGHT-1,tssw);
	redraw_timestep();

	lrectwrite(10,100,10+TWS_WIDTH-1,100+TWS_HEIGHT-1,twsw);
	redraw_time();

	for(i=0;i<numjoints;i++) {
		jptr=global_joint[i];
		lrectwrite(HOFF+10+220*i,10,HOFF+10+220*i+SW_WIDTH-1,10+SW_HEIGHT-1,sw);
		lrectwrite(HOFF+20+220*i,20,HOFF+20+220*i+TW_WIDTH-1,20+TW_HEIGHT-1,tw);
		cpack(0x00ffffff);
        cmov2(HOFF+30+220*i,SW_HEIGHT-1-15);
        charstr(jptr->name);
	
		cmov2(HOFF+30+220*i,90);
		sprintf(s,"a: %lf",jptr->a);
		charstr(s);

		cmov2(HOFF+30+220*i,70);
        sprintf(s,"d: %lf",jptr->d);
        charstr(s);

		cmov2(HOFF+30+220*i,50);
        sprintf(s,"alpha: %lf",jptr->alpha);
        charstr(s);

		cmov2(HOFF+30+220*i,30);
        sprintf(s,"theta: %lf",jptr->theta);
        charstr(s);
	}
#endif
	draw_all_slidebars();
	draw_all_arrows();
	draw_all_buttons();
#ifdef R_SGI 
	winset(ow);
#endif
}

void create_visual()
{
	long xs,ys,xo,yo;
	long wid,i;
	
	visual_view=VIEW_WORLD;
#ifdef R_SGI 
	prefposition(30,530,380,760);
    wid_visual=winopen("visual");
    getsize(&xs,&ys);
    getorigin(&xo,&yo);
    RGBmode();
    concave(TRUE);
    doublebuffer();
    gconfig();
    zbuffer(TRUE);
	winconstraints();
    czclear(0x205020,getgdesc(GD_ZMAX));
    mmode(MVIEWING);
    loadmatrix(Identity);
    perspective(400,(float)xs/(float)ys,.25,15.0);
    lookat(0.0,10.0,3.0,0.0,0.0,2.0,1800);
    lmdef(DEFMATERIAL,MAT_PANEL,0,mat_panel);
    lmdef(DEFMATERIAL,MAT_FLOOR,0,mat_floor);
    lmdef(DEFMATERIAL,MAT_WALL,0,mat_wall);
    lmdef(DEFLIGHT,1,0,lt);
    lmdef(DEFLMODEL,1,0,lm);
    lmbind(LMODEL,1);
    lmbind(LIGHT0,1);

	for(i=0;i<numjoints;i++) {
		lmdef(DEFMATERIAL,MAT_USER+i,0,joint[i]->material);
	}
	for(i=0;i<numboxes;i++) {
        lmdef(DEFMATERIAL,MAT_BOX+i,0,box[i]->material);
    }
#endif
	make_visual_menu();
	redraw_visual();
}

void bind_timestep()
{
	SLIDEBAR *sptr;
	double value;

	sptr=slidebar[time_sid];
	value=slidebar_position(sptr);
	value=time_bound_min+(value*(time_bound_max-time_bound_min));
	time_step=value;
	redraw_timestep();
}

void bind_snua()
{
	step_num++;
	if(step_num>9999) step_num=9999;
	redraw_time();
}

void bind_snub()
{
    step_num+=10;
    if(step_num>9999) step_num=9999;
    redraw_time();
}

void bind_snuc()
{
    step_num+=100;
    if(step_num>9999) step_num=9999;
    redraw_time();
}

void bind_snud()
{
    step_num+=1000;
    if(step_num>9999) step_num=9999;
    redraw_time();
}

void bind_snda()
{
    step_num--;
    if(step_num<0) step_num=0;
    redraw_time();
}

void bind_sndb()
{
    step_num-=10;
    if(step_num<0) step_num=0;
    redraw_time();
}

void bind_sndc()
{
    step_num-=100;
    if(step_num<0) step_num=0;
    redraw_time();
}

void bind_sndd()
{
    step_num-=1000;
    if(step_num<0) step_num=0;
    redraw_time();
}

void set_control()
{
    JOINT *jptr;
    VARIABLE *vptr;
    SLIDEBAR *sptr;
    ARROW *aptr;
    int i;
    double value;
	long ow;

    for(i=0;i<numjoints;i++) {
        jptr=global_joint[i];
        if(jptr->ca==ROB_CON) {
            vptr=variable[jptr->a_var];
            sptr=slidebar[jptr->s_a];
			value=jptr->a;
			vptr->value=value;
			position_slidebar(sptr,(value-vptr->min)/(vptr->max-vptr->min));
        }
        if(jptr->cd==ROB_CON) {
            vptr=variable[jptr->d_var];
            sptr=slidebar[jptr->s_d];
			value=jptr->d;
			vptr->value=value;
			position_slidebar(sptr,(value-vptr->min)/(vptr->max-vptr->min));
        }
        if(jptr->calpha==ROB_CON) {
            vptr=variable[jptr->alpha_var];
            aptr=arrow[jptr->a_alpha];
			value=jptr->alpha;
            vptr->value=value;
            aptr->a=value;
        }
        if(jptr->ctheta==ROB_CON) {
            vptr=variable[jptr->theta_var];
            aptr=arrow[jptr->a_theta];
			value=jptr->theta;
            vptr->value=value;
            aptr->a=value;
        }
    }
#ifdef R_SGI 
	ow=winget();
	winset(wid_control);
#endif
    redraw_all_text();
    draw_all_slidebars();
    draw_all_arrows();
#ifdef R_SGI 
	winset(ow);
#endif
}

void bind_control()
{
	JOINT *jptr;
	VARIABLE *vptr;
	SLIDEBAR *sptr;
	ARROW *aptr;
	int i;
	double value;

	for(i=0;i<numjoints;i++) {
		jptr=global_joint[i];
		if(jptr->ca==ROB_CON) {
			vptr=variable[jptr->a_var];
			sptr=slidebar[jptr->s_a];
			value=slidebar_position(sptr);
			value=vptr->min+(value*(vptr->max-vptr->min));
			vptr->value=value;
			jptr->a=value;
		}
		if(jptr->cd==ROB_CON) {
            vptr=variable[jptr->d_var];
            sptr=slidebar[jptr->s_d];
            value=slidebar_position(sptr);
            value=vptr->min+(value*(vptr->max-vptr->min));
            vptr->value=value;
            jptr->d=value;
        }
		if(jptr->calpha==ROB_CON) {
            vptr=variable[jptr->alpha_var];
            aptr=arrow[jptr->a_alpha];
            value=(double)(aptr->a);
            vptr->value=value;
            jptr->alpha=value;
        }
		if(jptr->ctheta==ROB_CON) {
            vptr=variable[jptr->theta_var];
            aptr=arrow[jptr->a_theta];
            value=(double)(aptr->a);
            vptr->value=value;
            jptr->theta=value;
        }
	}
	redraw_text();
	redraw_visual();
}

void set_button_colours(m_id)
int m_id;
{
	long ow;

#ifdef R_SGI 
	ow=winget();
	winset(wid_control);
#endif

	button[from_id]->textrgb=0x00ffffff;
	button[to_id]->textrgb=0x00ffffff;
	button[run_id]->textrgb=0x00ffffff;
	button[pause_id]->textrgb=0x00ffffff;

	if(m_id>=0) button[m_id]->textrgb=0x0000ff00;

	draw_button(button[from_id]);
	draw_button(button[to_id]);
	draw_button(button[run_id]);
	draw_button(button[pause_id]);
#ifdef R_SGI 
	winset(ow);
#endif
}

void bind_from_mode()
{
	global_mode=MODE_FROM;
	global_joint=from_joint;
	set_control();
	set_button_colours(from_id);
	redraw_visual();
}

void bind_to_mode()
{
	global_mode=MODE_TO;
	global_joint=to_joint;
	set_control();
	set_button_colours(to_id);
	redraw_visual();
}

void bind_pause_mode()
{
	global_mode=MODE_PAUSE;
	global_joint=joint;
	set_control();
	set_button_colours(pause_id);
	redraw_visual();
	redraw_graph();
}

void bind_run_mode()
{
	if(step_count==0) {	/* no itterations yet */
		init_joints();
	}
	global_mode=MODE_RUN;
	global_joint=joint;
	set_button_colours(run_id);
	set_control();
	redraw_visual();
}

void bind_reset_mode()
{
	init_joints();
	global_mode=MODE_RESET;
	global_joint=joint;
	set_control();
	set_button_colours(-1);
	redraw_visual();
	step_count=0;
	current_time=0.0;
	redraw_time();
	redraw_graph();
	reset_schematic();
}

void run_robot_run()	/* this function is called when the robot is running */
{
	if(step_count>=step_num) { /* stop */
		global_mode=MODE_RESET;
		set_button_colours(-1);
		set_control();
		redraw_graph();
		return;
	}
	next_itteration();
	record_graph();
	actual_time[step_count]=current_time;
	current_time+=time_step;
	step_count++;
	redraw_time();	
	redraw_visual();
}

void create_control()
{
	int width,height;
	int i;
	int id;

	width=HOFF+10+numjoints*(210+10);
	height=340;
	numslidebars=0;
	numarrows=0;
	numbuttons=0;

#ifdef R_SGI 
	prefposition(147,147+width-1,381-height,380);
	wid_control=winopen("control");
	RGBmode();
	singlebuffer();
	concave(FALSE);
	gconfig();
	zbuffer(FALSE);
	cpack(0x00402080);
	clear();
#endif

	make_control_menu();

	time_sid=create_slidebar(20,250,190,50,30,0x00406080,bind_timestep,NULL);

	create_button(190,205,20,20,0x00808080,NULL,NULL,bind_snua);
	create_button(160,205,20,20,0x00808080,NULL,NULL,bind_snub);
	create_button(130,205,20,20,0x00808080,NULL,NULL,bind_snuc);
	create_button(100,205,20,20,0x00808080,NULL,NULL,bind_snud);
	create_button(190,105,20,20,0x00808080,NULL,NULL,bind_snda);
	create_button(160,105,20,20,0x00808080,NULL,NULL,bind_sndb);
	create_button(130,105,20,20,0x00808080,NULL,NULL,bind_sndc);
	create_button(100,105,20,20,0x00808080,NULL,NULL,bind_sndd);

	from_id=create_button(10,60,60,30,0x00808080,"From",NULL,bind_from_mode);
	to_id=create_button(80,60,60,30,0x00808080,"To",NULL,bind_to_mode);
	run_id=create_button(150,60,60,30,0x00808080,"Run",NULL,bind_run_mode);
	pause_id=create_button(10,20,60,30,0x00808080,"Pause",NULL,bind_pause_mode);
	create_button(80,20,60,30,0x00808080,"Reset",NULL,bind_reset_mode);
	create_button(150,20,60,30,0x00808080,"Plan",robot_plan,NULL);

	for(i=0;i<numjoints;i++) {
		if(global_joint[i]->ca==ROB_CON) {
			id=create_slidebar(HOFF+20+i*220,160,190,50,30,0x00808080,bind_control,NULL);
			global_joint[i]->s_a=id;
		} else {
			global_joint[i]->s_a= -1;
		}
		if(global_joint[i]->cd==ROB_CON) {
			id=create_slidebar(HOFF+20+i*220,120,190,50,30,0x00808080,bind_control,NULL);
			global_joint[i]->s_d=id;
		} else {
			global_joint[i]->s_d= -1;
		}
		if(global_joint[i]->calpha==ROB_CON) {
			id=create_arrow(HOFF+20+i*220,200,0.0,0x00808080,bind_control,NULL);
			global_joint[i]->a_alpha=id;
		} else {
			global_joint[i]->a_alpha= -1;
		}
		if(global_joint[i]->ctheta==ROB_CON) {
			id=create_arrow(HOFF+120+i*220,200,0.0,0x00808080,bind_control,NULL);
        	global_joint[i]->a_theta=id;
		} else {
			global_joint[i]->a_theta= -1;
		}
	}
	redraw_control();
}

void process_visual(event,val)
int event;
short val;
{
}

void process_control(event,val)
int event;
short val;
{
	long x,y;

#ifdef R_SGI 
	switch(event) {
		case LEFTMOUSE:
			getorigin(&x,&y);
			if((getvaluator(MOUSEX)-x)>(HOFF+20)) {
				if((global_mode==MODE_FROM)||(global_mode==MODE_TO)) {
					if(find_slidebar()<0) {
						if(find_arrow()<0) {
							find_button();
						}
					}
				}
			} else {
				if(find_slidebar()<0) {
	               	if(find_arrow()<0) {
   	                 	find_button();
					}
				}
			}
			break;
	
		default:
			break;
	}
#endif
}

void main(argc,argv)
int argc;
char **argv;
{
	float rx,ry;
	FILE *in;
	int i;
	int event;
    short val;
	long cw;

	if(argc!=2) {
		printf("Usage: r <filename>");
		exit(0);
	}
	init_tl_index();
	init_tr_index();
	init_bl_index();
	init_br_index();
	graph_active=0;
	graph_width=0;
	graph_height=0;
	robot_created=0;
	step_count=0;
	pathnum=0;
	plan_nx=1;
	plan_ny=1;
	plan_nz=1;
	plan_passes=0;
	runmode=RMODE_NONE;
	world_mode=WORLD_SIMPLE;
	visual_mode=VISUAL_ORIGIN;
	numpictures=0;
	visual_x=0.0;
	visual_y=10.0;
	visual_z=3.0;
	visual_tilt=1800;

	save_name=strdup(argv[1]);
	schematic_print_name=strdup(argv[1]);	/* default */
	in=fopen(argv[1],"r");
	if(!in) {
		fprintf(stderr,"Error: cannot open input file %s\n",argv[1]);
		exit(1);
	}
	read_file(in);
	global_joint=joint;
	create_visual();
	create_control();
	create_graph();
	create_schematic();

	if(robot_created==0) {
		setup_joints();		/* copies everything */
	} else {
		update_joints();	/* update some things */
	}
	global_mode=MODE_FROM;
	global_joint=from_joint;
	set_button_colours(from_id);
	set_control();

#ifndef R_SGI

	printf("ROBOT SIMULATOR\n");
	printf("===============\n");
	printf("\n\n");
	printf("By Andrew Conway and Craig Dillon\n");
	printf("University of Melbourne\n");
	printf("\n");
	printf("Running in NO_GRAPHICS mode.\n");
	printf("Please wait whilst processing proceeds.\n");
	printf("\n");

#endif

#ifdef R_SGI 	
    qdevice(REDRAW);
    qdevice(LEFTMOUSE);
    qdevice(RIGHTMOUSE);
    qdevice(INPUTCHANGE);
	qdevice(UPARROWKEY);
	qdevice(DOWNARROWKEY);
	qdevice(LEFTARROWKEY);
	qdevice(RIGHTARROWKEY);
	qdevice(ESCKEY);
	qdevice(JKEY);
	qdevice(HKEY);
	qdevice(CKEY);
	qdevice(FKEY);

	redraw_visual();
    while(1) {	
		if(qtest()) {
            event=qread(&val);
            switch(event) {

				case ESCKEY:
					exit(0);

				case RIGHTMOUSE:	
					cw=winget();
					if(cw==wid_control) {
						if(val==1) do_control_menu(dopup(control_menu));
					}
                    if(cw==wid_visual) {
                        if(val==1) do_visual_menu(dopup(visual_menu));
                    }
					if(cw==wid_schematic) {
                        if(val==1) do_schematic_menu(dopup(schematic_menu));
                    }
					if(cw==wid_graph) {
                        if(val==1) do_graph_menu(dopup(graph_menu));
                    }
					break;
					
                case LEFTMOUSE:
					cw=winget();
					if(cw==wid_visual) process_visual(event,val);
					if(cw==wid_control) process_control(event,val);
					if(cw==wid_graph) process_graph(event,val);
					if(cw==wid_schematic) process_schematic(event,val);
                    break;

				case REDRAW:
					winset(val);
					if(val==wid_visual) redraw_visual();
					if(val==wid_control) redraw_control();
					if(val==wid_graph) redraw_graph();
					if(val==wid_schematic) redraw_schematic();
					break;

				case INPUTCHANGE:	
					if(val==wid_visual) winset(wid_visual);
					if(val==wid_control) winset(wid_control);
					if(val==wid_graph) winset(wid_graph);
					if(val==wid_schematic) winset(wid_schematic);
					break;

				case UPARROWKEY:
					if(!getbutton(UPARROWKEY)) break;
					cw=winget();
					if(cw==wid_schematic) schematic_up();
					if(cw==wid_graph) graph_up();
					break;

				case DOWNARROWKEY:
					if(!getbutton(DOWNARROWKEY)) break;
                    cw=winget();
                    if(cw==wid_schematic) schematic_down();
                    if(cw==wid_graph) graph_down();
                    break;
			
				case LEFTARROWKEY:
					if(!getbutton(LEFTARROWKEY)) break;
                    cw=winget();
                    if(cw==wid_schematic) schematic_left();
                    if(cw==wid_graph) graph_left();
					if(cw==wid_visual) visual_left();
                    break;

				case RIGHTARROWKEY:
					if(!getbutton(RIGHTARROWKEY)) break;
                    cw=winget();
                    if(cw==wid_schematic) schematic_right();
                    if(cw==wid_graph) graph_right();
					if(cw==wid_visual) visual_right();
                    break;

				case JKEY:
					if(!getbutton(JKEY)) break;
					cw=winget();
					if(cw==wid_schematic) schematic_jump();
					if(cw==wid_graph) graph_jump();
					break;

				case HKEY:
					if(!getbutton(HKEY)) break;
                    cw=winget();
                    if(cw==wid_schematic) schematic_home();
					if(cw==wid_graph) graph_home();
                    break;

				case CKEY:
					if(!getbutton(CKEY)) break;
                    cw=winget();
                    if(cw==wid_visual) visual_toggle_mode();
					break;

				case FKEY:
					/* used by redraw visual */
					break;

				default:
                    printf("main.c: unspecified event\n");
					exit(0);
            }
		}
		if(global_mode==MODE_RUN) {
			run_robot_run();
		} else {
			if(winget()==wid_visual) redraw_visual();
		}
    }
#else
	bind_run_mode();
	while(global_mode==MODE_RUN) {
		run_robot_run();
	}
	print_graph();
#endif
}
