/* #########################################
 * sonars.c
 *
 * sonar sensor processing routines
 * for the controller
 *#########################################
 */

/*
 * 12 sonars are distributed  almost uniformly around the body of
 *  the robot
 *
 * This should be parameterizable
 */

#include "structures.h"
#include "constants.h"
#include "parameters.h"
#include "display.h"

/* sonar_buffer is an array holding the readings */

float sonar_buffer[MAX_SONARS];
int sonar_num = 12;		/* current number of sonars */

#define CP4 0.7071067811865
#define SP4 0.7071067811865

/* sonar specifications: x,y coord and direction */
float sonar_spec_array[MAX_SONARS][5] =	
{
  {+290,+80,1,0,0},
  {+290,-80,1,0,0},
  {+240,-200,0,-1,(- ANG90)},
  {+200,-240,1,0,0},
  {-200,-240,-1,0,ANG180},
  {-240,-200,0,-1,(- ANG90)},
  {+220,+220,CP4,SP4,ANG45},
  {+220,-220,CP4,(- SP4),(- ANG45)},
  {-240,+200,0,+1,ANG90},
  {-200,+240,-1,0,ANG180},
  {+200,+240,+1,0,0},
  {+240,+200,0,+1,ANG90}
};

/* Masks for which sonars are front, sides */

int right_sonar_mask =   0x024;
int left_sonar_mask =    0x900;
int front_sonar_mask =   0xCCF;

#define FRONT_ANGLE 47.0	/* in degrees */
#define SIDE_ANGLE  66.0	/* in degrees */

calc_sonar_masks()
{
  int i;
  float ang;
  right_sonar_mask = left_sonar_mask = front_sonar_mask = 0;
  for (i=0; i<sonar_num; i++)
    {
      ang = RAD_TO_DEG * sonar_spec_array[i][4];
      if (ang > SIDE_ANGLE) left_sonar_mask |= 1<<i;
      if (ang < -SIDE_ANGLE) right_sonar_mask |= 1<<i;
      if (ABS(ang)  < FRONT_ANGLE) front_sonar_mask |= 1<<i;
    }
}



float sonar_x(int i,int r)	/* returns x coord of sonar reading I range R */
{
  return((float)(sonar_spec_x(i) + (float)r * sonar_spec_dx(i)));
}

float sonar_y(int i,int r)	/* returns y coord of sonar reading I range R */
{
  return((float)(sonar_spec_y(i) + (float)r * sonar_spec_dy(i)));
}


sonar_coords(i)			/* calculates the RW x,y coordinates of a sonar reading */
{
  sdata *r;
  float x, y;
  r = &sbucket[i];
  x = sonar_x(i,r->range);
  y = sonar_y(i,r->range);
  prot(&x,&y,(- r->fth));	/* rotate to current direction */
  x += r->fx;
  y += r->fy;
  r->x = x;
  r->y = y;
}

draw_sonars()			/* draws the current sonar readings */
{
  int i;
  for (i=0; i<sonar_num; i++)
    {
      if (sbucket[i].new > 0)
	{
	  draw_sonar_pt(sbucket[i].x,sbucket[i].y,3);
	  sbucket[i].new = 2;
	}
    }
  display_sbuf();
}


/* July 1990  Konolige
 *
 * Functions to store, update, and display the state of
 *   Flakey's sonars in windows.
 * 
 * Basic idea is to keep a circular buffer of N (~100) last significant 
 *  readings of the sonars, display these and keep them registered
 *  with Flakey's movements on the screen.
 *
 */

cbuf c1, c2, c3;
cbuf *sraw_buf = &c1;		/* holds raw sonar readings */
cbuf *sl_buf = &c2;		/* holds left/right readings */
cbuf *sr_buf = &c3;


empty_sbuf(w,n)			/* reset the buffer, max N readings */
{
  switch(w)
    {
    case 0: empty_it(sraw_buf,n); break;
    case 1: empty_it(sl_buf,n); empty_it(sr_buf,n); break;
    }
}

empty_it(buf,n)
     cbuf *buf;
     int n;
{
  buf->start = 0;
  buf->end   = 0; 
  if (n == 0) return;		/* don't change it */
  if (n < CBUF_LEN) buf->limit = n;
  else buf->limit = (CBUF_LEN - 1);
}

reset_sbuf(w)			/* reset the params */
{
     switch(w)
     {
     case 0:
	  empty_it(sraw_buf,0);
	  break;
     case 1:
	  empty_it(sl_buf,0);
	  empty_it(sr_buf,0);
	  break;
     }
}


/* process to add new raw readings to circular buffers */

#define MAX_RANGE 2500.0

sonar_proc()
{
  int i;
  for (i=0; i<sonar_num; i++)
    {
      if (sbucket[i].new == 1) /* have a new reading */
	{
	  sbucket[i].new = 2;	/* change to display only */
	  if (sbucket[i].range < MAX_RANGE && flakey.status == STATUS_MOVING)
	    {
	      if (1<<i & front_sonar_mask) add_sbuf_sonar(i);
	      if (1<<i & left_sonar_mask) add_sbuf_left_sonar(i);
	      if (1<<i & right_sonar_mask) add_sbuf_right_sonar(i);
	    }
	}
    }
}


add_sbuf(cbuf *buf,float x,float y) /* add the entry, rw coords */
{
  int end, st, mx;
  end = buf->end; st = buf->start; mx = buf->limit;
  buf->xbuf[end] = x;		/* add the entry */
  buf->ybuf[end] = y;
  end++;
  if (end > mx) end = 0;	/* check end of circular buffer */
  if (end == st)		/* need to delete oldest entry */
    { st++; if (st > mx) st = 0; }
  buf->end = end; buf->start = st;
}


add_sbuf_sonar(int i)		/* adds this entry, rw coords */
{
  add_sbuf(sraw_buf,sbucket[i].x,sbucket[i].y);
}

add_sbuf_left_sonar(i)		/* adds to the left-side sonar */
{
  add_sbuf(sl_buf,sbucket[i].x,sbucket[i].y);
}

add_sbuf_right_sonar(i)		/* adds to the left-side sonar */
{
  add_sbuf(sr_buf,sbucket[i].x,sbucket[i].y);
}


display_sb(cbuf *buf,int n)	/* current sonar points */
{
  int i;
  int st, end, mx;
  float *xbuf, *ybuf;
  xbuf = buf->xbuf;    ybuf = buf->ybuf;
  st = buf->start; end = buf->end; mx = buf->limit;
  for (i = st; i != end; i++)
    { if ( i > mx ) { i = -1; continue; }
      draw_sonar_pt(xbuf[i],ybuf[i],n);
    }
}

print_sraw()			/* current sonar points */
{
  int i;
  int st, end, mx;
  float *xbuf, *ybuf;
  xbuf = sraw_buf->xbuf;    ybuf = sraw_buf->ybuf;
  st = sraw_buf->start; end = sraw_buf->end; mx = sraw_buf->limit;
  for (i = st; i != end; i++)
    { if ( i > mx ) { i = -1; continue; }
      printf("%f %f\n",xbuf[i],ybuf[i]);
    }
}

print_right_side()			/* current sonar points */
{
  int i;
  int st, end, mx;
  float *xbuf, *ybuf;
  xbuf = sr_buf->xbuf;    ybuf = sr_buf->ybuf;
  st = sr_buf->start; end = sr_buf->end; mx = sr_buf->limit;
  for (i = st; i != end; i++)
    { if ( i > mx ) { i = -1; continue; }
      printf("%f %f\n",xbuf[i],ybuf[i]);
    }
}

print_left_side()			/* current sonar points */
{
  int i;
  int st, end, mx;
  float *xbuf, *ybuf;
  xbuf = sl_buf->xbuf;    ybuf = sl_buf->ybuf;
  st = sl_buf->start; end = sl_buf->end; mx = sl_buf->limit;
  for (i = st; i != end; i++)
    { if ( i > mx ) { i = -1; continue; }
      printf("%f %f\n",xbuf[i],ybuf[i]);
    }
}

display_sbuf()			/* graph it on the display pixrect */
{
  display_sb(sraw_buf,2);
  display_sb(sr_buf,2);
  display_sb(sl_buf,2);
}


update_sbuf(float dx,float dy,float dth) /* decrement by this amount, rw coords */
{
  int st, end, mx;
  float *xbuf, *ybuf;
  int i;
  float cth, sth;
  float tx, ty;
  int n;
  sdata *r;


  dth = - dth;			/* historical reasons... */
  cth = cos(dth);		/* we should only do this when dth > 0... */
  sth = sin(dth); 

/* printf("up: %f %f %f\n",dx,dy,dth); */


  for (n=0; n<sonar_num; n++)
    {
      r = &sbucket[n];
      r->fx -= dx;
      r->fy -= dy;
      r->fth -= dth;
      sonar_coords(n);
    }

  /* SRAW_BUF */

  xbuf = sraw_buf->xbuf;    ybuf = sraw_buf->ybuf;
  st = sraw_buf->start; end = sraw_buf->end; mx = sraw_buf->limit;
  for (i = st; i != end; i++)
    { if (i > mx) {if (end == 0) { break; } 
		   else {i = -1; continue; }}	/* reset if beyond the end */
      tx = xbuf[i] - dx;
      ty = ybuf[i] - dy;
      xbuf[i] = (tx * cth) - (ty * sth);
      ybuf[i] = (ty * cth) + (tx * sth);
    }


  /* SL_BUF */

  xbuf = sl_buf->xbuf;    ybuf = sl_buf->ybuf;
  st = sl_buf->start; end = sl_buf->end; mx = sl_buf->limit;
  for (i = st; i != end; i++)
    { if (i > mx) {if (end == 0) { break; } 
		   else {i = -1; continue; }}	/* reset if beyond the end */
      tx = xbuf[i] - dx;
      ty = ybuf[i] - dy;
      xbuf[i] = (tx * cth) - (ty * sth);
      ybuf[i] = (ty * cth) + (tx * sth);
    }

  /* SR_BUF */

  xbuf = sr_buf->xbuf;    ybuf = sr_buf->ybuf;
  st = sr_buf->start; end = sr_buf->end; mx = sr_buf->limit;
  for (i = st; i != end; i++)
    { if (i > mx) {if (end == 0) { break; } 
		   else {i = -1; continue; }}	/* reset if beyond the end */
      tx = xbuf[i] - dx;
      ty = ybuf[i] - dy;
      xbuf[i] = (tx * cth) - (ty * sth);
      ybuf[i] = (ty * cth) + (tx * sth);
    }

}




/*
 *
 *  Low-level functions to check occupancy in rectangular areas,
 *    using the front sonar buffer
 */

occupied(cx,cy,h,w,xy)		/* cx, cy are center coords, h and w are height and width */
				/*   all distances in mm */
				/* xy is 0 for x distance (front), 1 for y distance */
{
  float *x_buf, *y_buf;
  int start, end, mx;
  float x1, y1, x2, y2;
  int ret, i;
  x1 = cx - (h >> 1);
  x2 = x1 + h;
  y1 = cy - (w >> 1);
  y2 = y1 + w;

  ret = 5000;
  start = sraw_buf->start;	/* front buffer */
  end   = sraw_buf->end;
  mx    = sraw_buf->limit;
  x_buf = sraw_buf->xbuf;
  y_buf = sraw_buf->ybuf;

  /* check sonar points for containment in rectangle */

  for (i = start; i != end; ++i) /* put sonar readings into buckets */
    { if (i > mx) {i = -1; continue; }    /* reset if beyond the end */
      if (x_buf[i] >= x1 && x_buf[i] <= x2 && y_buf[i] >= y1 && y_buf[i] <= y2)
	{ if (xy == 0)		/* use x-distance */
	    { if (x_buf[i] < ret) ret = x_buf[i]; }
	  else  
	    { if (y_buf[i] < 0)
		{ if ((- y_buf[i]) < ret) ret = - y_buf[i]; }
	      else 
		{ if (y_buf[i] < ret) ret = y_buf[i]; }
	    }
	}
    }

  return(ret > 4999? 0 : ret);
}


occ(cx,cy,h,w,xy)		/* cx, cy are center coords, h and w are height and width */
				/*   all distances in mm */
				/* xy is 0 for x distance (front), 1 for y distance */
{
  float *x_buf, *y_buf;
  int start, end, mx;
  float x1, y1, x2, y2;
  int ret, i;
  x1 = cx - (h >> 1);
  x2 = x1 + h;
  y1 = cy - (w >> 1);
  y2 = y1 + w;

  ret = 5000;
  start = sraw_buf->start;	/* front buffer */
  end   = sraw_buf->end;
  mx    = sraw_buf->limit;
  x_buf = sraw_buf->xbuf;
  y_buf = sraw_buf->ybuf;

  /* check sonar points for containment in rectangle */

  for (i = start; i != end; ++i) /* put sonar readings into buckets */
    { if (i > mx) {i = -1; continue; }    /* reset if beyond the end */
      if (x_buf[i] >= x1 && x_buf[i] <= x2 && y_buf[i] >= y1 && y_buf[i] <= y2)
	{ if (xy == 0)		/* use x-distance */
	    { if (x_buf[i] < ret) ret = x_buf[i]; }
	  else  
	    { if (y_buf[i] < 0)
		{ if ((- y_buf[i]) < ret) ret = - y_buf[i]; }
	      else 
		{ if (y_buf[i] < ret) ret = y_buf[i]; }
	    }
	}
    }

  return(ret);
}


occupied2(d,s1,s2,xy)		/* d is line distance from center,
				 * s1 is min point on line, s2 is max.
				 * Returns min distance of sonar point from
				 *   line segment.
				 * all distances in mm 
				 * xy is +1 for +x dir (front), -1 for -x,
				 *   +2 for +y dir (left), -2 for -y (right) 
				 */
{
  float *x_buf, *y_buf;
  int start, end, mx;
  float df, s1f, s2f;
  int ret, i;

  start = sraw_buf->start;	/* front buffer */
  end   = sraw_buf->end;
  mx    = sraw_buf->limit;
  x_buf = sraw_buf->xbuf;
  y_buf = sraw_buf->ybuf;


  if (xy < 0)
    { df = -d; }
  else { df = d; }
  if ( s1 > s2)
    { s1f = s2; s2f = s1; }
  else 
    { s1f = s1; s2f = s2; }

  ret = 5000;

  /* check sonar points for containment in rectangle */

  if (xy == 1)			/* +X direction */
    { for (i = start; i != end; ++i) /* put sonar readings into buckets */
	{ if (i > mx) {i = -1; continue; }    /* reset if beyond the end */
	  if (x_buf[i] >= df && y_buf[i] >= s1f && y_buf[i] <= s2f)
	    { if (x_buf[i] < ret) ret = x_buf[i]; }
	}
    }
  else if (xy == -1)		/* -X direction */
    { for (i = start; i != end; ++i) /* put sonar readings into buckets */
	{ if (i > mx) {i = -1; continue; }    /* reset if beyond the end */
	  if (x_buf[i] <= df && y_buf[i] >= s1f && y_buf[i] <= s2f)
	    { if ((- x_buf[i]) < ret) ret = - x_buf[i]; }
	}
    }
  else if (xy == -2)		/* -Y direction */
    { for (i = start; i != end; ++i) /* put sonar readings into buckets */
	{ if (i > mx) {i = -1; continue; }    /* reset if beyond the end */
	  if (y_buf[i] <= df && x_buf[i] >= s1f && x_buf[i] <= s2f)
	    { if ((- y_buf[i]) < ret) ret = - y_buf[i]; }
	}
    }
  else if (xy == 2)		/* -Y direction */
    { for (i = start; i != end; ++i) /* put sonar readings into buckets */
	{ if (i > mx) {i = -1; continue; }    /* reset if beyond the end */
	  if (y_buf[i] >= df && x_buf[i] >= s1f && x_buf[i] <= s2f)
	    { if (y_buf[i] < ret) ret = y_buf[i]; }
	}
    }


  return(ret);
}



occupied3(d,s1,s2,xy)		/* Same as occupied2, but uses side sonars too */
{
  float *x_buf, *y_buf;
  int start, end, mx;
  float df, s1f, s2f;
  int ret, i;

  start = sraw_buf->start;	/* front buffer */
  end   = sraw_buf->end;
  mx    = sraw_buf->limit;
  x_buf = sraw_buf->xbuf;
  y_buf = sraw_buf->ybuf;


  if (xy < 0)
    { df = -d; }
  else { df = d; }
  if ( s1 > s2)
    { s1f = s2; s2f = s1; }
  else 
    { s1f = s1; s2f = s2; }

  ret = 5000;

  /* check sonar points for containment in rectangle */

  if (xy == 1)			/* +X direction */
    { for (i = start; i != end; ++i) /* put sonar readings into buckets */
	{ if (i > mx) {i = -1; continue; }    /* reset if beyond the end */
	  if (x_buf[i] >= df && y_buf[i] >= s1f && y_buf[i] <= s2f)
	    { if (x_buf[i] < ret) ret = x_buf[i]; }
	}
    }
  else if (xy == -1)		/* -X direction */
    { for (i = start; i != end; ++i) /* put sonar readings into buckets */
	{ if (i > mx) {i = -1; continue; }    /* reset if beyond the end */
	  if (x_buf[i] <= df && y_buf[i] >= s1f && y_buf[i] <= s2f)
	    { if ((- x_buf[i]) < ret) ret = - x_buf[i]; }
	}
    }
  else if (xy == -2)		/* -Y direction */
    { for (i = start; i != end; ++i) /* put sonar readings into buckets */
	{ if (i > mx) {i = -1; continue; }    /* reset if beyond the end */
	  if (y_buf[i] <= df && x_buf[i] >= s1f && x_buf[i] <= s2f)
	    { if ((- y_buf[i]) < ret) ret = - y_buf[i]; }
	}
    }
  else if (xy == 2)		/* +Y direction */
    { for (i = start; i != end; ++i) /* put sonar readings into buckets */
	{ if (i > mx) {i = -1; continue; }    /* reset if beyond the end */
	  if (y_buf[i] >= df && x_buf[i] >= s1f && x_buf[i] <= s2f)
	    { if (y_buf[i] < ret) ret = y_buf[i]; }
	}
    }


 if (xy == -2)		/* -Y direction */
    { 
      start = sr_buf->start;	/* right buffer */
      end = sr_buf->end;
      mx = sr_buf->limit;
      x_buf = sr_buf->xbuf;
      y_buf = sr_buf->ybuf;

      for (i = start; i != end; ++i) /* put sonar readings into buckets */
	{ if (i > mx) {i = -1; continue; }    /* reset if beyond the end */
	  if (y_buf[i] <= df && x_buf[i] >= s1f && x_buf[i] <= s2f)
	    { if ((- y_buf[i]) < ret) ret = - y_buf[i]; }
	}
    }
  else if (xy == 2)		/* +Y direction */
    {
      start = sl_buf->start;	/* left buffer */
      end   = sl_buf->end;
      mx    = sl_buf->limit;
      x_buf = sl_buf->xbuf;
      y_buf = sl_buf->ybuf;

      for (i = start; i != end; ++i) /* put sonar readings into buckets */
	{ if (i > mx) {i = -1; continue; }    /* reset if beyond the end */
	  if (y_buf[i] >= df && x_buf[i] >= s1f && x_buf[i] <= s2f)
	    { if (y_buf[i] < ret) ret = y_buf[i]; }
	}
    }





  return(ret);
}


