/*
 * plp.c
 *
 * written by Hynek Hermanski
 *
 * translated by f2c (version of 30 January 1990  16:02:04).
 * Modified by Chuck Wooters, ICSI, 7/6/90
 *
 * further modified at OGI -- array indices in some inner loops were
 * replaced by pointers.  the fft routine was replaces by TrigRecombFFT.
 *
 * Cleaned Up by Johan Schalkwyk (March 1993)
 *
 */

/* include file directives       */
#ifdef RAP_COMPILE
#include <rap.h>
#else
#include <stdio.h>
#include <stdlib.h>
#include <speech.h>
#include <math.h>
#endif

/* global constant definitions    */
#define two_to_the(N) (int)(pow(2.0,(N))+0.5)

/* external variable declaration  */
extern int errno, sys_nerr;
extern char *sys_errlist[];


/* Table of constant values       */
static double c_b17 = 0.33;
static double c_b28 = 10.0;


/* local module header declarations */
static int plp_ (float *speech, int nwind, int m, float *a, 
		 float *rc, float *gain, double sf);
static int hwind_ (float *weight, int npoint);
static int audw_ (int npoint, int *nfilt, float *cb, int *ibegen, double sf);
static int cosf_ (int m, int nfilt, float *wcos);
static int a2gexp_ (float *a, float *gexp, int i, int nc, double expon);



/*
 * 1/23/91 -- modified main to return data instead of write to stdout
 * mark fanty, OGI
 *
 */

/*
 * int ComputePLP(idata, num_idata, Win_offset, Win_size, Step_size,
 *                Sf, Expon, M, Np, Gain, plpout_p, numframes_p)
 *
 * Calculates the PLP coefficients of an input speech signal using the
 * algorithm of Hynek Hermanski.
 *
 * idata        (in): pointer to input speech data of type short.
 * num_idata    (in): number of data points available in array 
 * Win_offset   (in): adjustment to window relative to 0 start
 * Win_size     (in): analysis window in samples
 * Step_size    (in): step size in samples
 * Sf           (in): Sampling frequency (why double -- figure out)
 * Expon        (in): peak enhancement factor
 * M            (in): order of PLP model
 * Np           (in): Number of parameters to calculate
 * Gain         (in): gain flag (1 = gain; 0 no)
 * plpout_p    (out): pointer to output plp data of type float
 * numframes_p (out): return number of frames
 *
 * On exit returns (+1) for success and (-1) for failure
 *
 */

int ComputePLP(short *idata, int num_idata, int Win_offset, int Win_size, 
	       int Step_size, double Sf, double Expon, int M, int Np, 
	       int Gain, float ***plpout_p, int *numframes_p)
{
    static int   ibeg;                  /* Local variables                  */
 
    static float gain;                  /* gain parameter of model          */
    static float a[16];                 /* auto regressive coefficients     */
    static int   ianfr;                
    static int   nwind;                 /* number of windows in signal      */
    static int   istep;   
    static int   ii;                    /* loop counter variable            */
    static float rc[15];                /* reflection coefficients          */
    static float speech[512];           /* array of 512 samples in speech   */
           int   frame;

#ifdef RAP_COMPILE
DEFAULT_RAM = DRAM;
#endif

    nwind = Win_size;                   /* number of points in anal window  */
    istep = Step_size;                  /* number of points in each step    */

    ianfr = (num_idata - nwind) / istep + 1;   /* how many analysis frames  */
    *numframes_p = ianfr;	               /* return value              */

    /*
     * allocate memory for return vector of plp coefficients
     */

    *plpout_p = (float **) Alloc2d(ianfr, (int) Np ,sizeof(float));
    if(*plpout_p == NULL) {                   
          if( errno < sys_nerr ) 
	         ErrorString = sys_errlist[errno];
	  else
	         ErrorString = "ComputePLP:plpout_p - Alloc2d failed";
	return( -1 );
    }


    /* 
     * start of analysis loop                   
     */

    ibeg = Win_offset;
    for (frame = 0; frame < ianfr; frame++)
      {
	/* current beginning of the analysis window */
	for (ii = 1; ii <= nwind; ++ii)
	  {
	    if(ibeg + ii - 2 >= 0)
	      speech[ii - 1] = (float) idata[ibeg + ii - 2];
	    else
	      speech[ii - 1] = 0.0;
	  }

	/* 
	 * here is the analysis  
	 */
	plp_(speech, nwind, M, a, rc, &gain, Sf);
	if(Gain) {
	    for (ii = 0; ii < M + 1; ii++) {
		a[ii] = a[ii]/gain;
	      }
	  }

	a2gexp_(a, (*plpout_p)[frame], M, Np, Expon);
	(*plpout_p)[frame][0] = -(*plpout_p)[frame][0];

	ibeg += istep;
    }
    return (1);
}


/*
 * int plp_ (speech, nwind, m, a, rc, gain, sf)
 *
 * subroutine which computes m-th order (max 15) PLP model (described by 
 * m+1 autoregressive coefficients a and gain or by m reflection 
 * coefficients rc) from nwind-points (max 512) of speech signal with
 * sampling frequency sf (max 20 000 Hz)
 *
 * speech (in): pointer to speech signal (of type float)
 * nwind  (in): window (512 points typically) extractracted from speech
 * m      (in): order of model. (m+1) autoregressive coefficients
 * a      (in): pointer to auto regressive coefficients
 * rc     (in): pointer to reflection coefficients
 * gain   (in): pointer to model gain parameters
 * sf     (in): sampling frequency
 *
 */

static int plp_ (float *speech, int nwind, int m, float *a, 
		 float *rc, float *gain, double sf)
{
    static int icall = 0;               /* Initialized data                 */

    double d_1;                         /* temp double calculation variable */

    static float hwei[512];             /* hamming window -- weigh speech   */
    static int   nfft;                  /* number of points in fft          */
    static float wcos[368];	        /* was [23][16]                     */
    static int   nspt;
    static float r[16], s;              /* reflection coefficients          */
    static int   jfilt;                 /* loop counter variable            */
    static float rcmct;
    static int   nfilt;                 /* loop counter variable            */
    static float cb[900];
    static int   ib, ii, kk, mh, ibegen[69], ip; 	/* was [23][3]      */

    static float   alpmin;
    static float   audspe[23];          /* auditory spectrum coeff          */
    static float   spectr[512];         /* FFT spectrum                     */
    static int     npoint;              /* number of spectral points        */
    static int     nsize;                        
    static float   aib;
    static float   pai, aip, alp[17];
    static int     mct, mct2;
    static Complex cx[512], cy[512];

    Complex *cxp, *cyp, *cend;
    float   *sp;

    --speech;                            /* Parameter adjustments           */
    --a;
    --rc;

    if (icall == 0)                     /* if called for the first time,    */
      {
	hwind_(hwei, nwind);	        /* compute window array             */

	nfft = (int ) (log((double) (nwind)) / 0.693148) + 1;
                                   /* get fft size from the window length   */

	npoint = two_to_the((double)nfft) / 2 + 1;
                                   /* get number of spectral points         */
	nsize = two_to_the((double)nfft);

	audw_(npoint, &nfilt, cb, ibegen, sf);  
	                           /* compute spectral weights coefficients */

	cosf_(m, nfilt, wcos);	   /* compute IDFT coefficient table        */

	icall = 1;
      }

    for (ii = 1; ii <= nwind; ++ii) {        /*     window input speech     */
	speech[ii] = hwei[ii - 1] * speech[ii];
    }


    /**********************************/
    /*  compute speech power spectrum */
    /**********************************/

    /* pack real signal into complex array for fast TrigRecombFFT */
    cend = cx + nwind/2;
    sp = speech + 1;
    for(cxp = cx;cxp < cend;cxp++)
      {
	cxp->c_real = *sp++;
	cxp->c_imag = *sp++;
      }

    /* for odd length input */
    if( ((int)(nwind/2)) * 2 != nwind)
      {
	cxp->c_real = *sp;
	cxp->c_imag = 0.0;
	cxp++;
      }

    cend = cx + (1 << (nfft -1));
    for(;cxp < cend;cxp++)
      {
	cxp->c_real = 0.0;
	cxp->c_imag = 0.0;
      }

    /* whoops; need to check for odd.. */

    TrigRecombFFT(cx,cy,nfft-1);

    sp = spectr;
    cend = cy + (1 << (nfft -1));
    for(cyp = cy;cyp < cend;cyp++)
      *sp++ = cyp->c_imag * cyp->c_imag + cyp->c_real * cyp->c_real;


    /*****************************/
    /* compute auditory spectrum */
    /*****************************/
    {
      float *audp, *cbp, *specp, *specend;

      for (jfilt = 2; jfilt <= nfilt - 1; ++jfilt) {
	audp = audspe + (jfilt -1);
	*audp = 0.0;
	specp = spectr + (ibegen[jfilt - 1] - 1);
	specend = spectr + ibegen[jfilt + 22];
	cbp = cb + ibegen[jfilt + 45] - 1;
	while(specp < specend) {
	    *audp += *specp++ * *cbp++;
	}
      }
    }


    /*********************************************/
    /* cubic root intensity-loudness compression */
    /*********************************************/
      for (ii = 2; ii <= nfilt - 1; ++ii)
	{
	  d_1 = (double) audspe[ii - 1];
	  audspe[ii - 1] = (float) pow(d_1, c_b17);
	}

      /* the first and last point of auditory spectrum */
      audspe[0] = audspe[1];
      audspe[nfilt - 1] = audspe[nfilt - 2];


    /**************************************/
    /* inverse discrete fourier transform */
    /**************************************/
    {
      float rtmp, *audp, *audend, *wcp;

      nspt = nfilt - 1 << 1;
      for (kk = 1; kk <= m + 1; ++kk)
	{
	  rtmp = audspe[0];
	  audend = audspe + nfilt;

	  /* translation for indices and start point */
	  wcp = wcos + (2 + kk * 23 - 24);
	  for (audp = audspe + 1; audp < audend; audp++)
	    {
	      rtmp += *audp * *wcp++;
	    }
	  r[kk - 1] = rtmp/nspt;
	}
    }


    /*************************************/
    /* solution for autoregressive model */
    /*************************************/
    a[1]   = 1.0;
    alp[0] = r[0];
    rc[1]  = -r[1] / r[0];
    a[2]   = rc[1];
    alp[1] = r[0] + r[1] * rc[1];

    for (mct = 2; mct <= m; ++mct)
      {
	s    = 0.0;
	mct2 = mct + 2;
	alpmin = alp[mct - 1];

	for (ip = 1; ip <= mct; ++ip)
	  {
	    s += r[(mct2 - ip) - 1] * a[ip];
	  }

	rcmct = -s / alpmin;
	mh = mct / 2 + 1;

	for (ip = 2; ip <= mh; ++ip)
	  {
	    ib = mct2 - ip;
	    aip = a[ip];
	    aib = a[ib];
	    a[ip] = aip + rcmct * aib;
	    a[ib] = aib + rcmct * aip;
	  }
	a[mct + 1] = rcmct;
	alp[mct] = alpmin - alpmin * rcmct * rcmct;
	rc[mct] = rcmct;
      }
    *gain = alp[m];

    return (1);
} 


/*
 * int hwind_(weight, npoint)
 *
 * Computes the hamming window on speech signal.
 *
 * weight (in): pointer to input signal that has to be windowed.
 * npoint (in): number of points in window
 *
 */

static int hwind_ (float *weight, int npoint)
{
    static int ii;                         /* loop counter variable         */

    --weight;                              /* Parameter adjustments         */

    for (ii = 1; ii <= npoint; ++ii)
	weight[ii] = (float) (0.54 - 0.46 * cos(2.0 * M_PI * (ii - 1) / 
					(npoint - 1)));

    return 0;
} 


/*
 * int audw_ (npoint, nflit, cb, ibegen, sf)
 *
 * Computes auditory weighting functions           
 *
 * npoint (in): number of points in the fft spectrum                 
 * nfilt  (in): number of samples of auditory spectrum               
 *              equally spaced on the bark scale;                     
 *              1st filter at dc and last at the Nyquist frequency   
 *                                                                    
 * cb    (out): array of weighting coefficients to simulate               
 *              critical band spectral resolution                        
 *              and equal loudness sensitivity of hearing                
 *              on npoint speech power spectrum                          
 * ibegen(out): three-dimensional array which indicates where         
 *              to begin and where to end integration                
 *              of speech spectrum and where the given            
 *              weighting function starts in array cb                
 *              get Nyquist frequency in Bark                
 *
 */

static int audw_ (int npoint, int *nfilt, float *cb, int *ibegen, double sf)
{
    float  r_1, r_2;                          /* temp calculation variables */
    double d_1;

    static float freq, zdel, rsss;            /* Local variables            */
    static int i, j;
    static float x, z, f0, z0, f2samp, fh, fl, fnqbar;
    static int icount;
    static float fsq;

    --cb;                                     /* Parameter adjustments      */
    ibegen -= 24;                             /* [j=1 + 23 - 24] --> [0]    */

    d_1    = sf / 1200.0;                     
    fnqbar = (float) 6.0 * log(d_1 + sqrt(d_1 * d_1 + 1.0));  
    
    *nfilt = (int) fnqbar + 2;
                  /* compute number of filters for less than 1 Bark spacing */
    
    f2samp = (float) ((double) (npoint - 1)) / (sf / 2.0);
                  /* frequency -> fft spectral sample conversion            */
    
    zdel = fnqbar / (float) (*nfilt - 1);    /* compute filter step in Bark */
    
    icount = 1;                       /* loop over all weighting functions  */
    for (j = 2; j <= (*nfilt - 1); ++j)
      {
	ibegen[j + 69] = icount;      /* ibgen[23][3] --> [j][3] -->[j+69]  */ 
	
	z0 = zdel * (float) (j - 1);
                  /* get center frequency of the j-th filter in Bark        */
	
	f0 = (float) 600.0* (exp( ((double)z0)/6.0) - 
			     exp(-((double)z0)/6.0))/2.0;
                  /* get center frequency of the j-th filter in Hz          */
	
	fl = (float) 600.0* (exp( ( ((double)z0) - 2.5)/6.0) - 
			     exp(-( ((double)z0) - 2.5)/6.0))/2.0;
                  /* get low-cut frequency of j-th filter in Hz             */

	r_1 = fl * f2samp;
	ibegen[j + 23] =  ((int) (r_1+0.5)) + 1;    /* ibgen[j.1] -> [j+23] */
	if (ibegen[j + 23] < 1)
	    ibegen[j + 23] = 1;
	
	fh = (float) 600.0*(exp( ( ((double)z0) + 1.3)/6.0) - 
			    exp(-( ((double)z0) + 1.3)/6.0)) /2.0;
                   /* get high-cut frequency of j-th filter in Hz           */

	r_1 = fh * f2samp;
	ibegen[j + 46] = ((int) (r_1 +0.5)) + 1;   /* ibgen[j.2] -> [j+46]  */
	if (ibegen[j + 46] > npoint)
	    ibegen[j + 46] = npoint;
	
	for (i = ibegen[j + 23]; i <= ibegen[j + 46]; ++i)
	  {
	    freq = ((float) (i - 1)) / f2samp;
                            /* get frequency of j-th spectral point in Hz   */
	    
	    x = freq / 600.0;
                            /* get frequency of j-th spectral point in Bark */
	    
	    d_1 = (double) x;
	    z = (float) 6.0 * log(d_1 + sqrt(d_1 * d_1 + 1.0));
	    
	    z -= z0;	    /* normalize by center frequency in barks:      */

	    if (z <= -0.5)                             /* compute weighting */
	      {
		d_1 = (double) ( ((double)z) + 0.5);
		cb[icount] = (float) pow(c_b28, d_1);
	      }
	    else if (z >= 0.5)
	      {
		d_1 = (-2.5) * ( ((double)z) - 0.5);
		cb[icount] = (float) pow(c_b28, d_1);
	      }
	    else
	      {
		cb[icount] = 1.0;
	      }
	    
	    /*     calculate the 40 db equal-loudness curve        */
	    /*     at center frequency and add it to the weighting */

	    fsq = f0 * f0;
	    r_2 = fsq + 1.6e5;
	    rsss = fsq * fsq * (fsq + 1.44e6) / (r_2 * r_2 * (fsq + 9.61e6));
	    
	    cb[icount] = rsss * cb[icount];
                       /* add the qual-loundness curve to the weighting    */
	    ++icount;
	}
    }
    return (0);
} 


/*
 * int cosf_(m, nfilt, wcos)
 *
 * computes the cosine weightings for the Inverse Discrete Fourier Transfrom
 *
 * m    (in): order of plp auto regressive model
 * nfilt(in):
 * wcos (in): cosine weightings of inverse dft
 *
 */
 
static int cosf_ (int m, int nfilt, float *wcos)
{
    static int ii, jj;                       /* Loop counter variables     */

    wcos -= 24;                              /* Parameter adjustments      */
                                             /* [23][16] --> [j=1+24]      */

    for (ii = 1; ii <= (m+1); ++ii) 
      {
	for (jj = 2; jj <= (nfilt-1); ++jj) 
	  {
	    wcos[jj + ii * 23] = 2.0 * cos(2.0 * M_PI * (ii - 1) * (jj - 1) / 
					   (2.0 * ( nfilt - 1 )));
	  }
	wcos[nfilt + ii * 23] = cos(2.0 * M_PI * (ii - 1) * (jj - 1) /
				    (2.0 * ( nfilt - 1 )));
      }

    return (0);
} 


/* 
 * int a2gexp_ (a, gexp, i, nc, expon)
 *
 *
 * a    (in): pointer to autoregressive coefficients array
 * gexp(out):
 * i    (in): number of elements in autoregressice coefficient array
 * nc   (in): number of elements in output array
 * expon(in):  
 */

static int a2gexp_ (float *a, float *gexp, int i, int nc, double expon)
{
    int i_2;      

    static float c[257];                       /* Local variables          */
    static int j, l, jb, ii, lp;
    static float sum;

    --a;                                       /* Parameter adjustments    */
    --gexp;

    c[0] = log(a[1]);
    c[1] = -a[2] / a[1];
    for (l = 2; l <= nc; ++l)
      {
	lp = l + 1;
	if (l <= i + 1)
	    sum = l * a[lp] / a[1];
	else
	    sum = 0.0;
	i_2 = l;
	if(l < i + 1) i_2 = l;
	else i_2 = i + 1;
	for (j = 2; j <= i_2; ++j)
	  {
	    jb = l - j + 2;
	    sum += a[j] * c[jb - 1] * (jb - 1) / a[1];
	  }
	c[lp - 1] = -sum / l;
      }
    gexp[1] = c[0];

    if (expon != 0.0)
      for (ii = 2; ii <= nc; ++ii)
	  gexp[ii] = (float) pow((double)(ii-1),expon ) * c[ii - 1];
    else
      for (ii = 2; ii <= nc; ++ii)
	  gexp[ii] = c[ii - 1];

    return (0);
} 

