/*
 * lpc.c
 *
 * This module implements the following procedures for
 * calculating acoustic parameters.
 *
 * 1. speech-wave
 * 2. frame by frame
 * 3. Hamming window
 * 4. auto-correlation
 * 5. pre-emphasis
 * 6. LPC-coefficients
 * 7. LPC-auto-correlation
 * 8. LPC cepstrum
 * 9. LPC-delta-cepstrum
 *
 */

/* include file directives */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <speech.h>

/* external variable declarations */
extern int  errno, sys_nerr;
extern char *sys_errlist[];


/*
 * eerror(where)
 *
 * Prints an error message on screen and exits program
 *
 * Usage: eerror("main:LPC");
 *
 */

void eerror(char *where)
{
   printf("Error: %s\n", where);
   exit(1);
}


/*
 * Hamming(speech, windowed, Samples)
 *
 * Calculates the hamming window. Used to weight the speech
 * signal with. Used to define a frame of speech (window)
 *
 * Speech  (in): pointer to speech signal
 * window (out): pointer to allocated memory for window speech
 * Samples (in): number of points to calculate.
 *
 * Usage: Hamming(Speech, HammingWindow, 128);
 *
 */

void Hamming(float *speech, float *windowed, int Samples)
{
  register int i, n;

  n = Samples;
  for (i=0; i<n; i++) 
    windowed[i] = 0.54 - 0.46*cos(2.0 * M_PI * ((double) i)/((double) (n-1)));

  for (i=0; i<n; i++) 
    windowed[i] *= speech[i];
}


/*
 * void PreEmphasize(Speech, PreAmp, Samples)
 *  
 * Procedure pre emphasize a given window of speech.
 *
 * Speech (in): pointer to speech samples of type float
 * PreAmp(out): pointer to pre emphasized output signal, must be allocated
 * Samples(in): Number of samples in to process
 *
 * Usage: PreEmphasize(Signal, PreEmp, 128)
 *
 */

void PreEmphasize(float *Speech, float *PreAmp, int Samples)
{
   int   i;

   PreAmp[0] = Speech[0];
   for (i = 1; i < Samples; i++)
      PreAmp[i] = Speech[i] - 0.98*Speech[i - 1];    
}


/*
 *  AutoCorrelation(Signal, Samples, m, CorCoeff)
 *
 * Calculate the autocorrelationmatrix for a given signal. These
 * auto correlation coefficients are then used to determine the
 * Linear Predictor Coefficients
 *
 * Signal  (in): pointer to input speech signal of type float
 * Samples (in): size of input window (number of speech samples)
 * order   (in): the order, number of correlation coefficients to return
 * CorCoeff(out): pointer to correlation coeficient vector, must be allocated
 *                size = (m+1)
 * 
 * Usage: AutoCorrelation(PreAmp, 128, 12, Correlations)
 *
 */

void AutoCorrelation(float *Signal, int Samples, int m, float *CorCoeff)
{
   int   i, j;
   float Normal;

   for (i = 0; i <= m; i++) {
      CorCoeff[i] = 0.0;
      for (j = 0; j < Samples - i; j++)
	 CorCoeff[i] += Signal[j] * Signal[j + i];
   };

/* Now normalizing the autocorrelation.  */
   Normal = CorCoeff[0];

   for (i = 0; i <= m; i++)
     CorCoeff[i] /= Normal;
}


/*
 * DUrbin(p, CorrCoef, LPC, Reflect)
 *
 * Function to compute LPC-parameters with D'Urbin's formula. The
 * LPC coefficients and reflection coefficients are calculated using
 * the AutoCorrelation coefficient matrix
 *
 * p       (in): number of LPC coefficients (p<=m)
 * CorrCoef(in): pointer to Auto Correlation coefficients (m)
 * LPC    (out): Output LPC-coefficients, must be allocated, size --> (p+2)
 * Reflect(out): Corresponding reflection coefficients, must be allocated
 *               size (p+2)
 *
 * Usage: DUrbin (12, Correlation, LPCcoeff, Reflection);
 *
 */

void DUrbin(int p, float *CorrCoeff, float *LPC, float *Reflect)
{
   int       i, j;
   float     Error, alpha;
   float    *A;

   if (!(A = (float *)calloc((p + 1), sizeof(float)))) 
     eerror("DUrbin:A");

   LPC[1] = 1.0;
   Reflect[1] = -CorrCoeff[1]/CorrCoeff[0];
   alpha = CorrCoeff[0]*(1 - Reflect[1]*Reflect[1]);
   if (alpha == 0)
     printf("alpha == 0\n");
   LPC[2] = Reflect[1];

   for (i = 2; i <= p; i++) {
      Error = 0.0;

      for (j = 1; j <= i; j++)
	 Error += LPC[j]*CorrCoeff[i + 1 - j];

      Reflect[i] = -Error/alpha;
      alpha *= (1 - Reflect[i]*Reflect[i]);
      LPC[i + 1] = Reflect[i];

      for (j = 2; j <= i; j++)
	 A[j] = LPC[j] + Reflect[i]*LPC[i + 2 - j];

      for (j = 2; j <= i; j++)
	 LPC[j] = A[j];
   };

   LPC[0] = 1.0;
   for (i = 1; i <= p; i++)
      LPC[i] = -LPC[i + 1];

   free((char *)A);
}


/*
 * LPC_AutoCorrelation(n, m, p, LPC, CorCoeff)
 *
 * Extended auto correlation coefficients.
 *
 * n        (in): the number of extended coefficients = (n-p)
 * m        (in): the number of normal auto correlation coefficients
 * p        (in): number of LPC coefficients
 * LPC      (in): LPC coefficients
 * CorCoeff(out): output coefficients must be of dimension (n-p)
 *
 * Usage: LPC_AutoCorrelation(16, 13, 10, LPC, Correlations)
 *
 */

void LPC_AutoCorrelation(int n, int m, int p, float *LPC, float *CorCoeff)
{
  int   i, k;
  float sum;

  for (i=m+1; i<=n; i++) {
    sum = 0.0;
    for (k=1; k<=p; k++) 
      sum += LPC[k]*CorCoeff[i-k];
    CorCoeff[i] = sum;
  }
}


/*
 * LPC_Cepstrum(n, p, LPC, LPCCep)
 *
 * Calculates the LPC cepstrum coefficients.
 *
 * n     (in): number of cepstrum coefficients to calculate
 * p     (in): number of LPC parameters available
 * LPC   (in): pointer to LPC parameters
 * LPCCep(out): pointer to Cepstrum Coefficients of dimension (n)
 *
 * Usage: LPC_Cepstrum(16, 10, LPC, Cepstrum)
 *
 */

void LPC_Cepstrum(int n, int p, float *LPC, float *LPCCep)
{
  int   i, k;
  float sum;
  float *hhat;

  if (!(hhat = (float *) calloc(n + 1, sizeof(float))))
    eerror("LPC+Cepstrum:hhat");

    /* Calculate Complex cepstrum */
  hhat[1] = LPC[1];

  for (i=2; i<=n ; i++) {
    if (i<=p)
      sum = LPC[i];
    else
      sum = 0.0;
    for (k=1; k <= i-1; k++) {
      if (i-k <= p)
	sum += ((float)k)/((float)i) * hhat[k] * LPC[i-k];
    }
    hhat[i] = sum;
  }

  /* save LPC Cepstrum */
  LPCCep[1] = 0.0;
  for (i=1; i<=n; i++)
    LPCCep[i] = hhat[i];

  free ((char *) hhat);
}


/* 
 * DeltaCEP(n, LPCCep, DelCEP, reset)
 *
 * Calculates the slope through the LPC cepstral coefficients
 * through the points c(t-q), c(t), c(t+q). Due to the
 * time diff, the lag for correct answers are 2q.
 *
 * n      (in): number of LPC Cepstrum coefficients available
 * LPCCEP (in): LPC cepstral coefficients
 * DelCep(out): Delta Cepstral coefficients
 * reset  (in): flag to indicate to reset the time delay flag
 *
 * Usage: DeltaCEP(16, LPCCep, DelCep, 1)
 *
 */

void DeltaCEP(int n, float *LPCCep, float *DelCep, int reset)
{
  static float Cprev[50];
  static int   flag = 0;

  int   i, j;
  float grad;

  if (reset)
    flag = 0;

  if (!flag) {
    for (i=0; i<=n; i++)
      Cprev[i] = LPCCep[i];
    flag = 1;
    return;
  }

  /* Calculate gradient through points t, t-q */
  for (i=0; i<=n; i++) {
    grad  = LPCCep[i] - Cprev[i];
    DelCep[i] = grad;
  }

  /* Save LPC Cepstral coefficients as prev */
  for (i=0; i<=n; i++)
    Cprev[i] = LPCCep[i];
}


/*
 * int LPC_Analyse(speech, length, window, step, fSamp, 
 *                 Ri, Ai, RFi, Ci, DelCi, m, n, p)
 *
 * Executes a series of LPC analysis techniqeus on an incoming speech
 * signal. These include
 *   Auto Correlation Coefficients
 *   LPC Coefficients
 *   Extended Auto Correlation Coefficients
 *   LPC Cepstrum
 *   Delta Cepstrum
 *
 * speech  (in): pointer to speech signal of type short
 * length  (in): size of speech signal in samples
 * window  (in): size of window in milli seconds
 * step    (in): step size in milli seconds
 * fsamp   (in): sampling frequency, in Hz
 * Ri     (out): Auto Correlation coefficients (pointer to 2D array)
 * Ai     (out): LPC coefficients matrix
 * RFi    (out): reflection coefficients
 * Ci     (out): LPC cepstrum coefficients matrix
 * DelCi  (out): Derivative of LPC cepstrum over time
 * m       (in): order of auto correlation coefficients returned
 * n       (in): number of total Correlation coefficients (n-m = LPC-AutoCorr)
 * p       (in): order of LPC filter
 *
 * On exit returns the number of frames calculated, or (-1) for error
 *
 */

int LPC_Analyses(short *speech, int length, float window, float step, 
		 float fSamp, float ***Ri, float ***Ai, float ***RFi,
		 float ***Ci, float ***DelCi, int m, int n, int p)
{
  float *signal, *preemp, *hamm;        /* speech signal frame per frame  */
  int   win, st;                        /* step size and window size      */
  int   numWin;
  int   i, j, k;

  win = (int) ((window /1000.0) * fSamp);
  st  = (int) ((step / 1000.0) * fSamp);
  numWin = length / st;

  /* allocate memory for speech windows */
  if (!(signal = (float *) malloc(win * sizeof(float)))) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "LPC_Analyses:signal - malloc failed";
    return (-1);
  }

  if (!(preemp = (float *) malloc(win * sizeof(float)))) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "LPC_Analyses:preemp - malloc failed";
    return (-1);
  }

  if (!(hamm = (float *) malloc(win * sizeof(float)))) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "LPC_Analyses:hamm - malloc failed";
    return (-1);
  }

  /* Allocate memory for Coefficient arrays */
  if (!((*Ai) = (float **) Alloc2d(numWin+1, p+1, sizeof(float)))) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "LPC_Analyses:Ai - malloc failed";
    return (-1);
  }

  if (!((*RFi) = (float **) Alloc2d(numWin+1, p+1, sizeof(float)))) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "LPC_Analyses:Rfi - malloc failed";
    return (-1);
  }

  if (!((*Ri) = (float **) Alloc2d(numWin+1, n+2, sizeof(float)))) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "LPC_Analyses:Ri - malloc failed";
    return (-1);
  }

  if (!((*Ci) = (float **) Alloc2d(numWin+1, n+1, sizeof(float)))) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "LPC_Analyses:Ci - malloc failed";
    return (-1);
  }

  if (!((*DelCi) = (float **) Alloc2d(numWin+1, n+1, sizeof(float)))) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "LPC_Analyses:DelCi - malloc failed";
    return (-1);
  }

  /* do Calculations */
  for (i=st, k=0; i<length; i+= st, k++) {
    for (j=0; j<win; j++)
      signal[j] = (float) speech[i-st+j];

    PreEmphasize(signal, preemp, win);
    Hamming(preemp, hamm, win);
    AutoCorrelation(hamm, win, m, (*Ri)[k]);
    DUrbin(p, (*Ri)[k], (*Ai)[k], (*RFi)[k]);
    LPC_AutoCorrelation(n, m, p, (*Ai)[k], (*Ri)[k]);
    LPC_Cepstrum(n, p, (*Ai)[k], (*Ci)[k]);
    DeltaCEP(n, (*Ci)[k], (*DelCi)[k], 0);
  }

  free ((char *) signal);
  free ((char *) preemp);
  free ((char *) hamm);
  return (numWin);
}

