/*
 * $Id: dft.c,v 2.4 1993/06/09 20:58:40 johans Exp $
 *
 * fft.c
 *
 */


/* include file directives */
#include <malloc.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>

/* C library include file directives */
#include <speech.h>


/* external variable declaration */
extern int errno;                       /* System error number               */
extern int sys_nerr;                    /* index in sys error message list   */
extern char *sys_errlist[];             /* pointer to syster error message   */

/* local module header declarations */
static int CalculateW (int m );
static int CalculateCF( int m );

int FloatFFT (float *x, Complex *y, int m);
int TrigRecombFFT (Complex *cx, Complex *y, int m);
int FFT( Complex *x, int m );


/* local module variable declarations */
static Complex *W = NULL;               /* used by FFT() to hold W twiddle   */
static int MofW = 0;

static Complex *CF = NULL;             /* Trigonometrix Recombination Coeff  */
static int MofCF = 0;


/*
 * ComputeDFT( wav, wavlen, offset, winsize, int, coeffs, alpha, numframes_p )
 *
 * This procedure computes the discrete fourier transform on a signal
 * using speech as basis. The following filtering is done on the speech
 * signal before the dft commences:
 *    Pre Emphasis
 *    Hanning Windowing
 *
 * parameters:
 *  wav    (in): waveform data, pointer of type short
 *  wavlen (in): length of wav (number of sample points in wav[])
 *  offset (in): offset relative to start of data
 *  winsize(in): window size in data points
 *  inc    (in): increament in data points
 *  coeffs (in): number of coefficients in resulting DFT
 *  alpha  (in): alpha value to use in preemphasis; 
 *               if 0 then no preemphasis is done
 * numframes_p (out): number of usefule feature frames.
 *
 * returns:
 *  a pointer to the resulting DFT (Alloc2d type) otherwise
 *  a NULL is returned and ErrorString set.
 *
 */

short **ComputeDFT (short *wav, int wavlen, int offset, int winsize, int inc, 
		    int coeffs, double alpha, int *numframes_p)
{
  Complex *y;                       /* FFT of speech signal           */
  short **dft;                      /* pointer to dft structure       */
  float *twav;                      /* waveform converted to floats   */
  float start;                      /* offset value in signal - start */
  int iwav;                    
  int i, j;                         /* loop counter variables         */
  double val;                   
  int ncoeffs;                      /* number of coefficients compute */
  int ndft;                         /* number of coefficients in dft  */
  int m;


  /* Find log base 2 of coeffs * 2	      */
  ncoeffs = coeffs * 2;
  m = (int)( log( (double)ncoeffs ) / log( 2.0 ) );

  /* actual number of useful frames */
  *numframes_p = (((wavlen - offset) - winsize) + inc) / inc;

  /* allocate memory for floating point wave    */
  ndft = (wavlen - offset)/inc;
  if( ncoeffs > winsize ) {
    twav = (float *)malloc( sizeof(float) * ncoeffs );
  } else {
    twav = (float *)malloc( sizeof(float) * winsize );
  }
  if( twav == NULL ) {
    if( errno < sys_nerr ) {
      ErrorString = sys_errlist[errno];
    } else {
      ErrorString = "ComputeDFT:twav - malloc failed";
    }
    return( NULL );
  }

  /* allocate memory for FFT return             */
  y = (Complex *)malloc( sizeof(Complex) * coeffs );
  if( y == NULL ) {
    if( errno < sys_nerr ) {
      ErrorString = sys_errlist[errno];
    } else {
      ErrorString = "ComputeDFT:y - malloc failed";
    }
    free( (char *) twav);
    return( NULL );
  }
  
  /* allocate memory for dft per segment        */
  dft = (short **)Alloc2d( ndft, coeffs, sizeof(short) );
  if( dft == NULL ) {
    if( errno < sys_nerr ) {
      ErrorString = sys_errlist[errno];
    } else {
      ErrorString = "ComputeDFT:dft - Alloc2d failed";
    }
    free( (char *) twav);
    free( (char *) y);
    return( NULL );
  }
  
  /* 
   * set start value for preemphasis
   */
  start = 0.0;
  iwav = offset;

  for( i = 0; i < ndft; i++ ) {
    /* 
     * load temp wav array                
     */
    if( (iwav + winsize) > wavlen ) {
      
      for( j = 0; j < (iwav+winsize) - wavlen; j++ ) {
	twav[j] = (float)wav[j+iwav];
      }
      
      for( ; j < winsize; j++ ) {
	twav[j] = 0.0;
      }
      
    } else {
      for( j = 0; j < winsize; j++ ) {
	
	if( (j+iwav) >= 0 ) {
	  twav[j] = (float)wav[j+iwav];
	} else {
	  twav[j] = 0.0;
	}
      }
    }
    
    /* 
     * Do Preemphasis                      
     */
    
    if( alpha != 0.0 ) {
      Preemphasis( alpha, start, twav, winsize );
    }

    /*
     * Do Windowing                        
     */

    HanningWindow( twav, winsize );
    
    /* 
     * Do overlap or pading                
     */
    if( ncoeffs < winsize ) {
      
      for( j = ncoeffs; j < winsize; j++ ) {
	twav[j-ncoeffs] += twav[j];
      }
      
    } else {
      
      for( j = winsize; j < ncoeffs; j++ ) {
	twav[j] = 0.0;
      }
    }
    
    /* 
     * Compute DFT                        
     */
    if (FloatFFT( twav, y, m ) < 0) {
      free( (char *) twav);
      free( (char *) y);
      Free2d( (char **) dft);
      return( NULL );
    }
	       
    /*
     * Convert to db                      
     */
    
    for( j = 0; j < coeffs; j++ ) {
      
      val = ( y[j].c_real * y[j].c_real ) +
	( y[j].c_imag * y[j].c_imag );
      
      /******  take out second condition when
       * correct calling sequence and ndft computed
       */
      if( val == 0.0 || (iwav + winsize > wavlen)) {
	dft[i][j] = 0;
      } else {
	dft[i][j] = (short) ( 10.0 * log10( val ) );
      }
    }
    
    iwav += inc;
    start = (float) *(wav+iwav-1);
  }

  free((char *) twav );
  free((char *) y );

  return( dft );
}



/*
 * int FloatFFT( x, y, m )
 *
 * Computes the Fast Fourier Transform of a signal represent by
 * a sequence of floating point values.
 *
 * x  (in): pointer to input signal of type float 
 * y (out): pointer to output FFT signal of type complex
 * m  (in): number of points in signal (2^x), where x = number of points
 *
 * On exit returns (+1) for success and (-1) for failure.
 *
 * Usage: success = FloatFFT(InSignal, OutFFT, 8)
 *
 */

int FloatFFT (float *x, Complex *y, int m)
{
	Complex *cx;                     /* Computer FFT output (temp space) */
	int     p, j, k;                 /* Loop counter variables           */
	int     num;                     /* Number of DFT coefficients       */

	p   = m - 1;                     /* number complex points in signal  */
	num = 1 << p;                    /* 2^p = total complex points       */

	/* allocate memory for trigonometrix recombination input   */
	cx = (Complex *) calloc (num, sizeof(Complex));   
	if( cx == NULL ) {
		if( errno < sys_nerr ) {
			ErrorString = sys_errlist[errno];
		} else {
			ErrorString = "FloatFFT:cx malloc failed";
		}
		return( -1 );            /* memory allocation failure        */
	}
	
	/* save signal for in combined format                      */
	j = 0;
	for( k = 0; k < num; k++ ) {     /* for all complex points in signal */

		cx[k].c_real = x[j];     /* real      - 0,2,4,...            */
		cx[k].c_imag = x[j+1];   /* imaginary - 1,3,5,...            */

		j += 2;
	}

	if( TrigRecombFFT( cx, y, m-1 ) < 0 ) { /* Compute trig recombine FFT*/
		return( -1 );              /* failure of calling procedure   */
	} else {
	  y[num-1].c_real = 0.0;
	  y[num-1].c_imag = 0.0;  
	  for (j=0; j<2*num; j+=2) {       /* Calculate at Half Nyquist      */
	    y[num-1].c_real += x[j];
	    y[num-1].c_real -= x[j+1];
	  }
	}
	
	free((char *) cx );                /* free allocated memory - exit   */
	return( 1 );
}


/*
 * int TrigRecombFFT( cx, y, m )
 *
 * Computers the FFT of a complex signal, using a trigonometric recombination
 * principle in order to speed up matters somewhat.
 *
 * cx (in): pointer to input complex signal of type Complex
 * y (out): pointer to output FFT signal of type Complex
 * m  (in): the #point fft to computer (2^x format)
 *
 * On exit returns (+1) for success and (-1) for failure
 *
 * Usage: success = TrigRecombFFT( InSignal, FFTout, 8)
 *
 */

int TrigRecombFFT (Complex *cx, Complex *y, int m)
{
	Complex *ck, *xk, *xnk;             /* array pointer variables      */
	float Realsum, Realdif;             /* real part of trig recombine  */
	float Imagsum, Imagdif;             /* imag part of trig recombine  */
	int num, k, j, i;                   /* loop counter variables       */

	/* Do FFT on combined data                  */
	num = 1 << m;
	FFT( cx, m );

	/* Calculate Recombination Factors          */
	if( m != MofCF ) {
		if( CalculateCF( m ) < 0 ) {
			return( -1 );
		}
	}

	/* DC component, no multiplies              */
	y[0].c_real = cx[0].c_real + cx[0].c_imag;
	y[0].c_imag = 0.0;

	/* other frequencies by trig recombination  */
	ck = CF;
	xk = cx + 1;
	xnk = cx + num - 1;

	for( k = 1; k < num; k++ ) {
		Realsum = ( xk->c_real + xnk->c_real ) / 2.0;
		Imagsum = ( xk->c_imag + xnk->c_imag ) / 2.0;
		Realdif = ( xk->c_real - xnk->c_real ) / 2.0;
		Imagdif = ( xk->c_imag - xnk->c_imag ) / 2.0;

		y[k].c_real = Realsum + ck->c_real * Imagsum -
					ck->c_imag * Realdif;

		y[k].c_imag = Imagdif - ck->c_imag * Imagsum -
					ck->c_real * Realdif;

		ck++;
		xk++;
		xnk--;
	}

	return( 1 );
}


/*
 * FFT( x, m )
 *
 * In-place radix 2 decimation in time FFT
 * Computes the Fast Fourier transform of a signal
 * 
 * x (in): pointer to complex array, power of 2 size of FFT
 * m (in): size fo fft = 2^m
 *
 * Places FFT output on top of input COMPLEX array.
 * 
 */

int FFT( Complex *x, int m )
{
	Complex u, temp, tm;
	Complex *xi, *xip, *xj, *wptr;
	int i, j, k, l, le, windex;
	int n;

	/* Calculate Twiddle factors for FFT */
	if( m != MofW ) {
		if( CalculateW( m ) < 0 ) {
			return( -1 );
		}
	}

	/* n = 2^m = fft length   */
	n = 1 << m;

	/* start fft              */
	le = n;
	windex = 1;
	for( l = 0; l < m; l++ ) {
		le = le / 2;
		
		/* first iteration with no multiplies */
		for( i = 0; i < n; i = i + 2*le ) {
			xi = x + i;
			xip = xi + le;
			temp.c_real = xi->c_real + xip->c_real;
			temp.c_imag = xi->c_imag + xip->c_imag;
			xip->c_real = xi->c_real - xip->c_real;
			xip->c_imag = xi->c_imag - xip->c_imag;
			*xi = temp;
		}

		/* remaining iterations use to store w */
		wptr = W + windex - 1;
		for( j = 1; j < le; j++ ) {
			u = *wptr;
			for( i = j; i < n; i = i + 2*le ) {
				xi  = x + i;
				xip = xi + le;
				temp.c_real = xi->c_real + xip->c_real;
				temp.c_imag = xi->c_imag + xip->c_imag;
				tm.c_real = xi->c_real - xip->c_real;
				tm.c_imag = xi->c_imag - xip->c_imag;
				xip->c_real = tm.c_real * u.c_real - 
							tm.c_imag * u.c_imag;
				xip->c_imag = tm.c_real * u.c_imag +
							tm.c_imag * u.c_real;
				*xi = temp;
			}
			wptr = wptr + windex;
		}
		windex = 2 * windex;
	}

	/* rearrage data by bit reversing */
	j = 0;
	for( i = 1; i < ( n - 1 ); i++ ) {
		k = n / 2;
		while( k <= j ) {
			j = j - k;
			k = k / 2;
		}
		j = j + k;
		if( i < j ) {
			xi = x + i;
			xj = x + j;
			temp = *xj;
			*xj = *xi;
			*xi = temp;
		}
	}

	return( 1 );
}


/* 
 * static int CalculateW( m )
 *
 * This procedure calculates the W twiddle factors for the 
 * FFT routine, thus needed only once. Then check to see if already
 * called, and do not call again.
 * 
 * m (in): number of points in FFT = (2^m)
 *
 */

static int CalculateW (int m )
{
	double wrecur_real, wrecur_imag;  /* twiddle factors recurrence vars */
	double w_real, w_imag;            /* current twiddle factor          */
	double wtmp_real;                 
	double arg;                       /* current angle for sine/cosine   */
	Complex *xj;                      /* pointer variable to W array     */
	int n, le, j;                     /* loop counter variables          */

	/* if memory allocated, free allocated memory              */
	if( W != NULL ) {
		free( (char *) W );
	}

	MofW = m;

	n = 1 << m;
	le = n / 2;
	
	/* Allocate memory for FFT twiddle factors                */
	W = (Complex *)calloc( le - 1, sizeof( Complex ) );
	if( W == NULL ) {
		if( errno < sys_nerr ) {
			ErrorString = sys_errlist[errno];
		} else {
			ErrorString = "CalculateW:W calloc failed";
		}
		return( -1 );
	}

	/* Calculate FFT twiddle factors here                     */
	arg = 4.0 * atan( 1.0 ) / ((double)le);
	wrecur_real = w_real = cos( arg );
	wrecur_imag = w_imag = -sin( arg );
	xj = W;

	for( j = 1; j < le; j++ ) {

		xj->c_real = (float)wrecur_real;
		xj->c_imag = (float)wrecur_imag;

		xj++;
		wtmp_real = wrecur_real * w_real - wrecur_imag * w_imag;
		wrecur_imag = wrecur_real * w_imag + wrecur_imag * w_real;
		wrecur_real = wtmp_real;
	}

	return( 1 );
}


/*
 * static int CalculateCF( m )
 *
 * Calculates the Trigonometric Recombination Coefficients.
 * Used only once, and never called again. With multiple FFT calls
 * saves computation time.
 *
 * m (in): number of points in FFT transform = (2^m)
 *
 */

static int CalculateCF( int m )
{
	double arg, factor;
	int num, k;

	/* if memory is allocated, free allocated memory               */
	if( CF != NULL ) {
		free( CF );
	}

	MofCF = m;
	num = 1 << m;

	/* allocate memory for trigonometric recombination coefficient */
	CF = (Complex *)calloc( num - 1, sizeof(Complex) );
	if( CF == NULL ) {
		if( errno < sys_nerr ) {
			ErrorString = sys_errlist[errno];
		} else {
			ErrorString = "CalculateCF:CF calloc failed";
		}
		return( -1 );
	}

	/* calculate trigonometric recombination coefficients          */
	factor = 4.0 * atan( 1.0 ) / ((double)num);
	for( k = 1; k < num; k++ ) {

		arg = factor * ((double)k);

		CF[k-1].c_real = (float)cos( arg );
		CF[k-1].c_imag = (float)sin( arg );
	}

	return( 1 );
}



