/*
 * $Id: asciihdr.c,v 2.2 1993/05/07 18:12:22 johans Exp $
 *
 * header.c
 *
 */

/* standard C library include file directives */
#include <string.h>
#include <ctype.h>
#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>

/* speech library include file directivs */
#include <speech.h>

/* external varaible declaration */
extern int errno;
extern int sys_nerr;
extern char *sys_errlist[];

#define READLINE_LENGTH         (1024)

/* local module header declarations */
static char **ReadArrayData( FILE *fp, int *len );
static void SplitLine (char *line, char **key, char **value );
static char *CleanStr (char *str );
static char *GetLine( FILE *fp );
static void ErrorFreeHeader( AsciiHdr *hdr, int len );


/*
 * ReadHeader( fp )
 *
 *  Reads header from fp.  returns pointer to read header.
 *
 * parameters:
 *   fp - stream to read from
 *
 * returns:
 *  pointer to AsciiHdr array or NULL on errors.
 *
 */

AsciiHdr *ReadHeader( FILE *fp )
{
  AsciiHdr *hdr;
  char     **ad;
  int      size;
  char     *line;
  char     *key, *value;
  int      len;

  /* allocate memory for Ascii header */
  hdr = (AsciiHdr *)malloc( sizeof(AsciiHdr) );
  if( hdr == NULL ) {
    if( errno < sys_nerr ) {
      ErrorString = sys_errlist[errno];
    } else {
      ErrorString = "malloc failed";
    }
    return( NULL );
  }

  hdr->ah_key = NULL;
  hdr->ah_value = NULL;
  size = 1;

  /* read until end of header is reached */
  while( 1 ) {
    line = GetLine( fp );
    SplitLine( line, &key, &value );

    /* if header starts with NIST_1A it is not an Ascii header */
    if ( strcmp (key, "NIST_1A" ) == 0) {
      return (NULL);
    }

    if( strcmp( key, "END OF HEADER" ) == 0 ) {
      free( line );
      break;
    }
    
    if( strcmp( key, "Begin" ) == 0 ) {

      ad = ReadArrayData( fp, &len );
      
      if( ad == NULL ) {
	return( NULL );
      }
      
      (hdr+size-2)->ah_value = ad;
      (hdr+size-2)->ah_vlen = len;
      
    } else {
      
      size++;
      hdr = (AsciiHdr *)realloc( (char *)hdr,
				sizeof(AsciiHdr) * size );

      if( hdr == NULL ) {
	if( errno < sys_nerr ) {
	  ErrorString = sys_errlist[errno];
	} else {
	  ErrorString = "realloc failed";
	}
	return( NULL );
      }
      
      (hdr+size-2)->ah_key = (char *)malloc( strlen( key ) + 1 );
      
      if( (hdr+size-2)->ah_key == NULL ) {
	if( errno < sys_nerr ) {
	  ErrorString = sys_errlist[errno];
	} else {
	  ErrorString = "malloc failed";
	}
	ErrorFreeHeader( hdr, size - 1 );
	return( NULL );
      }
      strcpy( (hdr+size-2)->ah_key, key );

      if( value == NULL ) {
	(hdr+size-2)->ah_value = NULL;
	continue;
      }
      
      (hdr+size-2)->ah_value = (char **)malloc( sizeof(char *));
      
      if( (hdr+size-2)->ah_value == NULL ) {
	if( errno < sys_nerr ) {
	  ErrorString = sys_errlist[errno];
	} else {
	  ErrorString = "malloc failed";
	}
	ErrorFreeHeader( hdr, size - 1 );
	return( NULL );
      }
      
      *((hdr+size-2)->ah_value) = (char *)malloc( strlen( value ) + 1 );
      
      if( *((hdr+size-2)->ah_value) == NULL ) {
	if( errno < sys_nerr ) {
	  ErrorString = sys_errlist[errno];
	} else {
	  ErrorString = "malloc failed";
	}
	ErrorFreeHeader( hdr, size -1 );
	return( NULL );
      }
      strcpy( *((hdr+size-2)->ah_value), value );
      (hdr+size-2)->ah_vlen = 1;
      
    }

    free( line );
  }
  
  (hdr+size-1)->ah_key = NULL;
  (hdr+size-1)->ah_value = NULL;
  
  return( hdr );
}


/*
 * WriteHeader( fp, hdr )
 *
 *  Writes hdr AsciiHdr stucture to fp ASCII form.
 *
 * parameters:
 *   fp - stream to write to
 *   hdr - pointer to AsciiHdr array to write out.
 *
 * returns:
 *  1 - if ok
 *  -1 - if errors
 */

int WriteHeader( FILE *fp, AsciiHdr *hdr )
{
  int c, i;

  while( hdr->ah_key != NULL ) {

    c = fprintf( fp, "%s:", hdr->ah_key );
    if( c == EOF ) {
      if( errno < sys_nerr ) {
	ErrorString = sys_errlist[errno];
      } else {
	ErrorString = "end of file";
      }
      return( -1 );
    }
    
    if( hdr->ah_vlen == 1 ) {
      c = fprintf( fp, " %s\n", *(hdr->ah_value) );
      
      if( c == EOF ) {
	if( errno < sys_nerr ) {
	  ErrorString = sys_errlist[errno];
	} else {
	  ErrorString = "end of file";
	}
	return( -1 );
      }
      
    } else {
      
      c = fprintf( fp, "\nBegin\n" );
      if( c == EOF ) {
	if( errno < sys_nerr ) {
	  ErrorString = sys_errlist[errno];
	} else {
	  ErrorString = "end of file";
	}
	return( -1 );
      }
			
      for( i = 0; i < hdr->ah_vlen; i++ ) {
	
	c = fprintf( fp, "%s\n", *(hdr->ah_value+i) );
	
	if( c == EOF ) {
	  if( errno < sys_nerr ) {
	    ErrorString = sys_errlist[errno];
	  } else {
	    ErrorString = "end of file";
	  }
	  return( -1 );
	}
      }
      
      c = fprintf( fp, "End\n" );
      
      if( c == EOF ) {
	if( errno < sys_nerr ) {
	  ErrorString = sys_errlist[errno];
	} else {
	  ErrorString = "end of file";
	}
	return( -1 );
      }
    }
    
    hdr++;
  }
  
  c = fprintf( fp, "END OF HEADER\n" );
  
  if( c == EOF ) {
    if( errno < sys_nerr ) {
      ErrorString = sys_errlist[errno];
    } else {
      ErrorString = "end of file";
    }
    return( -1 );
  }
  
  return( 1 );
}


/*
 * ReadArrayData( fp, len )
 *
 *  reads extra data from fp and returns a pointer to
 *  a extradata structure that holds the data.
 *
 * parameters:
 *   fp - stream to read from.
 *   len - out variable, len of array is placed here.
 *
 * returns:
 *  a pointer to an array of char strings if everything went
 *  ok, otherwise ErrorString is set and NULL is returned.
 *
 */

static char **ReadArrayData( FILE *fp, int *len )
{
  char **data;
  char **tmp;
  char *str;
  char *line;
  int dlen = 0;
  
  data = (char **)malloc( sizeof(char *) );
  
  if( data == NULL ) {
    if( errno < sys_nerr ) {
      ErrorString = sys_errlist[errno];
    } else {
      ErrorString = "malloc faild";
    }
    return( NULL );
  }

  while( 1 ) {
    line = GetLine( fp );
    
    if( line == NULL ) {
      return( NULL );
    }
    
    str = CleanStr( line );
    
    if( strcmp( str, "End" ) == 0 ) {
      break;
    }
    
    dlen++;
    
    tmp = (char **)realloc( (char *)data, sizeof(char *) * dlen );
    
    if( tmp == NULL ) {
      if( errno < sys_nerr ) {
	ErrorString = sys_errlist[errno];
      } else {
	ErrorString = "realloc failed";
      }
      FreeStrArray( data, dlen - 1 );
      *len = 0;
      return( NULL );
    }
    
    data = tmp;
    
    *(data+dlen-1) = (char *)malloc( strlen( str ) + 1 );
    
    if( *(data+dlen-1) == NULL ) {
      if( errno < sys_nerr ) {
	ErrorString = sys_errlist[errno];
      } else {
	ErrorString = "malloc failed";
      }
      FreeStrArray( data, dlen-1 );
      *len = 0;
      return( NULL );
    }
    strcpy( *(data+dlen-1), str );
    
    free( line );
  }

  free( line );
  
  *len = dlen;
  return( data );
}


/*
 * Splitline( line, key, value )
 *
 *  Used to ReadHeader() to find the key and value
 *  in header lines.  The key and value are separated by
 *  a ':'.
 * 
 * parameters:
 *   key - pointer to a char pointer.  The address of the key will 
 *         be placed here.
 *   value - pointer to a char pointer.  The address of the value
 *           will be placed here.
 *
 * returns:
 *   nothing.
 */

static void SplitLine (char *line, char **key, char **value )
{
  char *tmp;

  tmp = strchr( line, ':' );
  
  if( tmp == NULL ) {
    *key = CleanStr( line );
    *value = NULL;
    return;
  }
  
  *tmp = NULL;
  
  *key = CleanStr( line );
  *value = CleanStr( tmp+1 );
}


/*
 * CleanStr( str )
 *
 *  Stripps white space from the begining and end of
 *  str.  Returns a char pointer the the begining of the
 *  cleaned str.
 *
 * NOTE: str is NOT copied, the cleaning is done in-place.
 *  
 * parameters:
 *   str - pointer the NULL terminated string to be cleaned.
 *
 * returns:
 *   a pointer to the cleaned string.  
 *
 */

static char *CleanStr (char *str )
{
  char *tmp;

  while( isspace( *str ) && ( *str != NULL ) ) {
    str++;
  }
  
  tmp = str;
  
  tmp = str + strlen( str ) - 1;
  
  while( isspace( *tmp ) ) {
    tmp--;
  }
  tmp++;
  *tmp = NULL;
  
  return( str );
}


/*
 * GetLine( fp )
 *
 *  reads the next line from fp.  A line is everything
 *  from the current possion up to and including the next newline (\n).
 *  The space for the line is allocated using malloc().  It is
 *  the responcibilty of the caller to free this space.
 *
 * parameters:
 *  fp - FILE pointer to read from.
 *
 * returns:
 *  returns a null terminated string containing the line.
 *  returns NULL on EOF or ERRORS.  If and ERROR occur ErrorString will 
 *   be set.
 */

static char *GetLine( FILE *fp )
{
  char buf[READLINE_LENGTH];
  char *str = NULL;
  int s_len = 0;
  int len;
  
  if( fgets( buf, sizeof(buf), fp ) == NULL ) {
    return( NULL );
  }
  
  len = strlen( buf );
  str = (char *)malloc( len + 1 );
  if( str == NULL ) {
    if( errno < sys_nerr ) {
      ErrorString = sys_errlist[errno];
    } else {
      ErrorString = "malloc failed";
    }
    return( NULL );
  }
		
  strcpy( str, buf );
  s_len = len + 1;

  /*
   * Did the easy case work??
   */
  if( buf[s_len-2] == '\n' ) {
    return( str );
  } 

  if( feof( fp )
#ifdef ultrix
     || ferror( fp )
#endif
     ) {
    return( str );
  }
	
  while( fgets( buf, sizeof(buf), fp ) != NULL ) {

    len = strlen( buf );
    
    str = (char *)realloc( str, s_len + len );
    
    if( str == NULL ) {
      if( errno < sys_nerr ) {
	ErrorString = sys_errlist[errno];
      } else {
	ErrorString = "realloc failed";
      }
      return( NULL );
    }
    
    strcat( str, buf );
    
    s_len += len;

    if( buf[s_len-2] == '\n' ) {
      return( str );
    } 
  }
  
  return( NULL );
}


/*
 * ErrorFreeHeader( hdr, size )
 *
 *
 */

static void ErrorFreeHeader( AsciiHdr *hdr, int len )
{
  int i;

  for( i = 0; i < len; i++ ) {
    
    if( hdr[i].ah_key != NULL ) {
      free( hdr[i].ah_key );
    }

    if( hdr[i].ah_value != NULL ) {
      FreeStrArray( hdr[i].ah_value, hdr[i].ah_vlen );
    }
  }
  
  free( hdr );

}


/*
 * FreeHeader( hdr )
 *
 * parameters:
 *	hdr - pointer to header
 *
 * returns:
 *
 */

void FreeHeader( AsciiHdr *hdr )
{
  AsciiHdr *thdr;

  thdr = hdr;
  
  while( thdr->ah_key != NULL ) {
    
    if( thdr->ah_key != NULL ) {
      free( thdr->ah_key );
    }
    
    if( thdr->ah_value != NULL ) {
      FreeStrArray( thdr->ah_value, thdr->ah_vlen );
    }
    thdr++;
  }
  
  free( hdr );
}


/*
 * FindHeaderVal( hdr, key )
 *
 *  Look for key in hdr and returns a pointer to 
 *  the first value in the ah_value array.  
 *  Returns NULL if key is not found.
 *
 * parameters:
 *   hdr - pointer to AsciiHdr array.
 *   key - NULL terminated string.
 *
 * returns:
 *   pointer to a NULL terminated string if key is found,
 *   otherwise NULL is returned.
 */

char *FindHeaderVal( AsciiHdr *hdr, char *key )
{

  while( hdr->ah_key != NULL ) {
    
    if( strcasecmp( hdr->ah_key, key ) == 0 ) {
      return( *(hdr->ah_value) );
    }
    
    hdr++;
  }
  
  return( NULL );
}


/*
 * GetValueArray( hdr, key, len )
 *
 *  returns pointer to ah_value array for key.
 *  Return NULL is key is not found.
 *
 * parameters:
 *    hdr - pointer to AsciiHdr array.
 *    key - NULL terminated string.
 *    len - pointer to and int.
 *
 * returns:
 *    len is set to length of array.
 *    pointer to a an char array if key is
 *    found, otherwise NULL is returned.
 *
 */

char **GetValueArray( AsciiHdr *hdr, char *key, int *len )
{

  while( hdr->ah_key != NULL ) {

    if( strcasecmp( hdr->ah_key, key ) == 0 ) {
      *len = hdr->ah_vlen;
      return( hdr->ah_value );
    }

    hdr++;
  }
  
  return( NULL );
  
}

