/*
 * buildlola.c
 *
 * The buildlola program loops through all files defined in the from
 * file id data base, and generates the correspoding lola speech label
 * files, using the current directory as output directory.
 *
 */

/* standard C library include file directives */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


/* Speech tools include file directives */
#include <speech.h>


/* data base include file directives */
#include <phonebase.h>


/* external variable declaration */
extern int  errno, sys_nerr;
extern char *sys_errlist[];


/* global variable declarations */
int        CREATE_DATABASE = 0;
int        *PreviousLink;            /* array of most recent previous links */
PhonemeRec *PhonemeIndex;            /* phoneme for index file              */

FILE       *contextfile;             /* contextual phoneme data base        */
FILE       *idfile;                  /* from file id data base              */


char       headorder, machineorder;  /* byte ordering of record and machine */
char       byteswap;                 /* must i do byte swapping flag        */


/* global command line parameter variables */
char       contextfilename[200];     /* name of contextual database file    */
char       indexfilename[200];       /* name of index file for context base */
char       fileidfilename[200];      /* name of file id data base           */

char       dbname[200]="";           /* name of data base to search         */


/*
 * Usage()
 *
 * print out the usage parameters of the buildlola program
 *
 */

void Usage()
{
  fprintf(stderr, "Usage: buildlola {dbname}\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "dbname: name of data base to search\n");
  fprintf(stderr, "\n");
  fprintf(stderr, " \t -h Help. This help message\n");
  exit(1);
}


/*
 * get_comline (argc, argv)
 *
 * read an interpret the command line parameters
 *
 * argc, argv (in): command line variables
 *
 */

void get_comline (int argc, char **argv)
{
  int c;
  extern int  optind;
  extern char *optarg;

  while ( (c = getopt (argc, argv, "h")) != -1) {
    switch (c) 
      {

      case 'h':
      default:
	Usage();
      }
  }

  if ((argc-optind) != 1)
    Usage();

  strcpy(dbname, argv[optind]);
  
  /* create data base and index file names */
  strcpy(contextfilename, dbname);
  strcat(contextfilename, CONTEXTBASE);
  
  strcpy(indexfilename, dbname);
  strcat(indexfilename, INDEX);
  
  strcpy(fileidfilename, dbname);
  strcat(fileidfilename, IDBASE);
}


/*
 * SegnentList *Add_Segment(lolasegment, begin, end, phoneme)
 *
 * This procedure add a segment to the end of the segment list
 * and then returns to the end fo the segment list.
 *
 * lolasegment (in): the segment list to use
 * begin       (in): the starting location
 * end         (in): ending location in units
 * phoneme     (in): pointer to phone label
 *
 */

SegmentList *Add_Segment(SegmentList *lolasegment, int begin, int end,
			 char *phoneme)
{
  SegmentList *tmp;
  SegmentList *pointer;

  pointer = lolasegment;
  if (pointer->sl_forward != NULL) {
    do {
      pointer = pointer->sl_forward;
    } while (pointer->sl_forward);
  }

  if (!(tmp = (SegmentList *) malloc(sizeof(SegmentList)))) {
    if (errno < sys_nerr) 
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "tmp --> malloc failed";
    fprintf(stderr, "Add_Segment: %s\n", ErrorString);
    exit(1);
  }

  if ((tmp->sl_label = (char *) malloc(strlen(phoneme) + 1)) == NULL) {
    if (errno < sys_nerr)
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "temp->sl_label malloc failed";
    fprintf(stderr, "Add_Segment: %s\n", ErrorString);
    exit(1);
  }

  tmp->sl_begin = begin;
  tmp->sl_end   = end;
  strcpy(tmp->sl_label, phoneme);
  tmp->sl_forward =  NULL;
  tmp->sl_backward = pointer;
  pointer->sl_forward = tmp;

  return (tmp);
}


/*
 * Fix_Segment_List(SegmentList *lolasegment)
 *
 * this procedure sweeps through the segment list
 * and sets the borders correct for start and end mgn
 *
 * lolasegment (in/out): the segment list to align
 *
 */

void Fix_Segment_List(SegmentList *lolasegment)
{
  int start, end;
  SegmentList *segpointer;

  segpointer = lolasegment->sl_forward;
  start = segpointer->sl_begin;
  end   = segpointer->sl_end;
  
  while (segpointer) {
    segpointer->sl_begin = start;
    segpointer->sl_end   = end;

    start = segpointer->sl_end;
    segpointer = segpointer->sl_forward;
    if (segpointer) 
      end = start + (segpointer->sl_end - segpointer->sl_begin);
  }
}


/*
 * int Extract_Lola_File(contextfile, idfile, PhonemeIndex, contextlola, idnum)
 *
 * This procedure extracts the lola file from the contextual data base
 *
 * contextfile (in): pointer to contextual file stream
 * idfile      (in): pointer to id file stream
 * PhonemeIndex(in): pointer to contextual data base phoneme index
 * contextlola (in): segment listing
 * idnum       (in): index number of the id file
 *
 * On error returns -1, else +1
 *
 */

int Extract_Lola_File(FILE *contextfile, FILE *idfile, 
		      PhonemeRec *PhonemeIndex, SegmentList **contextlola,
		      int idnum)
{
  int        record;                       /* loop counter variables     */
  FileIDRec  IDFile;                       /* id file record for lola    */
  ContextRec OneContext;                   /* context file entry         */
  SegmentList *curr_lola;                  /* pointer to current lola    */
  char       phoneme[10];                  /* phoneme label              */
  int        begin, end;                   /* segment borders of phoneme */

  /* create segment list */
  *contextlola = NewSegmentList();
  curr_lola    = *contextlola;
  if (curr_lola == NULL) {
    if (errno < sys_nerr) 
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "NewSegmentList --> malloc failed";
    return(-1);
  }

  if (Read_FileID_Record(idfile, &IDFile, idnum)<0)
    return (-1);

  record = IDFile.start_point;

  /* loop until end of lola chain */
  do {
    if (Read_Context_Record(contextfile, &OneContext, record) < 0)
      return (-1);

    strcpy(phoneme, PhonemeIndex[OneContext.ph_curr-1].phoneme);
    begin = OneContext.startloc;
    end   = OneContext.endloc;

    curr_lola = Add_Segment(curr_lola, begin, end, phoneme);

    record = OneContext.ph_nextlink;
  } while (record>=0);

  Fix_Segment_List(*contextlola);
  
  return (1);
}


/* 
 * master magician code
 *
 */

main (int argc, char **argv)
{
  int         i;                            /* loop counter variable     */
  float       ms;                           /* units of lola segments    */
  SegmentList *contextlola;                 /* contextual lola file      */
  char        lolafilename[200];            /* name of lola output file  */
  FileIDRec   IDFile;                       /* name of from file id      */
  
  /* process command line directives */
  get_comline(argc, argv);

  machineorder = LittleIndian();

  /* do data base setup */
  if (Read_Index_File (indexfilename, &PhonemeIndex, &PreviousLink)<0) {
    fprintf(stderr, "Read_Index_File: %s\n", ErrorString);
    exit(1);
  }
  
  if (Open_Contextual_Base (&contextfile, contextfilename) < 0) {
    fprintf(stderr, "Open_Contextual_Base: %s\n", ErrorString);
    free((char *) PhonemeIndex);
    free((char *) PreviousLink);
    exit(1);
  }

  if (Open_FileID_Base(&idfile, fileidfilename) < 0) {
    fprintf(stderr, "Open_FileID_Base: %s\n", ErrorString);
    free((char *) PhonemeIndex);
    free((char *) PreviousLink);
    Close_Contextual_Base(&contextfile);
    exit(1);
  }

  i = 0;
  do {
    if (Read_FileID_Record(idfile, &IDFile, i)<0)
      break;

    strcpy(lolafilename, strrchr(IDFile.phnfilename, '/')+1);
    printf("%d -- %s\n", i+1, lolafilename);
    
    if (Extract_Lola_File(contextfile, idfile, PhonemeIndex, &contextlola,i)>0)
      WriteSegmentList(lolafilename, contextlola, 1.0);
    i++;
  } while (!(feof(idfile)));

  /* do system cleanup before exiting */
  Close_Contextual_Base (&contextfile);
  Close_FileID_Base(&idfile);

  free((char *) PhonemeIndex);
  free((char *) PreviousLink);
}

