static char *rcsident = "$Header: /projects/cslu/speech/work/src/bin/nopt/nopt/RCS/nopt.c,v 1.43 1993/06/10 17:17:58 johans Exp johans $";

/**************************************************************************
 *
 * nopt.c - NOPT main program
 *
 **************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <sys/types.h>
#include <sys/times.h>
#include <signal.h>
#include <malloc.h>
#include <ctype.h>
#include <speech.h>

#include "nopt.h"
#include "version.h"

#ifdef solaris
#define RMAX  0.5
#define SRANDOM srand48
#define RANDOM drand48
#else
	extern long random();
#define RMAX 		(1.07374e+09) /* (maximum random number)/2 */
#define SRANDOM srandom
#define RANDOM random
#endif

#define TABSZ 		(50000)	/* number of entries in sigmoid tbl */
#define TABSTEP		(0.0005) /* increment between entries in sigmoid tbl */
#define EMAX 		(70.0)	/* cut-off for exp-function */
#define EMIN	 	(-70.0)	/* cut-off for exp-function */
#define MAXFEAT 	(100)	/* if any input feature exceeds this warn */

#define TABSZ2 (((float)TABSZ)/2.0)
#define ALPHA	((float)TABSTEP)

static float f(float s);
static void eerror (char *where);
static int ReadNetworkConfig(char *file, float *tol, int *rnd );
static int ReadVectorFile(char *file);
static int ReadWeightsFile(char *file,int iter);
static int WriteWeightsFile(char *file);
static void InitRandomWeights(int seed);
static void InitExpTable(void);
static void IntDump(void);
static void ReportCPUTime(int iter);
static void PrintTimeHeader(void);


static char *WeightsBaseName;
int DumpTime = 100;

static float OutHi = 0.7;	
static float OutLow = 0.3;

static struct tms LastTimes;
static int ReportTime = 0;

static float **NeuralAct, *ExpTable, *Weights; 
static float **NeuralDetla, **InputAct;
static int NumberOfLayers, NumInputVectors; 
static int MaxNumInterations, nconn;
static int *SizeOfLayers, *Classes, NumberOfLayers1; 
static int NumberOfLayers2, *SumLayerWeights;
static int hiddennodes=0, seed=0;
static int prt_flag;

void print_usage()
{

  fprintf( stderr, "NOPT - Conjugate-gradient Neural-Net classifier\n\n");
  fprintf( stderr, "usage: nopt [-i count] [-c iter] [-v] [-h target_hi]" );
  fprintf( stderr, " [-l target_low] [-n hidden] [-r seed] configfile" );
  fprintf( stderr, " vectorfile weightbase\n");
  fprintf( stderr, "\ndescription:\n\n");
  fprintf( stderr, "configfile : the network description for nopt\n");
  fprintf( stderr, "vectorfile : the input vectors for training with nopt\n");
  fprintf( stderr, "weightfile : the binary weight file used by nopt\n\n");
  fprintf( stderr, "count : interval count at which nopt should dump weight file\n");
  fprintf( stderr, "target_hi : target hi value, defaults to 0.7\n" );
  fprintf( stderr, "target_low : target low value, defaults to 0.3\n" );
  fprintf( stderr, "-c : continue from a previous training session\n");
  fprintf( stderr, "-v : run in verbose mode\n" );
}

/*************************************************************************
 *
 * Calculate sigmoid 
 *
 *************************************************************************/

static float f(float s)
{
	float x;

	x = s / ALPHA;
	if (x > EMAX)
		x = EMAX;
	else if (x < EMIN)
		x = EMIN;
	return (float) 1. / (1. + exp (-(float) x));
}

/*************************************************************************
 *
 *     Exit with appropriate error message 
 *
 *************************************************************************/

static void eerror (char *where)
{
	perror (where);
	exit (1);
}

/*************************************************************************
 *
 *     Calculate value of energy function at w; store gradient in dw 
 *
 *************************************************************************/

float errfc (float *w,float *dw,
	     int nconn)		/* num of connections=dim of w, dw */
{
	int      i2, il, ct, iin;
	float    s, tot_err;
	register float  rfl;
	register float *pv1, *pv2, *pend;

	/* clear dw */

	rfl = 0.000;
	pv1 = dw;
	pend = pv1 + nconn;

	for (; pv1 < pend;)
		(*pv1++) = rfl;

	tot_err = 0.;
	for (iin = 0; iin < NumInputVectors; iin++)
	{
		pend = w;
		for (i2 = 0; i2 < SizeOfLayers[1]; i2++)
		{
			rfl = 0.00;
			pv1 = pend;
			pend = pv1 + SizeOfLayers[0];
			pv2 = InputAct[iin];

			for (; pv1 < pend;)
				rfl += (*pv2++) * (*pv1++);

			ct = rfl / ALPHA + TABSZ2;

			if (ct >= TABSZ)
				ct = TABSZ - 1;
			else if (ct < 0)
				ct = 0;

			NeuralAct[1][i2] = ExpTable[ct];
		}

		for (il = 1; il < NumberOfLayers1; il++)
			for (i2 = 0; i2 < SizeOfLayers[il + 1]; i2++)
			{
				rfl = 0.;
				pv1 = pend;
				pend = pv1 + SizeOfLayers[il];
				pv2 = NeuralAct[il];

				for (; pv1 < pend;)
					rfl += (*pv1++) * (*pv2++);

				ct = rfl / ALPHA + TABSZ2;

				if (ct >= TABSZ)
					ct = TABSZ - 1;
				else if (ct < 0)
					ct = 0;

				NeuralAct[il + 1][i2] = ExpTable[ct];
			}

		for (i2 = 0; i2 < SizeOfLayers[NumberOfLayers1]; i2++)
		{
			if (i2 == Classes[iin] - 1)
			{
				rfl = min (NeuralAct[NumberOfLayers1][i2] - 
								OutHi, 0);
			} else
				rfl = max (NeuralAct[NumberOfLayers1][i2] - 
								OutLow, 0);

			tot_err += rfl * rfl;

			NeuralDetla[NumberOfLayers1][i2] = 
				(rfl *= NeuralAct[NumberOfLayers1][i2] *
				(1.0 - NeuralAct[NumberOfLayers1][i2]) );
			pv1 = NeuralAct[NumberOfLayers2];
			pv2 = &dw[SumLayerWeights[NumberOfLayers2] + i2 * 
						SizeOfLayers[NumberOfLayers2]];
			pend = pv2 + SizeOfLayers[NumberOfLayers2];

			for (; pv2 < pend;)
				(*pv2++) += rfl * (*pv1++);

		}

		for (il = NumberOfLayers2; il > 1; il--)
		{
			pv1 = NeuralDetla[il];
			pend = pv1 + SizeOfLayers[il];
			rfl = 0.000;

			for (; pv1 < pend;)
				(*pv1++) = rfl;

			for (i2 = 0; i2 < SizeOfLayers[il + 1]; i2++)
			{
				pv2 = &w[SumLayerWeights[il] + i2 * 
							SizeOfLayers[il]];
				pend = pv2 + SizeOfLayers[il];
				pv1 = NeuralDetla[il];
				rfl = NeuralDetla[il + 1][i2];
				for (; pv2 < pend;)
					(*pv1++) += (*pv2++) * rfl;
			}

			for (i2 = 0; i2 < SizeOfLayers[il]; i2++)
			{
				rfl = NeuralDetla[il][i2] *= NeuralAct[il][i2] 
						* (1.0 - NeuralAct[il][i2]);
				pv1 = NeuralAct[il - 1];
				pv2 = &dw[SumLayerWeights[il - 1] + i2 * 
							SizeOfLayers[il - 1]];
				pend = pv2 + SizeOfLayers[il - 1];
				for (; pv2 < pend;)
					(*pv2++) += rfl * (*pv1++);
			}
		}

		pv1 = NeuralDetla[1];
		pend = pv1 + SizeOfLayers[1];
		rfl = 0.000;

		for (; pv1 < pend;)
			(*pv1++) = rfl;

		for (i2 = 0; i2 < SizeOfLayers[2]; i2++)
		{
			pv2 = &w[SumLayerWeights[1] + i2 * SizeOfLayers[1]];
			pend = pv2 + SizeOfLayers[1];
			pv1 = NeuralDetla[1];
			rfl = NeuralDetla[2][i2];

			for (; pv2 < pend;)
				(*pv1++) += (*pv2++) * rfl;
		}

		for (i2 = 0; i2 < SizeOfLayers[1]; i2++)
		{
			rfl = NeuralDetla[1][i2] *= NeuralAct[1][i2] * 
						(1.0 - NeuralAct[1][i2]);
			pv1 = InputAct[iin];
			pv2 = &dw[SizeOfLayers[0] * i2];
			pend = pv2 + SizeOfLayers[0];

			for (; pv2 < pend;)
				(*pv2++) += rfl * (*pv1++);
		}
	}

	tot_err *= 0.5000;

	if (prt_flag)
		printf ("%10.6f\n", tot_err);

	return tot_err;
}

/*************************************************************************/

main(int argc, char *argv[] )
{
        extern char *optarg;
        extern int optind;
	extern double atof();
	char *config_file;
	char *input_file;
	int cont_flag = 0;
	int ContIter;
	int data_set = 0;
	float ferr, tol;
	int vflag = 0;
	int rnd;
	int c;

	int i;


	if( argc < 4 ) {
		print_usage();
		exit( 2 );
	}

	while( (c = getopt(argc, argv, "n:r:i:l:h:c:vt")) != EOF ) {

		switch (c) {

			case 'c':
				cont_flag++;
				ContIter = atoi( optarg );
				break;
			case 't':
				ReportTime++;
				break;
			case 'v':
				vflag++;
				prt_flag++;
				break;
			case 'i':
				DumpTime = atoi( optarg );
				break;
			case 'h':
				OutHi = (float)atof( optarg );
				break;
			case 'l':
				OutLow = (float)atof( optarg );
				break;
                        case 'n':
				hiddennodes = atoi(optarg);
				break;
                        case 'r':
				seed = atoi(optarg);
				break;
				
			case '?':
				print_usage();
				exit( 2 );
			
		}
	}

	if( (argc - optind) != 3 ) {
		print_usage();
		exit( 2 );
	}

	config_file = argv[optind];
	input_file = argv[optind+1];
	WeightsBaseName = argv[optind+2];
	

	if( vflag ) {

		printf("\nNOPT - A conjugate-gradient neural-net classifier\n\n");
		printf("      Version:       %s \n", opt_version);
		printf("      Config file:   %s \n", config_file);
		printf("      Vector file:   %s \n", input_file);
		printf("      Weight Base:   %s \n\n", WeightsBaseName );

		if( cont_flag ) {
			printf("NOPT - Continuing from a previously trained set.\n\n"); 
		}
	}

	ReadNetworkConfig( config_file, &tol, &rnd );

	if (vflag) {
		printf("NOPT - Parameters for this training session:\n\n");
		printf("      Number of layers:              %6d\n",
							NumberOfLayers);
		printf("      Maximum iterations:            %6d\n",
							MaxNumInterations);
		printf("      Stopping criterion:                 %10.8f\n",tol);
		printf("      Random weight seed:            %6d\n",rnd);
		printf("      Size of feature (input) layer: %6d\n",
							SizeOfLayers[0]-1);
		printf("      Size of hidden layer:          %6d\n",
							SizeOfLayers[1]);
		printf("      Size of output layer:          %6d\n\n",
					SizeOfLayers[NumberOfLayers-1]);
	}

	ReadVectorFile( input_file );

	/* Initsialize weights */

	if( vflag ) {
		printf("NOPT - Initialize weights\n\n");
	}

	if( cont_flag ) {
		ReadWeightsFile( WeightsBaseName, ContIter );
	} else {
	   if (seed!=0)
	     rnd = seed;
	   InitRandomWeights( rnd );
	}

	/* Set up lookup table for exp-function */
	InitExpTable();

	/* optimize */

	if( vflag ) {
		printf("NOPT - Begin training\n\n");
	}

	signal( SIGINT, IntDump );
	signal( SIGTERM, IntDump );

	if( ReportTime > 0 ) {
		PrintTimeHeader();
		ReportCPUTime( 0 );
	}

	ferr = CGrd( &Weights, nconn, MaxNumInterations, tol );

	if( ReportTime > 0 ) {
		ReportCPUTime( MaxNumInterations );
	}

	/* save weights */

	if (vflag)
		printf("\nNOPT - Finished training. Save weight file\n\n");

	WriteWeightsFile( WeightsBaseName );

	if (vflag)
		printf("NOPT - Execution completed\n\n");

	exit( 0 );

}

static int ReadNetworkConfig(char *file, float *tol, int *rnd )
{
	FILE *fp;
	int szmax = 0;
	int i;

	if( (fp = fopen( file, "r" )) == NULL ) {
		eerror ("Error reading network configuration file");
	}

	fscanf( fp, " %d %d %f %d", &NumberOfLayers, &MaxNumInterations, 
								tol, rnd);

	if( NumberOfLayers < 3 ) {
		eerror ("Network has less than 3 layers");
	}

	if(NumberOfLayers > 10 ) {
		eerror ("Network has more than 10 layers");
	}

	NumberOfLayers1 = NumberOfLayers - 1;
	NumberOfLayers2 = NumberOfLayers - 2;

	/*
	 * Read in size of each layer; increment size of first layer for
	 * extra neuron. Szmax set to size of largest layer
	 */

	SizeOfLayers = (int *)malloc( NumberOfLayers * sizeof(int) );

	if( SizeOfLayers == NULL ) {
		eerror( "SizeOfLayers malloc error" );
	}

	szmax = 0;
	for( i = 0; i < NumberOfLayers; i++) {

	  fscanf( fp, " %d", &SizeOfLayers[i] );
	  if ((i==1) && (hiddennodes != 0))
	    SizeOfLayers[i] = hiddennodes;

		if( i == 0 ) {
			SizeOfLayers[i]++;
		}

		if( SizeOfLayers[i] > szmax ) {
			szmax = SizeOfLayers[i];
		}
	}

	fclose( fp );		/* done with config file */

	/*
	 * Set SumLayerWeights[i] to the total number of weights up to 
	 * layer i
	 */

	SumLayerWeights = (int *)malloc( NumberOfLayers1 * sizeof(int) );

	if( SumLayerWeights == NULL ) {
		eerror( "SumLayerWeights malloc error" );
	}

	nconn = 0;

	for( i = 0; i < NumberOfLayers1; i++ ) {
		SumLayerWeights[i] = nconn;
		nconn += SizeOfLayers[i] * SizeOfLayers[i + 1];
	}

	/*
	 * NeuralAct = matrix of neural activities 
	 * NeuralDetla = matrix of "delta" values, see PDP. 
	 * Weights = weight matrix
	 */

	NeuralAct = (float **)Alloc2d( NumberOfLayers, szmax, sizeof(float) );

	if( NeuralAct == NULL ) {
		eerror( "NeuralAct Alloc2d error" );
	}

	NeuralDetla = (float **)Alloc2d( NumberOfLayers, szmax, sizeof(float) );

	if(NeuralDetla == NULL ) {
		eerror( "NeuralDetla Alloc2d error" );
	}

	Weights = (float *)malloc( nconn * sizeof (float) );

	if( Weights == NULL ) {
		eerror( "Weights malloc error" );
	}

}

static int ReadVectorFile(char *file )
{
	FILE *fp;
	int i,j;
	int sr;

	if( ( fp = fopen( file, "r" ) ) == NULL ) {
		eerror( "Error reading input vector file" );
	}

	fscanf( fp, "%d", &NumInputVectors );

	if( NumInputVectors < 2 ) {
		eerror( "Suspicious value for number of input vectors" );
	}

#ifdef not
	if( vflag ) {
		printf("NOPT - Number of input vectors:       %6d\n\n",
							NumInputVectors);
	}
#endif

	Classes = (int *)malloc( NumInputVectors * sizeof(int) );

	if( Classes == NULL ) {
		eerror( "Classes malloc error" );
	}

	InputAct = (float **) Alloc2d( NumInputVectors, SizeOfLayers[0], 
							sizeof(float) );

	if( InputAct == NULL ) {
		eerror( "InputAct Alloc2d error" );
	}

	for( i = 0; i < NumInputVectors; i++ )
	{
		fscanf( fp, "%d", &Classes[i] );

		if( Classes[i] > SizeOfLayers[NumberOfLayers1] ) {
			eerror( "More classes than output neurons" );
		}

		for( j = 0; j < SizeOfLayers[0] - 1; j++)
		{

			if( fscanf( fp, "%f", &InputAct[i][j] ) == EOF ) {
				eerror( "Insufficient data in vector file" );
			}

			if( ( InputAct[i][j] > MAXFEAT ) || 
			    ( InputAct[i][j] < -MAXFEAT ) ) {
				fprintf (stderr, "\nWARNING: sample #%d has a large feature!\n", i );
			}
		}

		InputAct[i][SizeOfLayers[0] - 1] = 1.0;

	}
	
	while( (sr = fgetc( fp )) != EOF ) {

		if( !isspace(sr) ) {

			fprintf( stderr, "nopt: data left over in vector file!\n" );
			fprintf( stderr, "check to see that the network config is correct.\n" );
			fclose( fp );
			exit( 1 );
		}

	} 

	fclose( fp );
}

static int ReadWeightsFile(char *file,int iter)
{
	char buf[1024];
	NeuralNet nn;
	FILE *fp;
	int i;

	sprintf( buf, "%s.%d", WeightsBaseName, iter );

	if( ( fp = fopen( buf, "r" )) == NULL ) {
		eerror( "Error opening weight file from previous run" );
	}

	if( NNRead( fp, &nn ) < 0 ) {
		fprintf( stderr, "nopt: %s: %s\n", buf, ErrorString );
		exit( 1 );
	}
	fclose( fp );

	Weights = nn.nn_w;
	
	SetCGrdIteration( nn.nn_iter );
}

static int WriteWeightsFile(char *file)
{
	NeuralNet nn;
	FILE *fp;
	int iter;


	if( ( fp = fopen( file, "w" ) ) == NULL ) {
		eerror( "outfile" );
	}

	iter = GetCGrdIteration();

	nn.nn_iter = iter;
	nn.nn_nlayers = NumberOfLayers;
	nn.nn_sz = SizeOfLayers;
	nn.nn_w = Weights;

	NNWrite( fp, &nn );

	fclose( fp );
}

static void InitRandomWeights(int seed)
{
	float ftemp;
	int i, j, k;

	SRANDOM((long) seed );

	for( i = 0; i < NumberOfLayers1; i++ ) {

		for( j = 0; j < SizeOfLayers[i]; j++) {

			ftemp = SizeOfLayers[i];

			ftemp = sqrt ((double) ftemp) / 3.0;

			for( k = 0; k < SizeOfLayers[i + 1]; k++) {
				Weights[SumLayerWeights[i] + 
					k * SizeOfLayers[i] + j] =
					 ( ((float)RANDOM()) / RMAX - 1.0) 
					 / ftemp;
			}
		}
	}
}

static void InitExpTable()
{
	float ftemp;
	int i;


        ExpTable = (float *)malloc( TABSZ * sizeof (float) );

        if( ExpTable == NULL ) {
                eerror( "ExpTable malloc error" );
	}

        ftemp = ((float)TABSZ) * TABSTEP / 2.0;

        for( i = 0; i < TABSZ; i++ ) {
                ExpTable[i] = (float)( 1.0 / 
			(1.0 + exp( -((double)(((double)i)*TABSTEP-ftemp)) ) ));
	}

}

void DumpWeights(int iter)
{
	char buf[1024];

	if( ReportTime > 0 ) {
		ReportCPUTime( iter );
	}

	sprintf( buf, "%s.%d", WeightsBaseName, iter );

	WriteWeightsFile( buf );

}

static void IntDump()
{
	char buf[1024];

	signal( SIGINT, SIG_IGN );
	signal( SIGTERM, SIG_IGN );

	sprintf( buf, "%s.%d", WeightsBaseName, GetCGrdIteration() );

	WriteWeightsFile( buf );

	exit( 3 );
}

static void ReportCPUTime(int iter)
{
	struct tms clk;
	double user, sys;
	double luser;

	if( times( &clk ) == -1 ) {
			fprintf( stderr, "    %d\ttimes() failed\n", iter );
	} else {
		user = (double) clk.tms_utime;
		user /= 60.0;

		luser = (double) LastTimes.tms_utime;
		luser /= 60.0;

		sys = (double) clk.tms_stime;
		sys /= 60.0;

		fprintf( stderr, "    %8d   %10.4g    %10.4g    %10.4g\n", iter, user, 
						user - luser, sys );

		LastTimes = clk;
	}

}

static void PrintTimeHeader()
{

	fprintf( stderr, "    Iteration    Total User    Delta User      System\n" );
	fprintf( stderr, "----------------------------------------------------------\n\n" );

}
