static char *rcsident = "$Header: /projects/cslu/speech/work/src/bin/auto_lyre/RCS/Lspec.c,v 4.4 1993/05/28 21:27:40 johans Exp $";

/*
 * Lspec.c - Line Spectrum Scale Widget
 *------------------------------------------------------------*
 * Copyright 1988, Fil Alleva and Carnegie Mellon University
 *------------------------------------------------------------*
 * HISTORY
 * 12-Jun-89  Fil Alleva (faa) at Carnegie-Mellon University
 *	Added pointer initialization for lspec.segments in Initialize()
 *
 * 13-Apr-93 Johan Schalkwyk at Oregon Graduate Institute changed
 *       to ANSI style header declarations
 *
 */

/* Standard C library include file directives */
#include <c.h>
#include <math.h>
#include <malloc.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

/* X library include file directives */
#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/StringDefs.h>
#include <X11/IntrinsicP.h>

/* Lyre include file directives */
#include <LspecP.h>
#include <Scale.h>
#include <ToolUtil.h>
#include <LyreStringDefs.h>
#include <form_peak.h>

/* Private Definitions */

#define MyXtSetArg(args,i,arg,val)	{XtSetArg (args[i], arg, val); i++;}

static char *labels[] = {
  "0", "1", "2", "3", "4", "5", "6", 0
};

/* NB.
 *     The KeyPress translations for the digits have to be encoded
 * with their ascii representation (48 == 0, 49 == 1, etc)
 */

static char defaultTranslations[] =
    "<KeyPress>48:   TrackMouse(l) PutLabel(0) TrackMouse(e) \n\
     <KeyPress>49:   TrackMouse(l) PutLabel(1) TrackMouse(e) \n\
     <KeyPress>50:   TrackMouse(l) PutLabel(2) TrackMouse(e) \n\
     <KeyPress>51:   TrackMouse(l) PutLabel(3) TrackMouse(e) \n\
     <KeyPress>52:   TrackMouse(l) PutLabel(4) TrackMouse(e) \n\
     <KeyPress>53:   TrackMouse(l) PutLabel(5) TrackMouse(e) \n\
     <KeyPress>54:   TrackMouse(l) PutLabel(6) TrackMouse(e) \n\
     <Key>l:   TrackMouse(l) NextFrame() TrackMouse(e) \n\
     <Key>h:   TrackMouse(l) LastFrame() TrackMouse(e) \n\
     <Key>k:   TrackMouse(l) NextPeak() TrackMouse(e) \n\
     <Key>j:   TrackMouse(l) LastPeak() TrackMouse(e) \n\
     <Key>n:   TrackMouse(l) NextFrame() TrackMouse(e) \n\
     <Key>b:   TrackMouse(l) LastFrame() TrackMouse(e) \n\
     <Key>x:   TrackMouse(l) CopyFrame() TrackMouse(e) \n\
     <Key>\\ :   TrackMouse(l) SkipPeak() TrackMouse(e) \n\
     <BtnDown>:  TrackMouse(l) PositionCursor() TrackMouse(e) \n\
     <Enter>:       TrackMouse(e) \n\
     <Leave>:       TrackMouse(l) \n\
     <Motion>:      TrackMouse(m) \n\
";

/* private procedure declarations */
static void PutLabel (LspecWidget w, XEvent *event, String *params,
		      Cardinal *num_params);
static void NextFrame (LspecWidget w, XEvent *event, String *params,
		       Cardinal *num_params);
static void LastFrame (LspecWidget w, XEvent *event, String *params,
		       Cardinal *num_params);
static void CopyFrame (LspecWidget w, XEvent *event, String *params,
		       Cardinal *num_params);
static void NextPeak (LspecWidget w, XEvent *event, String *params,
		      Cardinal *num_params);
static void LastPeak (LspecWidget w, XEvent *event, String *params,
		      Cardinal *num_params);
static void SkipPeak (LspecWidget w, XEvent *event, String *params,
		      Cardinal *num_params);
static void PositionCursor (LspecWidget w, XEvent *event, String *params,
			    Cardinal *num_params);


static XtActionsRec actions[] = {
  {"PutLabel",    (XtActionProc) PutLabel},
  {"NextFrame",   (XtActionProc) NextFrame},
  {"LastFrame",   (XtActionProc) LastFrame},
  {"CopyFrame",   (XtActionProc) CopyFrame},
  {"NextPeak",    (XtActionProc) NextPeak},
  {"LastPeak",    (XtActionProc) LastPeak},
  {"SkipPeak",    (XtActionProc) SkipPeak},
  {"PositionCursor", (XtActionProc) PositionCursor},
  {"TrackMouse",  (XtActionProc) LyreDispTrackMouse},
  {NULL,NULL}
};


/* Initialization of defaults */

#define OFFSET(field) XtOffset(LspecWidget,lspec.field)
#define GOFFSET(field) XtOffset(Widget,core.field)

static XtResource resources[] = {
    {XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
        OFFSET(foreground_pixel), XtRString, "Black"},
    {XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
	GOFFSET(width), XtRString, "256"},
    {XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
	GOFFSET(height), XtRString, "128"},
    {XtNreverseVideo, XtCReverseVideo, XtRBoolean, sizeof (Boolean),
	OFFSET (reverse_video), XtRString, "FALSE"},
    {XtNdata, XtNdata, XtRPointer, sizeof(caddr_t),
	OFFSET (input_data), XtRPointer, NULL},
    {XtNfcand, XtNfcand, XtRPointer, sizeof(caddr_t),
	OFFSET (fcand), XtRPointer, NULL},
    {XtNsync, XtNsync, XtRPointer, sizeof(caddr_t),
	OFFSET (sync), XtRPointer, NULL},
    {XtNasync, XtNasync, XtRPointer, sizeof(caddr_t),
	OFFSET (async), XtRPointer, NULL},
    {XtNsynccnt, XtNsynccnt, XtRInt, sizeof(int),
	OFFSET(sync_cnt), XtRString, "0"},
    {XtNsyncdelta, XtNsyncdelta, XtRInt, sizeof(int),
	OFFSET(sync_delta), XtRString, "0"},

    {XtNstartRow, XtNstartRow, XtRInt, sizeof(int),
	OFFSET(start_row), XtRString, "0"},
    {XtNrowCount, XtNrowCount, XtRInt, sizeof(int),
	OFFSET(row_count), XtRString, "0"},
    {XtNallRows, XtNallRows, XtRBoolean, sizeof (Boolean),
	OFFSET (all_rows), XtRString, "TRUE"},
    {XtNdispPeakExtent, XtCParameter, XtRBoolean, sizeof (Boolean),
	OFFSET (disp_peak_extent), XtRString, "FALSE"},
    

    {XtNminOffset, XtCMinOffset, XtRInt, sizeof(int),
	OFFSET(min_offset), XtRString, "50"},
    {XtNmaxHeight, XtNmaxHeight, XtRInt, sizeof(int),
	OFFSET(max_height), XtRString, "0"},
    {XtNrightUp, XtCRightUp, XtRBoolean, sizeof(Boolean),
	OFFSET(right_up), XtRString, "TRUE"},
    {XtNskew, XtCSkew, XtRFloat, sizeof(float),
	OFFSET(skew), XtRString, "0.0"},

    {XtNhscaleWidget, XtCParameter, XtRPointer, sizeof(caddr_t),
	OFFSET (hscale_widget), XtRPointer, NULL},
    {XtNvscaleWidget, XtCParameter, XtRPointer, sizeof(caddr_t),
	OFFSET (vscale_widget), XtRPointer, NULL},
};

#undef OFFSET
#undef GOFFSET

/* Private procedure declarations */
static void    Initialize (Widget request, Widget new, ArgList arglist,
			   Cardinal *num_args);
static void    Realize (Widget w_in, XtValueMask *valueMask, 
			XSetWindowAttributes *attrs);
static void    Destroy (Widget gw);
static void    Resize (Widget gw);
static void    Redisplay (Widget gw, XEvent *event_in, Region region);
static Boolean SetValues (Widget gcurrent, Widget grequest, Widget gnew,
			  ArgList arglist, Cardinal *num_args);

static void    ComputeImage (register LspecWidget w, Boolean resize);

static void    ComputeImageX (LspecWidget w, float PixPerDatum);

static void    ComputeGlobalMax (LspecWidget w);

static void    ComputeGMax1 (LspecWidget w);
static void    ComputeGMax2 (LspecWidget w);
static void    ComputeGMax4 (LspecWidget w);
static void    ComputeGlobalMin (LspecWidget w);
static void    ComputeGMin1 (LspecWidget w);
static void    ComputeGMin2 (LspecWidget w);
static void    ComputeGMin4 (LspecWidget w);


LspecClassRec
lspecClassRec = {
    /* Core Fields */
    {
	     /* superclass		 */ (WidgetClass) &lyreDispClassRec,
	     /* class_name		 */ "Lspec",
	     /* widget_size		 */ sizeof (LspecRec),
	     /* class_initialize	 */ NULL,
	     /* class_part_initialize	 */ NULL,
	     /* class_inited		 */ FALSE,
	     /* initialize		 */ Initialize,
	     /* initialize_hook		 */ NULL,
	     /* realize			 */ Realize,
	     /* actions			 */ actions,
	     /* num_actions		 */ XtNumber(actions),
	     /* resources		 */ resources,
	     /* resource_count		 */ XtNumber (resources),
	     /* xrm_class		 */ NULL,
	     /* compress_motion		 */ TRUE,
	     /* compress_exposure	 */ (TRUE | XtExposeGraphicsExpose),
	     /* compress_enterleave	 */ TRUE,
	     /* visible_interest	 */ FALSE,
	     /* destroy			 */ Destroy,
	     /* resize			 */ Resize,
	     /* expose			 */ Redisplay,
	     /* set_values		 */ SetValues,
	     /* set_values_hook		 */ NULL,
	     /* set_values_almost	 */ XtInheritSetValuesAlmost,
	     /* get_values_hook		 */ NULL,
	     /* accept_focus		 */ NULL,
	     /* version			 */ XtVersion,
	     /* callback_private	 */ NULL,
	     /* tm_table	         */ defaultTranslations,
	     /* query_geometry           */ NULL,
	     /* display_accelerator	 */ XtInheritDisplayAccelerator,	
	     /* extension		 */ NULL,
    }
};

WidgetClass lspecWidgetClass = (WidgetClass) & lspecClassRec;


/*
 * Initialize (request, new)
 *
 */

static void Initialize (Widget request, Widget new, ArgList arglist,
			Cardinal *num_args)
{
    LspecWidget w = (LspecWidget) new;
    XtGCMask valuemask;
    XGCValues myXGCV;

    valuemask = GCForeground | GCBackground | GCLineWidth;

    if (w->lspec.reverse_video) {
	Pixel fg = w->lspec.foreground_pixel;
	Pixel bg = w->core.background_pixel;

	if (w->core.border_pixel == fg)
	    w->core.border_pixel = bg;
	w->lspec.foreground_pixel = bg;
	w->core.background_pixel = fg;
    }

    myXGCV.foreground = w->lspec.foreground_pixel;
    myXGCV.background = w->core.background_pixel;
    myXGCV.line_width = 0;
    w->lspec.myGC = XtGetGC ((Widget) w, valuemask, &myXGCV);

    myXGCV.foreground = w->core.background_pixel;
    myXGCV.background = w->lspec.foreground_pixel;
    myXGCV.line_width = 0;
    w->lspec.labelGC = XtGetGC ((Widget) w, valuemask, &myXGCV);

    w->lspec.data = w->lspec.input_data;
    w->lspec.input_data = 0;

    w->lspec.points = 0;
    w->lspec.segments = 0;
    w->lspec.current_frame = 0;
    w->lspec.current_peak = 0;
    SetScale ((LyreDispWidget)w->lspec.hscale_widget, w->lyreDisp.secs_ppix);
}


/*
 * Realize (w, valueMask, attrs)
 *
 */

static void Realize (Widget w_in, XtValueMask *valueMask, 
		     XSetWindowAttributes *attrs)
{
    LspecWidget w = (LspecWidget) w_in;
    XtCreateWindow ((Widget) w, InputOutput, (Visual *) CopyFromParent,
		    *valueMask, attrs);
    Resize ((Widget) w);
}


/*
 * Destroy (gw)
 *
 */

static void Destroy (Widget gw)
{
    LspecWidget w = (LspecWidget) gw;

    XtDestroyGC (w->lspec.myGC);
}


/* 
 * Resize (gw)
 *
 */

static void Resize (Widget gw)
{
    LspecWidget w = (LspecWidget) gw;

    /* don't do this computation if window hasn't been realized yet. */
    if (XtIsRealized ((Widget) w)) {
	ComputeImage (w, TRUE);
    }
}


/*
 * Redisplay (ge, event, region)
 *
 */

static void Redisplay (Widget gw, XEvent *event_in, Region region)
{
    LspecWidget  w      = (LspecWidget) gw;
    XExposeEvent *event = (XExposeEvent *) event_in;
    int                 start;
    int                 left_extra, right_extra;
    register int        i, j, end;
    register float      PixPerDatum = w->lspec.pix_per_datum;
    register XSegment  *seg;
    int                 as_idx;
    int                 last_as_idx;
    int                 base_x;

    if (w->lspec.data == NULL)
	return;

    left_extra = (w->lspec.skew + 1) * w->lspec.max_height;
    right_extra = (w->lspec.skew + 1) * w->lspec.max_height;

    start = (w->lyreDisp.virtual_x + event->x - left_extra) /
	(PixPerDatum * w->lspec.sync_delta);
    start = MAX (0, start);
    end = (w->lyreDisp.virtual_x + event->x + event->width +
	   right_extra) / (PixPerDatum * w->lspec.sync_delta);
    end += 1;
    end = MIN (end, w->lspec.sync_cnt);

    if (w->lspec.points) {
	last_as_idx = -1;
	for (i = start; i < end; i++) {
	    /*
	     * If the current as_idx is the same as the last then we've
	     * already displayed this data. 
	     */
	    as_idx = w->lspec.sync[i];
	    if (as_idx == last_as_idx)
		continue;
	    last_as_idx = as_idx;

	    base_x = (w->lspec.async[as_idx] * PixPerDatum) -
		w->lyreDisp.virtual_x;

	    w->lspec.points[as_idx * w->lspec.row_count].x += base_x;
	    XDrawLine (XtDisplay (w), XtWindow (w), w->lspec.myGC,
		       base_x, 0, base_x, w->core.height);

	    XDrawLines (XtDisplay (w), XtWindow (w), w->lspec.myGC,
			w->lspec.points + (as_idx * w->lspec.row_count),
			w->lspec.row_count, CoordModePrevious);
	    w->lspec.points[as_idx * w->lspec.row_count].x -= base_x;
	}
    }
#if 1
    /*
     * Make sure we have fcand data as well as segments
     */
    if (w->lspec.fcand && w->lspec.segments && (start < end)) {
	int segs_per_form = (w->lspec.disp_peak_extent) ? 3 : 1;

	seg = &w->lspec.segments[w->lspec.sync[start] * NUMCAND*segs_per_form];
	last_as_idx = -1;
	for (i = start; i < end; i++) {
	    /*
	     * If the current as_idx is the same as the last then we've
	     * already displayed this data. 
	     */
	    as_idx = w->lspec.sync[i];
	    if (as_idx == last_as_idx)
		continue;
	    last_as_idx = as_idx;

	    base_x = (w->lspec.async[as_idx] * PixPerDatum) -
		w->lyreDisp.virtual_x;

	    for (j = 0; j < NUMCAND * segs_per_form; j++, seg++) {
		seg->x1 = base_x;
		seg->x2 += base_x;
	    }
	}
	seg = &w->lspec.segments[w->lspec.sync[start] * NUMCAND*segs_per_form];
	XDrawSegments (XtDisplay (w), XtWindow (w), w->lspec.myGC,
		   seg, (1 + w->lspec.sync[end - 1] - w->lspec.sync[start]) *
		       NUMCAND * segs_per_form);
	/*
	 * Display the labels of formants 
	 */
	last_as_idx = -1;
	for (i = start; i < end; i++) {
	    int    cp = w->lspec.current_peak;
	    int    cf = w->lspec.current_frame;
	    int    num_cand;
	    int    row_offset = w->lspec.row_count + w->lspec.start_row;
	    SPEAK  *sp;
	    float  ppr = w->lspec.pix_per_row;

	    /*
	     * If the current as_idx is the same as the last then we've
	     * already displayed this data. 
	     */
	    as_idx = w->lspec.sync[i];
	    if (as_idx == last_as_idx)
		continue;
	    last_as_idx = as_idx;

	    base_x = (w->lspec.async[as_idx] * PixPerDatum) -
		w->lyreDisp.virtual_x;

	    sp = w->lspec.fcand[as_idx].cands;
	    num_cand = w->lspec.fcand[as_idx].numpeaks;

	    for (j = 0; j < num_cand; j++, sp++) {
		if ((j == cp) && (as_idx == cf))
		  XDrawImageString (XtDisplay (w), XtWindow (w), 
				    w->lspec.labelGC,  base_x + 2,
				    (int) ((row_offset - sp->peak) * ppr) -7,
				    labels[sp->label], 
				    strlen (labels[sp->label]));
		else
		  XDrawString (XtDisplay (w), XtWindow (w), w->lspec.myGC,
			       base_x + 2,
			       (int) ((row_offset - sp->peak) * ppr) -7,
			       labels[sp->label], strlen (labels[sp->label]));
	    }
	}
	/*
	 * Restore the x2 coordinate of the format lines 
	 */
	last_as_idx = -1;
	for (i = start; i < end; i++) {
	    /*
	     * If the current as_idx is the same as the last then we've
	     * already displayed this data. 
	     */
	    as_idx = w->lspec.sync[i];
	    if (as_idx == last_as_idx)
		continue;
	    last_as_idx = as_idx;

	    base_x = (w->lspec.async[as_idx] * PixPerDatum) -
		w->lyreDisp.virtual_x;

	    for (j = 0; j < NUMCAND * segs_per_form; j++, seg++) {
		seg->x2 -= base_x;
	    }
	}
    }
#endif
}


/* 
 * Boolean SetValues (gcurrent, grequest, gnew)
 *
 */

static Boolean SetValues (Widget gcurrent, Widget grequest, Widget gnew,
			  ArgList arglist, Cardinal *num_args)
{
    LspecWidget         current = (LspecWidget) gcurrent;
    LspecWidget         new = (LspecWidget) gnew;
    LspecWidget         req = (LspecWidget) grequest;
    Boolean             redisplay = FALSE;
    Boolean             recompute = FALSE;
    XtGCMask            valuemask;
    XGCValues           myXGCV;

    if ((new->lspec.foreground_pixel != current->lspec.foreground_pixel)
	|| (new->core.background_pixel != current->core.background_pixel)) {
	valuemask = GCForeground | GCBackground | GCFont | GCLineWidth;
	myXGCV.foreground = new->lspec.foreground_pixel;
	myXGCV.background = new->core.background_pixel;
	myXGCV.line_width = 0;
	XtDestroyGC (current->lspec.myGC);
	new->lspec.myGC = XtGetGC (gcurrent, valuemask, &myXGCV);
	redisplay = TRUE;
    }

    if (new->core.background_pixel != current->core.background_pixel) {
	valuemask = GCForeground | GCLineWidth;
	myXGCV.foreground = new->core.background_pixel;
	myXGCV.line_width = 0;
	redisplay = TRUE;
    }

    if (new->lyreDisp.secs_ppix != current->lyreDisp.secs_ppix) {
	SetScale ((LyreDispWidget)new->lspec.hscale_widget, 
		  new->lyreDisp.secs_ppix);
	recompute = TRUE;
    }

    if (new->lspec.input_data) {
	new->lspec.data = new->lspec.input_data;
	new->lspec.input_data = 0;
	ComputeGlobalMin (new);
	ComputeGlobalMax (new);
	recompute = TRUE;
    }

    /* If all_rows isn't true then inspect the values for
     * row_start and row_count and make sure that they make sense
     */
    if ((new->lspec.row_count != current->lspec.row_count) ||
	(new->lspec.start_row != current->lspec.start_row)) {
       new->lspec.all_rows = FALSE;
       recompute = TRUE;
    }

    if (new->lspec.data) {
    if (new->lspec.all_rows) {
      new->lspec.start_row = 0;
      new->lspec.row_count = new->lspec.data->rows;
    }
    else {
      new->lspec.start_row = MAX (0, new->lspec.start_row);
      new->lspec.start_row =MIN(new->lspec.data->rows-2,new->lspec.start_row);
      new->lspec.row_count = MAX (2, new->lspec.row_count);
      new->lspec.row_count = MIN (new->lspec.data->rows,new->lspec.row_count);
      if ((new->lspec.start_row + new->lspec.row_count) > new->lspec.data->rows)
	new->lspec.row_count = new->lspec.data->rows -new->lspec.start_row;
    }
  }

    if (new->lspec.disp_peak_extent != current->lspec.disp_peak_extent)
      recompute = TRUE;

    if (new->lspec.min_offset != current->lspec.min_offset)
      recompute = TRUE;

    if (recompute && new->lspec.data) {
	new->lspec.pix_per_datum = (float) new->lspec.data->ns_per_col /
	    			   (new->lyreDisp.secs_ppix * 1000000000.0);
	ComputeImage (new, FALSE);
	new->lyreDisp.virtual_width = new->lspec.sync_cnt * 
	  new->lspec.sync_delta *
	    new->lspec.pix_per_datum;
	redisplay = TRUE;
    }

    if (current->lyreDisp.virtual_width != new->lyreDisp.virtual_width)
	XtCallCallbacks ((Widget) new, XtNwidthProc, 
			 (XtPointer) new->lyreDisp.virtual_width);

    return (redisplay);
}


/*
 * ComputeImage (w, resize)
 *
 * resize (in): TRUE if this request is from resize
 *
 */

static void ComputeImage (register LspecWidget w, Boolean resize) 
{
    register float      PixPerDatum;
    register adata_t   *adata = w->lspec.data;

    if (adata == 0)
	return;

    if ((adata->ptr == 0) || (adata->count == 0))
	return;

    PixPerDatum = w->lspec.pix_per_datum;

    if (resize && (w->lspec.height == w->core.height))
	return;

    w->lspec.pix_per_row = (float) w->core.height / w->lspec.row_count;
    w->lspec.height = w->core.height;

    if (w->lspec.vscale_widget) {
	int arg_cnt;
	Arg args[20];
	union {
	    float f;
	    unsigned int i;
	}                   unit_per_pix;

	unit_per_pix.f = (((float) w->lspec.row_count)/16.0) /
	  ((float)w->core.height);

	arg_cnt = 0;
	MyXtSetArg (args, arg_cnt, XtNunitPerPix, unit_per_pix.i);
	MyXtSetArg (args, arg_cnt, XtNzeroPixOffset,
		    (w->core.height +
		     (int)(w->lspec.start_row * w->lspec.pix_per_row)));
	XtSetValues ((Widget) w->lspec.vscale_widget, args, arg_cnt);
    }

    XtFree ((char *) w->lspec.points);
    XtFree ((char *) w->lspec.segments);

    /*
     * Allocate points and segments memory
     */
    w->lspec.points = (XPoint *) malloc (adata->count * sizeof (XPoint));
    if (w->lspec.fcand)
      w->lspec.segments = (XSegment *) malloc (sizeof(XSegment) *
					       adata->cols * NUMCAND * 3);

    switch (adata->bsize) {
    case 1:
	ComputeImageX (w, PixPerDatum);
	break;
    case 2:
	ComputeImageX (w, PixPerDatum);
	break;
    case 4:
	ComputeImageX (w, PixPerDatum);
	break;
    }
}


/*
 * ComputeImageX (w, PixPerDatum)
 *
 */
 
static void ComputeImageX (LspecWidget w, float PixPerDatum)
{
    adata_t            *adata = w->lspec.data;
    char               *data_ptr = (char *) adata->ptr;
    register int        dval;		   /* Data value to display */
    register int        scale_i;	   /* i12.0 representaion the scale */
    register int        floor;
    register char      *ptr;		   /* Data pointer */
    register float      pix_per_row = w->lspec.pix_per_row;
    register int        width;		   /* Width of current diplay column */
    register int        x;		   /* X coord of current display col */
    register int        col, row;	   /* Data row and col indecies */
    register XPoint    *points = w->lspec.points;
    register int        range;
    register Boolean    right_up = w->lspec.right_up;
    int                *skew;
    FCAND              *fp = w->lspec.fcand;
    XSegment           *seg = w->lspec.segments;

    floor = w->lspec.global_min + w->lspec.min_offset;
    range = (w->lspec.global_max - floor);

    skew = (int *) malloc (sizeof (int) * w->lspec.row_count);
    for (row = 0; row < w->lspec.row_count; row++)
	skew[row] = ((row * w->lspec.skew) / w->lspec.row_count) *
	    w->lspec.max_height;

    scale_i = (w->lspec.max_height << 12) / range;
    width = 0;
    x = 0;
    for (col = 0; col < adata->cols; col++) {
	ptr = data_ptr + (col * adata->rows) + w->lspec.start_row;

	if (w->lspec.max_height == 0) {
	    /*
	     * If we aren't using a constant height specified by max_height
	     * then we have to compute the amount room availiable for each
	     * frame. Note we need to special case the last frame and out of
	     * order frames. 
	     */
	    int                 diff;

	    if (col + 1 < adata->cols)
		diff = w->lspec.async[col + 1] - w->lspec.async[col];
	    else
		diff = w->lspec.sync_delta;
	    if (diff <= 0)
		diff = w->lspec.sync_delta;
	    scale_i = ((int) (PixPerDatum * diff) << 12) / range;
	}

	if (fp) {
	    int    i, lower, upper, peak, rc;
	    SPEAK  *sp = 0;
	    int    segs_per_form = (w->lspec.disp_peak_extent) ? 3 : 1;

	    rc = w->lspec.row_count - 1;
	    sp = fp[col].cands;
	    for (i = 0; i < fp[col].numpeaks; i++, sp++) {
		lower = sp->lower - w->lspec.start_row;
		upper = sp->upper - w->lspec.start_row;
		peak = sp->peak - w->lspec.start_row;
		if (w->lspec.disp_peak_extent) {
		  seg->x2 = ((MAX (0, (ptr[lower] - floor)) * scale_i) >> 12);
		  seg->y1 = seg->y2 = (rc - lower) * pix_per_row;
		  seg++;
		}
		seg->x2 = 4;
		seg->y1 = seg->y2 = (rc - peak) * pix_per_row;
		seg++;
		if (w->lspec.disp_peak_extent) {
		    seg->x2 = ((MAX (0, (ptr[upper] - floor))* scale_i) >> 12);
		    seg->y1 = seg->y2 = (rc - upper) * pix_per_row;
		    seg++;
		}
	    }
	    if (NUMCAND - i)
		memset ((char *) seg, 0, 
			sizeof (XSegment)*(NUMCAND - i)*segs_per_form);
	    seg += (NUMCAND - i) * segs_per_form;
	}

	for (row = (w->lspec.row_count - 1); row >= 0; row--, points++) {
	    dval = *ptr++ - floor;
	    if (dval < 0)
		dval = 0;
	    else if (dval > range)
		dval = range;
	    dval = (dval * scale_i) >> 12;
	    if (right_up)
		points->x = dval + skew[row];
	    else
		points->x = dval + skew[row];
	    points->y = (row * pix_per_row);
	}
	/*
	 * Make points 1 thru (w->lspec.rows-1) relative.points[0] is
	 * absolute. NB. To do this computation in place we must do it from
	 * last to first. 
	 */
	for (row = -1; row > -(w->lspec.row_count); row--) {
	    points[row].x -= points[row - 1].x;
	    points[row].y -= points[row - 1].y;
	}
    }
    free (skew);
}


/*
 * ComputeGlobalMax(w)
 *
 */

static void ComputeGlobalMax (LspecWidget w)
{
    switch (w->lspec.data->bsize) {
    case 1:
	ComputeGMax1 (w);
	break;
    case 2:
	ComputeGMax2 (w);
	break;
    case 4:
	ComputeGMax4 (w);
	break;
    }
}

#define COMPUTEGMAX(name, type) \
static void name ( LspecWidget w) \
{ \
    register            c; \
    register adata_t   *adata = w->lspec.data; \
    register type      *ptr = (type *) adata->ptr; \
    register int        max; \
 \
    max = *ptr; \
    for (c = 0; c < adata->count; c++, ptr++) { \
	if (*ptr > max) \
	    max = *ptr; \
    } \
    w->lspec.global_max = max; \
}

COMPUTEGMAX (ComputeGMax1, char)
COMPUTEGMAX (ComputeGMax2, short)
COMPUTEGMAX (ComputeGMax4, int)

#undef COMPUTEGMAX

/* 
 * ComputeGlobalMin (w)
 *
 * compute global data min
 *
 */

static void ComputeGlobalMin (LspecWidget w)
{
    switch (w->lspec.data->bsize) {
    case 1:
	ComputeGMin1 (w);
	break;
    case 2:
	ComputeGMin2 (w);
	break;
    case 4:
	ComputeGMin4 (w);
	break;
    }
}


#define COMPUTEGMIN(name, type) \
static void name ( LspecWidget w) \
{ \
    register            c; \
    register adata_t   *adata = w->lspec.data; \
    register type      *ptr = (type *) adata->ptr; \
    register int        min; \
 \
    min = *ptr; \
    for (c = 0; c < adata->count; c++, ptr++) { \
	if (*ptr < min) \
	    min = *ptr; \
    } \
    w->lspec.global_min = min; \
}

COMPUTEGMIN (ComputeGMin1, char)
COMPUTEGMIN (ComputeGMin2, short)
COMPUTEGMIN (ComputeGMin4, int)

#undef COMPUTEGMIN


/*
 * PutLabel (w, event, params, num_params)
 *
 */

static void PutLabel (LspecWidget w, XEvent *event, String *params,
		      Cardinal *num_params )
{

    if (w->lspec.fcand == NULL)
      return;

    w->lspec.fcand[w->lspec.current_frame].cands[w->lspec.current_peak].label =
      atoi(params[0]);

    if (w->lspec.fcand[w->lspec.current_frame].numpeaks-1 ==
	w->lspec.current_peak) {
      w->lspec.current_peak = 0;
      NextFrame (w, event, params, num_params);
    }
    else
      NextPeak (w, event, params, num_params);
}


/*
 * SkipPeak(w, event, params, num_params)
 *
 */

static void SkipPeak (LspecWidget w, XEvent *event, String *params, 
		      Cardinal *num_params )
{

    if (w->lspec.fcand == NULL)
      return;

    if (w->lspec.fcand[w->lspec.current_frame].numpeaks-1 ==
	w->lspec.current_peak) {
      w->lspec.current_peak = 0;
      NextFrame (w, event, params, num_params);
    }
    else
      NextPeak (w, event, params, num_params);
}


/*
 * NextFrame (w, event, params, num_params)
 *
 */

static void NextFrame (LspecWidget w, XEvent *event, String *params, 
		       Cardinal *num_params )
{
    int                *async = w->lspec.async;
    int                 width;

    if (w->lspec.fcand == NULL)
	return;

    if (w->lspec.current_frame + 1 < w->lspec.data->cols)
	w->lspec.current_frame++;
    else
	return;

    if (w->lspec.current_peak >= 
	w->lspec.fcand[w->lspec.current_frame].numpeaks)
	w->lspec.current_peak = 
	  w->lspec.fcand[w->lspec.current_frame].numpeaks - 1;

    if (((int)w->core.width) <
	((int) ((async[w->lspec.current_frame - 1]) * w->lspec.pix_per_datum -
		w->lyreDisp.virtual_x + (int) (2 * w->lspec.pix_per_datum *
					       w->lspec.sync_delta))))
	LScroll (w->lyreDisp.scroll_widget, 0.80);

    width = ((async[w->lspec.current_frame] - 
	      async[w->lspec.current_frame - 1]) * 2) *	w->lspec.pix_per_datum;
    XClearArea (XtDisplay (w), XtWindow (w),
		(int) ((async[w->lspec.current_frame - 1]) * 
		       w->lspec.pix_per_datum - w->lyreDisp.virtual_x),
		0, width, w->core.height, TRUE);
}


/*
 * LastFrame (w, event, params, num_params )
 *
 */

static void LastFrame (LspecWidget w, XEvent *event, String *params, 
		       Cardinal *num_params )
{
    int *async = w->lspec.async;
    int width;

    if (w->lspec.fcand == NULL)
	return;

    if (w->lspec.current_frame - 1 >= 0)
	w->lspec.current_frame--;
    else
	return;

    if (w->lspec.current_peak >=
	w->lspec.fcand[w->lspec.current_frame].numpeaks)
	w->lspec.current_peak =
	    w->lspec.fcand[w->lspec.current_frame].numpeaks - 1;

    if (0 > ((int) (w->lspec.async[w->lspec.current_frame] * 
		    w->lspec.pix_per_datum -
		    w->lyreDisp.virtual_x)))
      LScroll (w->lyreDisp.scroll_widget, -0.80);

    width = ((async[w->lspec.current_frame + 1] - 
	      async[w->lspec.current_frame]) * 2) * w->lspec.pix_per_datum;
    XClearArea (XtDisplay (w), XtWindow (w),
		(int) (w->lspec.async[w->lspec.current_frame] * 
		       w->lspec.pix_per_datum -
		       w->lyreDisp.virtual_x), 0, width, w->core.height, TRUE);
}


/*
 * CopyFrame (w, event, params, num_params )
 *
 */

static void CopyFrame (LspecWidget w, XEvent *event, String *params, 
		       Cardinal *num_params )
{
    register int i, j;
    int cf = w->lspec.current_frame;
    SPEAK *sp_from, *sp_to;

    if (w->lspec.fcand == NULL)
      return;

    if (w->lspec.current_frame+1 >= w->lspec.data->cols)
      return;

    sp_from = w->lspec.fcand[cf].cands;
    sp_to = w->lspec.fcand[cf+1].cands;
    for (i = 0; i < NUMCAND; i++) {
      for (j = 0; j < NUMCAND; j++) {
	if ((sp_to[i].lower <= sp_from[j].peak) &&
	                       (sp_from[j].peak <= sp_to[i].upper)) {
	  sp_to[i].label = sp_from[j].label;
	  break;
        }
      }
    }

    NextFrame (w, event, params, num_params);
}


/*
 * NextPeak (w, event, params, num_params )
 *
 */

static void NextPeak (LspecWidget w, XEvent *event, String *params, 
		      Cardinal *num_params )
{

    if (w->lspec.fcand == NULL)
      return;

    if (w->lspec.current_peak+1 <
	w->lspec.fcand[w->lspec.current_frame].numpeaks)
      w->lspec.current_peak++;
    else
      return;

    XClearArea (XtDisplay (w), XtWindow (w),
                (int)(w->lspec.async[w->lspec.current_frame] * w->lspec.pix_per_datum -
		w->lyreDisp.virtual_x),
                0, (int)(w->lspec.pix_per_datum * w->lspec.sync_delta),
		w->core.height, TRUE);

}


/*
 * LastPeak (w, event, params, num_params )
 *
 */

static void LastPeak (LspecWidget w, XEvent *event, String *params, 
		      Cardinal *num_params )
{
    if (w->lspec.fcand == NULL)
      return;

    if (w->lspec.current_peak-1 >= 0)
      w->lspec.current_peak--;
    else
      return;

    XClearArea (XtDisplay (w), XtWindow (w),
                (int)(w->lspec.async[w->lspec.current_frame] * 
		      w->lspec.pix_per_datum -
		      w->lyreDisp.virtual_x),
                0, (int)(w->lspec.pix_per_datum * w->lspec.sync_delta), 
		w->core.height, TRUE);
}


/*
 * static void PositionCursor (w, event, params, num_params )
 *
 * DESCRIPTION
 *       Position the cursor at the nearest peak.
 *
 */

static void PositionCursor (LspecWidget w, XEvent *event, String *params, 
			    Cardinal *num_params )
{
  register int        i, row;
  register int        best_row_dist, best_peak;
  register SPEAK     *sp;

    if (w->lspec.fcand == NULL)
	return;

    XClearArea (XtDisplay (w), XtWindow (w),
     (int) (w->lspec.async[w->lspec.current_frame] * w->lspec.pix_per_datum -
	    w->lyreDisp.virtual_x),
		0, (short) (w->lspec.pix_per_datum * w->lspec.sync_delta),
		w->core.height, TRUE);

    {
	int                 sync_idx =
	(event->xbutton.x + w->lyreDisp.virtual_x) /
	(w->lspec.pix_per_datum * w->lspec.sync_delta);

	if (sync_idx >= w->lspec.sync_cnt)
	    sync_idx = w->lspec.sync_cnt - 1;

	w->lspec.current_frame = w->lspec.sync[sync_idx];
    }

    row = (event->xbutton.y) / w->lspec.pix_per_row;
    row = (w->lspec.row_count + w->lspec.start_row) - row;
    best_row_dist = 1000;
    best_peak = 0;
    sp = w->lspec.fcand[w->lspec.current_frame].cands;
  for (i = 0; i < w->lspec.fcand[w->lspec.current_frame].numpeaks; i++, sp++) {
	if (abs (sp->peak - row) < best_row_dist) {
	    best_peak = i;
	    best_row_dist = abs (sp->peak - row);
	}
    }
    w->lspec.current_peak = best_peak;

    XClearArea (XtDisplay (w), XtWindow (w),
     (int) (w->lspec.async[w->lspec.current_frame] * w->lspec.pix_per_datum -
	    w->lyreDisp.virtual_x),
		0, (int) (w->lspec.pix_per_datum * w->lspec.sync_delta),
		w->core.height, TRUE);

}
