static char *rcsident = "$Header: /amd/lydia/projects/cslu/MD4a/C/work/src/bin/auto_lyre/RCS/Gspec.c,v 4.14 1993/05/28 21:27:40 johans Exp $";

/*
 * Gspec.c - Gray Scale Widget
 *------------------------------------------------------------*
 * Copyright 1988, Fil Alleva and Carnegie Mellon University
 *------------------------------------------------------------*
 * HISTORY
 *  1-Mar-89  Fil Alleva (faa) at Carnegie-Mellon University
 *	- Fixed bug in Redisplay() that occured when the left edge of the
 *	bit map was position to the right of the left edge of the view
 *	port.
 *	- Fixed minor bug the ComputeImageX() code that allowed
 *	a gray scale byte to be used twice in a row.
 *	- Fixed bug that was cloberring column zero. It occured because
 *	I wan't allocating space for column N (which would clobber column
 *	0 when displayed).
 *
 * 23-Jan-89  Fil Alleva (faa) at Carnegie-Mellon University
 *	- Added code to free gspec.data when new comes in.
 *	- Fixed code that computed gspec.max_col_unit field for the
 *	  variable width frames used by pitch synchronous data.
 *
 * 19-Jan-89  Fil Alleva (faa) at Carnegie-Mellon University
 *	Added initialization of
 *		gspec.image
 *		gspec.gs.data
 *		gspec.height.data
 *	to 0 so that I could always do an XtFree() on on that field before
 *	reallocating the space.
 *
 *  4-Jan-89  Fil Alleva (faa) at Carnegie-Mellon University
 *	- Fixed fence post bug in ComputeImageX().
 *	- Added code to display asynchronous data.
 *	- Added some XtFree() calls to close some memory leaks that
 *	  occured when Destroy() was called.
 *
 *  13-Apr-93 Johan Schalkwyk at Oregon Graduate Institute
 *      - Changed to ANSI style header declarations
 *
 *------------------------------------------------------------*
 * BUGS
 */

/* Standard C file include file directives */
#include <c.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>

/* X library include file directives */
#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/StringDefs.h>
#include <X11/IntrinsicP.h>
#include <X11/Xatom.h>
#include <X11/cursorfont.h>
/* #include <X11/Xatomtype.h>  */

/* lyre include file directives */
#include <LyreDisp.h>
#include <ScaleP.h>
#include <Scale.h>
#include <GspecP.h>
#include <Gspectool.h>
#include <ToolUtil.h>
#include <color.h>
#include <color_lyre.h>
#include <LyreStringDefs.h>


/* Private Definitions */
#define MyXtSetArg(args,i,arg,val) {XtSetArg (args[i],arg,(XtArgVal)val); i++;}


/* Initialization of defaults */
#define OFFSET(field) XtOffset(GspecWidget,gspec.field)
#define GOFFSET(field) XtOffset(Widget,core.field)

static XtResource resources[] = {
    {XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
        OFFSET(foreground_pixel), XtRString, "Black"},
    {XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
	GOFFSET(width), XtRString, "10"},
    {XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
	GOFFSET(height), XtRString, "0"},
    {XtNreverseVideo, XtCReverseVideo, XtRBoolean, sizeof (Boolean),
	OFFSET (reverse_video), XtRString, "FALSE"},
    {XtNdata, XtNdata, XtRPointer, sizeof(caddr_t),
	OFFSET (input_data), XtRPointer, NULL},
    {XtNsampsPerSec, XtNsampsPerSec, XtRFloat, sizeof (XtArgVal),
        OFFSET(sampsPerSec), XtRString, "16000.0"},
    {XtNstart_time, XtNstart_time, XtRFloat, sizeof (XtArgVal),
        OFFSET(start_time), XtRString, "0.0"},
    {XtNaoffset, XtNaoffset, XtRFloat, sizeof (XtArgVal),
        OFFSET(aoffset), XtRString, "0.0"},
    {XtNLYRElow_color, XtCLYRElow_color, XtRPixel, sizeof(Pixel), /* default colors */
        OFFSET(gs.low_color), XtRString, "blue"},
    {XtNLYREzero_color, XtCLYREzero_color, XtRPixel, sizeof(Pixel),
        OFFSET(gs.zero_color), XtRString, "white"},
    {XtNLYREhigh_color, XtCLYREhigh_color, XtRPixel, sizeof(Pixel),
        OFFSET(gs.high_color), XtRString, "red"},
    {XtNlinearData, XtNlinearData, XtRBoolean, sizeof (Boolean),
	OFFSET (linear_data), XtRString, "FALSE"},
    {XtNrowlabel, XtNrowlabel, XtRString, sizeof (String),
	OFFSET (RowLabels), XtRString, NULL},
    {XtNhscaleWidget, XtCParameter, XtRPointer, sizeof(caddr_t),
	OFFSET (hscale_widget), XtRPointer, NULL},
    {XtNvscaleWidget, XtCParameter, XtRPointer, sizeof(caddr_t),
	OFFSET (vscale_widget), XtRPointer, NULL},

    {XtNpminWidget, XtCParameter, XtRPointer, sizeof(caddr_t),
	OFFSET (pmin_widget), XtRPointer, NULL},
    {XtNpmaxWidget, XtCParameter, XtRPointer, sizeof(caddr_t),
	OFFSET (pmax_widget), XtRPointer, NULL},
    {XtNnminWidget, XtCParameter, XtRPointer, sizeof(caddr_t),
	OFFSET (nmin_widget), XtRPointer, NULL},
    {XtNnmaxWidget, XtCParameter, XtRPointer, sizeof(caddr_t),
	OFFSET (nmax_widget), XtRPointer, NULL},

    {XtNposmin, XtNposmin, XtRFloat, sizeof(float),
    	OFFSET(gs.posmin), XtRString, "0"},
    {XtNposmax, XtNposmax, XtRFloat, sizeof(float),
    	OFFSET(gs.posmax), XtRString, "100"},
    {XtNposcolors, XtNposcolors, XtRInt, sizeof(int),
    	OFFSET(gs.input_poscolors), XtRString, "50"},
    {XtNnegmin, XtNnegmin, XtRFloat, sizeof(float),
    	OFFSET(gs.negmin), XtRString, "-20"},
    {XtNnegmax, XtNnegmax, XtRFloat, sizeof(float),
    	OFFSET(gs.negmax), XtRString, "0"},
    {XtNnegcolors, XtNnegcolors, XtRInt, sizeof(int),
    	OFFSET(gs.input_negcolors), XtRString, "20"},
    {XtNexp, XtNexp, XtRFloat, sizeof(float),
    	OFFSET(gs.exp), XtRString, "2.0"},

};

#undef OFFSET
#undef GOFFSET

static void Initialize (Widget request, Widget new, ArgList arglist,
			Cardinal *num_args);
static void Realize (Widget w_in, XtValueMask *valueMask, 
		     XSetWindowAttributes *attrs);
static void Destroy (Widget gw);
static void Resize (Widget w_in);
static void Redisplay (Widget gw, XEvent *event_in, Region region);
static Boolean SetValues (Widget gcurrent, Widget grequest, Widget gnew,
			  ArgList arglist, Cardinal *num_args);
static void ComputeImage (register GspecWidget w, Boolean resize, 
			                          Boolean RecomputeGS);
static int MakeGS (GspecWidget w);
static int MakeHeight (GspecWidget w);
static PutGrayMap (XImage *image, gs_t *gs, Pixel pixel, int x, int y,
		   int width, int height);
static void ComputeColMax (GspecWidget w);
static void FEUpdate (LyreDispWidget w, XMotionEvent *event, String *params,
		      Cardinal *num_params);
static void FEZero (LyreDispWidget w, XMotionEvent *event, String *params,
		    Cardinal *num_params);
static void SmoothImage (GspecWidget w);

void   ComputeImageX (GspecWidget w, double PixPerDatum);
double rint(double x);
int    LocalMax (GspecWidget w, register int col);
int    GlobalMax (GspecWidget w);
int    Compute_Max_Col_Unit (register adata_t *adata);
int   HZToPix(GspecWidget w, float hza);
float  PixToHZ(GspecWidget w, float pix);
int   TabHZToPix(GspecWidget w, float hza);
String PixToTabHZ(GspecWidget w, int pix);
void   spix(int x, int y, XImage *i);
int    gpix(int x, int y, XImage *i);

static char defaultTranslations[] = "\
<Enter>:       TrackMouse(e) FEUpdate() \n\
<Leave>:       TrackMouse(l) FEZero() \n\
<Motion>:      TrackMouse(m) FEUpdate() \n\
<Key>S:        SmoothImage() \n\
<Key>s:        SmoothImage() \n\
";

static XtActionsRec actions[] = {
	{"TrackMouse",		(XtActionProc) LyreDispTrackMouse},
        {"FEUpdate",            (XtActionProc) FEUpdate},
        {"FEZero",              (XtActionProc) FEZero},
        {"SmoothImage",        (XtActionProc) SmoothImage},
};



GspecClassRec
gspecClassRec = {
    /* Core Fields */
    {
	     /* superclass		 */ (WidgetClass) &lyreDispClassRec,
	     /* class_name		 */ "Gspec",
	     /* widget_size		 */ sizeof (GspecRec),
	     /* class_initialize	 */ NULL,
	     /* class_part_initialize	 */ NULL,
	     /* class_inited		 */ FALSE,
	     /* initialize		 */ Initialize,
	     /* initialize_hook		 */ NULL,
	     /* realize			 */ Realize,
	     /* actions			 */ actions,
	     /* num_actions		 */ XtNumber(actions),
	     /* resources		 */ resources,
	     /* resource_count		 */ XtNumber (resources),
	     /* xrm_class		 */ NULL,
	     /* compress_motion		 */ TRUE,
	     /* compress_exposure	 */ (TRUE | XtExposeGraphicsExpose),
	     /* compress_enterleave	 */ TRUE,
	     /* visible_interest	 */ FALSE,
	     /* destroy			 */ Destroy,
	     /* resize			 */ Resize,
	     /* expose			 */ Redisplay,
	     /* set_values		 */ SetValues,
	     /* set_values_hook		 */ NULL,
	     /* set_values_almost	 */ XtInheritSetValuesAlmost,
	     /* get_values_hook		 */ NULL,
	     /* accept_focus		 */ NULL,
	     /* version			 */ XtVersion,
	     /* callback_private	 */ NULL,
	     /* tm_table	         */ defaultTranslations,
	     /* query_geometry           */ NULL,
	     /* display_accelerator	 */ XtInheritDisplayAccelerator,	
	     /* extension		 */ NULL,
    }
};

WidgetClass gspecWidgetClass = (WidgetClass) & gspecClassRec;

/*
 * Initialize (request, new)
 *
 * Initialize all resources and widgets description concerning the 
 * Gspec tool widget.
 *
 */

static void Initialize (Widget request, Widget new, ArgList arglist,
			Cardinal *num_args)
{
    GspecWidget w = (GspecWidget) new;
    XtGCMask valuemask;
    XGCValues myXGCV;
    Display	*t_dpy = XtDisplay(w);

    valuemask = GCForeground | GCBackground | GCLineWidth;

    if (w->gspec.reverse_video) {
	Pixel fg = w->gspec.foreground_pixel;
	Pixel bg = w->core.background_pixel;

	if (w->core.border_pixel == fg)
	    w->core.border_pixel = bg;
	w->gspec.foreground_pixel = bg;
	w->core.background_pixel = fg;
    }

    w->gspec.RowLabels = NULL;

    myXGCV.foreground = w->gspec.foreground_pixel;
    myXGCV.background = w->core.background_pixel;
    myXGCV.line_width = 0;
    w->gspec.myGC = XtGetGC ((Widget) w, valuemask, &myXGCV);

    w->gspec.data = 0;
    w->gspec.input_data = 0;
    w->gspec.max_col_data = 0;
    w->gspec.min_col_data = 0;
    w->gspec.height.data = 0;

    w->gspec.gs.width = 0;
    w->gspec.gs.height = 0;
    w->gspec.gs.gmap = 0;

    w->gspec.gs.ncolors = NULL;
    w->gspec.gs.pcolors = NULL;
    w->lyreDisp.hzToPix = 0; /* for now */
    w->lyreDisp.pixToHZ = PixToTabHZ;

    SetScale ((LyreDispWidget)w->gspec.hscale_widget, w->lyreDisp.secs_ppix);

    if ((LYREpseudo_color_flag) &&
	(!(LYREaux_value & LyreShowDithering))) {/*if we're gonna blit color*/
      w->gspec.image = XCreateImage (XtDisplay (w), (Visual *) CopyFromParent,
				     DisplayPlanes(t_dpy,DefaultScreen(t_dpy)),
				     ZPixmap, 0, 0, 0, 0, 32, 0); 
    }/*endif gonna work in color*/
    else {
      w->gspec.image = XCreateImage (XtDisplay (w), (Visual *) CopyFromParent,
				     DisplayPlanes(t_dpy,DefaultScreen(t_dpy)),
				     XYPixmap, 0, 0, 0, 0, 32, 0); 
      
    }/*else*/
    
}			
    
/*
 * Realize (w, valueMask, attrs)
 *
 */

static void Realize (Widget w_in, XtValueMask *valueMask, 
		     XSetWindowAttributes *attrs)
{
    GspecWidget w = (GspecWidget) w_in;
    attrs->cursor = XCreateFontCursor( XtDisplay(w), XC_tcross );
    *valueMask |= CWCursor;

    XtCreateWindow ((Widget) w, InputOutput, (Visual *) CopyFromParent,
		    *valueMask, attrs);
    Resize ((Widget) w);

}

/* 
 * Destroy (gw)
 *
 */

static void Destroy (Widget gw)
{
    GspecWidget w = (GspecWidget) gw;

    if (w->gspec.data) {
      free (w->gspec.data->ptr);
      if (w->gspec.data->col_time)
	    free (w->gspec.data->col_time);
      free (w->gspec.data);
    }

    XtFree ((char *) w->gspec.height.data);
    XtFree ((char *) w->gspec.max_col_data);
    XtFree ((char *) w->gspec.min_col_data);
    XDestroyImage (w->gspec.image);
    XtDestroyGC (w->gspec.myGC);
    XtFree((char *) w->gspec.pixmap);	/* free this */
    if (LYREpseudo_color_flag)
      XFreeColormap(XtDisplay(w),LYRElocal_colormap); /* make X happy */
    if (LYREaux_value & LyreSynchronize)
	XSynchronize(XtDisplay(w),0);

}

/*
 * Resize (w)
 *
 */

static void Resize (Widget w_in)
{
    GspecWidget w = (GspecWidget) w_in;

    /* don't do this computation if window hasn't been realized yet. */
    if (XtIsRealized ((Widget) w)) {
	ComputeImage( w, TRUE, FALSE );
    }
}

/* 
 * Redisplay (gw, event, region)
 *
 */

static void Redisplay (Widget gw, register XEvent *event_in, Region region)
{
    GspecWidget  w = (GspecWidget) gw;
    XExposeEvent *event = (XExposeEvent *) event_in;
    register	int src_x;
    int		tmp_width;
    Display	*t_dpy = XtDisplay(w);
    int		row;
    int		column;
    XImage	*image;
    Pixel	shade;
    int offset;

    offset = (int)( (w->gspec.start_time + w->gspec.aoffset) 
					/ (w->lyreDisp.secs_ppix * 1000.0) );

    if (w->gspec.image->data) {
      src_x = ( w->lyreDisp.virtual_x + event->x ) - offset;
      if (src_x < 0) {
	event->x -= src_x;
	event->width -= src_x;
	src_x = 0;
      }

      XPutImage (XtDisplay (w), XtWindow (w), w->gspec.myGC,
		 w->gspec.image,
		 src_x, event->y,
		 event->x, event->y,
		 event->width, event->height);
      }
}


/* 
 * Boolean SetValues (gcurrent, grequest, gnew)
 *
 */

static Boolean SetValues (Widget gcurrent, Widget grequest, Widget gnew,
			  ArgList arglist, Cardinal *num_args)
{
    GspecWidget         current = (GspecWidget) gcurrent;
    GspecWidget         new = (GspecWidget) gnew;
    GspecWidget         req = (GspecWidget) grequest;
    Boolean             redisplay = FALSE;
    Boolean             recompute = FALSE;
    Boolean             RecomputeGS = FALSE;
    XtGCMask            valuemask;
    XGCValues           myXGCV;
    int i;

    if ((new->gspec.foreground_pixel != current->gspec.foreground_pixel)
	|| (new->core.background_pixel != current->core.background_pixel)) {
	valuemask = GCForeground | GCBackground | GCFont | GCLineWidth;
	myXGCV.foreground = new->gspec.foreground_pixel;
	myXGCV.background = new->core.background_pixel;
	myXGCV.line_width = 0;
	XtDestroyGC (current->gspec.myGC);
	new->gspec.myGC = XtGetGC (gcurrent, valuemask, &myXGCV);
	redisplay = TRUE;
    }

    if (new->core.background_pixel != current->core.background_pixel) {
	valuemask = GCForeground | GCLineWidth;
	myXGCV.foreground = new->core.background_pixel;
	myXGCV.line_width = 0;
	redisplay = TRUE;
    }

    if (new->lyreDisp.secs_ppix != current->lyreDisp.secs_ppix) {
      SetScale ((LyreDispWidget)new->gspec.hscale_widget, 
		new->lyreDisp.secs_ppix);
      recompute = TRUE;
    }

    /*
     * This has to be done before new->gspec.data is freed
     */
    if( new->gspec.RowLabels != current->gspec.RowLabels ) {

	if( current->gspec.RowLabels != NULL ) {

		for( i = 0; i < current->gspec.data->rows; i++ ) {
			XtFree((char *) current->gspec.RowLabels[i] );
		}

		XtFree((char *) current->gspec.RowLabels );
	}


    }

    if (new->gspec.input_data) {
	if (new->gspec.data) {

	    XtFree ((char *) new->gspec.data->col_time);
	    XtFree ((char *) new->gspec.data->ptr);
	    XtFree ((char *) new->gspec.data);
	}
	new->gspec.data = new->gspec.input_data;
	new->gspec.input_data = 0;
	if (new->gspec.data->col_time)
	    new->gspec.max_col_unit = Compute_Max_Col_Unit (new->gspec.data);
	else
	    new->gspec.max_col_unit = 1;
	ComputeColMax (new);
	ComputeColMin (new);	/* do the min too */
	recompute = TRUE;
    }

	if( current->gspec.gs.posmin != new->gspec.gs.posmin ||
		current->gspec.gs.posmax != new->gspec.gs.posmax ||
		current->gspec.gs.negmin != new->gspec.gs.negmin ||
		current->gspec.gs.negmax != new->gspec.gs.negmax ||
		current->gspec.gs.poscolors != new->gspec.gs.input_poscolors ||
		current->gspec.gs.negcolors != new->gspec.gs.input_negcolors ||
		current->gspec.gs.exp != new->gspec.gs.exp ) {

		recompute = TRUE;
		RecomputeGS = TRUE;

	}

    if (recompute) {
	ComputeImage (new, FALSE, RecomputeGS);
	new->lyreDisp.virtual_width = new->gspec.image->width;
	redisplay = TRUE;
    }

    if (current->lyreDisp.virtual_width != new->lyreDisp.virtual_width)
	XtCallCallbacks ((Widget) new, XtNwidthProc, 
			 (XtPointer) new->lyreDisp.virtual_width);

    return (redisplay);
}


/* 
 * ComputeImage (w, resize, RecomputeGS)
 *
 * w          (in):
 * resize     (in): TRUE if this request is from resize
 * RecomputeGS(in): TRUE when the GS needs to computed
 *
 */

static void ComputeImage (register GspecWidget w, Boolean resize, 
			                          Boolean RecomputeGS)
{
    double             PixPerUnit;
    register adata_t   *adata = w->gspec.data; /* spectral data */
    register gs_t      *gs = &w->gspec.gs;
    int                 max, min;
    int                 new_pix_per_row;
    Display		*t_dpy = XtDisplay(w);
    Status		status;
    
    if( adata == 0 ) {
      return;
    }

    if( (adata->ptr == 0) || (adata->count == 0) ) {
      return;
    }
    
    /*
     * No need to recompute image if only the width changed 
     */
    if( resize && (w->core.height == w->gspec.image->height) ){
      return;
    }

    if( adata->ns_per_col == 0 ) {
      adata->ns_per_col = 1.0e9 / w->gspec.sampsPerSec;
    } else {
      w->gspec.sampsPerSec = 1.0e9 / adata->ns_per_col;
    }

    PixPerUnit = (float) adata->ns_per_col / (w->lyreDisp.secs_ppix *
					      1000000000.0);

    /*
     * Compute number of pixels per row + 2. A coefficient can 
     * be be as many as 1.5 pixels higher than the average. 
     */
    new_pix_per_row = (w->core.height / w->gspec.data->rows) + 2;

    if (w->gspec.vscale_widget) {
      int arg_cnt;
      Arg args[5];
      union {
	float f;
	unsigned int i;
      } unit_per_pix;
      
      /*
       * This gets us to kHz per pixel
       * 1.0e9 - nano-secs/secs, 2.0 - to get Nyquist freq.,
       * 1000.0 - to get kHz
       */
      unit_per_pix.f = ((1.0e9 / (float)adata->ns_per_col) / 2.0 /  1000.0) 
	/ ((float)w->core.height);

      arg_cnt = 0;
      MyXtSetArg (args, arg_cnt, XtNunitPerPix, unit_per_pix.i);
      MyXtSetArg (args, arg_cnt, XtNzeroPixOffset, w->core.height);
      XtSetValues ((Widget) w->gspec.vscale_widget, args, arg_cnt);
    }

    w->gspec.pix_per_row = new_pix_per_row;

    MakeHeight( w ); /* Recompute Height Vectors */

    if( (w->gspec.pix_per_row > gs->height) || 
       (RecomputeGS) ||
       (((PixPerUnit * w->gspec.max_col_unit) + 1.0) > gs->width) ) {
      
      gs->height = w->gspec.pix_per_row + 1.0;
      gs->width = (PixPerUnit * w->gspec.max_col_unit) + 2.0;

      if( LYREbest_colormap_installed ) {

	XClearWindow( XtDisplay(w), XtWindow(w) );

	status = FillNewColormap( w, LYRElocal_colormap, 
				 LYRErgbs );

	if( status != TRUE ) {
	  fprintf( stderr, "could not allocate all colors\n" );
	}
	
      } else {
	MakeGS( w );
      }
    }
    
    XtFree (w->gspec.image->data);

    /*
     * Setup image structure 
     */
   if (adata->col_time) {
	/*
	 * Note that we have to account for the width of the last
	 * column. Since we don't explicity have this info we
	 * estimate its width to be that of the second to the last
	 * column
	 */
	w->gspec.image->width = PixPerUnit *
		(adata->col_time[adata->cols - 1] +
			(adata->col_time[adata->cols - 1] - 
			 adata->col_time[adata->cols - 2]));
    }
    else
	w->gspec.image->width = PixPerUnit * adata->cols;

    w->gspec.image->height = w->core.height;
    if (LYREpseudo_color_flag)	/* a stitch in time . . . */
      w->gspec.image->bytes_per_line = /* if it's color, then */
	w->gspec.image->width * sizeof(Pixel); /* we're talking big!! */
    else
      w->gspec.image->bytes_per_line = ((w->gspec.image->width + 7) >> 3) + 1;
    w->gspec.image->data = (char *)
      XtCalloc (1, w->gspec.image->height * w->gspec.image->bytes_per_line);

    if (LYREpseudo_color_flag) {
      ColorComputeImageX( w, PixPerUnit);
    } else {
      ComputeImageX( w, PixPerUnit);
    }
}

/* 
 * ComputeImageX (w, PixPerDatum)
 *
 */

void ComputeImageX (GspecWidget w, double PixPerDatum)
{
	gs_t *gs = &w->gspec.gs;
	adata_t *adata = w->gspec.data;
	float *data_ptr = adata->ptr;
	XImage *image = w->gspec.image;
	float *rptr;
	float dval;
	Pixel cur;
	int col, row;
	int x, y;
	int rh;
	int *heightvec;
	int cw;
	float pscale;
	int width;
	int i;



	heightvec = w->gspec.height.data[0];
	x = 0;

	pscale = fabs( (gs->posmax - gs->posmin) );
	pscale = ( 1.0 / pscale ) * ((float)gs->poscolors);

	for( col = 0; col < adata->cols; col++ ) {

		rptr = data_ptr + ( col * adata->rows );
		y = 0;

		if( (col + 1) < adata->cols ) {

			if( adata->col_time ) {
				width = (adata->col_time[col + 1 ] *
						PixPerDatum) - x;
			} else {
				width = ((col + 1) * PixPerDatum) - x;
			}
		}


		for( row = adata->rows - 1; row >= 0; row-- ) {

			dval = *(rptr+row);


			if( dval <= gs->posmin ) {
				cur = 0;
			} else if( dval >= gs->posmax ) {
				cur = gs->poscolors - 1;
			} else {

				i = (int)( (dval - ((float)gs->posmin) )
						* pscale );

				i = abs(i);

				if( i >= gs->poscolors ) {
					i = gs->poscolors - 1;
				}

				cur = (Pixel)i;
			}

			if( heightvec[row] ) {

				PutGrayMap(image, gs, cur, x, y, 
						width, heightvec[row] );

				y += heightvec[row];
			}
		}
		x += width;
	}
}


/*
 * PutGrayMap( image, gs, pixel, x, y, width, height )
 *
 */

static PutGrayMap(XImage *image, gs_t *gs, Pixel pixel, int x, int y, 
		  int width, int height )
{
  int i, j;
  Pixel *pp = gs->gmap[pixel];

  
  for( i = 0; i < height; i++ ) {
    for( j = 0; j < width; j++ ) {
      XPutPixel( image, x+j, y, *pp++ );
    }
    y++;
  }
}


#ifdef sequent
/*
 * double rint (x)
 *
 * this is adequate for here, I believe;  rint is supposed to handle
 * large values which may not fit in an int 
 *
 */

double rint(double x)
{
  int i;
  i = (int) (x + 0.5);
  return (double)i;
}
#endif


/* 
 * MakeGS (w)
 *
 * make the gray scale vectors
 *
 */

static int MakeGS( GspecWidget w )
{
#ifdef ultrix
	extern double rint();
#endif
	gs_t *gs = &w->gspec.gs;
	adata_t *adata = w->gspec.data;
	int *heightvec;
	Display *dpy;
	Pixel white;
	Pixel black;
	Pixel *pp;
	int i, j;
	int maxh;
	int maxw;
	int c;
	int step, bcount;
	int numpixels;
	float scale;
	double PixPerUnit;


	dpy = XtDisplay(w);
	white = WhitePixel(dpy,DefaultScreen(dpy));
	black = BlackPixel(dpy,DefaultScreen(dpy));

	gs->poscolors = gs->input_poscolors;

	heightvec = w->gspec.height.data[0];

	PixPerUnit = (float) adata->ns_per_col / (w->lyreDisp.secs_ppix *
					      1000000000.0);

	maxh = -1;
	for( i = 0; i < adata->rows; i++ ) {
		if( heightvec[i] > maxh ) {
			maxh = heightvec[i];
		}
	}
	
	maxw = -1;
	c = 0;
	for( i = 0; i < adata->cols; i++ ) {

		if( adata->col_time ) {
			j = (adata->col_time[i + 1 ] * PixPerUnit) - c;
		} else {
			j = ((i + 1) * PixPerUnit) - c;
		}

		if( j > maxw ) {
			maxw = j;
		}
		c += j;
	} 

	numpixels = maxw * maxh;

	if( gs->gmap ) {
		Free2d( gs->gmap );
	}

	gs->gmap = (Pixel **)Alloc2d( gs->poscolors, numpixels,
							sizeof(Pixel) );

	if( gs->gmap == NULL ) {
		XtAlert( "Could not allocate storage for gray scale bitmaps" );
		return( -1 );
	}

	for( i = 0; i < gs->poscolors; i++ ) {

		scale = ((float)i) / ((float)gs->poscolors);
		scale = (float)pow( (double)scale, (double)(gs->exp) );

		if( scale > 1.0 ) {
			scale = 1.0;
		}

		bcount = (int)( ((float)numpixels) * scale);

		pp = gs->gmap[i];

		/* 
		 * Clear out bitmap
		 */
		for( j = 0; j < numpixels; j++ ) {
			pp[j] = white;
		}

		if( bcount <= 0 ) {
			continue;
		}

		if( bcount >= (numpixels - 2) ) {

			for( j = 0; j < numpixels; j++ )  {
				pp[j] = black;
			}

		} else {

			for( j = 0; j < bcount; j++ )  {

				c = (int)lrand48();
				c %= numpixels;

				if( pp[c] == black ) {
					j--;
				} else {
					pp[c] = black;
				}
			}
		}


	}

}


/* 
 * MakeHeight (w)
 *
 * make the height vectors
 *
 * Description
 *	The height of the spectrogram on the display will almost certainly not
 * be an intregal number of coefficients high so we must display a coefficient
 * in some non-intregal number of pixels. The way we choose to do this is
 * through sampling rather than smoothing since it is much faster expecially
 * if the sampling is predetermined. Each 'height' vector specifies a
 * particular sampling. We use several height vectors to avoid an undesirable
 * hysteriesis from developing.
 *
 */

static int MakeHeight (GspecWidget w)
{
    height_t           *height = &w->gspec.height;
    register int       *data;
    register int        i, c;
    int                *datap;
    int                 byte_cnt;
    int                 y_end;
    int                 rows = w->gspec.data->rows;
    int                 pix_height = w->core.height;
    register float      base_coeff_pix_height = (float) pix_height / rows;
    register float      coeff_pix_height;


    /*
     * Check for a row count first 
     */
    if (rows <= 0){
	return;
	}/*endif*/

    /*
     * Get rid of old height buffer. 
     */
    XtFree ((char *) height->data);

    height->cnt = 1;		/* Be arbitrary here */

    byte_cnt = rows * sizeof (int);

    height->data = (int **)
	XtCalloc ((sizeof (int *) + byte_cnt) * height->cnt +
		  (sizeof (int **) * height->cnt),
		  1);
    datap = (int *) (height->data + height->cnt);

    /*
     * Compute each height vector. Each entry cooresponds to the height in
     * pixels for that coefficient 
     */
    for (i = 0; i < height->cnt; i++) {
	data = height->data[i] = datap + (rows * i);
	coeff_pix_height = 0.0;
	/*
	 * Compute the pixel height for each row. Note that the sum of the row
	 * heights will equal the pixel height for the window since
	 * base_coeff_pix_height * rows = pix_height 
	 */
	for (c = 0; c < rows; c++, data++) {
	    *data = 0;
	    coeff_pix_height += base_coeff_pix_height;
	    while (coeff_pix_height >= 1.0) {
		(*data)++;
		coeff_pix_height -= 1.0;
	    }
	}
    }
}
		    

/*
 * LocvalMax (w, col)
 *
 */

int LocalMax (GspecWidget w, register int col)
{
    register int start_col;
    register int end_col;
    register int *ptr;
    register int max;

    start_col = 0;
    end_col = w->gspec.data->cols;

    ptr = w->gspec.max_col_data + start_col;
    max = *ptr++;

    for (col = start_col + 1; col < end_col; col++, ptr++) {
	if (max < *ptr)
	    max = *ptr;
    }

    return (max);
}


/*
 * GlobalMax (w)
 *
 */

int GlobalMax (GspecWidget w)
{
    register int start_col;
    register int end_col;
    register int *ptr;
    register int max;
    register int col;

    start_col = 0;
    end_col = w->gspec.data->cols;

    ptr = w->gspec.max_col_data + start_col;

    max = *ptr;
    for (col = start_col; col < end_col; col++, ptr++) {
	if (max < *ptr)
	    max = *ptr;
    }

    return (max);
}


/* 
 * ComputeColMax (GspecWidget w)
 *
 * compute column max
 *
 */

static void ComputeColMax (GspecWidget w)
{
    register c, r;
    register adata_t *adata = w->gspec.data;
    register float *ptr = (float *) adata->ptr;
    register float *tptr;
    register int max;
    register int *max_col_data;

    if (adata == 0)
	return;

    if ((adata->ptr == 0) || (adata->count == 0))
	return;

    XtFree ((char *) w->gspec.max_col_data);
    w->gspec.max_col_data =
	(int *) XtCalloc (w->gspec.data->cols, sizeof (int));

    max_col_data = w->gspec.max_col_data;

    for (c = 0; c < adata->cols; c++) {
	tptr = ptr + (c * adata->rows);

  	max = (int) *tptr;

	for (r = 0; r < adata->rows; r++, tptr++) {
	    if( ((int)*tptr) > max)
		max = (int) *tptr;
	}
	*max_col_data++ = max;
    }
}


/* 
 * Compute_Max_Col_Unit (adata)
 *
 * Compute the deltas in the data vector col_time and return the
 * maximum delta.
 *
 */

int Compute_Max_Col_Unit (register adata_t *adata)
{
    register int        i, tmp, max;
    register int       *time = adata->col_time;

    max = 0;
    for (i = 0; i < adata->cols - 1; i++) {
	tmp = time[i + 1] - time[i];
	if (tmp > max)
	    max = tmp;
    }
    return (max);
}


/*
 * int HZToPix(w, hza)
 *
 */
 
int HZToPix(GspecWidget w, float hza)
{
  float nyquist = w->gspec.sampsPerSec / 2.0;

  return ((int)w->core.height-(int)(hza / (nyquist / 
					     (float)w->core.height)) - 1);
}


/*
 * float PixToHZ (w, pix)
 *
 */

float PixToHZ(GspecWidget w, float pix)
{
  return( 1.0 );
}


/*
 * int TabHZTToPix (w, hza)
 *
 */

int TabHZToPix(GspecWidget w, float hza)
{
/*
  register int i;

  for (i = 1; i <= w->gspec.data->rows; i++)
    if ((w->gspec.freqTab->tab[i-1] <= hza) && (hza < w->gspec.freqTab->tab[i]))
      break;

  if (i > w->gspec.freqTab->size)
    return (0);
  else {
    float pixPerBin = (float) w->core.height / w->gspec.freqTab->size;
    int  pix = pixPerBin * ((i-1) +
                             ((hza - w->gspec.freqTab->tab[i-1]) /
                              (w->gspec.freqTab->tab[i] -
                               w->gspec.freqTab->tab[i-1])));

    return (w->core.height - (pix+1));
  }
*/
  return( 0 );
}


/*
 * String PixToTabHZ (w, pix)
 *
 */

String PixToTabHZ(GspecWidget w, int pix)
{
  float binsPerPix;
  float floatBin;
  int  bin;
  String str;


  if( w->gspec.RowLabels == NULL ) {
	return( " " );
  }

  if( pix < 0 ) {

	pix = 0;

  } else if( pix > (int)w->core.height-1 ) {

	pix = w->core.height-1;

  }

  binsPerPix = (float) w->gspec.data->rows / (float)w->core.height;
  floatBin =   ((float)w->core.height - (float)(pix + 1)) * binsPerPix;
  bin = (int) floatBin;

  if( bin >= w->gspec.data->rows ) {

    bin = w->gspec.data->rows - 1;

  }

  str = w->gspec.RowLabels[bin];

  return( str );

}


/*
 * FEUpdata(w, event, params, num_param)
 *
 */

static void FEUpdate (LyreDispWidget w, XMotionEvent *event, 
		      String *params, Cardinal *num_params)
{
  int arg_cnt = 0;
  Arg  args[20];
  String str;


  str = w->lyreDisp.pixToHZ(w, event->y);

  MyXtSetArg( args, arg_cnt, XtNrowlabel, str );
  XtSetValues( w->lyreDisp.toolWidget, args, arg_cnt );

}


/*
 * FEZeor (w, event, params, num_params)
 *
 */

static void FEZero (LyreDispWidget w, XMotionEvent *event, String *params, 
		    Cardinal *num_params)
{
  int arg_cnt = 0;
  Arg  args[20];

  MyXtSetArg( args, arg_cnt, XtNrowlabel, " " );
  XtSetValues( w->lyreDisp.toolWidget, args, arg_cnt );

}


/* 
 * SmoothImage (w)
 *
 * Description
 *      This subroutine is simple, slow and doesn't work very
 * well so it is ifdef'd out.
 *
 */

static void SmoothImage (GspecWidget w)
{

}


/*
 * spix(x, y, i)
 *
 */

void spix(int x, int y, XImage *i)
{
    int idx = (i->bytes_per_line * y) + (x >> 3);
    i->data[idx] |= (1 << (x & 0x7));
}


/*
 * gpix(x, y, i)
 *
 */

int gpix(int x, int y, XImage *i)
{
    int idx = (i->bytes_per_line * y) + (x >> 3);
    return ((i->data[idx] >> (x & 0x7)) & 0x1);
}
