/*
  title: timer.c
  purpose: Interrupt timer functions (used by projector.c)
  
  authors:  Gareth Lee.
  date:     10-02-94
  modified: 04-05-94

  Copyright (C) 1994 Gareth Lee.
     
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  changes:
  10-02-94: routines extracted from projector.c during rationalization of this
    module.
  16-04-94: Minor changes during port to i386 Linux.  AlarmHandler parameters
    changed to increase compilance with header files.
  19-04-94: AlarmHandler() modified to reinstall itself on each `alarm' call
    in preparation for the next.  This seems to be a quirk of Linux's
    interrupt handling.  Under SunOS the driver remains installed until
    removed (and default behaviour reinstated).
  04-05-94: The previous alarm handler is now stored when StartTimer() is
    called and reistalled  with StopTimer().
*/
#include <stdio.h>
#include <signal.h>
#include <sys/time.h>
#include <assert.h>

void (*PreviousAlarmHandler)(int);  /* pointer to the previous alarm handler */

/*****************************************************************************/

/*
  AlarmHandler: Interrupt handler for the SIGALRM signal produced when
    drawing in sweep mode.
*/
void AlarmHandler(int x)
{
#ifdef LINUX
  assert(signal(SIGALRM, AlarmHandler) == SIG_DFL);
#endif
}

/*****************************************************************************/

/*
  StartTimer: setup the interval timer to generate regular interrupts with a
    period of `period' micro-seconds.  A dummy handler routine is registered
    to service the interrupts.  The previous handler is stored for
    reinstatement in StopTimer().
*/
void StartTimer(unsigned int period)
{
  struct itimerval interval;

  interval.it_interval.tv_sec = period / 1000000;
  interval.it_interval.tv_usec = period % 1000000;
  interval.it_value = interval.it_interval;
  setitimer(ITIMER_REAL, &interval, NULL);
  PreviousAlarmHandler = signal(SIGALRM, AlarmHandler);
}

/*****************************************************************************/

/*
  StopTimer: disable the periodic interrupts previously established by
    StartTimer().  Reinstall the previous alarm handler routine.
*/
void StopTimer(void)
{
  struct itimerval interval;

  interval.it_interval.tv_sec = 0;
  interval.it_interval.tv_usec = 0;
  interval.it_value.tv_sec = 0;
  interval.it_value.tv_usec = 0;
  setitimer(ITIMER_REAL, &interval, NULL);
  assert(signal(SIGALRM, PreviousAlarmHandler) == AlarmHandler);
}

/*****************************************************************************/

/* end of timer.c */
