/*
  title:   fvw2ascii.c
  purpose: read fview feature file formatted feature vectors and save in
    fview ascii format (which can be read using the ascii format converter.

  author:   Gareth Lee.
  date:     06-06-94
  modified: 07-06-94

  Copyright (C) 1994 Gareth Lee.
     
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  changes:
  07-06-94:  Compatibility added with previous Fview V1.0 feature format
    allowing conversion of old data files.
*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <string.h>
#include <math.h>
#include <assert.h>

#include "../fviewio.h"

/* Version One data structures to allow backward compatibility */
#define FVIEW_V1_FEATURE_MAGIC "fview1fh"

typedef struct                          /*fview feature header*/
{
  char magic[8];                        /*magic number*/
  unsigned vector_dimension;            /*dimension of data*/
  unsigned number_observations;         /*number of data points*/
  unsigned info_size;                   /*set to zero*/
} FviewV1FeatureHeader;

#define VERSION "1.2"

/*****************************************************************************/

char* file_name = NULL;
int header_option = 1;
int data_option = 1;
FviewFeatureHeader ffh;
char* text_header;

#ifdef SUNOS
extern char* optarg;                                /* for use with getopt() */
#endif
        
/*****************************************************************************/

/*
  DetectFileFormat: ascertian file format by reading and checking an 8 byte
    magic number (string) from the start of the file.
*/
int DetectFileFormat(FILE *fd)
{
  int format;
  char magic[8];
  assert(fread(magic, sizeof(char), 8, fd) == 8);

  /* compare magic against the known magic numbers */
  if (strncmp(magic, FVIEW_V1_FEATURE_MAGIC, 8) == 0)
    format = 1;                                         /* fview version 1.0 */
  else
    if (strncmp(magic, FVIEW_FEATURE_MAGIC, 8) == 0)
      format = 2;                                       /* fview version 1.2 */
    else
      format = 0;

  rewind(fd);
  return format;
}

/*****************************************************************************/

/*
  ReadNewHeaderRecord: reads version 1.2 feature file headers.
*/
void ReadNewHeaderRecord(FILE* fd)
{
  assert(fread(&ffh, sizeof(FviewFeatureHeader), 1, fd) == 1);
  /* Note: magic number already checked */
  text_header = malloc(ffh.info_size + 10);
  assert(text_header != NULL);
  assert(fread(text_header, sizeof(char), ffh.info_size, fd) == ffh.info_size);
}

/*****************************************************************************/

/*
  ReadOldHeaderRecord: reads version 1.0 feature file headers.
*/
void ReadOldHeaderRecord(FILE* fd)
{
  FviewV1FeatureHeader fv1;
  
  assert(fread(&fv1, sizeof(FviewV1FeatureHeader), 1, fd) == 1);
  /* Note: magic number already checked */

  /* convert the header to V2 data format */
  memcpy(ffh.magic, fv1.magic, 8);
  ffh.vector_dimension = fv1.vector_dimension;
  ffh.number_observations = fv1.number_observations;
  ffh.info_size = fv1.info_size;
  ffh.logged = FVIEW_FEAT_UNKNOWN;
  ffh.color_preference = FVIEW_COLR_NOPREF;

  text_header = malloc(ffh.info_size + 10);
  assert(text_header != NULL);
  assert(fread(text_header, sizeof(char), ffh.info_size, fd) == ffh.info_size);
}

/*****************************************************************************/

void WriteHeaderRecord(FILE* fd)
{
  int i, text_records;
  char *compression_status;

  /* count the text records */
  text_records = 0;
  for (i = 0; i < ffh.info_size; i++)
    if (text_header[i] == '\n')
      text_records++;

  /* print header line on standard out */
  switch(ffh.logged)
  {
  case FVIEW_FEAT_LINEAR:
    compression_status = "linear";
    break;
  case FVIEW_FEAT_UNKNOWN:
    compression_status = "unknown";
    break;
  case FVIEW_FEAT_LOGGED:
    compression_status = "logged";
    break;
  }
  printf("fview %d %d %d %s\n", ffh.vector_dimension, ffh.number_observations,
                                       (text_records + 1), compression_status);
  printf("%s\n", text_header);
  fflush(stdout);
}

/*****************************************************************************/

/*
  CopyDataRecords: copy data across to the output converting to ascii.
*/
void CopyDataRecords(FILE* fd)
{
  int i, j;
  double *buf;
  
  buf = malloc(ffh.vector_dimension * sizeof(double));
  assert(buf != NULL);
  for (i = 0; i < ffh.number_observations; i++)
  {
    assert(fread(buf, sizeof(double), ffh.vector_dimension, fd)
           == ffh.vector_dimension);
    for (j = 0; j < ffh.vector_dimension; j++)
      if (j < (ffh.vector_dimension - 1))
        printf("%f ", buf[j]);
      else
        printf("%f\n", buf[j]);
    fflush(stdout);
  }
  free(buf);

  fprintf(stderr, "fvw2ascii: %d records converted\n",ffh.number_observations);
  fflush(stderr);
}

/*****************************************************************************/

void DisplayUsage(void)
{
  fprintf(stderr, "usage: fvw2ascii [-d] [-h] -f file.fvw\n");
  fflush(stderr);
}

/*****************************************************************************/

int main(int argc, char* argv[])
{
  FILE* fd;
  int opt, fmt;

  while ((opt = getopt(argc, argv, "df:h")) != -1)
    switch (opt)
    {
    case 'd':
      header_option = 0;                         /* output data records only */
      break;
    case 'f':
      file_name = optarg;
      break;
    case 'h':
      data_option = 0;                         /* output header records only */
      break;
    default:
      DisplayUsage();
      exit(-1);
    }

  if (file_name == NULL)
  {
    fprintf(stderr, "fvw2ascii: error: no file name specified.\n");
    DisplayUsage();
    exit(-1);
  }
  fd = fopen(file_name, "r");
  assert(fd != NULL);

  fmt = DetectFileFormat(fd);
  switch (fmt)
  {
  case 1:
    fprintf(stderr, "fvw2ascii: warning: file has v1.0 header record\n");
    fflush(stderr);
    ReadOldHeaderRecord(fd);                   /* convert v1.0 header format */
    break;
  case 2:
    ReadNewHeaderRecord(fd);             /* read v1.2 header format directly */
    break;
  default:
    fprintf(stderr, "fvw2ascii: error: file format not recognised\n");
    exit(-1);
  }
  if (header_option)
    WriteHeaderRecord(fd);
  if (data_option)
    CopyDataRecords(fd);

  assert(fclose(fd) == 0);
  return 0;
}

/*****************************************************************************/

/* end of fvw2ascii.c */
