/*
  title:    colours.c
  purpose:  Allows 3D color set to be modified during operation by
    creating a separate sub-window for which RGB values may be
    controlled using sliders.  Intended for linking into fview.

  author:   Gareth Lee & Hugh Sanderson
  date:     06-12-93
  modified: 15-02-94

  Copyright (C) 1994 Gareth Lee and Hugh Sanderson.
     
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  changes:
  08-12-93: 3D colours menu completed.
  15-02-94: ColorExposeWindow() and ColorHideWindow() added.
*/

/* Standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <math.h> 
#include <string.h>
#include <unistd.h>

/* X lib headers */
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>

/* XView headers */
#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/cursor.h>
#include <xview/cms.h>

typedef REAL real;

/* Header files associated with other compilation units */
#include "projectors.h"
#include "callbacks.h"
#include "fview.h"
#include "interface.h"
#include "colours.h"
#include "dbuffer.h"

typedef unsigned char byte;
typedef struct
{
   byte r, g, b;
} R_G_B;

#include <xview/xview.h>
#include <xview/panel.h>
 
Xv_opaque  ColorFrame;
Xv_opaque  ColorPanel;
Xv_opaque  ColorSliderR;
Xv_opaque  ColorSliderG;
Xv_opaque  ColorSliderB;
Xv_opaque  ColorDepthSlider;
Xv_opaque  ColorSelector;
Xv_opaque  ColorCanvas;
Window     ColorWindow;

int SelectedColour = 0;

int ColorFrameExposed = 0;                         /* frame initially hidden */
static R_G_B ColDat[4];
static unsigned long Colours[4];

/*****************************************************************************/
 
void ChangeSliderCallback(void)
{
  SelectedColour = xv_get(ColorSelector, PANEL_VALUE);
  
  xv_set(ColorSliderR, PANEL_VALUE, ColDat[SelectedColour].r, NULL);
  xv_set(ColorSliderG, PANEL_VALUE, ColDat[SelectedColour].g, NULL);
  xv_set(ColorSliderB, PANEL_VALUE, ColDat[SelectedColour].b, NULL);
}

/*****************************************************************************/

void SliderNotify(void)
{
  XColor xc;
  
  xc.flags = DoRed | DoGreen | DoBlue;
  xc.pixel = Colours[SelectedColour];

  ColDat[SelectedColour].r = xv_get(ColorSliderR, PANEL_VALUE);
  ColDat[SelectedColour].g = xv_get(ColorSliderG, PANEL_VALUE);
  ColDat[SelectedColour].b = xv_get(ColorSliderB, PANEL_VALUE);

  xc.red   = ColDat[SelectedColour].r * 256;
  xc.green = ColDat[SelectedColour].g * 256;
  xc.blue  = ColDat[SelectedColour].b * 256;

  XStoreColor(display, DefColormap, &xc);
}

/*****************************************************************************/

void SetupSliders(void)
{
  int i;
  XColor xc;

  /* Use the current colour values */
  for (i = 0; i < 4; i++)
  {
    ColDat[i].r = dim3_defs[i].red / 256;
    ColDat[i].g = dim3_defs[i].green / 256;
    ColDat[i].b = dim3_defs[i].blue / 256;

    xc = dim3_defs[i];    
    xc.pixel = Colours[i];
    
    XStoreColor(display, DefColormap, &xc);
  }
  
  ChangeSliderCallback();  
}

/*****************************************************************************/

void DepthSliderCallback(Panel_item item, int value)
{
  DepthScaling3D = value;
  RefreshGraphics();
}

/*****************************************************************************/

void CreateColorWindow(Xv_opaque parent)
{
  unsigned long plane_mask;
  
  if (!XAllocColorCells(display,DefColormap,FALSE,&plane_mask,0,Colours,4))
  {
    printf("Error: could not allocate demo colors\n");
    Quit(-1);
  }

  ColorFrame =
    xv_create(parent, FRAME_CMD,
              XV_X,      100,
              XV_Y,      200,
              XV_WIDTH,  250,
              XV_HEIGHT, 280,
              XV_LABEL, "3D Graphics Controls",
              FRAME_SHOW_RESIZE_CORNER, FALSE,
              FRAME_CMD_DEFAULT_PIN_STATE, FRAME_CMD_PIN_IN,
              FRAME_DONE_PROC, ColorDismissCallback,
              XV_SHOW, FALSE,
              NULL);
  ColorCanvas =
    xv_create(ColorFrame, CANVAS,
              XV_X, 0,
              XV_Y, 0,
              XV_WIDTH, WIN_EXTEND_TO_EDGE,
              XV_HEIGHT, 100,
              CANVAS_X_PAINT_WINDOW, TRUE,
              CANVAS_RETAINED, TRUE,
              CANVAS_FIXED_IMAGE, TRUE,
              NULL);
  ColorWindow = (Window) xv_get(canvas_paint_window(ColorCanvas), XV_XID);
  
  ColorPanel =
    xv_create(ColorFrame, PANEL,
              XV_X, 0,
              XV_Y, 100,
              XV_WIDTH , WIN_EXTEND_TO_EDGE,
              XV_HEIGHT, WIN_EXTEND_TO_EDGE,
              WIN_BORDER, FALSE,
              NULL);

  CreateRGBSlider(ColorPanel,  5, 10, 25);
  ColorSelector =
    xv_create(ColorPanel, PANEL_CHOICE,
              XV_X, 5,
              XV_Y, 95,
              PANEL_CHOICE_NROWS, 1,
              PANEL_LAYOUT, PANEL_VERTICAL,
              PANEL_CHOOSE_NONE, FALSE,
              PANEL_LABEL_STRING, "Colour selected:",
              PANEL_CHOICE_STRINGS,
                "BackGround",
                "Text",
                "Left",
                "Right",
                NULL,
              PANEL_NOTIFY_PROC, ChangeSliderCallback,
              NULL);

  ColorDepthSlider =
    xv_create(ColorPanel, PANEL_SLIDER,
              XV_X, 5,
              XV_Y, 150,
              PANEL_LABEL_STRING, "Depth: ",
              PANEL_SLIDER_WIDTH, 120,
              PANEL_TICKS, 11,
              PANEL_DIRECTION, PANEL_HORIZONTAL,
              PANEL_SHOW_VALUE, FALSE,
              PANEL_MIN_VALUE, 10,
              PANEL_MAX_VALUE, 70,
              PANEL_VALUE, DepthScaling3D,
              PANEL_NOTIFY_PROC, DepthSliderCallback,
              NULL);

  SetupSliders();
  DrawTestCanvas();
}

/*****************************************************************************/

void CreateRGBSlider(Xv_opaque panel, int x, int y, int space)
{
  ColorSliderR =
    xv_create(panel, PANEL_SLIDER,
              XV_X, x,
              XV_Y, y,
              PANEL_LABEL_STRING, "R: ",
              PANEL_SLIDER_WIDTH, 160,
              PANEL_TICKS, 16,
              PANEL_DIRECTION, PANEL_HORIZONTAL,
              PANEL_SHOW_RANGE, FALSE,
              PANEL_MIN_VALUE, 0,
              PANEL_MAX_VALUE, 255,
              PANEL_VALUE, 0,
              PANEL_NOTIFY_PROC, SliderNotify,
              NULL);
  ColorSliderG =
    xv_create(panel, PANEL_SLIDER,
              XV_X, x,
              XV_Y, y + space,
              PANEL_LABEL_STRING, "G: ",
              PANEL_SLIDER_WIDTH, 160,
              PANEL_TICKS, 16,
              PANEL_DIRECTION, PANEL_HORIZONTAL,
              PANEL_SHOW_RANGE, FALSE,
              PANEL_MIN_VALUE, 0,
              PANEL_MAX_VALUE, 255,
              PANEL_VALUE, 0,
              PANEL_NOTIFY_PROC, SliderNotify,
              NULL);
  ColorSliderB =
    xv_create(panel, PANEL_SLIDER,
              XV_X, x,
              XV_Y, y + (space * 2),
              PANEL_LABEL_STRING, "B: ",
              PANEL_SLIDER_WIDTH, 160,
              PANEL_TICKS, 16,
              PANEL_DIRECTION, PANEL_HORIZONTAL,
              PANEL_SLIDER_END_BOXES, FALSE,
              PANEL_SHOW_RANGE, FALSE,
              PANEL_SHOW_VALUE, TRUE,
              PANEL_MIN_VALUE, 0,
              PANEL_MAX_VALUE, 255,
              PANEL_VALUE, 0,
              PANEL_NOTIFY_PROC, SliderNotify,
              NULL);
}

/*****************************************************************************/

int DrawTestCanvas()
{
  XSetForeground(display, gc_3d, Colours[0]);
  XFillRectangle(display, ColorWindow, gc_3d, 0, 0, 250, 100);
  XSetForeground(display, gc_3d, Colours[2]);
  XFillRectangle(display, ColorWindow, gc_3d, 50, 0, 50, 100);
  XSetForeground(display, gc_3d, Colours[3]);
  XFillRectangle(display, ColorWindow, gc_3d, 150, 0, 50, 100);
  XSetForeground(display, gc_3d, Colours[1]);
  XDrawString(display, ColorWindow, gc_3d, 5, 90,
              "An example of the chosen text colour", 36);
}

/*****************************************************************************/

void ColorDismissCallback(void)
{
  int i;
  XColor xc;
  
  xv_set(ColorFrame, XV_SHOW, FALSE, NULL);
  xv_set(DisplayColorsButton, PANEL_INACTIVE, FALSE, NULL);

  /* return colours to the dim3_defs structure */
  for (i = 0; i < 4; i++)
  {
    xc.red   = ColDat[i].r * 256;
    xc.green = ColDat[i].g * 256;
    xc.blue  = ColDat[i].b * 256;
    xc.flags = DoRed | DoGreen | DoBlue;
    
    dim3_defs[i] = xc;
  }

  /* Incorporate the new colours into the double buffering scheme */
  SwapDisplayBuffer();
}

/*****************************************************************************/

/*
  ColorExposeWindow: Expose colour settings window and disable button.
*/
void ColorExposeWindow(void)
{
  SetupSliders();
  xv_set(ColorFrame, XV_SHOW, TRUE, NULL);
  ColorFrameExposed = 1;
  xv_set(DisplayColorsButton, PANEL_INACTIVE, TRUE, NULL);
}

/*****************************************************************************/

/*
  ColorHideWindow: Hide colour settings window without reading widget states.
    This call is generated when moving to a different display mode whilst
    the colour setting window is open.
*/
void ColorHideWindow(void)
{
  if (ColorFrameExposed)
  {
    ColorFrameExposed = 0;

    /*
      Note: must remove the pin from the window before it will allow itself
      to be closed.
    */
    xv_set(ColorFrame,
           FRAME_CMD_PIN_STATE, FRAME_CMD_PIN_OUT,
           XV_SHOW, FALSE,
           NULL);
    xv_set(DisplayColorsButton, PANEL_INACTIVE, FALSE, NULL);
  }
}

/*****************************************************************************/

/* end of colours.c */
