/**************************************************************************
*
* ROUTINE
*               distortion
*
* FUNCTION
*                
*               computes distortion/distance measures and likelihood ratio
*
* SYNOPSIS
*
*	subroutine distortion(s, ts, ws, l, no, dm, sumdm, framedm)
*
*   formal 
*
*                       data    I/O
*       name            type    type    function
*       -------------------------------------------------------------------
*	s		float	i	"reference" speech input
*	ts		short	i	"test" speech input
*	ws		float	i	Hamming window, length l
*	l		int	i	length of comparison
*	no		int	i	filter order
*	dm		float	o	distances array (subframe)
*	sumdm		float	o	distances array (current sum)
*	framedm		int	o	number of subframes  
*
***************************************************************************
*
* DESCRIPTION
*
*	Calculate distortions/distances (log spectral error, etc.).
*	See the first reference below for a complete description.  A
*	"reference" system is compared against a "test" system.  Because
*	of the nonsymetric nature of the Itakura-Saito measure which some
*	of these distortion measures are based, poorer measures will be
*	obtained if the "reference" and "test" systems are reversed.
*	Because of gain uncertainties, a few measures are reported.
*	(Peter Kroon generally uses the measure DM(4).)
*
***************************************************************************
*
* CALLED BY
*
*	celp
*
* CALLS
*
*	cor   dist
*
***************************************************************************
* 
* REFERENCES
*
*	"Distance Measures for Speech Processing", A.H. Gray and J.D. Markel,
*	IEEE Trans. on ASSP, Vol. ASSP-24, no. 5, Oct. 1976
*
*	"Quantization and Bit Allocation in Speech Processing",
*	A.H. Gray and J.D. Markel,IEEE Trans. on ASSP, Vol. ASSP-24
*	no. 6, Dec. 1976
*
*	"A Note on Quantization and Bit Allocation in Speech Processing",
*	A.H. Gray and J.D. Markel,IEEE Trans. on ASSP, Vol. ASSP-25
*	no. 3, June 1977
*		
**************************************************************************/
#include "ccsub.h"
distortion(s, ts, ws, l, no, dm, sumdm, framedm)
int l, no, *framedm;
short ts[];
float s[], ws[], dm[], sumdm[];
{
  float tsw[MAXL], ssw[MAXL], c0hp, chp[MAXNO+1], c0tsw, ctsw[MAXNO+1];
  int j;

/*	*apply Hamming window						*/

  for (j = 0; j < l; j++)
  {
    ssw[j] = ws[j] * s[j];
    tsw[j] = ws[j] * (float)ts[j];
  }

  /*	*calculate autocorrelation sequences				*/

  cor(ssw, l, no, &c0hp, chp);
  cor(tsw, l, no, &c0tsw, ctsw);

  /*	*rearrange arrays for dist					*/

  for (j = no; j > 0; j--)
  {
    chp[j] = chp[j-1];
    ctsw[j] = ctsw[j-1];
  }
  chp[0] = c0hp;
  ctsw[0] = c0tsw;

  /*	*find distances							*/

  if (chp[0] != 0.0 && ctsw[0] != 0.0)
  {
    j = no * 4;
    dist(no, j, chp, ctsw, dm, sumdm, framedm);
  }
}
