/**************************************************************************
*
* ROUTINE
*               cbsearch
*
* FUNCTION
*               find optimal MSPE excitation code word
*
* SYNOPSIS
*               subroutine cbsearch(l, v)
*
*   formal 
*
*                       data    I/O
*       name            type    type    function
*       -------------------------------------------------------------------
*       l               int     i       length of vectors
*       v(l)            float   o       optimum excitation segment found
*
*   external 
*                       data    I/O
*       name            type    type    function
*       -------------------------------------------------------------------
*	cbgbits		int	i	code book gain bit allocation
*	cbindex		int	i/o	code book index
*	gindex		int	i/o	gain index
*	ncsize		int	i	code book size
*	no		int	i	filter order  predictor
*	h[]		float	i	impulse response
*	x[];		float	i	code book
*	cbgtype[];	char	i	code book gain quantizer type
*	cb_gain_vid	int	i
*	cb_qgain_vid	int	i
*	cb_match_vid	int	i
*	cb_exc_vid	int	i
*	cb_ir_vid	int	i
*	dbcon_vid	int     i
*	mxsw		int	i	modified excitation switch
*
***************************************************************************
*	
* DESCRIPTION
*
*  Code book search is performed by closed-loop analysis using conventional
*  minimum squared prediction error (MSPE) criteria of the perceptually
*  weighted error signal.  The code book is overlaped to allow recursive
*  computational savings in routine cgain:
*
*	index	code book
*		+-------------------------+
*	1	| 2(M-1)       2(M-1)+L-1 |
*	2	| 2(M-2)       2(M-2)+L-1 |
*	:	| :            :	  |
*	N-1	| .            .	  |
*	N	| .            .	  |
*	:	| 2            61	  |
*	:	| 0            59	  |
*		+-------------------------+
*
*	where: M = maximum code book size
*	       N = actual code book search size (any value between 1 & M)
*	       L = code word length
*
*	each code word is:  2(M-index) -> 2(M-index)+L-1
*  	
***************************************************************************
*
*   global 
*
*
*	CODE BOOK VARIABLES:
*	err	real	error for each code book entry
*	gain	real	gain for each code book entry
*
***************************************************************************
* CALLED BY
*
*       csub
*
* CALLS
*
*	cgain   [gainencode]   save_sg   mexcite3
*
***************************************************************************
*
*
* REFERENCES
*
*	Tremain, Thomas E., Joseph P. Campbell, Jr and Vanoy C. Welch,
*	"A 4.8 kbps Code Excited Linear Predictive Coder," Proceedings
*	of the Mobile Satellite Conference, 3-5 May 1988, pp. 491-496.
*
*	Campbell, Joseph P. Jr., Vanoy C. Welch and Thomas E. Tremain,
*	"An Expandable Error-Protected 4800 bps CELP Coder (U.S. Federal
*	Standard 4800 bps Voice Coder)," Proceedings of ICASSP, 1989.
*	(and Proceedings of Speech Tech, 1989.)
*
*************************************************************************/
#define TRUE		1
#define FALSE		0
#define LEN		30 /* length of truncated impulse response */
#include "ccsub.h"
#include <math.h>
extern int cbgbits, cbindex, gindex, ncsize, mxsw;
extern float h[MAXLP], x[MAXCODE];
extern char cbgtype[10];
#ifdef SUNGRAPH
extern int cb_gain_vid, cb_qgain_vid, cb_match_vid, cb_exc_vid, cb_ir_vid;
extern int dbcon_vid;
#endif
cbsearch(l, v)
int l;
float v[];
{
  int i, codeword;
  float emax, gain[MAXNCSIZE], err[MAXNCSIZE], cgain(), gainencode();
  float quangain, oldgain;

#ifdef SUNGRAPH
  float dbcon;
#endif

#ifdef SUNGRAPH

  /*		*save impulse response in file 'codebook'		*/
  /*		*(same as impulse response in file 'pitch')             */
  
  save_sg(cb_ir_vid, h, l, "save cb_ir_vid");
#endif
  
  /*		*find gain and -error term for each code word		*/
  /*		*and search for best code word (max -error term)	*/
  /*		*(codewords are overlapped by shifts of -2		*/
  /*		* along the code vector x)				*/
  /*		*NOTE: gain(i) & err(i) can be replaced by scalars  	*/  
  codeword = 2*MAXNCSIZE - 2;
  cbindex = 1;
  gain[0] = cgain(&x[codeword], l, TRUE, LEN, &err[0]);
  emax = *err;
  codeword -= 2;
  for (i = 1; i < ncsize; i++)
  {
    gain[i] = cgain(&x[codeword], l, FALSE, LEN, &err[i]);
    codeword -= 2;
    if (err[i] >= emax)
    {
      emax = err[i];
      cbindex = i + 1;
    }
  }

  /*   if (err(cbindex).lt.0.0)print *,' CB match<0',frame,err(cbindex) */

  /*		*pointer to best code word				*/

  codeword = 2*(MAXNCSIZE - cbindex);

  /*		*OPTIONAL (may be useful for integer DSPs)		*/
  /*		*given best code word, recompute its gain to		*/
  /*		*correct any accumulated errors in recursions		*/
  gain[cbindex-1] = cgain(&x[codeword], l, TRUE, l, &err[cbindex-1]);

#ifdef SUNGRAPH

  /*		*save code book gain in file 'codebook'			*/
  /*		*if closed-loop quant, this is QUANTIZED gain		*/
  
  save_sg(cb_gain_vid, &gain[cbindex-1], 1, "save cb_gain_vid");

  /*		*save code book error in file 'codebook'		*/
  
  save_sg(cb_match_vid, err, ncsize, "save cb_error_vid");
#endif

  oldgain = gain[cbindex - 1];

  /* *constrained excitation						*/
  if (mxsw) 
    mexcite3(&gain[cbindex - 1]);

  /*		*gain quantization, UNNECESSARY for closed-loop quant	*/
  
  if (strncmp(cbgtype, "none", 4) != 0)
    if (cbgbits == 5)
      gain[cbindex-1] = gainencode(gain[cbindex-1], &gindex);
  else
    printf("cbsearch: not quantizing cbgain\n");
    
  /*		*scale selected code word vector -> excitation array	*/
  /*		*call VDECODE?						*/
  
  for (i = 0; i < l; i++)
    v[i] = gain[cbindex - 1] * x[i + codeword];
#ifdef SUNGRAPH

  /*		*save quantized code book gain in file 'codebook'	*/
  
  if (oldgain != 0)
    dbcon = 20.0 * log10(fabs(oldgain / gain[cbindex - 1]));
  quangain = gain[cbindex - 1];
  
  save_sg(cb_qgain_vid, &quangain, 1, "save cb_qgain_vid");
  save_sg(dbcon_vid, &dbcon, 1, "save dbcon_vid");
  
  /*		*save code book excitation in file 'codebook'		*/
  
  save_sg(cb_exc_vid,v,l, "save cb_exc_vid");
  
#endif
}
