/**************************************************************************
*
* NAME
*	actorc 
*
* FUNCTION
*
*	Schur recursion to do autocorrelation analysis.
*	Converts autocorrelation sequence to reflection coefficients.
*
* SYNOPSIS
*
*	subroutine actorc (c0, c, rc, n, err)
*
*   formal 
*                       data    I/O
*       name            type    type    function
*       -------------------------------------------------------------------
*       c0              float   i       c(0)
*	c(n)		float	i	auto-correlation coefficients
*       rc(n)		float   o       reflection coefficients (voiced-> +rc1)
*       n               int     i       number of reflection coefficients
*	err		float	i/o	normalized prediction error
*       
***************************************************************************
*       
* DESCRIPTION
*
*       This performs the classical Schur recursion (rediscovered by
*	LeRoux & Guegen) on the correlation sequence c0, c(1), c(2) . . .
*	to obtain n reflection coefficients (rc).  The recursion can be
*	performed entirely in fractional arithemetic because the
*	correlation sequence is normalized by c0, so all the quantities
*	in the recursion are less than unity magnitude (except c0).
*
*	The sign convention used defines the first reflection coefficient
*	as the normalized first autocorrelation coefficient, which results
*	in positive values of rc(1) for voiced speech.
*
***************************************************************************
*
* CALLED BY
*
*	autohf
*
* CALLS
*
*
*
***************************************************************************
*       
* REFERENCES
*
*	Parsons, Voice and Speech Processing, McGraw-hill, 1987, p.160&378.
*
***************************************************************************
*       
* EXAMPLES
*
*	call cor(data,np,c0,c,n)
*	call actorc(c0,c,rc,n)
***************************************************************************
*
* PROCESS DESCRIPTION
*
*       Name    Type    Function
*
*       d       float      Recursion sequence (dim n)
*       g       float      Recursion sequence (dim n)
*       p       int        Orderindex recursion
*       err     float      Backward error
*       err0    float      Backward error of order 0
**************************************************************************/
#include "ccsub.h"
actorc(c0, c, rc, n, err)
int n;
float c0, c[], rc[], *err;
{
  float d[MAXNO], g[MAXNO], rch;
  int i, p;

  if (c0 <= 0.0) 	/* If zero energy, set rc's to zero & return	*/
  {
    for (i = 0; i < n; i++)
      rc[i] = 0.0;
    return;
  }
  for (i = 0; i < n; i++)
    g[i] = d[i] = c[i] / c0;
  rch = g[0];
  rc[0] = rch;
  *err = 1. - rch * rch;
  for (p = 1; p < n; p++)
  {
    for (i = 0; i < n - p; i++)
    {
      g[i] = g[i+1] - rch * d[i];
      d[i] = d[i] - rch * g[i+1];
    }

    rch = g[0] / *err;		/* Extract the reflection coefficient	*/
    rc[p] = rch;

    *err = *err * (1. - rch * rch); /* The mean squares error recursion */
  }
}
