;;; -*- Package: Toolbed; Syntax: Common-Lisp; Mode: Lisp; Base: 10 -*-

(in-package 'toolbed)

(defun create-monitor-handle (associate-with report-to eval-form 
                                    children-and-args) 

;;;  This function creates a new Monitor-Handle and links it into the
;;; monitor handle structure. This function should only be called
;;; after Get-Monitor-Handle has failed. This function returns the new
;;; handle.
 

;;;  Associate-With should be a clos object with a  Monitors slot.
;;; This is the object which this spell has been cast on.

;;;  Report-To is the thing that should be poked when this monitor is
;;; tripped.

;;; Eval-Form is the LISP executable code whose value this spell is
;;; keeping track of

;;;  Children-and-Args is a list of dotted pairs. The car is a clos
;;; object upon which this handle depends.  The cdr is the arguments
;;; which should be passed to this clos object when a spell is
;;; atempted on this object.

 (let ((handle (pcl::*make-instance 'monitor-handle)))
       
       ;;  Cast-spells on Children. Generate a list of children
       ;; whose value this spell is keeping track of.

      (setf (slot-value handle 'children)
            (remove 'spell-failed
                   (mapcar #'(lambda (child-arg)
                                    (cast-spell (car child-arg)
                                           (cdr child-arg)
                                           handle)) children-and-args))
            )
       
       ;;  Compute the first value for handle

      (setf (slot-value handle 'eval-result)
            (eval eval-form))
      (setf (slot-value handle 'associate-with)
            associate-with)
      (setf (slot-value handle 'parents)
            (list report-to))
      (setf (slot-value handle 'eval-form)
            eval-form)
      (setf (slot-value associate-with 'monitors)
            (cons handle (slot-value associate-with 'monitors)))
      handle))


(defun get-monitor-handle (associated-with report-to eval-form 
                                 children-and-args) 

;;;  This function installs Report-to on a current Monitor-handle with
;;; proper Eval-form, or Create-monitor-handle is called. The  handle
;;; is returned. See Create-monitor-handle for definition of  args.
 (do ((rest (slot-value associated-with 'monitors)
            (cdr rest)))
     ((null rest)
      (create-monitor-handle associated-with report-to eval-form 
             children-and-args))
     (if (equal eval-form (slot-value (car rest)
                                 'eval-form))
       
       ;; We have appropriate handle. We must poke it to verify
       ;; that its value is correct. Then add Report-To to its
       ;; list of parents.

         (progn (poke-monitor (car rest))
                (setf (slot-value (car rest)
                             'parents)
                      (cons report-to (slot-value (car rest)
                                             'parents)))
                (return (car rest))))))


(defun poke-monitor (monitor-handle) 

;;; This function is used to implement the message pass when a
;;; monitored CLOS object wishes to report a change to its parent.
;;; Poke-monitor accepts the handle being poked. The action to be
;;; performed is defined by the Monitor-handle.
 (cond 
       ;; if  Monitor-handle is not an instance of the class 
       ;; MONITOR-HANDLE, assume it is a form to be eval'ed

    ((not (typep monitor-handle 'monitor-handle))
     (eval monitor-handle))
    (t (let ((new-value (eval (slot-value monitor-handle 'eval-form))))
            (cond
       
       ;; check if value being monitored has changed, if so
       ;; then update Eval-Result and poke parents

               ((not (equal new-value (slot-value monitor-handle
                                             'eval-result)))
                (setf (slot-value monitor-handle 'eval-result)
                      new-value)
                (dolist (parent (slot-value monitor-handle 'parents))
                       (poke-monitor parent))))))))


(defun remove-monitor (monitor-handle) 

;;; This function removes a monitor which was placed on a Dragon.
;;; Remove-monitor will recursively remove all dependent monitors with
;;; no other parents. WARNING: do not use this function unless you
;;; know there are no other handles which depend on this handle. Use
;;; Remove-Parent-From-Monitor instead.
 (let ((associate-with (slot-value monitor-handle 'associate-with)))
      (dolist (child (slot-value monitor-handle 'children))
             (remove-parent-from-monitor child monitor-handle))
      (setf (slot-value associate-with 'monitors)
            (remove monitor-handle (slot-value associate-with
                                          'monitors)))))


(defun remove-parent-from-monitor (child parent) 

;;; This function removes Parent from the list of Monitor-handles
;;; which Child should report changes to. If the Child has no parents
;;; it is to report to, then remove the Child.
 (setf (slot-value child 'parents)
       (remove parent (slot-value child 'parents)))
 (if (null (slot-value child 'parents))
     (remove-monitor child)))
