;;; -*- Package: Toolbed; Syntax: Common-Lisp; Mode: Lisp; Base: 10 -*-

(in-package 'toolbed)

(DEFUN CREATE-ENGRAM (DRAGON VERB MORE-ARGS)
   (DECLARE (SPECIAL *CURRENT-DRAGON* *CURRENT-CASE* 
                   *DECISION-SUPPORT-MODE*))
   (LET ((INVOKEES-ENGRAM (PCL::*make-instance 'ENGRAM))
         (CALLERS-ENGRAM (SLOT-VALUE *CURRENT-DRAGON* 'CURRENT-ENGRAM))
	 new-action-record temp-engram)
       
       ;; This function creates an Engram for an invoked
       ;; dragon.  It then connects the dragon to the invoking
       ;; dragon by means of a transaction or consultation
       ;; record.  The substructure created is maintained by
       ;; this function and the Fill-Engram function
       
       ;; If in decision-support-mode, use a
       ;; consultation-record else use a transaction record.

        (IF *DECISION-SUPPORT-MODE* (SETF NEW-ACTION-RECORD
                                          (PCL::*make-instance 'CONSULTATION-RECORD))
            (SETF NEW-ACTION-RECORD (PCL::*make-instance 'TRANSACTION-RECORD
                                           )))
       
       ;; Connect the action-record to current-engram and the
       ;; invoking dragons current engram.  Also fill in the
       ;; slots of the newly created engram and action record.

        (IF (NOT (toolset::slot-empty-p callers-engram 'last-action-record))
            (PROGN (SETF (SLOT-VALUE NEW-ACTION-RECORD 'PREVIOUS-RECORD)
                         (SLOT-VALUE CALLERS-ENGRAM 'LAST-ACTION-RECORD))
                   (SETF (SLOT-VALUE (SLOT-VALUE CALLERS-ENGRAM
                                            'LAST-ACTION-RECORD)
                                'NEXT-RECORD)
                         NEW-ACTION-RECORD)
                   (SETF (SLOT-VALUE CALLERS-ENGRAM 'LAST-ACTION-RECORD)
                         NEW-ACTION-RECORD))
            (PROGN (SETF (SLOT-VALUE CALLERS-ENGRAM 'FIRST-ACTION-RECORD)
                         NEW-ACTION-RECORD)
                   (SETF (SLOT-VALUE CALLERS-ENGRAM 'LAST-ACTION-RECORD)
                         NEW-ACTION-RECORD)))
        (SETF (SLOT-VALUE INVOKEES-ENGRAM 'INVOKERS-TRANSACTION-RECORD)
              NEW-ACTION-RECORD)
        (SETF (SLOT-VALUE NEW-ACTION-RECORD 'INCLUDING-ENGRAM)
              CALLERS-ENGRAM)
        (SETF (SLOT-VALUE INVOKEES-ENGRAM 'WHOSE-MEMORY?)
              DRAGON)
        (SETF (SLOT-VALUE INVOKEES-ENGRAM 'CASE)
              *CURRENT-CASE*)
        (SETF (SLOT-VALUE DRAGON 'CURRENT-ENGRAM)
              INVOKEES-ENGRAM)
        (SETF (SLOT-VALUE NEW-ACTION-RECORD 'CONTACTEE)
              (SLOT-VALUE DRAGON 'UNIQUE-NAME))
        (SETF (SLOT-VALUE NEW-ACTION-RECORD 'CONTACTEE-ENGRAM-POINTER)
              INVOKEES-ENGRAM)
        (SETF (SLOT-VALUE NEW-ACTION-RECORD 'INVOKE-ARGS)
              (CAR MORE-ARGS))
        (SETF (SLOT-VALUE NEW-ACTION-RECORD 'INVOKE-VERB)
              VERB)
       
       ;; If the size of memory is too large, chop off the
       ;; oldest engram from the memory list

        (IF (EQUAL (SLOT-VALUE DRAGON 'MEMORY-CURRENT-SIZE)
                   (SLOT-VALUE DRAGON 'MEMORY-RETENTION-LIMIT))
            (PROGN (SETF TEMP-ENGRAM (LAST (SLOT-VALUE DRAGON
                                                  'MEMORY)))
                   (SETF (SLOT-VALUE DRAGON 'MEMORY)
                         (NBUTLAST (SLOT-VALUE DRAGON 'MEMORY)))
                   (DESTROY TEMP-ENGRAM)
                   (SETF TEMP-ENGRAM NIL)))
       
       ;; Now add the latest engram to memory 

        (PUSH INVOKEES-ENGRAM (SLOT-VALUE DRAGON 'MEMORY))))


(DEFUN FILL-ENGRAM (dragons-engram callers-engram USER-ANSWER RESULT)
   (DECLARE (SPECIAL *CURRENT-DRAGON* *DECISION-SUPPORT-MODE*))
       
       ;; This function fills in the result of the invocation
       ;; into the appropriate slots of the current-dragon's
       ;; engram and the newly created action-record (this will
       ;; either be a transaction record or a consultation
       ;; record. 
       
       ;; If *decision-support-mode* is not on, then use result
       ;; from the dragon as the value to be placed into Body
       ;; and Final-judgement.
       
       ;; Else, if *decision-support-mode* is on, then check
       ;; user-answer .  If user-answer is "not-used" then use
       ;; result as if *decision-support-mode* were off by
       ;; placing result in Body and Final-judgement, also
       ;; place it in Dragon-recommendation, and make
       ;; User-comment "not-used" implying that the user did
       ;; not use his/her option of offering a different value. 
       
       ;;  If user-answer is some other value, then use the
       ;; user-answer for Body and Final-judgement, while
       ;; placing the dragon's result as 
       ;; Dragon-recommendation.       

   (LET ((NEW-ACTION-RECORD (SLOT-VALUE callers-engram
					'LAST-ACTION-RECORD)))
        (IF (NOT (TYPEP NEW-ACTION-RECORD 'CONSULTATION-RECORD))
            (PROGN (SETF (SLOT-VALUE callers-engram
                                'BODY)
                         RESULT)
                   (SETF (SLOT-VALUE NEW-ACTION-RECORD 'FINAL-JUDGEMENT
                                )
                         RESULT))
            (PROGN (IF (NOT (EQUAL USER-ANSWER 'NOT-USED))
                       (PROGN (SETF (SLOT-VALUE callers-engram
                                           'BODY)
                                    USER-ANSWER)
                              (SETF (SLOT-VALUE NEW-ACTION-RECORD
                                           'FINAL-JUDGEMENT)
                                    USER-ANSWER)
                              (SETF (SLOT-VALUE NEW-ACTION-RECORD
                                           'DRAGON-RECOMMENDATION)
                                    RESULT)
                              (SETF (SLOT-VALUE NEW-ACTION-RECORD
                                           'USER-COMMENT)
                                    NIL))
                       (PROGN (SETF (SLOT-VALUE callers-engram
                                           'BODY)
                                    RESULT)
                              (SETF (SLOT-VALUE NEW-ACTION-RECORD
                                           'FINAL-JUDGEMENT)
                                    RESULT)
                              (SETF (SLOT-VALUE NEW-ACTION-RECORD
                                           'DRAGON-RECOMMENDATION)
                                    RESULT)
                              (SETF (SLOT-VALUE NEW-ACTION-RECORD
                                           'USER-COMMENT)
                                    'NOT-USED)))))))


(DEFUN GET-USER-RESULT (DRAGON RESULT)
   (DECLARE (SPECIAL *QUERY-IO*))
   (LET (USER-ANSWER USER-PROMPT)
        (FORMAT *QUERY-IO* " The return value from ~A was ~A.~%"
               (SLOT-VALUE DRAGON 'DISPLAY-NAME)
               RESULT)
        (FORMAT *QUERY-IO* " Do you want to use this answer?~%")
        (SETF USER-PROMPT (YES-OR-NO-P "  Yes or No? "))
        (IF USER-PROMPT (SETF USER-ANSWER 'NOT-USED)
            (PROGN (FORMAT *QUERY-IO* "Please enter user result. ")
                   (SETF USER-ANSWER (READ *QUERY-IO*))))))


(DEFUN INVOKE (DRAGON VERB &REST MORE-ARGS) "Invoke an agent"
                                           ; Invokes an agent, and
                                           ; creates the associated
                                           ; engram links

   (DECLARE (SPECIAL *CURRENT-DRAGON* *INVOCATION-TRACE-FLAG* 
                   *TRACE-OUTPUT* *QUERY-IO* *CURRENT-CASE* 
                   *INVOCATION-SINGLE-STEP* *DECISION-SUPPORT-MODE* 
                   *MEMORY-ON-FLAG*))
   (if (not (typep dragon 'dragon))
       (error
	"~S is not a DRAGON (it is a ~S), and therefore cannot be invoked."
	dragon (if (eq (type-of dragon) 'pcl::iwmc-class)
		   (class-of dragon)
		   (type-of dragon))))
   (LET ((SELF *CURRENT-DRAGON*)
          USER-ANSWER RESULT (DRAGONS-ENGRAM nil) 
	 (callers-engram nil))
       
       ;; If decision-support-mode is on, consult the dragon, 
       
       ;; else use the dragon's response as the answer
       
       ;; Print out trace information if flag is on

        (IF *INVOCATION-TRACE-FLAG* (FORMAT *TRACE-OUTPUT* "~% The calling dragon is ~A.~% The called dragon is ~A.~% The verb is ~A.~% The &REST args are ~S *CC* is ~S.~%"
                                           (SLOT-VALUE *CURRENT-DRAGON*
                                                  'DISPLAY-NAME)
                                           (SLOT-VALUE DRAGON
                                                  'DISPLAY-NAME)
                                           VERB MORE-ARGS
					   *current-case*))
       
       ;; If *memory-on-flag* is set to true, then call
       ;; create-engram to create an engram and connect it to
       ;; the invoked dragon.
       
       ;; an engram and connect it to the invoked dragon.  
       
       ;; Then connect the engram to *current-dragon* and
       ;; adjust the substructure appropriately. Set local variable
       ;; MY-ENGRAM to engram we just created, to avoid problems referring
       ;; to it via current-engram slot (if we end up getting invoked
       ;; sometime while in the midst of performing our action)...

        (IF *MEMORY-ON-FLAG* 
	    (progn
	      (CREATE-ENGRAM DRAGON VERB MORE-ARGS)
	      (setf dragons-engram (slot-value dragon 'current-engram)
		    callers-engram (slot-value *current-dragon*
					       'current-engram))))
       
       ;; Activates single-step feature if global flag is true

        (IF *INVOCATION-SINGLE-STEP* (PROGN (FORMAT *QUERY-IO* 
                                          "~% Type a <CR> to continue."
                                                   )
                                            (READ-LINE *QUERY-IO*)))
       
       ;; Call controller for the dragon and deal with the
       ;; result

        (SETF *CURRENT-DRAGON* DRAGON)
	(if (assoc verb (slot-value dragon 'actions))
	    (SETF RESULT (APPLY (CDR (ASSOC VERB (SLOT-VALUE DRAGON
							     'ACTIONS)))
				MORE-ARGS))
	    (error "Dragon ~S has no ~S action."
		   (slot-value dragon 'unique-name)
		   verb))
       
       ;; If *Decision-Support-Mode* is on then ask user for
       ;; the final result

        (IF *DECISION-SUPPORT-MODE* (SETF USER-ANSWER (GET-USER-RESULT
                                                       DRAGON RESULT)))
       
       ;; Execute function in Afterwards slot if there is one

        (EVAL (SLOT-VALUE DRAGON 'AFTERWARDS))
        (SETF *CURRENT-DRAGON* SELF)
       
       ;; Stuff results in appropriate slots by calling the
       ;; fill-engram function

        (IF *MEMORY-ON-FLAG* (FILL-ENGRAM dragons-engram
					  callers-engram
					  USER-ANSWER RESULT))
       
       ;; If *Invocation-trace-flag* is on, then print out
       ;; result information to the *Trace-output* window

        (IF *INVOCATION-TRACE-FLAG*
            (PROGN (FORMAT *TRACE-OUTPUT* 
                       " The return value from the call to ~A is ~A.~%"
                          (SLOT-VALUE DRAGON 'DISPLAY-NAME)
                          RESULT)
                   (IF (AND *DECISION-SUPPORT-MODE*
                            (NOT (EQUAL RESULT USER-ANSWER)))
                       (FORMAT *TRACE-OUTPUT* 
                              " The return value from the user ~A~%" 
                              USER-ANSWER))))
        (RETURN-FROM INVOKE RESULT)))

