  
(proclaim '(special *CONSTANT-LIST* *TYPE-LIST* *PREDICATE-LIST* *PRINT-TYPE*
		    *OPERATORS* *INFERENCE-RULES* *OPS* *TYPE-ARRAY*
		    *CONNECTIONS* *GROUP-GRAPH* *TRANSFORMATIONS* *DROPPED*
		    *GROUP-INTERACTION* *VARIABLE-TYPING* *PRIMARY*))

;;;
;;;
;;;=============================================================================
;;;
;;; Determine the primary effects.
;;;
;;;
;;; This function determines which predicates are primary and which ones are 
;;; secondary.  In this version, all deletes are made secondary.
;;; 
;;; Creates a type array with the operators along the x-axis and the predicate
;;; types along the y-axis.  Each of the effects of all the operators are placed
;;; in the corresponding slot in the array and then the function attempts to 
;;; lower triangularize.  If it is successful, the primary effects are along the 
;;; diagonal.
;;;
(defun determine-primary-effects (ops)
  (cond (*PRIMARY*
	 (set-primary-effects ops *PRIMARY*))
	(t 
	 (let* ((effects (find-effects ops))
		(types (delete-duplicates (find-types effects) :test #'equal))
		(type-array (make-array (list (1+ (length ops)) (1+ (length types)))
					:initial-element nil)))
	   (setq *TYPE-ARRAY* type-array)
	   (put-ops-in-array ops type-array 1)
	   (put-types-in-array types type-array 1)
	   (put-lits-in-array effects type-array ops types)
	   (cond ((lower-triangularize type-array)
		  (mark-primary-effects type-array))
		 (t (format t "~%Type array cannot be made lower triangular!")
		    (format t "~%Entering interactive mode...")
		    (format t "~%Please specify 't' if the effect is primary.")
		    (ask-about-primary-effects ops)))))))
;;;
;;;
;;;
;;; Set which effects are primary.
;;;
(defun set-primary-effects (ops primary)
  (cond ((null primary))
	((eq 'all primary)
	 (set-all-effects ops))
	(t (set-primary-effects-of (caar primary)(cdar primary))
	   (set-primary-effects ops (cdr primary)))))
	   

(defun set-primary-effects-of (primary-effect ops)
  (cond ((null ops))
	(t
	 (set-effects-of (get (car ops) 'typed-effects) primary-effect)
	 (set-primary-effects-of primary-effect (cdr ops)))))

(defun set-all-effects (ops)
  (cond ((null ops))
	(t (set-effects-of (get (car ops) 'typed-effects) 'all)
	   (set-all-effects (cdr ops)))))
;;;
;;;
;;;
;;; Set the effects of a particular operator. 
(defun set-effects-of (effects primary-effect)
  (cond ((null effects))
	((eq 'all primary-effect)
	 (setf (formula-status (car effects)) 'primary)
;	 (format t "~%   Primary:   ~a" 
;		 (formula-instance (car effects)))
	 (set-effects-of (cdr effects) primary-effect))
	((equal (formula-type (car effects))
		(extract primary-effect))
	 (setf (formula-status (car effects)) 'primary)
;	 (format t "~%   Primary:   ~a" 
;		 (formula-instance (car effects)))
	 (set-effects-of (cdr effects) primary-effect))
	(t 
	 (if (null (formula-status (car effects)))
	     (setf (formula-status (car effects)) 'secondary))
	 (set-effects-of (cdr effects) primary-effect))))
;;;
;;;
;;;
;;; Ask the user which effects are primary.
;;;
(defun ask-about-primary-effects (ops)
  (cond ((null ops))
	(t (format t "~%Operator: ~a" (car ops))
	   (ask-about-effects-of (get (car ops) 'typed-effects))
	   (ask-about-primary-effects (cdr ops)))))
;;;
;;;
;;;
;;; Ask about the effects of a particular operator. 
(defun ask-about-effects-of (effects)
  (cond ((null effects))
	((eq 'secondary (formula-status (car effects)))
	 (ask-about-effects-of (cdr effects)))
	(t (format t "~%   ~a" (formula-instance (car effects)))
	   (cond ((read)(setf (formula-status (car effects)) 'primary)))
	   (ask-about-effects-of (cdr effects)))))
;;;
;;;
;;;
;;; Given the list of operators, returns a list of all the effects.
;;;
(defun find-effects (ops)
    (cond ((null ops) nil)
	  (t (append (get (car ops) 'typed-effects)
		     (find-effects (cdr ops))))))
;;;
;;;
;;;
;;; Given the list of operators, returns a list of all the preconditions.
;;;
(defun find-preconds (ops)
    (cond ((null ops) nil)
	  (t (append (get (car ops) 'typed-preconds)
		     (find-preconds (cdr ops))))))
;;;
;;;
;;;
;;; Given a list of effects, returns the corresponding list of types.
;;;
(defun find-types (effects)
    (cond ((null effects) nil)
;	  ((not-subgoalable (formula-instance (car effects)))
;	   (find-types (cdr effects)))
	  (t (cons (strip-negation (formula-type (car effects)))
		   (find-types (cdr effects))))))

(defun not-subgoalable (literal)
  (if *PRIMARY*
	(let ((primary-ops (assoc literal *PRIMARY* 
			 :test #'(lambda (x y)(match (list x)(list y)
						     '(((nil))))))))
	  (cond ((null primary-ops) nil)
		(t (null (cdr primary-ops)))))))


;;;
;;;
;;;
;;; Places the list of operators along the top of the x-axis.
;;;
(defun put-ops-in-array (ops type-array index)
  (cond ((null ops) nil)
	(t (setf (aref type-array index 0)(list (car ops)))
	   (put-ops-in-array (cdr ops) type-array (1+ index)))))
;;;
;;;
;;;
;;; Places the list of types along the y-axis.
;;;
(defun put-types-in-array (types type-array index)
  (cond ((null types) nil)
	(t (setf (aref type-array 0 index)(list (car types)))
	   (put-types-in-array (cdr types) type-array (1+ index)))))
;;;
;;;
;;;
;;; Places the formula structures in the array based on their corresponding
;;; operator and type.
;;;
(defun put-lits-in-array (effects array ops types)
    (cond ((null effects) nil)
	  ((null (position (strip-negation (formula-type (car effects)))
			   types 
			   :test #'equal))
	   (put-lits-in-array (cdr effects) array ops types))
	  (t (let ((col (1+ (position (formula-operator (car effects)) ops)))
		   (row (1+ (position (strip-negation 
				       (formula-type (car effects))) 
				      types 
				      :test #'equal))))
	       (setf (aref array col row)
		     (cons (car effects)(aref array col row)))
	       (put-lits-in-array (cdr effects) array ops types)))))
;;;
;;;
;;;
;;; This function verifies that the array to be triangularized is square
;;; (i.e., that each operator has a single corresponding primary effect).
;;;
(defun lower-triangularize (iarray)
  (let ((x-dim (1- (array-dimension iarray 0)))
	(y-dim (1- (array-dimension iarray 1))))
    (cond ((/= x-dim y-dim)
	   (format t "~%Lower triangularize only handles square arrays!")
	   nil)
	  (t (triangularize iarray x-dim)))))
;;;
;;;
;;;
;;; Attempts to triangularize the array.  Finds the row with the most entries
;;; and places that at the bottom of the array.  Then it finds a column with
;;; no entires and places that above the diagonal of that row.  This process 
;;; continues until the entire array is triangular.
;;;
(defun triangularize (iarray size)
  (cond ((zerop size) t)
	(t (let ((next-row (row-with-most-entries iarray size 0 0)))
	     (if (/= size next-row)(swap-row iarray size next-row)))
	   (let ((next-col (column-with-no-entries iarray size size)))
	     (cond ((null next-col) nil)
		   (t (if (/= size next-col)(swap-column iarray size next-col))
		      (triangularize iarray (1- size))))))))
;;;
;;;
;;;
;;; Finds the row with the most entries.
;;;
(defun row-with-most-entries (iarray row most entries)
  (cond ((zerop row) most)
	(t (let ((row-entries (number-of-entries iarray row 1)))
	     (cond ((> row-entries entries)
		    (row-with-most-entries iarray (1- row) row row-entries))
		   (t (row-with-most-entries iarray (1- row) most entries)))))))
;;; 
;;;
;;;
;;; Determines the number of entries in a given row.
;;;
(defun number-of-entries (iarray row col)
  (cond ((= col (array-dimension iarray 1)) 0)
	((null (aref iarray col row))
	 (number-of-entries iarray row (1+ col)))
	(t (1+ (number-of-entries iarray row (1+ col))))))
;;; 
;;;
;;;
;;; Finds a column with no entries above a particular row.
;;;
(defun column-with-no-entries (iarray row col)
  (cond ((zerop col) nil) ;;; Exhausted all columns
	((and (non-secondary-lit (aref iarray col row))
	      (column-clear-above-row iarray (1- row) col))
	 col)
	(t (column-with-no-entries iarray row (1- col)))))
;;;
;;;
;;;
;;; Verifies that there is a non-conditional add that can be placed on the
;;; diagonal
(defun non-secondary-lit (literals)
  (cond ((null literals) nil)
	((not (eq 'secondary (formula-status (car literals)))))
	(t (non-secondary-lit (cdr literals)))))
;;;
;;;
;;;
;;; Determines whether a particular column is clear above the given row.
;;;
(defun column-clear-above-row (iarray row col)
  (cond ((zerop row) t)
	((aref iarray col row) nil)
	(t (column-clear-above-row iarray (1- row) col))))
;;;
;;;
;;;
;;; Given a triangularized array, marks the elements on the diagonal as primary.
;;;
(defun mark-primary-effects (iarray)
  (if (/= (array-dimension iarray 0)(array-dimension iarray 1))
      (error "Assuming square matrix..."))
  (do ((diagonal 1 (1+ diagonal)))
      ((= (array-dimension iarray 0) diagonal))
    (set-status (aref iarray diagonal diagonal) 'primary)))
;;;
;;;
;;;
;;; Sets the status of a particular formula to the given status.  The status
;;; must be either 'secondary or 'primary.  If the effect is conditional, the
;;; status will already be set to secondary.
;;;
(defun set-status (formula status)
  (cond ((not (or (eq status 'secondary)(eq status 'primary)))
	 (error "~%Invalid status: ~a" status))
	((null formula) nil)
	((eq 'secondary (formula-status (car formula)))
	 (set-status (cdr formula) status))
	(t (setf (formula-status (car formula)) status)
	   (set-status (cdr formula) status))))
  
;;;
;;;==============================================================================
;;;
;;;
;;;

(defun mark-all-static-predicates (ops types)
    (setq *STATIC* (remove-duplicates 
		    (mark-static-predicates ops types) :test #'equal)))

(defun mark-static-predicates (ops types)
  (cond ((null ops) nil)
	(t (append (mark-static-preconditions (get (car ops) 'typed-preconds) types)
		   (mark-static-predicates (cdr ops) types)))))


(defun mark-static-preconditions (preconds types)
  (cond ((null preconds) nil)
	((not (member (strip-negation (formula-type (car preconds))) 
		      types :test #'equal))
	 (setf (formula-status (car preconds)) 'static)
	 (cons (strip-negation (formula-type (car preconds)))
	       (mark-static-preconditions (cdr preconds) types)))
	(t (mark-static-preconditions (cdr preconds) types))))
;;;
;;;=============================================================================
;;;
