/* Intercept: intercept input and output from a server, and print it nicely.
 * 
 */
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include "socket_server.h"
#include "open_socket.h"

/* record of client connections to me: */
struct x {
  int c_fd;   /* client's fd */
  int s_fd;   /* server fd */
  int c_pos;  /* 0-15; position for me to write nice output to next */
  int s_pos;  /* 0-15; position for me to write nice output to next */
  };
  
struct x lookup[64];
struct x *lookup_s[64];

static char localhost[] = "localhost";
char *server_machine = localhost;
char *server_port = NULL;


void intercept_dead_server()
{
  socket_server_shutdown();
  exit(0);
  }


void intercept_close_server_connection(sd)
     int sd;
{
  int sd2;

  socket_server_end_of_file(sd);
  sd2 = lookup_s[sd]->c_fd;
  close(sd);
  lookup_s[sd] = NULL;
  if (sd2 != -1)
    socket_server_end_of_file(sd2);
  }


void intercept_close_file(sd)
     int sd;
{
  int sd2;

  if (sd == lookup[sd].c_fd) {
    sd2 = lookup[sd].s_fd;
    printf(" % 2d->S  connection closed to %d\n", sd, sd2);
    lookup[sd].s_fd = -1;
    lookup[sd].c_fd = -1;
    lookup[sd].c_pos = 0;
    lookup[sd].s_pos = 0;
    if ((sd2 != -1) && (lookup_s[sd2] != NULL))
      intercept_close_server_connection(sd2);
    }
  }    


void intercept_new_client(sd)
     int sd;
{
  int ssd;

  if (server_port == NULL)
    ssd = -1;
  else {
    ssd = open_socket(server_machine, server_port);
    if (ssd == -1)
      {
      fprintf(stderr, "intercept: unable to connect to %s at \"%s\".\n",
	      server_port, server_machine);
      exit(-1);
      }
    fcntl(ssd, F_SETFL, O_NONBLOCK);
    socket_server_add_input_fd(ssd);
    lookup_s[ssd] = &(lookup[sd]);
    }
  printf(" % 2d->S  connection established\n", sd);
  fcntl(sd, F_SETFL, O_NONBLOCK);
  lookup[sd].c_fd = sd;
  lookup[sd].s_fd = ssd;
  lookup[sd].c_pos = 0;
  lookup[sd].s_pos = 0;
  }


void intercept_read_client_input(sd)
     int sd;
{
  int n, x, i;  
  unsigned char buf[16];

  x = lookup[sd].c_pos;
  while (1) {
    /* Get as much of the new input as I can fit onto a line. */
    n = read(sd, buf + x, 16-x);
    if (n <= 0)
      break;

    /* Print out the transaction. */
    printf(" % 2d->S   ", sd);
    /* Left hand characters: */
    for (i = 0; i < x; i++)
      fputc('_', stdout);
    for (; i < x+n; i++)
      fputc(((buf[i] >= 0x20) && (buf[i] < 0x80)) ? buf[i] : '.', stdout);
    for (; i < 16; i++)
      fputc('_', stdout);
    fputc(' ', stdout);
    /* Right hand numbers: */
    for (i = 0; i < x; i++)
      printf(" --");
    for (; i < x+n; i++)
      printf(" %02X", buf[i]);
    for (; i<16; i++)
      printf(" --");
    fputc('\n', stdout);

    /* Pass this input along to the server: */
    if (lookup[sd].s_fd != -1)
      for (i=0; i<n; i++)
	i += write(lookup[sd].s_fd, buf + x + i, n - i);
    
    /* Tweak my state to reflect the new reality. */
    x = lookup[sd].c_pos = ((x + n) % 16);
    }
  
  if ((n == -1) && (errno == EWOULDBLOCK))
    return;

  if (n == 0) {
    socket_server_end_of_file(sd);
    return;
    }
  }




void intercept_read_server_input(ssd)
     int ssd;
{
  int n, x, i;  
  unsigned char buf[16];

  x = lookup_s[ssd]->s_pos;
  while (1) {
    /* Get as much of the new input as I can fit onto a line. */
    n = read(ssd, buf + x, 16-x);
    if (n <= 0)
      break;

    /* Print out the transaction. */
    printf("+ S->%-2d  ", lookup_s[ssd]->c_fd);
    /* Left hand characters: */
    for (i = 0; i < x; i++)
      fputc('_', stdout);
    for (; i < x+n; i++)
      fputc(((buf[i] >= 0x20) && (buf[i] < 0x80)) ? buf[i] : '.', stdout);
    for (; i < 16; i++)
      fputc('_', stdout);
    fputc(' ', stdout);
    /* Right hand numbers: */
    for (i = 0; i < x; i++)
      printf(" --");
    for (; i < x+n; i++)
      printf(" %02X", buf[i]);
    for (; i<16; i++)
      printf(" --");
    fputc('\n', stdout);

    /* Pass this input along to the client: */
    write(lookup_s[ssd]->c_fd, buf + x, n);
    
    /* Tweak my state to reflect the new reality. */
    x = lookup_s[ssd]->s_pos = (x + n) % 16;
    }
  
  if ((n == -1) && (errno == EWOULDBLOCK))
    return;

  if (n == 0) {
    intercept_close_server_connection(ssd);
    return;
    }
  }





int main(argc, argv)
     int argc;
     char *argv[];
{
  char *intercept_port = NULL;
  int x;

  /* close all files except standards so that we can use them: */
  for (x = 3; x < 64; x++)
    close(x);

  x = 1;
  while (x < argc)
    {
    if (argv[x][0] == '-')
      switch (argv[x][1]) {
      case 'h': /* server host */
	server_machine = argv[x+1];
	break;
      case 'p': /* server port */
	server_port = argv[x+1];
	break;
      case 'P': /* my Port */
	intercept_port = argv[x+1];
	break;
	}
    x++;
    }

  if ((intercept_port == NULL) && (server_machine != localhost))
    intercept_port = server_port;

  if (intercept_port == NULL)
    {
    fprintf(stderr, "\n\
Usage: intercept [-p port] [-h host] [-P port2] [-s L]\n\
         -p and -h specify the port and host where the real server is;\n\
            specify neither one if you do not want to connect to a server.\n\
         -P will be the interception port number.\n\
         -s allows you to limit the number of clients allowed at once to L.\n\
\n");
    exit(0);
    }

  /* Catch control-c for smooth termination. */
  signal(SIGINT,  intercept_dead_server);
  signal(SIGTERM, intercept_dead_server);
  signal(SIGQUIT, intercept_dead_server);

  /* Call  socket_server.c:  */
  socket_server_initialize(intercept_read_server_input, NULL);

  if (socket_server_startup(intercept_port, intercept_read_client_input, 0,
                            intercept_new_client, intercept_close_file))
    return;
  socket_server_poll(NULL);
  }

